/*!
  * \file spm_SystemStateManagerBase.cpp
  *  \brief
  *    Creation of state machine. It also handles the internal state transitions.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *   27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SystemStateManagerBase.h"

// interfaces class definitions
#include "spm_ISuperVisionManager.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_IOsalProxy.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IOsLinux.h"
#include "spm_IWakeupHandler.h"
#include "spm_LxProcInfo.h"
#include "spm_ISystemStateStatistics.h"

#include "spm_IFactory.h"

// spm helper
#include "timeConvert.h"

// needed for static trigger functions
#include "spm_SubStateHandlerConfig.h"
#include "spm_SystemStateStatistics.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
#include "trcGenProj/Header/spm_SystemStateManagerBase.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_SYSTEMSTATEMANAGER

#define FC_SPM_SYSSTATE_HISTORY_COUNT   20

#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

/******************************************************************************
  | declarations
  |-----------------------------------------------------------------------*/

spm_tclSystemStateManagerBase*spm_tclSystemStateManagerBase::_poMyBaseRef = 0;


/******************************************************************************
  | implementation
  |-----------------------------------------------------------------------*/
spm_tclSystemStateManagerBase::spm_tclSystemStateManagerBase( const ISpmFactory& factory )

/*!
  * \fn
  *  \brief
  *    constructor.
  *    - initialize member variable
  *    - semaphore/timer create
  *    - check to restore old system state (in case of reset)
  *
  *  \param[in] factory : spm factory object.
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   : ISpmSystemStateManager( factory )
#ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   , _hDeboounceTimer( OSAL_C_INVALID_HANDLE )
#endif
   , _hStateChangeAccess( OSAL_C_INVALID_HANDLE )
   , _u32SystemState( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND )
   , _u32IntermediateSystemState(spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND)
   , _u32NewSystemState( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND )
   , _u32InitialSystemState( spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND )
   , _bPointOfNoReturnReached( FALSE )
   , _hStateTimeoutTimer( OSAL_C_INVALID_HANDLE )
   , _u32CurrentInternSubState( 0 )
   , _u32StateChangeCounter( 0 )
   , _poclGlobalApplicationManager( NULL )
   , _poclSupervisionManager( NULL )
   , _poclCcaServiceHandler( NULL )
   , _poclSubStateHandler( NULL )
   , _poclSystemPowerManager( NULL )
   , _poSpmOsLinux( NULL )
   , _poclOsalProxy( NULL )
   , _poclWorkerServer( NULL )
   , _poclWakeupHandler( NULL ){
   _poMyBaseRef = this;

   _oHistoryList.clear( );

   /* create semaphore to protect list access */
   if ( OSAL_s32SemaphoreCreate( SPM_STATE_CHANGE_SEM_NAME, &_hStateChangeAccess, (tU32)1 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclSystemStateManagerBase !!!!!! Error detected !!!!!! cannot create Semaphore _hStateChangeAccess: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   if ( OSAL_s32TimerCreate( vStateTimerCallback, this, &_hStateTimeoutTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclSystemStateManagerBase !!!!!! Error detected !!!!!! cannot create Timer _hStateTimeoutTimer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
      if ( OSAL_s32TimerCreate( vDebounceStateChangeTimerCallback, this, &_hDeboounceTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclSystemStateManagerBase !!!!!! Error detected !!!!!! cannot create  Timer _hDeboounceTimer: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      } else if ( OSAL_s32TimerSetTime( _hDeboounceTimer, SPM_U32_DEBOUNCE_STATE_CHANGE_TIME, SPM_U32_DEBOUNCE_STATE_CHANGE_TIME ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclSystemStateManagerBase !!!!!! Error detected !!!!!! cannot start Timer _hDeboounceTimer: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif // ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   dp_tclSpmDpInternDataSystemState oNewSystemState;
   _u32OrgResetSystemState = oNewSystemState.tGetData( );
   ETG_TRACE_USR1( ( "spm_tclSystemStateManagerBase(): INITSTATE in CONSTRUCTOR system state %u after reset.", ETG_ENUM( SPM_SYSTEM_STATES, _u32OrgResetSystemState ) ) );
}

spm_tclSystemStateManagerBase::~spm_tclSystemStateManagerBase( ){
/*!
  * \fn
  *  \brief
  *    destructor.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   _poclWorkerServer->vRemoveClient( this );

   if ( OSAL_s32SemaphoreClose( _hStateChangeAccess ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }

   if ( OSAL_s32SemaphoreDelete( SPM_STATE_CHANGE_SEM_NAME ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }

   if ( OSAL_s32TimerDelete( _hStateTimeoutTimer ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }

   #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
      if ( OSAL_s32TimerDelete( _hDeboounceTimer ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   #endif

   _poclCcaServiceHandler        = NULL;
   _poclGlobalApplicationManager = NULL;
   _poclSubStateHandler          = NULL;
   _poclSystemPowerManager       = NULL;
   _poclSupervisionManager       = NULL;
   _poclOsalProxy                = NULL;
   _poclWorkerServer             = NULL;

   _poMyBaseRef                  = NULL;
   _poSpmOsLinux                 = NULL;
   _poclWakeupHandler            = NULL;
}

tVoid spm_tclSystemStateManagerBase::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    SPM factory invokes this method to initialize the dependency of the spm_tclSystemStateManagerBase.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
// get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,          ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager,       ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWakeupHandler,            ISpmWakeupHandler );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,             ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager,       ISpmSupervisionManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poSpmOsLinux,                 ISpmOsLinux );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,                ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,          ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,        ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclGlobalApplicationManager, ISpmGlobalApplicationManager );

   _poclSubStateHandler->vRemoveAllTrigger( &_tSubStateTrigger );
   _poclSubStateHandler->vRemoveAllTrigger( &_tCurSubStateTrigger );

   _poclWorkerServer->vAddClient( this );
} // vGetReferences

tVoid spm_tclSystemStateManagerBase::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  *   Before starting all other SPM components we have to set the current system state
  *   evaluate first system state (check for POR / reset / Warmstart / registry)
  *
  *  \param
  *  \note
  *  \version
  ******
  */
// ***************************************************************************
// Before starting all other SPM components we have to set the current system state
// evaluate first system state (check for POR / reset / Warmstart / registry
// ***************************************************************************
   dp_tclSpmDpInternDataSystemState oNewSystemState;

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poclWakeupHandler );
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   // initialize FSM and connect FSM with Behaviour
   vInitStateMachine( );

   _u32SystemState    = spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND; // SPM_SYSTEM_INITIALIZED;
   _u32NewSystemState = spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_BACKGROUND;

   // check if inititial system state is "ON" after POR
   tU8  u8StartMode     = _poclWakeupHandler->u8GetStartMode( );
   tU32 u32WakeUpReason = _poclWakeupHandler->u32GetWakeupReason( );

   ETG_TRACE_USR1( ( "spm_tclSystemStateManagerBase::vStartCommunication(): StartType '%u' and WakeUpReason '%u'.",
                     ETG_ENUM( SPM_STARTTYPE,     u8StartMode ),
                     ETG_ENUM( SPM_WAKEUP_REASON, u32WakeUpReason ) ) );
#  ifdef SPM_ENABLE_OFF_AFTER_RESET_IN_SHUTDOWN
   if (u8StartMode == SPM_U8_STARTTYPE_RESTART_AP){
      if ( ( _u32OrgResetSystemState == SPM_SYSTEM_SHUTDOWN )
           || ( _u32OrgResetSystemState == SPM_SYSTEM_PREPARE_SHUTDOWN )
           ){

               dp_tclSpmDpInternDataResetsInShutdown oResetsInShutdown;
               if (oResetsInShutdown.tGetData() == 0) {
                   oResetsInShutdown.vSetData(1);
               } else {

                   if (!_poclOsalProxy->bPrepareForShutdown() ){
                       ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
                   }
                   if (!_poclOsalProxy->bShutdownSystem(SPM_C_S32_SYSTEMINIT_SWITCH_OFF) ){
                       ETG_TRACE_USR4( ( "ERROR using DEV_SPM  SPM_C_S32_SYSTEMINIT_SWITCH_OFF " ) );
                   }
               }

      } else {
         // restore state active before reset
         _u32NewSystemState = oNewSystemState.tGetData();
         vRestoreLastState(_u32NewSystemState);
         ETG_TRACE_USR1( ( "spm_tclSystemStateManagerBase(): Restore system state %u after reset.", ETG_ENUM(SPM_SYSTEM_STATES, _u32NewSystemState) ) );
      }

   } else {
#  endif

   dp_tclSpmDpConfigInitialSystemState oStartupState;
   if ( DP_U8_ELEM_STATUS_VALID == oStartupState.u8DataValid( ) ){

      _u32NewSystemState = oStartupState.tGetData( );
      ETG_TRACE_USR1( ( "spm_tclSystemStateManagerBase::vStartCommunication(): Found initial state in configuration: Start with system state %u.", ETG_ENUM( SPM_SYSTEM_STATES, _u32NewSystemState ) ) );
   } else {
      _u32NewSystemState = _u32InitialSystemState;
      ETG_TRACE_ERR( ( "spm_tclSystemStateManagerBase::vStartCommunication(): Initial state in configuration is not valid: Start with system state %u.", ETG_ENUM( SPM_SYSTEM_STATES, _u32NewSystemState ) ) );
   }
   vRestoreLastState( _u32NewSystemState );
#  ifdef SPM_ENABLE_OFF_AFTER_RESET_IN_SHUTDOWN
   }
#  endif

   {
      OSAL_trTimeDate               rCurrentTime = { 0,0,0,0,0,0,0,0,0 };
      _poclOsalProxy->bGetUtcTime( &rCurrentTime );
      rCurrentTime.s32Year    += 1900;

      tclTimeConvert                oTimeConv;

      TSpmSystemStateStatisticsData oStateElem;
      oStateElem.u32Time       = oTimeConv.u32GetSeconds( &rCurrentTime );
      oStateElem.u32TimeMs     = OSAL_ClockGetElapsedTime( );
      oStateElem.u8SystemState = (tU8)_u32NewSystemState;
      oStateElem.u8EntryType   = SPM_STATE_HISTORY_ENTRY_WAKEUP;
      oStateElem.u.u8Wakeup    = (tU8)u32WakeUpReason;

      SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSystemStateStatistics, spm_tclSystemStateStatistics, ISpmSystemStateStatistics );
      poclSystemStateStatistics->vUpdateSystemStateHistory( oStateElem );
   }

   oNewSystemState.vSetData( _u32NewSystemState );

   ETG_TRACE_USR1( ( "spm_tclSystemStateManagerBase::vStartCommunication(): Initial system state %u.", ETG_ENUM( SPM_SYSTEM_STATES, _u32NewSystemState ) ) );

   // start system state interval timer
   vStartStateTimeoutTimer( u32GetStateTime( _u32NewSystemState ) ); // start timeout for current state

   #ifndef LCM_UNIT_TESTS
      spm_corefi_tclMsgSystemStateStatus systemstate;

      SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

      systemstate.SystemState.enType = (spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType)_u32SystemState;
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_SYSTEMSTATE, &systemstate );
   #endif
} // vStartCommunication

// state handling
tVoid spm_tclSystemStateManagerBase::u32PostCalcNewSystemState( tU32 u32NewSystemState,
                                                                tU32 u32OldSystemState,
                                                                tU32 u32TransitionId ){
/*!
  * \fn
  *  \brief
  *    do some history handling and othr stuff common for all state changes in all state machines
  *
  *  \param[in] u32OldSystemState : previous FSM state.
  *  \param[in] u32NewSystemState : current FSM state.
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   if ( ( u32NewSystemState != SPM_SYSTEM_ON )
     && ( u32OldSystemState == SPM_SYSTEM_ON )
      ){
      // connect to battry detected --> remember persitent
      dp_tclSpmDpInternDataConnectToBattery oCon2Battery;
      oCon2Battery.vSetData(FALSE);
      // TRACE_SPM_INFO_STRING("Now reset battery connection state");
   }

   if ( ( u32NewSystemState != u32OldSystemState )
        && ( u32TransitionId != SPM_U32_TRANSITION_ID_INVALID )
        ){
      _u32StateChangeCounter++;
   }
   {
      // add entry to history dump
      TSysStateHistoryEntry tHistoryEntry = {
         OSAL_ClockGetElapsedTime( ),
         0,
         _tCurSubStateTrigger,
         u32TransitionId
      };
      _oHistoryList.push_back( tHistoryEntry );
      if ( _oHistoryList.size( ) > FC_SPM_SYSSTATE_HISTORY_COUNT ){
         _oHistoryList.pop_front( );
      }
   }
   {
      OSAL_trTimeDate rCurrentTime = { 0,0,0,0,0,0,0,0,0 };
      if ( _poclOsalProxy != NULL ){
         (tVoid)_poclOsalProxy->bGetUtcTime( &rCurrentTime );
      }
      rCurrentTime.s32Year    += 1900;

      tclTimeConvert                oTimeConv;

      TSpmSystemStateStatisticsData oStateElem;
      oStateElem.u32Time       = oTimeConv.u32GetSeconds( &rCurrentTime );
      oStateElem.u32TimeMs     = OSAL_ClockGetElapsedTime( );
      oStateElem.u8SystemState = (tU8)u32NewSystemState;
      oStateElem.u8EntryType   = SPM_STATE_HISTORY_ENTRY_SUBSTATE;
      oStateElem.u.tTrigger    = _tCurSubStateTrigger;

      SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSystemStateStatistics, spm_tclSystemStateStatistics, ISpmSystemStateStatistics );
      poclSystemStateStatistics->vUpdateSystemStateHistory( oStateElem );
   }
} // u32PostCalcNewSystemState

tVoid spm_tclSystemStateManagerBase::vNewSubState( TTriggerMsg *tNewTrigger ){
/*!
  * \fn
  *  \brief
  *    New substate (OnOffReasons) detected. Called by substate handler.
  *
  *  \param[in] tNewTrigger:
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _poclSubStateHandler->vCopyTrigger( &_tSubStateTrigger, tNewTrigger );
   // _poclSubStateHandler->vTraceTrigger(SPM_SSM_NEWTRIGGER, tNewTrigger);

   if ( _bPointOfNoReturnReached ){
      // here filter for all substates required for shutdown e.g. CoproOff / StateTo / ...
      // no transition back to state != OFF possible
      _poclSubStateHandler->vRemoveAllTrigger( &_tSubStateTrigger );

      if ( _poclSubStateHandler->bIsTriggerSet( SPM_U32_SYNC_SHUTDOWN_ACK ) ){
         bActStateReached( SPM_SYSTEM_SHUTDOWN );
      }
   }
   vCheckNewStateTransition( );

} // vNewSubState

tVoid spm_tclSystemStateManagerBase::vNewSystemStateReached( ){
/*!
  * \fn
  *  \brief
  *    Called by Local Application Manager when new system state is reached -> all applications has acknowledged transition.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */

   _u32SystemState = _u32NewSystemState;
   dp_tclSpmDpInternDataCurrentSystemState oCurSysState;
   oCurSysState.vSetData( _u32SystemState );

   SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

   ETG_TRACE_FATAL( ( "vNewSystemStateReached, new system state %u reached at time: %dms",
                     ETG_ENUM(SPM_SYSTEM_STATES, _u32SystemState),
                     OSAL_ClockGetElapsedTime()
                     ) );

   if ( _oHistoryList.size( ) != 0 ){
      TSysStateHistoryEntry tLastElement = _oHistoryList.back( );
      tLastElement.u32TimeStampReached = OSAL_ClockGetElapsedTime( );
      _oHistoryList.pop_back( );
      _oHistoryList.push_back( tLastElement );
   }

   bActStateReached( _u32SystemState );

   spm_corefi_tclMsgSystemStateStatus systemstate;
   systemstate.SystemState.enType = (spm_fi_tcl_SPM_e32_SYSTEM_STATES::tenType)_u32SystemState;
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_SYSTEMSTATE, &systemstate );

   if ( SPM_SYSTEM_SHUTDOWN != _u32SystemState ){
      if ( FALSE == _poclGlobalApplicationManager->bProcessSystemState( _u32SystemState, _u32SystemState ) ){
         ETG_TRACE_ERRMEM( ( "spm_tclSystemStateManagerBase::vNewSystemStateReached: GlobalApplicationManager call system process failed in system state %u",
                            ETG_ENUM(SPM_SYSTEM_STATES, _u32SystemState) ) );
      }
   }
} // vNewSystemStateReached

tU32 spm_tclSystemStateManagerBase::u32GetRemainingStateTimeoutTime( ){
/*!
  * \fn
  *  \brief
  *    Get remaining time for current system state.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tU32 u32RemainingTime = 0;
   tU32 u32IntervalTime  = 0;

   if ( OSAL_s32TimerGetTime( _hStateTimeoutTimer, &u32RemainingTime, &u32IntervalTime ) != OSAL_OK ){
      // don't need to error trace -> on ADIT platform even if timer isn't running return value is != OK
      // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
   }

   return( u32RemainingTime );
}

tVoid spm_tclSystemStateManagerBase::vStateTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *
  *  \param
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   (tVoid)pArg;

   if ( NULL != _poMyBaseRef ){
      if ( NULL != _poMyBaseRef->_poclWorkerServer ){
         _poMyBaseRef->_poclWorkerServer->bPostMessage( "ISpmSystemStateManager", SPM_U32_WORKER_SSM_STATE_TIMEOUT );
      }
   }
}

tVoid spm_tclSystemStateManagerBase::vSystemStateTimeOut( ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *
  *  \param
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   ETG_TRACE_USR4( ( "vStateTimerCallback, state timer expired" ) );

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   if ( _u32SystemState == SPM_PRJ_SYSTEM_STATE_DOWNLOAD ){
      // force coldstart download failed
      #ifdef SPM_U32_INTERNAL_SUBSTATE_FORCE_DOWNLOAD_OFF
         _poclSubStateHandler->vSetInternalSubState( SPM_U32_INTERNAL_SUBSTATE_FORCE_DOWNLOAD_OFF, SPM_U32_INTERNAL_SUBSTATE_FORCE_DOWNLOAD_OFF );
      #else
         _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_ERROR );
      #endif

   }
   // trigger substate to change to thread SpmSubSt context

   _poclSubStateHandler->vSetSubStateType( SPM_U32_STATE_TO, TRUE );
} // vSystemStateTimeOut

#ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   tVoid spm_tclSystemStateManagerBase::vDebounceStateChangeTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *
  *  \param
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
      (tVoid)pArg;

      if ( NULL != _poMyBaseRef ){
         if ( NULL != _poMyBaseRef->_poclSystemPowerManager ){
            _poMyBaseRef->_poclWorkerServer->bPostMessage( "ISpmSystemStateManager", SPM_U32_WORKER_SSM_STATE_DEBOUNCE );
         }
      }
   }

   tVoid spm_tclSystemStateManagerBase::vDebounceStateChangeTo( ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *
  *  \param
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

      if ( _u32StateChangeCounter > SPM_U32_DEBOUNCE_STATE_CHANGE_COUNTER ){
         if ( _u32NewSystemState != SPM_SYSTEM_ON ){
            _poclSubStateHandler->vSetSubStateType( SPM_U32_STATE_DEBOUNCE, TRUE );
            _u32StateChangeCounter = 0;
         }
      } else {
         _u32StateChangeCounter = 0;
      }
      return;
   }

#endif // ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE


tVoid spm_tclSystemStateManagerBase::vStartStateTimeoutTimer( tU32 u32Time ) const

/*!
  * \fn
  *  \brief
  *    Start or stop state timer.
  *
  *  \param[in] u32Time : interval time in ms
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
{
   ETG_TRACE_USR3( ( "spm_tclSystemStateManagerBase::vStartStateTimeoutTimer(): SystemStateTimeout %u.", u32Time ) );

   if ( u32Time > 0 ){
      // set StateTimeout-Timer active
      if ( OSAL_s32TimerSetTime( _poMyBaseRef->_hStateTimeoutTimer, u32Time, 0 ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   } else {
      // reset StateTimeout-Timer
      if ( OSAL_s32TimerSetTime( _poMyBaseRef->_hStateTimeoutTimer, 0, 0 ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }

} // vStartStateTimeoutTimer

tVoid spm_tclSystemStateManagerBase::vHandleMessage( tU32 u32Message,
                                                     tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    Handles the Message from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm:
  *  \note
  *  \version
  ******
  */
   (tVoid)u32Parm;
   switch ( u32Message ){
      case SPM_U32_WORKER_SSM_STATE_TIMEOUT:
         vSystemStateTimeOut( );
         break;

      case SPM_U32_WORKER_SSM_STATE_DEBOUNCE:
         #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
            vDebounceStateChangeTo( );
         #endif
         break;

      case SPM_U32_WORKER_SSM_ACTIVATE_AUTOMATIC:
         if ( _poclSubStateHandler ){
            _poclSubStateHandler->vSetSubStateType( SPM_U32_AUTOMATIC, TRUE );
         }
         break;

      case SPM_U32_WORKER_SSM_DEACTIVATE_AUTOMATIC:
         if ( _poclSubStateHandler ){
            _poclSubStateHandler->vSetSubStateType( SPM_U32_AUTOMATIC, FALSE );
         }
         break;

      default:
         // nothing to do
         break;
   } // switch
}    // vHandleMessage
/*!
  * \fn
  *  \brief
  *    Update startup configuration depending of active COREs.
  *
  *  \note
  *  \version
  ******
  */ 
tVoid spm_tclSystemStateManagerBase::vCheckStartOrderConfiguration( ){

   // update startup configuration depending of active COREs
   tU32 u32NumberOfCores = 1;

   {
      spm_tclLxCpuInfo oCpuInfo;
      u32NumberOfCores = oCpuInfo.u32GetNumberOfCpus( );
   }

   tU32 u32Offset = 0; //for single core

   if ( u32NumberOfCores == 2 ){
      u32Offset = SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
   } else if ( u32NumberOfCores == 4 ){
      u32Offset = 2 * SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
   }

   dp_tclSpmDpInternDataStartupConfig oCfgConfig;
   tU32                               u32CfgNb                                        = oCfgConfig.tGetData( ) % SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
   u32CfgNb += u32Offset;
   u32CfgNb  = u32CfgNb % SPM_STARTUP_CONFIGURATION_MAX_NB;

   tChar                              cCfgRegs[SPM_STARTUP_CONFIGURATION_MAX_NB][200] = SPM_STARTUP_CONFIGURATION;
   ETG_TRACE_USR1( ( "spm_tclSystemStateManagerNiStd::vStateReachedPrepareshutdown(): Number of COREs: %d, Config: %d --> '%s'!", u32NumberOfCores, u32CfgNb, cCfgRegs[u32CfgNb] ) );

   //check if file is in FS
   OSAL_tIODescriptor                 fd                                              = OSAL_IOOpen( cCfgRegs[u32CfgNb], OSAL_EN_READONLY );
   if ( fd == OSAL_ERROR ){
      ETG_TRACE_USR1( ( "vReadStartupConfiguration(): File not found: '%s'. Set config w/o core dependencies.", cCfgRegs[u32CfgNb] ) );
      u32CfgNb = u32CfgNb % SPM_STARTUP_CONFIGURATION_MAX_NB_LAST_MODE;
      ETG_TRACE_USR1( ( "vReadStartupConfiguration(): set startup file: '%s'", cCfgRegs[u32CfgNb] ) );
   } else {
      OSAL_s32IOClose( fd );
   }

   oCfgConfig << u32CfgNb;
} // vCheckStartOrderConfiguration
