/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SystemStateManager.h
  * @brief        This is header file contains System State Manager functions.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include <list>
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#include "spm_SystemStateManagerBase.h"

#define DONT_CARE 0xffffffff

typedef tVoid tSPM_STATE_ENTRY_FUNC ( tVoid );
typedef tVoid tSPM_STATE_REACHED_FUNC ( tVoid );
typedef tVoid tSPM_STATE_EXIT_FUNC ( tVoid );

#define SPM_INTERN_SUBSTATE_STATE_TIMEOUT      (tU32)0x00000040     // intern
#define SPM_INTERN_SUBSTATE_OFF_VIA_IGNITION   (tU32)0x00000100     // intern navi in system
#define SPM_INTERN_SUBSTATE_OFF_VIA_TRANSPORT  (tU32)0x00000200     // intern navi in system
#define SPM_INTERN_SUBSTATE_DIAG_FROM_ON       (tU32)0x00000400
#define SPM_INTERN_SUBSTATE_WAIT_PHONE         (tU32)0x00000800
#define SPM_INTERN_SUBSTATE_FAST_SHUTDOWN      (tU32)0x00001000

#define SPM_U32_WAIT_FOR_OFF                   (tU32)0x80000000
#define SPM_U32_MSG_INVALID                    (tU32)0xFFFFFFFF

struct T_SYSTEM_STATE_ENTRY
{
   tU32 u32SystemState;
   tU32 u32Timeout;
   tSPM_STATE_ENTRY_FUNC *pfnvStateEntry;
   tSPM_STATE_REACHED_FUNC *pfnvStateReached;
   tSPM_STATE_EXIT_FUNC *pfnvStateExit;
};

/*!
  *  \class spm_tclSystemStateManager
  *  \brief This class creates state machine. It also handles the internal state transitions.
  *
  *         Any Substate change always gets notified first to SubStateHandler and SubStateHandler
  *         then decides, whether or not to inform SystemStateManager of the new SubState Trigger.
  *         Based on the value of certain SubStates, a new SystemState is calculated.
  *         This class is extended to meet the project specific needs.
  *
  *         All the registered connected applications are intimated of state change when the system
  *         state machine is updated to the next system state, through GlobalApplicationManager.
  *         The process of changing the software block states to the updated state is carried out by
  *         GlobalApplicationManager in conjunction with LocalApplicationManager and ApplicationDatabase
  *         and finally different processes are notified.
  */
class spm_tclSystemStateManager : public spm_tclSystemStateManagerBase
{

protected:
typedef struct
{
   tU32 u32TransitionId;
   tU32 u32CurrentFsmState;
   tU32 u32UpdateTrigger[SPM_TRIGGER_ARRAY_SIZE];
   tU32 u32Trigger[SPM_TRIGGER_ARRAY_SIZE];
   tU32 u32TriggerNotSet[SPM_TRIGGER_ARRAY_SIZE];
   tU32 u32NextFsmState;
   tU32 u32MsgToPost;
}TStateTransitionElement;
// static TStateTransitionElement _aSysStateTransitionTable[];
T_SYSTEM_STATE_ENTRY    *_stateEntries;
TStateTransitionElement *_transitionEntries;
tU32                     _u32NbTransionEntries;

tBool  bCheckForMatch( tU32       *pu32Data,
                       const tU32 *pu32TabEntry,
                       tBool       bCheckForNotSet = FALSE ) const;

public:
spm_tclSystemStateManager( const ISpmFactory& factory );
virtual ~spm_tclSystemStateManager( );

// trace handling
// virtual tVoid vHandleTraceMessage(const tUChar *puchData);
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Param = 0 );

virtual tVoid vSetStateTime( tU32 u32State,
                             tU32 u32Time );

virtual tU32 u32GetStateTime( tU32 u32State ) const;

protected:
virtual tBool bActStateReached( tU32 u32SystemState ) const;

virtual tU32  u32CalcNewSystemState( tU32 u32CurrentSystemState );
virtual tVoid vCheckNewStateTransition();
virtual tVoid vInitStateMachine( );

virtual tVoid vRestoreLastState( tU32 u32NewSystemState );

// default: empty statemachine, no transitions, no states
virtual tU32 u32GetNumberOfTransitionTableEntries( ) const { return( 0 ); }
virtual tU32 u32GetNumberOfStateEntries( ) const { return( 0 ); }

tVoid vSetStateEntries( T_SYSTEM_STATE_ENTRY *e ){ _stateEntries = e; }
tVoid vSetTransitionEntries( TStateTransitionElement *e,
                             tU32                     nb ){
   _transitionEntries    = e;
   _u32NbTransionEntries = nb;
}

virtual tU32   u32ActStateEntry( tU32 u32NewSystemState ) const;

virtual tVoid  vActStateExit( tU32 u32NewSystemState ) const;

protected:
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

virtual const tChar*getName( ) const { return( "spm_tclSystemStateManager" ); }
};

