/*!
  * \file spm_SystemStateManager.cpp
  *  \brief
  *    Creation of state machine. It also handles the internal state transitions.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *   27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SystemStateManager.h"

// spm class definitions
#include "spm_SystemStateStatistics.h"

// interfaces class definitions
#include "spm_ISuperVisionManager.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_IOsalProxy.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IOsLinux.h"
#include "spm_IWakeupHandler.h"

// spm helper
#include "timeConvert.h"

// needed for static trigger functions
#include "spm_SubStateHandlerConfig.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSM
#include "trcGenProj/Header/spm_SystemStateManager.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_SYSTEMSTATEMANAGER

/******************************************************************************
  | declarations
  |-----------------------------------------------------------------------*/

/******************************************************************************
  | implementation
  |-----------------------------------------------------------------------*/
spm_tclSystemStateManager::spm_tclSystemStateManager( const ISpmFactory& factory )

/*!
  * \fn
  *  \brief
  *    constructor.
  *    - initialize member variable
  *    - semaphore/timer create
  *    - check to restore old system state (in case of reset)
  *
  *  \param[in] factory : spm factory object.
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   : spm_tclSystemStateManagerBase( factory )
   , _stateEntries( NULL )
   , _transitionEntries( NULL )
   , _u32NbTransionEntries( 0 ){
}

spm_tclSystemStateManager::~spm_tclSystemStateManager( ){
/*!
  * \fn
  *  \brief
  *    destructor.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   _stateEntries      = NULL;
   _transitionEntries = NULL;
}

tVoid spm_tclSystemStateManager::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    SPM factory invokes this method to initialize the dependency of the spm_tclSystemStateManager.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
// get all needed references now -> all SPM objects are now available
   spm_tclSystemStateManagerBase::vGetReferences( );
} // vGetReferences

tVoid spm_tclSystemStateManager::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  *   Before starting all other SPM components we have to set the current system state
  *   evaluate first system state (check for POR / reset / Warmstart / registry)
  *
  *  \param
  *  \note
  *  \version
  ******
  */
// ***************************************************************************
// Before starting all other SPM components we have to set the current system state
// evaluate first system state (check for POR / reset / Warmstart / registry
// ***************************************************************************

   spm_tclSystemStateManagerBase::vStartCommunication( );
} // vStartCommunication

tVoid spm_tclSystemStateManager::vInitStateMachine( ){
/*!
  * \fn
  *  \brief
  *   initialize statemachine, set default state
  *
  *  \param
  *  \note
  *  \version
  ******
  */

   /* nothing to do for classical statemachine */
}

tVoid spm_tclSystemStateManager::vRestoreLastState( tU32 u32NewSystemState ){
/*!
  * \fn
  *  \brief
  *   set statemachine to initial state after reset
  *   this is not the default state
  *
  *  \param
  *  \note
  *  \version
  ******
  */

   /* nothing to do for classical statemachine */

   u32ActStateEntry( u32NewSystemState );
}

tVoid spm_tclSystemStateManager::vActStateExit( tU32 u32NewSystemState ) const {
/*!
  * \fn
  *  \brief
  *    Called when new system state detected by state machine and current state is leaved.
  *
  *  \param[in] u32NewSystemState: current system state to be leaved.
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool                       bFound;
   const T_SYSTEM_STATE_ENTRY *prStateDescr;

   SPM_NULL_POINTER_CHECK( _stateEntries );

   {
      bFound       = FALSE;
      // prStateDescr = _arStateTable;
      prStateDescr = _stateEntries;

      /* while state not found and not end of state table */
      while ( ( !bFound ) && ( prStateDescr < ( _stateEntries + u32GetNumberOfStateEntries( ) ) ) ){
         /* if state is active */
         if ( u32NewSystemState == prStateDescr->u32SystemState ){
            /* call state exit function */
            bFound = TRUE;
            ( * prStateDescr->pfnvStateExit )( );
         }
         prStateDescr++;
      }
   }
} // vActStateExit

tU32 spm_tclSystemStateManager::u32ActStateEntry( tU32 u32NewSystemState ) const {
/*!
  * \fn
  *  \brief
  *    Called when new system state detected by state machine and current state has to be entered.
  *
  *  \param[in] u32NewSystemState : system state detected by state machine
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool                       bFound;
   const T_SYSTEM_STATE_ENTRY *prStateDescr;
   tU32                        u32Timeout = 0;

   SPM_NULL_POINTER_CHECK_VAL( _stateEntries );

   {
      bFound       = FALSE;
      prStateDescr = _stateEntries;

      /* while state not found and not end of state table */
      while ( ( !bFound ) && ( prStateDescr < ( _stateEntries + u32GetNumberOfStateEntries( ) ) ) ){
         /* if state is active */
         if ( u32NewSystemState == prStateDescr->u32SystemState ){
            /* call state exit function */
            bFound     = TRUE;
            ( * prStateDescr->pfnvStateEntry )( );
            u32Timeout = prStateDescr->u32Timeout;
         }
         prStateDescr++;
      }
   }
   return( u32Timeout );
} // u32ActStateEntry

tBool spm_tclSystemStateManager::bActStateReached( tU32 u32SystemState ) const {
/*!
  * \fn
  *  \brief
  *    Called when new system state is reached -> all applications has acknowledged transition.
  *
  *  \param[in] u32SystemState : system state entered
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool                       bFound = FALSE;
   const T_SYSTEM_STATE_ENTRY *prStateDescr;

   SPM_NULL_POINTER_CHECK_VAL( _stateEntries );

   {
      prStateDescr = _stateEntries;

      /* while state not found and not end of state table */
      while ( ( !bFound ) && ( prStateDescr < ( _stateEntries + u32GetNumberOfStateEntries( ) ) ) ){
         /* if state is active */
         if ( u32SystemState == prStateDescr->u32SystemState ){
            /* call state exit function */
            bFound = TRUE;
            ( * prStateDescr->pfnvStateReached )( );
         }
         prStateDescr++;
      }
   }
   return( bFound );
} // bActStateReached

// state handling
tU32 spm_tclSystemStateManager::u32CalcNewSystemState( tU32 u32CurrentSystemState ){
/*!
  * \fn
  *  \brief
  *    check for new system state in state transition table _arStateTable[].
  *
  *  \param[in] u32NewSystemState : current FSM state.
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tU32 u32NewSystemState = SPM_SYSTEM_INVALID;
   tU32 u32Timeout;

   tU32 u32TransitionId   = SPM_U32_TRANSITION_ID_INVALID;
   tU32 u32MsgToPost      = SPM_U32_MSG_INVALID;

   SPM_NULL_POINTER_CHECK_VAL( _poclSubStateHandler );

   _poclSubStateHandler->vCopyTrigger( &_tCurSubStateTrigger, &_tSubStateTrigger );
   _poclSubStateHandler->vRemoveAllTrigger( &_tSubStateTrigger, FALSE );

   const TStateTransitionElement *prTransitionDescr;

   SPM_NULL_POINTER_CHECK_VAL( _transitionEntries );

   prTransitionDescr = _transitionEntries;


   /* while end state not found and not end of state table */
   // while ( (bFound==FALSE) && (prTransitionDescr < (_aStateTransitionTable + ELEMENTE(_aStateTransitionTable)) ) )
   while ( ( u32TransitionId == SPM_U32_TRANSITION_ID_INVALID )
           && ( prTransitionDescr < ( _transitionEntries + _u32NbTransionEntries ) )
           ){
      // checksystem state
      if ( ( prTransitionDescr->u32CurrentFsmState == u32CurrentSystemState ) || ( prTransitionDescr->u32CurrentFsmState == DONT_CARE ) ){
         // is update trigger active
         if ( bCheckForMatch( (tU32*)_tCurSubStateTrigger.u32TriggerUpdate, (const tU32*)prTransitionDescr->u32UpdateTrigger ) ){
            // is trigger data active
            if ( bCheckForMatch( (tU32*)_tCurSubStateTrigger.u32Trigger, (const tU32*)prTransitionDescr->u32Trigger ) ){
               // is a substate not active
               if ( bCheckForMatch( (tU32*)_tCurSubStateTrigger.u32Trigger, (const tU32*)prTransitionDescr->u32TriggerNotSet, TRUE ) ){
                  // found a table entry
                  u32NewSystemState = prTransitionDescr->u32NextFsmState;
                  u32TransitionId   = prTransitionDescr->u32TransitionId;
                  u32MsgToPost      = prTransitionDescr->u32MsgToPost;
               }
            }
         }
      }
      prTransitionDescr++;
   }

   if ( u32TransitionId != SPM_U32_TRANSITION_ID_INVALID ){
      if ( ( ( u32CurrentSystemState == SPM_SYSTEM_PREPARE_SHUTDOWN )
             || ( u32CurrentSystemState == SPM_SYSTEM_SHUTDOWN )
             )
           && ( u32NewSystemState == SPM_SYSTEM_PREPARE_SHUTDOWN )
           ){
         // no transition -> system is already in shutdown phase
         u32TransitionId   = SPM_U32_TRANSITION_ID_INVALID;
         u32NewSystemState = SPM_SYSTEM_INVALID;
      } else {
         if ( u32NewSystemState != u32CurrentSystemState ){
            vActStateExit( u32CurrentSystemState );
         }
         u32Timeout = u32ActStateEntry( u32NewSystemState );
         if ( ( SPM_SYSTEM_STANDBY == u32NewSystemState )
              && ( 0 == u32Timeout ) ){
            vStartStateTimeoutTimer( 10 );
         } else {
            vStartStateTimeoutTimer( u32Timeout );
         }
      }

      // clear highest bit for trace output
      u32TransitionId = u32TransitionId & ~SPM_U32_WAIT_FOR_OFF;

   } else {
      // no new state found
      vStateNoTransition( );
   }

   if (u32TransitionId == SPM_U32_TRANSITION_ID_INVALID) {
       ETG_TRACE_USR1( ( "vGetNewSystemState, new system detected: (%u) with state transition ID %d",
                         ETG_ENUM( SPM_SYSTEM_STATES, u32NewSystemState ),
                         u32TransitionId ) );
   } else {
       ETG_TRACE_FATAL( ( "vGetNewSystemState, new system detected: (%u) with state transition ID %d",
                         ETG_ENUM( SPM_SYSTEM_STATES, u32NewSystemState ),
                         u32TransitionId ) );
   }

   u32PostCalcNewSystemState( u32NewSystemState, u32CurrentSystemState, u32TransitionId );

   if ( SPM_U32_MSG_INVALID != u32MsgToPost ){
      SPM_NULL_POINTER_CHECK_VAL( _poclWorkerServer );
      _poclWorkerServer->bPostMessage( "ISpmSystemStateManager", u32MsgToPost );
   }

   return( u32NewSystemState );
} // u32CalcNewSystemState

tVoid spm_tclSystemStateManager::vCheckNewStateTransition() {
/*!
  * \fn
  *  \brief
  *    Entry point to check for system state transitions.
  *
  *  \param
  *  \return
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tU32 u32NewSysState;

   // point of no return reached
   if (_u32NewSystemState != SPM_SYSTEM_SHUTDOWN){

      _u32IntermediateSystemState = _u32NewSystemState;
      u32NewSysState              = u32CalcNewSystemState(_u32NewSystemState);

      if (SPM_SYSTEM_INVALID != u32NewSysState){
         _u32NewSystemState = u32NewSysState;

         // remember new system state
         dp_tclSpmDpInternDataSystemState oNewSystemState;
         oNewSystemState.vSetData(_u32NewSystemState);

         // inform immediatley via direct interface
         if (FALSE == _poclGlobalApplicationManager->bProcessSystemState(_u32NewSystemState) ){
         }

         _poclSubStateHandler->vProcessSystemState(_u32NewSystemState);

         // check if we have to debounce --> display should be off while debouncing
         if (_u32StateChangeCounter > SPM_U32_DEBOUNCE_STATE_CHANGE_COUNTER){
            if (_u32NewSystemState != SPM_SYSTEM_ON){
               _poclSubStateHandler->vSetSubStateType(SPM_U32_STATE_DEBOUNCE, TRUE);
               _u32StateChangeCounter = 0;
            }
         }

         // set again --> just to make sure that it was not changed meanwhile
         oNewSystemState.vSetData(_u32NewSystemState);


      } else {
         _poclSubStateHandler->vProcessSystemState(SPM_SYSTEM_INVALID);
      }
   }

}

// timer handling
tVoid spm_tclSystemStateManager::vSetStateTime( tU32 u32State,
                                                tU32 u32Time ){
/*!
  * \fn
  *  \brief
  *    Set new timer value for specific system state.
  *
  *  \param[in] u32State : system state
  *  \param[in] u32Time : time in ms
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tBool                 bFound = FALSE;
   T_SYSTEM_STATE_ENTRY *prStateDescr;

   SPM_NULL_POINTER_CHECK( _stateEntries );

   {
      prStateDescr = _stateEntries;

      /* while state not found and not end of state table */
      while ( ( !bFound ) && ( prStateDescr < ( _stateEntries + u32GetNumberOfStateEntries( ) ) ) ){
         /* if state is active */
         if ( u32State == prStateDescr->u32SystemState ){
            bFound                   = TRUE;
            prStateDescr->u32Timeout = u32Time;
         }
         prStateDescr++;
      }
   }

   if ( !bFound ){
      ETG_TRACE_FATAL( ( "spm_tclSystemStateManager::vSetStateTime(): State %d unknown", ETG_ENUM( SPM_SYSTEM_STATES, u32State ) ) );
   }
} // vSetStateTime

tU32 spm_tclSystemStateManager::u32GetStateTime( tU32 u32State ) const {
/*!
  * \fn
  *  \brief
  *    Get timer value for specific system state.
  *
  *  \param[in] u32State : system state
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   tU32                        u32ReturnVal = ( tU32 ) - 1;
   tBool                       bFound       = FALSE;
   const T_SYSTEM_STATE_ENTRY *prStateDescr;

   SPM_NULL_POINTER_CHECK_VAL( _stateEntries );

   {
      prStateDescr = _stateEntries;

      /* while state not found and not end of state table */
      while ( ( !bFound ) && ( prStateDescr < ( _stateEntries + u32GetNumberOfStateEntries( ) ) ) ){
         /* if state is active */
         if ( u32State == prStateDescr->u32SystemState ){
            bFound       = TRUE;
            u32ReturnVal = prStateDescr->u32Timeout;
         }
         prStateDescr++;
      }
   }

   if ( !bFound ){
      ETG_TRACE_FATAL( ( "spm_tclSystemStateManager::u32GetStateTime(): State %d unknown", ETG_ENUM( SPM_SYSTEM_STATES, u32State ) ) );
   }
   return( u32ReturnVal );
} // u32GetStateTime

tBool spm_tclSystemStateManager::bCheckForMatch( tU32       *pu32Data,
                                                 const tU32 *pu32TabEntry,
                                                 tBool       bCheckForNotSet )  const {
/*!
  * \fn
  *  \brief
  *    Checks data with transition table entry.
  *
  *  \param[in] pu32Data: data to be compared
  *  \param[in] pu32TabEntry: transition table entry
  *  \param[in] bCheckForNotSet: trigger set/not set flag
  *  \note
  *  \todo
  *  \version
  ******
  */
// check for
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      if ( * pu32TabEntry != 0xffffffff ){
         if ( !bCheckForNotSet ){
            if ( !( ( * pu32TabEntry & * pu32Data ) == * pu32TabEntry ) ){
               return( FALSE );
            }
         } else if ( !( ( * pu32TabEntry & ~( * pu32Data ) ) == * pu32TabEntry ) ){
            return( FALSE );
         }
      }
      pu32Data++;
      pu32TabEntry++;
   }
   return( TRUE );
} // bCheckForMatch

tVoid spm_tclSystemStateManager::vHandleMessage( tU32 u32Message,
                                                 tU32 u32Param ){
/*!
  * \fn
  *  \brief
  *    handle any message valid only for classical LCM state machine
  *
  *  \param[in] u32Message: message ID
  *  \param[in] u32Param: parameter of message
  *  \note
  *  \todo
  *  \version
  ******
  */
// check for
   spm_tclSystemStateManagerBase::vHandleMessage( u32Message, u32Param );
}
