/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     LifeCycleConsumerProxy.cpp
  * \brief    Implementation of the LifeCycleConsumerProxy component
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  * \todo implement the ASF callbacks "correctly" to also print content of parameter to remove LINT
  *
  * \par Copyright:
  * (c) 2013-2015 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
// include for etg support lib
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LCM_SERVER_LCMDBUS
#include "trcGenProj/Header/LifeCycleConsumerProxy.cpp.trc.h"
#endif

#include "IDbusProxy.h"
#include "ILcmConsumerStub.h"
#include "I_lcm_ServiceLcmDbusIf.h"

#include "LifeCycleConsumerProxy.h"
//  boost::shared_ptr is used here cause by ASF interface.

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

using namespace ::asf::core;
using namespace ::org::genivi::NodeStateManager::LifeCycleConsumer;

DEFINE_CLASS_LOGGER_AND_LEVEL("org/bosch/cm/lcm/LifeCycleConsumerProxy", LifeCycleConsumerProxy, Info);

/**
  *  \brief constructor of the class DBusProxy
  *
  *  \param [in] factory reference to the LCM-factory to get the references to other LCM classes
  *  \return class is created
  *
  *  \details constructs all stubs and permanent proxies. it checks if DBUS adress is defined
  */
LifeCycleConsumerProxy::LifeCycleConsumerProxy(lcm_tclAppMain *poMainAppl)
   : ILifeCycleConsumerProxy(poMainAppl){
   ETG_TRACE_USR1( ( "LifeCycleConsumerProxy called" ) );
}

/**
  *  \brief Destructor of class
  *
  *  \return none
  *
  *  \details remove all dynamically created clients
  */
LifeCycleConsumerProxy::~LifeCycleConsumerProxy(){
   ETG_TRACE_USR1( ( "~LifeCycleConsumerProxy called" ) );
}

/**
  *  \brief start working
  *
  *  \return
  *
  *  \details remove all dynamically created clients
  */
void LifeCycleConsumerProxy::vStartCommunication(){
   _poIDbusProxy = dynamic_cast < IDbusProxy* >( _cpoMain->getHandler("IDbusProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusProxy);
}           // vStartCommunication

/**
  *  \brief adds the unit- and service proxies
  *
  *  \param [in] strUnitName name of the unit
  *  \return none
  *
  *  \details escapes the unit name and adds the respective serviceproxy and unitproxy to the maps
  */
tVoid LifeCycleConsumerProxy::AddLifeCycleConsumerProxy(const std::string strBusName,
                                                        const std::string strObjectPath){
   bool                                   bFound = false;
//   ahl_bEnterCritical(_hProxyListAccess);
   AsfLcmConsumerProxyMap::const_iterator iterConsumerPos;

   for(iterConsumerPos = _lcmConsumerProxyMap.begin();
       iterConsumerPos != _lcmConsumerProxyMap.end();
       iterConsumerPos++){
      if(iterConsumerPos->second->getObjectName() == strObjectPath){
         bFound = true;
         break;
      }
   }
//      ahl_bReleaseCritical(_hProxyListAccess);
   if(!bFound){
      boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy > _lifecycleconsumerProxy = ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy::createProxy("DBusLifecycleConsumerPort", // portName
                                                                                                                                                                                                                         strBusName,                  // busName
                                                                                                                                                                                                                         strObjectPath,               // objectPath
                                                                                                                                                                                                                         DBUS_BUS_SYSTEM,             // busType
                                                                                                                                                                                                                         * this                       // ServiceAvailableIF
                                                                                                                                                                                                                         );
      ILcmConsumerStub                         *poILcmConsumerStub = dynamic_cast < ILcmConsumerStub* >( _cpoMain->getHandler("ILcmConsumerStub") );
      LCM_NULL_POINTER_CHECK(poILcmConsumerStub);
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      std::string                               strSwBlockName     = _poIDbusProxy->ExtractApplicationName(strObjectPath);
      boost::shared_ptr < AsfLcmConsumerProxy > pAsfLcmConsumerProxy(new AsfLcmConsumerProxy(_lifecycleconsumerProxy, strBusName, strObjectPath, poILcmConsumerStub->u32GetAppIdByName(strSwBlockName ) ) );
      _lcmConsumerProxyMap.insert(std::pair < const boost::shared_ptr < Proxy >, boost::shared_ptr < AsfLcmConsumerProxy > >(_lifecycleconsumerProxy, pAsfLcmConsumerProxy) );
      ETG_TRACE_USR1( ( "_lcmConsumerProxyMap has now a size of %u", (tUInt)_lcmConsumerProxyMap.size() ) );
   }
} // LifeCycleConsumerProxy::AddLifeCycleConsumerProxy

/**
  *  \brief send application change request
  *
  *  \param [in] u32AppId CCA application ID
  *  \return none
  *
  *  \details gets the LifeCycleConsumerProxy and sends the application state
  *           change request
  */
tVoid LifeCycleConsumerProxy::SendAppStateChangeRequest(const tU32 u32AppId,
                                                        const tU32 u32PowerData){
   // here we have to post message to LCM consumer
//    ahl_bEnterCritical(_hProxyListAccess);
   AsfLcmConsumerProxyMap::const_iterator iterConsumerPos;

   for (iterConsumerPos = _lcmConsumerProxyMap.begin(); iterConsumerPos != _lcmConsumerProxyMap.end(); ++iterConsumerPos){
      if (iterConsumerPos->second->u32GetAppId() == u32AppId){

         ETG_TRACE_USR1( ( "LifeCycleConsumerProxy::SendAppStateChangeRequest(): App %08x: Send powerstate %x on bus %s",
         ETG_ENUM(ail_u16AppId, u32AppId ),
         ETG_ENUM(SPM_APPSTATE_DEF, u32PowerData), 
         iterConsumerPos->second->getBusName().c_str() ) );

         iterConsumerPos->second->vSetRequestedState(u32PowerData);
         boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy > lifecycleconsumerProxy = iterConsumerPos->second->getProxy();
         lifecycleconsumerProxy->sendLifecycleRequestRequest(* this, SPM_U32_LCM_GET_PROXY_SHUTDOWN_TYPE(u32PowerData) /*request*/, iterConsumerPos->second->u32GetAppId() /*requestId*/);
      }
   }
//    ahl_bReleaseCritical(_hProxyListAccess);
} // LifeCycleConsumerProxy::SendAppStateChangeRequest

/**
  *  \brief handles received responces for LifeCycleRequests
  *
  *  \return True  if application was found
  *          False if application was NOT found
  *
  *  \details if the application is found the response is send to ProcBaseEarly
  */
tBool LifeCycleConsumerProxy::bHandleLifeCycleRequestResponses(const tU32 u32AppId,
                                                               const ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e eNsmErrorStatus){
   tBool                                  bFound = FALSE;
//   ahl_bEnterCritical(_hProxyListAccess);
   AsfLcmConsumerProxyMap::const_iterator iterConsumerPos;

   for (iterConsumerPos = _lcmConsumerProxyMap.begin(); iterConsumerPos != _lcmConsumerProxyMap.end(); ++iterConsumerPos){
      if (iterConsumerPos->second->u32GetAppId() == u32AppId){
         tU32 u32RequestedState = iterConsumerPos->second->u32GetRequestedState();
         iterConsumerPos->second->vSetCurrentState(u32RequestedState);

         Ilcm_tclServiceLcmDbusIf *poServerRef       = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
         LCM_NULL_POINTER_CHECK_VAL(poServerRef);

         ETG_TRACE_USR1( ( "LifeCycleConsumerProxy::bHandleLifeCycleRequestResponses(): state=%u send ACK to LAM for App %08x!",
         ETG_ENUM(SPM_APPSTATE_DEF, u32RequestedState),
         ETG_ENUM(ail_u16AppId, iterConsumerPos->second->u32GetAppId() ) ) );
//         ahl_bReleaseCritical(_hProxyListAccess);
         if (eNsmErrorStatus == ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e__NsmErrorStatus_Ok){
            poServerRef->sendAppStateChangeStatus(u32AppId, u32RequestedState);
            bFound = TRUE;
            break;
         }
      }
   }

   if(iterConsumerPos == _lcmConsumerProxyMap.end() ){
      ETG_TRACE_USR1( ( "LifeCycleConsumerProxy::bHandleLifeCycleRequestResponses(): no match found for %08x!", ETG_ENUM(ail_u16AppId, u32AppId) ) );
//      ahl_bReleaseCritical(_hProxyListAccess);
   }

   return( bFound );
} // LifeCycleConsumerProxy::bHandleLifeCycleRequestResponses

// ######################################
// ASF-Core implementations
// ######################################
#if 1       // allow editor to collapse this section
// ServiceAvailableIF implementation

/**
  *  \brief handles all service available indications by ASF
  *
  *  \param [in] proxy pointer to the proxy where the service is now available
  *  \param [in] stateChange contains the previous and new state of the service
  *  \return Return_Description
  *
  *  \details Details
  */
   void LifeCycleConsumerProxy::onAvailable(const boost::shared_ptr < Proxy >& proxy,
                                            const ServiceStateChange         & stateChange){
      ETG_TRACE_USR1( ( "onAvailable ..........." ) );
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);

//      ahl_bEnterCritical(_hProxyListAccess);
      AsfLcmConsumerProxyMap::const_iterator iterConsumerProxyPos;
      iterConsumerProxyPos = _lcmConsumerProxyMap.find(proxy);
      if( iterConsumerProxyPos != _lcmConsumerProxyMap.end() ){
         ETG_TRACE_USR1( ( "  - %40s ConsumerProxy is connected --> trigger LAM with state initialized --> app %08x",
                           iterConsumerProxyPos->second->getObjectName().c_str(),
                           ETG_ENUM(ail_u16AppId, iterConsumerProxyPos->second->u32GetAppId() ) ) );

         Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
         LCM_NULL_POINTER_CHECK(poServerRef);

         uint32                    u32AppId    = iterConsumerProxyPos->second->u32GetAppId();
//         ahl_bReleaseCritical(_hProxyListAccess);
         poServerRef->sendAppInitializedStatus(u32AppId);

      } else {
//         ahl_bReleaseCritical(_hProxyListAccess);
      }
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   /**
     *  \brief called if any service we are registered too is removed from DBus-Daemon
     *
     *  \param [in] proxy pointer to the proxy where the service state of the stub has been changed
     *  \param [in] stateChange structure containing the precvious and current service state
     *  \return void
     *
     *  \details Handles all actions to be taken when a service becomes unavailable.
     *  ATTENTION: This should never happen
     */
   void LifeCycleConsumerProxy::onUnavailable(const boost::shared_ptr < Proxy >& proxy,
                                              const ServiceStateChange         & stateChange){
      ETG_TRACE_USR1( ( "onUnavailable ..........." ) );
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);

//      ahl_bEnterCritical(_hProxyListAccess);
      AsfLcmConsumerProxyMap::const_iterator iterConsumerPos;
      iterConsumerPos = _lcmConsumerProxyMap.find(proxy);
      if( iterConsumerPos != _lcmConsumerProxyMap.end() ){
         // check AsfLcmConsumerProxy
         ETG_TRACE_USR1( ( "LifeCycleConsumerProxy %s is disconnected", iterConsumerPos->second->getObjectName().c_str() ) );
      }
//      ahl_bReleaseCritical(_hProxyListAccess);
   }        // onUnavailable

#endif      // #if 1 // allow editor to collapse this section

// ######################################
// LifecycleConsumerProxy implementations
// ######################################
#if 1       // allow editor to collapse this section
// LifecycleRequestMethodCallbackIF
   void LifeCycleConsumerProxy::onLifecycleRequestError(const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy >& proxy,
                                                        const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifecycleRequestError > & error){
      ETG_TRACE_ERRMEM(("Received onLifecycleRequestError"));
      std::string strProxyBusName = proxy->getDBusBusName();
      std::string strProxyInfName = proxy->getInterfaceName();
      std::string strProxyBusObj = proxy->getDBusObjectPath();
      std::string strProxyError = error->getName();
      ETG_TRACE_ERRMEM(("onLifecycleRequestError proxy bus %s", strProxyBusName.c_str()));
      ETG_TRACE_ERRMEM(("onLifecycleRequestError proxy interface %s", strProxyInfName.c_str()));
      ETG_TRACE_ERRMEM(("onLifecycleRequestError proxy Dbus obj path %s", strProxyBusObj.c_str()));
      ETG_TRACE_ERRMEM(("onLifecycleRequestError error name %s", strProxyError.c_str()));
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   void LifeCycleConsumerProxy::onLifecycleRequestResponse(const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifeCycleConsumerProxy >  & proxy,
                                                           const ::boost::shared_ptr < ::org::genivi::NodeStateManager::LifeCycleConsumer::LifecycleRequestResponse >& response){
      ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e eStatus = response->getErrorCode();
      ETG_TRACE_USR1(("LifeCycleConsumerProxy::onLifecycleRequestResponse(): Received onLifecycleRequestReply , %u", ETG_ENUM(ORG_GENIVI_NODESTATEMANAGER_LCMERRORTYPES_NSMERRORSTATUS, eStatus) ));

      //      ahl_bEnterCritical(_hProxyListAccess);
      AsfLcmConsumerProxyMap::const_iterator iterConsumerPos;
      iterConsumerPos = _lcmConsumerProxyMap.find(proxy);
      if (iterConsumerPos != _lcmConsumerProxyMap.end()){
         iterConsumerPos->second->vSetCurrentState(iterConsumerPos->second->u32GetRequestedState());

         Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast <Ilcm_tclServiceLcmDbusIf*>(_cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf"));
         LCM_NULL_POINTER_CHECK(poServerRef);

         tU32                      u32AppId = iterConsumerPos->second->u32GetAppId();
         tU32                      u32RequestedState = iterConsumerPos->second->u32GetRequestedState();
         //         ahl_bReleaseCritical(_hProxyListAccess);
         if ( eStatus == ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e__NsmErrorStatus_Ok ){
            ETG_TRACE_USR1(("LifeCycleConsumerProxy::onLifecycleRequestResponse(): send ACK to LAM for App %08x!", ETG_ENUM(ail_u16AppId, u32AppId)));
            poServerRef->sendAppStateChangeStatus(u32AppId, u32RequestedState);
         } else if (eStatus == ::org::genivi::NodeStateManager::LcmErrorTypes::_NsmErrorStatus_e__NsmErrorStatus_ResponsePending){
            ETG_TRACE_USR1(("LifeCycleConsumerProxy::onLifecycleRequestResponse(): waiting for asynchronous response for App %08x!", ETG_ENUM(ail_u16AppId, u32AppId)));
         }  else {
            ETG_TRACE_ERR(("LifeCycleConsumerProxy::onLifecycleRequestResponse(): unexpected response (%u) for App %08x!", ETG_ENUM(ORG_GENIVI_NODESTATEMANAGER_LCMERRORTYPES_NSMERRORSTATUS, eStatus), ETG_ENUM(ail_u16AppId, u32AppId)));
            ETG_TRACE_ERRMEM(("LifeCycleConsumerProxy::onLifecycleRequestResponse(): unexpected response (%u) for App %08x!", ETG_ENUM(ORG_GENIVI_NODESTATEMANAGER_LCMERRORTYPES_NSMERRORSTATUS, eStatus), (tUInt)ETG_ENUM(ail_u16AppId, u32AppId)));
         }
      }
      else {
         //         ahl_bReleaseCritical(_hProxyListAccess);
      }
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

#endif // #if 1 // allow editor to collapse this section
}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

