/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     LcmRecoveryClientStub.cpp
  * \brief    Implementation of the LcmRecoveryClientStub component
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  * \todo implement the ASF callbacks "correctly" to also print content of parameter to remove LINT
  *
  * \par Copyright:
  * (c) 2013-2015 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
// include for etg support lib
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LCM_SERVER_LCMDBUS
#include "trcGenProj/Header/LcmRecoveryClientStub.cpp.trc.h"
#endif

#include "IDbusProxy.h"

// include type definitions of Genivi
#include "org/genivi/NodeStateManager/LcmErrorTypesConst.h"

#include "LcmRecoveryClientStub.h"
//  boost::shared_ptr is used here cause by ASF interface.

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

using namespace ::asf::core;
using namespace ::org::bosch::cm::lcm::Generic_RecoveryClient;

DEFINE_CLASS_LOGGER_AND_LEVEL("org/bosch/cm/lcm/LcmRecoveryClientStub", LcmRecoveryClientStub, Info);

/**
  *  \brief constructor of the class LcmRecoveryClientStub
  *
  *  \param [in] factory reference to the LCM-factory to get the references to other LCM classes
  *  \return class is created
  *
  *  \details constructs all stubs and permanent proxies. it checks if DBUS adress is defined
  */
LcmRecoveryClientStub::LcmRecoveryClientStub(lcm_tclAppMain *poMainAppl)
   : ILcmRecoveryClientStub(poMainAppl),
   Generic_RecoveryClientStub("GenericRecoveryClientPort"),
   _bSupervisionActive(true){
   ETG_TRACE_USR1( ( "LcmRecoveryClientStub called" ) );
   RecoveryClientIgnoreSet.clear();
}

/**
  *  \brief Destructor of class
  *
  *  \return none
  *
  *  \details remove all dynamically created clients
  */
LcmRecoveryClientStub::~LcmRecoveryClientStub(){
   ETG_TRACE_USR1( ( "~LcmRecoveryClientStub called" ) );
}

/**
  *  \brief start working
  *
  *  \return
  *
  *  \details currenlty empty
  */
void LcmRecoveryClientStub::vStartCommunication(){
   _poIDbusProxy = dynamic_cast < IDbusProxy* >( _cpoMain->getHandler("IDbusProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusProxy);
}           // vStartCommunication

/**
  *  \brief add a Unit that is stopped to the ignore list of failed units
  *
  *  \details If a unit is not correctly configured it may go into failed state when
  *   it is stopped. To prevent resets it is added to a list of units that will be
  *   ignored for onSetFailedServiceNameRequest
  */
void LcmRecoveryClientStub::vUnitStopped(const std::string& UnitName){
   RecoveryClientIgnoreSet.insert(UnitName);
   ETG_TRACE_USR1( ( "LcmRecoveryClientStub::vUnitStopped(): Number of entries in RecoveryClientIgnoreSet %u!", (tUInt)RecoveryClientIgnoreSet.size() ) );
}

/**
  *  \brief Update value of _bSupervisionActive when state of Supervision is changed
  *
  *  \details when state of Supervision is changed from lcm early, 
  *   lcm early inform to lcm late to change value of _bSupervisionActive
  */
void LcmRecoveryClientStub::vSupervisionStateChange(tBool SupervisionActive){
   _bSupervisionActive = SupervisionActive;
   ETG_TRACE_USR1( ( "LcmRecoveryClientStub::bSupervisionStateChange(): New state of _bSupervisionActive: %d!", ETG_ENUM( SPM_STATE_COMMON,_bSupervisionActive ) ) );
}

// ######################################
// LcmRecoveryClientStub implementation
// ######################################
#if 1       // allow editor to collapse this section

/**
  *  \brief called by GenericRecoveryClient if a service entered failed state
  *
  *  \details Checks if this service was stopped by LCM and prints message to ErrMem
  */
   void LcmRecoveryClientStub::onSetFailedServiceNameRequest(const ::boost::shared_ptr < SetFailedServiceNameRequest >& request){
      ETG_TRACE_USR1( ( "LcmRecoveryClientStub::onSetFailedServiceNameRequest(): Received!" ) );
      std::string ErrorString = "Unit reported Error: ServiceName=";
      std::string ServiceName;
      if(request->hasServiceName() ){
         ServiceName  = request->getServiceName();
         ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetFailedServiceNameRequest(): ServiceName: %s", ServiceName.c_str() ) );
         ErrorString += request->getServiceName();
      } else {
         ServiceName  = "unknown";
         ErrorString += ServiceName;
         ETG_TRACE_ERRMEM( ( "No ServiceName found" ) );
      }
      if( RecoveryClientIgnoreSet.find(ServiceName) != RecoveryClientIgnoreSet.end() ){
         std::string IgnoreString = "Unit was stopped by StopUnit. Error is ignored for " + ServiceName;
         ETG_TRACE_ERRMEM( ( "Unit was stopped by StopUnit. Error is ignored for %s", ServiceName.c_str() ) );
      } else {
         ETG_TRACE_ERRMEM( ( "%s", ErrorString.c_str() ) );
      }
      ETG_TRACE_ERRMEM( ("LcmRecoveryClientStub::onSetFailedServiceNameRequest(): Send response sendSetFailedServiceNameResponse") );
      sendSetFailedServiceNameResponse(Response__E_OK);
   }        // onSetFailedServiceNameRequest

/**
  *  \brief called by GenericRecoveryClient if a service entered failed state
  *
  *  \details Checks if this service was stopped by LCM and prints message to ErrMem
  */
   void LcmRecoveryClientStub::onSetFailedServiceStatusRequest(const ::boost::shared_ptr < SetFailedServiceStatusRequest >& request){
      ETG_TRACE_USR1( ( "LcmRecoveryClientStub::onSetFailedServiceStatusRequest(): Received!" ) );
      std::string ServiceName = "undefined";
      if(request->hasServiceName() ){
         ServiceName = request->getServiceName();
      }
      std::string ErrorString = "Unit " + ServiceName + " reported State as ";
      if( RecoveryClientIgnoreSet.find(ServiceName) == RecoveryClientIgnoreSet.end() ){
         // if current service is not in the list of stopped units
         if(request->hasServiceState() ){
            UnitStates  ServiceState  = request->getServiceState();
            const char *UnitStateName = UnitStates_Name(ServiceState);
            if(UnitStateName != NULL){
               std::string UnitStateString(UnitStateName);
               ErrorString += UnitStateString;
            } else {
               ErrorString += "undefined";
            }
            ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetFailedServiceStatusRequest(): ServiceState: %d for %s", ETG_ENUM(SPM_SYSTEMD_UNIT_STATES, (int8)ServiceState), ServiceName.c_str() ) );
         } else {
            ErrorString += "unknown";
            ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetFailedServiceStatusRequest(): No ServiceState found" ) );
         }
         ETG_TRACE_ERRMEM( ( "%s", ErrorString.c_str() ) );
      }
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   void LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(const ::boost::shared_ptr < SetCallStackRequestCounterRequest >& request){
      ETG_TRACE_USR1( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): Received!" ) );
      std::string ServiceName = "undefined";
      if(request->hasServiceName() ){
         ServiceName = request->getServiceName();
      }
      if(request->hasCallStackRequestCounter() ){
         int8 u8CallStackRequestCounter = request->getCallStackRequestCounter();
         ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): CallStackRequestCounter: %d for %s", u8CallStackRequestCounter, ServiceName.c_str() ) );

         if( RecoveryClientIgnoreSet.find(ServiceName) == RecoveryClientIgnoreSet.end() ){
            // if current service is not in the list of stopped units
            if (u8CallStackRequestCounter >= SPM_MAX_NUMBER_OF_RECOVERCLIENT_ACTIONS){
               std::string ErrorString = "Max Number of Actions reached - resetting";
               ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): Max Number of Actions reached - resetting!" ) );
               if (_bSupervisionActive){
                  LCM_RESET_VIA_ASSERT;
               } else {
                  ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): Reset disabled via TTFis" ) );
               }
            }        // < SPM_MAX_NUMBER_OF_RECOVERCLIENT_ACTIONS
         } else {
            ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): Reset disabled due to StopUnit" ) );
         }
      } else {
         ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): No CallStackRequestCounter found" ) );
      }
      ETG_TRACE_ERRMEM( ("LcmRecoveryClientStub::onSetCallStackRequestCounterRequest(): Send respone sendSetCallStackRequestCounterResponse") );
      sendSetCallStackRequestCounterResponse(Response__E_OK);
   }        // onSetCallStackRequestCounterRequest
/**
  *  \brief called by GenericRecoveryClient When GenericRecoveryClient wants to know that GenericRecoveryClient can reset or not
  *
  *  \details No reset is intended if
  *           _bSupervisionActive==false     
  *           RecoveryClientIgnoreSet.find(ServiceName) != RecoveryClientIgnoreSet.end()     
  */
   void LcmRecoveryClientStub::onCheckResetConditionRequest (const ::boost::shared_ptr< CheckResetConditionRequest >& request){
      ETG_TRACE_USR1( ( "LcmRecoveryClientStub::onCheckResetConditionRequest(): Received!" ) );
      std::string ServiceName = "undefined";
      if(request->hasServiceName() ){
         ServiceName = request->getServiceName();
      }
      
      if( (RecoveryClientIgnoreSet.find(ServiceName) != RecoveryClientIgnoreSet.end()) && (!_bSupervisionActive)){
         ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onCheckResetConditionRequest(): No reset is intended" ) );

         sendCheckResetConditionResponse(LcmResetAllowed__NOT_ALLOW_TO_RESET);
      } else {
         ETG_TRACE_ERRMEM( ( "LcmRecoveryClientStub::onCheckResetConditionRequest(): Can reset via GenericRecoveryClient" ) );
         sendCheckResetConditionResponse(LcmResetAllowed__ALLOW_TO_RESET);
      }
   }        // onCheckResetConditionRequest

#endif      // #if 1 // allow editor to collapse this section
}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

