/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     DbusServiceProxy.cpp
  * \brief    Implementation of the DbusServiceProxy component
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  * \todo implement the ASF callbacks "correctly" to also print content of parameter to remove LINT
  *
  * \par Copyright:
  * (c) 2013-2015 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
// include for etg support lib
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LCM_SERVER_LCMDBUS
#include "trcGenProj/Header/DbusServiceProxy.cpp.trc.h"
#endif

#include "IDbusProxy.h"
#include "IDbusUnitProxy.h"
#include "I_lcm_ServiceLcmDbusIf.h"

#include "DbusServiceProxy.h"
#include <boost/algorithm/string/predicate.hpp>
//  boost::shared_ptr is used here cause by ASF interface.

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

using namespace ::asf::core;
using namespace ::org::freedesktop::systemd1::Service;

DEFINE_CLASS_LOGGER_AND_LEVEL("org/bosch/cm/lcm/DbusServiceProxy", DbusServiceProxy, Info);

/**
  *  \brief constructor of the class DBusServiceProxy
  *
  *  \param [in] factory reference to the LCM-factory to get the references to other LCM classes
  *  \return class is created
  *
  *  \details constructs all stubs and permanent proxies. it checks if DBUS adress is defined
  */
DbusServiceProxy::DbusServiceProxy(lcm_tclAppMain *poMainAppl)
   : IDbusServiceProxy(poMainAppl){
   ETG_TRACE_USR1( ( "DbusServiceProxy called" ) );
}

/**
  *  \brief Destructor of class
  *
  *  \return none
  *
  *  \details remove all dynamically created clients
  */
DbusServiceProxy::~DbusServiceProxy(){
   ETG_TRACE_USR1( ( "~DbusServiceProxy called" ) );

   AsfServiceProxyMap::iterator itServicePos;
   for (itServicePos = _serviceProxyMap.begin(); itServicePos != _serviceProxyMap.end();){
      itServicePos->second->getProxy()->sendDeregisterAll();
      _serviceProxyMap.erase(itServicePos++);
   }
}

/**
  *  \brief start working
  *
  *  \return
  *
  *  \details currently empty
  */
void DbusServiceProxy::vStartCommunication(){
   _poIDbusProxy = dynamic_cast < IDbusProxy* >( _cpoMain->getHandler("IDbusProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusProxy);
}           // vStartCommunication

/**
  *  \brief adds the unit- and service proxies
  *
  *  \param [in] strUnitName name of the unit
  *  \return none
  *
  *  \details escapes the unit name and adds the respective serviceproxy and unitproxy to the maps
  */
tVoid DbusServiceProxy::vAddServiceProxy(const std::string strUnitName,
                                         const std::string strObjectPath){

   // Do not create ServiceProxy for .target units as they do not implement org.freedesktop.systemd1.Service interface.
   if ( boost::ends_with(strUnitName, ".target") ){
      ETG_TRACE_ERR( ( "Service Proxy not applicable for %s", strUnitName.c_str() ) );
      return;
   }

   bool                               bFound = false;
//   ahl_bEnterCritical(_hProxyListAccess);
   AsfServiceProxyMap::const_iterator iterServicePos;

   for(iterServicePos = _serviceProxyMap.begin();
       iterServicePos != _serviceProxyMap.end();
       iterServicePos++){
      if(iterServicePos->second->getServiceName() == strUnitName){
         bFound = true;
         break;
      }
   }
//   ahl_bReleaseCritical(_hProxyListAccess);
   if(!bFound){
      boost::shared_ptr < ServiceProxy > _serviceProxy = ServiceProxy::createProxy("DBusServiceClientPort",    // portName
                                                                                   "org.freedesktop.systemd1", // busName
                                                                                   strObjectPath,              // objectPath
                                                                                   DBUS_BUS_SYSTEM,            // busType
                                                                                   * this                      // ServiceAvailableIF
                                                                                   );
      boost::shared_ptr < AsfServiceProxy > pAsfServiceProxy(new AsfServiceProxy(_serviceProxy, strUnitName) );
//      ahl_bEnterCritical(_hProxyListAccess);
      _serviceProxyMap.insert(std::pair < const boost::shared_ptr < Proxy >, boost::shared_ptr < AsfServiceProxy > >(_serviceProxy, pAsfServiceProxy) );
//      ahl_bReleaseCritical(_hProxyListAccess);
      ETG_TRACE_USR1( ( "_serviceProxyMap has now a size of %u", (tUInt)_serviceProxyMap.size() ) );
   } else {
      ETG_TRACE_USR1( ( "_serviceProxyMap already contains a proxy for %s", strUnitName.c_str() ) );
//      ahl_bReleaseCritical(_hProxyListAccess);
   }
}           // vAddServiceProxy

/**
  *  \brief send the request to get the PID of the main process for the service file
  *
  *  \param [in] strServiceName - name of the service file
  *  \return void
  *
  *  \details get the correct serviceProxy, or create one if it does not yet exist
  *  (this may happen if the service was started indirectly by another service)
  *  send the request to get the main PID
  */
void DbusServiceProxy::vTriggerGetMainPid(const std::string strServiceName){
      ETG_TRACE_USR1( ( "DbusServiceProxy::vTriggerGetMainPid for service: %s", strServiceName.c_str() ) );

   tBool                              bFound = false;
//   ahl_bEnterCritical(_hProxyListAccess);
   AsfServiceProxyMap::const_iterator iterServicePos;
   for(iterServicePos = _serviceProxyMap.begin();
       iterServicePos != _serviceProxyMap.end();
       iterServicePos++){
      if(iterServicePos->second->getServiceName() == strServiceName){
         bFound = true;
         iterServicePos->second->getProxy()->sendExecMainPIDGet(* this);
         break;
      }
   }
//   ahl_bReleaseCritical(_hProxyListAccess);
   if(!bFound){
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      std::string     escapedUnitName  = _poIDbusProxy->unitNameEscape(strServiceName);
      std::string     objectPath       = "/org/freedesktop/systemd1/unit/" + escapedUnitName;
      vAddServiceProxy(strServiceName, objectPath);
      IDbusUnitProxy *poIDbusUnitProxy = dynamic_cast < IDbusUnitProxy* >( _cpoMain->getHandler("IDbusUnitProxy") );
      LCM_NULL_POINTER_CHECK(poIDbusUnitProxy);
      poIDbusUnitProxy->vAddUnitProxy(strServiceName, objectPath);
   }
}           // DbusServiceProxy::vTriggerGetMainPid

// ######################################
// ASF-Core implementations
// ######################################
#if 1       // allow editor to collapse this section
// ServiceAvailableIF implementation

/**
  *  \brief handles all service available indications by ASF
  *
  *  \param [in] proxy pointer to the proxy where the service is now available
  *  \param [in] stateChange contains the previous and new state of the service
  *  \return Return_Description
  *
  *  \details Details
  */
   void DbusServiceProxy::onAvailable(const boost::shared_ptr < Proxy >& proxy,
                                      const ServiceStateChange         & stateChange){
      ETG_TRACE_USR1( ( "onAvailable ..........." ) );
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);

//      ahl_bEnterCritical(_hProxyListAccess);
      AsfServiceProxyMap::iterator iterServicePos;
      iterServicePos = _serviceProxyMap.find(proxy);
      if( iterServicePos != _serviceProxyMap.end() ){
         ETG_TRACE_USR1( ( "  - %s ServiceProxy is connected", iterServicePos->second->getServiceName().c_str() ) );
         ETG_TRACE_USR1( ( "  - register attributes and events" ) );

         boost::shared_ptr < ServiceProxy > serviceProxy = iterServicePos->second->getProxy();
         iterServicePos->second->_u32ExecMainPIDRegisterId = (tU32)serviceProxy->sendExecMainPIDRegister(* this);
         serviceProxy->sendExecMainPIDGet(* this);
      }
//      ahl_bReleaseCritical(_hProxyListAccess);
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   /**
     *  \brief called if any service we are registered too is removed from DBus-Daemon
     *
     *  \param [in] proxy pointer to the proxy where the service state of the stub has been changed
     *  \param [in] stateChange structure containing the precvious and current service state
     *  \return void
     *
     *  \details Handles all actions to be taken when a service becomes unavailable.
     *  ATTENTION: This should never happen
     */
   void DbusServiceProxy::onUnavailable(const boost::shared_ptr < Proxy >& proxy,
                                        const ServiceStateChange         & stateChange){
         ETG_TRACE_USR1( ( "onUnavailable ..........." ) );
         LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);

//      ahl_bEnterCritical(_hProxyListAccess);
      AsfServiceProxyMap::const_iterator iterServicePos;
      iterServicePos = _serviceProxyMap.find(proxy);
      if( iterServicePos != _serviceProxyMap.end() ){
         ETG_TRACE_USR1( ( "serviceProxy %s is disconnected", iterServicePos->second->getServiceName().c_str() ) );
         boost::shared_ptr < ServiceProxy > serviceProxy = iterServicePos->second->getProxy();
         serviceProxy->sendExecMainPIDDeregister(iterServicePos->second->_u32ExecMainPIDRegisterId);
      }
//      ahl_bReleaseCritical(_hProxyListAccess);
   }        // onUnavailable

#endif      // #if 1 // allow editor to collapse this section

// ######################################
// ServiceProxy implementations
// ######################################
#if 1       // allow editor to collapse this section

/**
  *  \brief called when the GetMainPid message is answered
  *
  *  \param [in] proxy pointer to the proxy where the service is now available
  *  \param [in] update - contains the MainPid of the service file
  *  \return none
  *  \todo sometimes, especially when "sh" is called instead of starting the *.out file directly
  *        returned PID is 0.
  *        - retry?
  *        - use other interface?
  *
  *  \details Details
  */
   void DbusServiceProxy::onExecMainPIDUpdate(const ::boost::shared_ptr < ServiceProxy >     & proxy,
                                              const ::boost::shared_ptr < ExecMainPIDUpdate >& update){
         ETG_TRACE_USR1( ( "DbusServiceProxy::onExecMainPIDUpdate(): Received ProcessID" ) );
//      ahl_bEnterCritical(_hProxyListAccess);
      AsfServiceProxyMap::const_iterator iterServicePos;
      iterServicePos = _serviceProxyMap.find(proxy);
      if( iterServicePos != _serviceProxyMap.end() ){
         ETG_TRACE_USR1( ( "  - %40s - ProcessID = %d",
                           iterServicePos->second->getServiceName().c_str(),
                           update->getExecMainPID() ) );
         iterServicePos->second->_u32ExecMainPID = update->getExecMainPID();

         std::string  strSwBlock = _oAppCfg.strGetNameByServiceName(iterServicePos->second->getServiceName());
         // send the process ID back to LCM-Early
         Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
         LCM_NULL_POINTER_CHECK(poServerRef);
         poServerRef->sendNewProcessIdStatus( strSwBlock.c_str(), iterServicePos->second->getServiceName().c_str(), iterServicePos->second->_u32ExecMainPID);
      } else {
            ETG_TRACE_ERRMEM( ( "  No client found " ) );
      }
//      ahl_bReleaseCritical(_hProxyListAccess);
   }        // onExecMainPIDUpdate

   void DbusServiceProxy::onExecMainPIDError(const ::boost::shared_ptr < ServiceProxy >    & proxy,
                                             const ::boost::shared_ptr < ExecMainPIDError >& error){
            ETG_TRACE_ERRMEM( ( "Received DbusServiceProxy::onExecMainPIDError" ) );
//      ahl_bEnterCritical(_hProxyListAccess);
      AsfServiceProxyMap::const_iterator iterServicePos;
      iterServicePos = _serviceProxyMap.find(proxy);
      if( iterServicePos != _serviceProxyMap.end() ){
         if(error->hasName() && error->hasMessage() ){
            std::string ErrMessage = error->getName() + ":" + error->getMessage();
            ETG_TRACE_ERRMEM( ( "  - %40s - %s",
                                iterServicePos->second->getServiceName().c_str(),
                                ErrMessage.c_str() ) );
         } else {
            ETG_TRACE_ERRMEM( ( "  - %40s - no error message found", iterServicePos->second->getServiceName().c_str() ) );
         }
      } else {
            ETG_TRACE_ERRMEM( ( "  No client found " ) );
      }
//            ahl_bReleaseCritical(_hProxyListAccess);
   }        // onExecMainPIDError

#endif      // #if 1 // allow editor to collapse this section

}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

