/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     DbusProxy.cpp
  * \brief    Implementation of the DbusProxy component
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  * \todo implement the ASF callbacks "correctly" to also print content of parameter to remove LINT
  *
  * \par Copyright:
  * (c) 2013-2015 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
// include for etg support lib
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LCM_SERVER_LCMDBUS
#include "trcGenProj/Header/DbusProxy.cpp.trc.h"
#endif

#include <sstream>
#include <boost/algorithm/string.hpp>

#include "DbusProxy.h"
#include "asf/core.h"

#include "I_lcm_ServiceLcmDbusIf.h"
#include "ILifeCycleConsumerProxy.h"

// interfaces class definitions
#include <systemd/sd-daemon.h>

namespace org {
namespace bosch {
namespace cm {
namespace lcm {

using namespace ::asf::core;

DEFINE_CLASS_LOGGER_AND_LEVEL("org/bosch/cm/lcm/DbusProxy", DbusProxy, Info);

/**
  *  \brief sets the service name to a given Job
  *
  *  \param [in] strServiceName name of the service
  *  \param [in] strJobName name of the job to set the service name for
  *
  *  \details unfortunately the StartUnit responce only gives the job, and the service is only known
  *      when NewJob is received, so the service name must be set later
  */
tBool SpmJobList::bSetServiceName(const std::string& strServiceName,
                                  const std::string& strJobName){
   tBool                 bFound = false;
   TSpmJobList::iterator iterJobListPos;

   for(iterJobListPos = _lJobList.begin();
       iterJobListPos != _lJobList.end();
       iterJobListPos++){
      if(iterJobListPos->getJobName() == strJobName){
         iterJobListPos->vSetServiceName(strServiceName);
         bFound = true;
         continue;
      }
   }

   return( bFound );
}           // SpmJobList::bSetServiceName

/**
  *  \brief sets the state of a job to started
  *
  *  \param [in] strJobName name of the job
  */
tVoid SpmJobList::vSetJobStarted(const std::string& strJobName){
   TSpmJobList::iterator iterJobListPos;

   for(iterJobListPos = _lJobList.begin();
       iterJobListPos != _lJobList.end();
       iterJobListPos++){
      if(iterJobListPos->getJobName() == strJobName){
         iterJobListPos->vSetJobStarted();
         continue;
      }
   }
}

/**
  *  \brief sets the state of a job to started
  *
  *  \param [in] strJobName name of the job
  */
tVoid SpmJobList::vSetJobDone(const std::string& strJobName){
   TSpmJobList::iterator iterJobListPos;

   for(iterJobListPos = _lJobList.begin();
       iterJobListPos != _lJobList.end();
       iterJobListPos++){
      if(iterJobListPos->getJobName() == strJobName){
         iterJobListPos->vSetJobDone();
         continue;
      }
   }
}

/**
  *  \brief sets the state of a job to error (e.g. JobNew/JobRemoved) returned an error
  *
  *  \param [in] strJobName name of the job
  */
tVoid SpmJobList::vSetJobError(const std::string& strJobName){
   TSpmJobList::iterator iterJobListPos;

   for(iterJobListPos = _lJobList.begin();
       iterJobListPos != _lJobList.end();
       iterJobListPos++){
      if(iterJobListPos->getJobName() == strJobName){
         iterJobListPos->vSetJobError();
         continue;
      }
   }
}

/**
  *  \brief get the service name related to the job name
  *
  *  \param [in] strServiceName name of the service
  *  \param [in] strJobName name of the job
  *
  *  \details adds a new job to the list
  */
tVoid SpmJobList::vAddJob(const std::string& strServiceName,
                          const std::string& strJobName){
   SpmJobElement oSpmJobElement(strServiceName, strJobName);

   _lJobList.push_back(oSpmJobElement);
}

/**
  *  \brief get the service name related to the job name
  *
  *  \param [in] strServiceName name of the service
  *
  *  \details if the job is not found "not found" will be returned
  */
std::string SpmJobList::strGetJobNameByServiceName(const std::string& strServiceName){
   std::string                 strJobName = "not found";
   TSpmJobList::const_iterator cIterJobListPos;

   for(cIterJobListPos = _lJobList.begin();
       cIterJobListPos != _lJobList.end();
       cIterJobListPos++){
      SpmJobElement oSpmJobElement = * cIterJobListPos;
      if(oSpmJobElement.getServiceName() == strServiceName){
         strJobName = oSpmJobElement.getJobName();
         continue;
      }
   }

   return( strJobName );
}           // SpmJobList::strGetJobNameByServiceName

/**
  *  \brief get the job name related to the service name
  *
  *  \param [in] strJobName name of the job
  *
  *  \details if the job is not found "not found" will be returned
  */
std::string SpmJobList::strGetServiceNameByJobName(const std::string& strJobName){
   std::string                 strServiceName = "not found";
   TSpmJobList::const_iterator cIterJobListPos;

   for(cIterJobListPos = _lJobList.begin();
       cIterJobListPos != _lJobList.end();
       cIterJobListPos++){
      SpmJobElement oSpmJobElement = * cIterJobListPos;
      if(oSpmJobElement.getJobName() == strJobName){
         strServiceName = oSpmJobElement.getServiceName();
         continue;
      }
   }

   return( strServiceName );
}           // SpmJobList::strGetServiceNameByJobName

/**
  *  \brief constructor of the class DBusProxy
  *
  *  \param [in] factory reference to the LCM-factory to get the references to other LCM classes
  *  \return class is created
  *
  *  \details constructs all stubs and permanent proxies. it checks if DBUS adress is defined
  */
DbusProxy::DbusProxy(lcm_tclAppMain *poMainAppl)
   : IDbusProxy(poMainAppl){

   ETG_TRACE_USR1( ( "constructor for DbusProxy called" ) );

   _posystemd_interface = new systemd_interface(this, CCA_C_U16_APP_SPM_SLV);
   LCM_NULL_POINTER_CHECK(_posystemd_interface);
   _posystemd_interface->systemd_notify_ready();
}

/**
  *  \brief Destructor of class
  *
  *  \return none
  *
  *  \details remove all dynamically created clients
  */
DbusProxy::~DbusProxy(){
   ETG_TRACE_USR1( ( "~DbusProxy called" ) );

   delete _posystemd_interface;
}

/**
  *  \brief start working
  *
  *  \return
  *
  *  \details reads list of units that are allowed to fail from registry
  *
  *  \todo add reading of failed list
  */
void DbusProxy::vStartCommunication(){
}           // vStartCommunication

/**
  *  \brief Periodic systemd watchdog callback.
  *
  *  \return TRUE  = systemd watchdog is acknowledged
  *          FALSE = systemd watchdog is NOT acknowledged
  *
  *  \details This method is periodically called and must return TRUE to acknowldge the systemd watchdog.
  *           This method shall add as many checks as possible to verify that all SW parts of the host process (LCM-late) are running and work as expected.
  *           If the checks fail this method shall return FALSE.
  */
bool DbusProxy::OnAppWatchdog(){
   // For the moment unconditionally return TRUE.
   return( true );
}

/**
  *  \brief function will send any of the LCM application state request via DBus interfaces
  *
  *  \param [in] u32AppId internal LCM applicationID of the application
  *  \param [in] u16PowerType type of CCA message
  *  \param [in] u32PowerData1 Data to be send
  *  \param [in] u32PowerData2 Data to be send
  *  \return none
  *
  *  \details wrapps the sending of application requests for the DBus framework for all applications using the NodeStateManager interfaces.
  *           It sends out all the application state change requests
  */
void DbusProxy::vPostLcmRequest(tU32 u32AppId,
                                tU16 u16PowerType,
                                tU32 u32PowerData1,
                                tU32 u32PowerData2){

   Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
   (tVoid)u32PowerData2;
      LCM_NULL_POINTER_CHECK(poServerRef);

   if (u16PowerType == AMT_C_U16_PWR_PROXY_START_CONF){
      // start confirmation is implicitly done by creating consumer object --> init is done when onAvailable for this proxy is called
   } else if (u16PowerType == AMT_C_U16_PWR_PROXY_END_APP){
      // send ack directly back to LAM
      poServerRef->sendAppEndStatus(u32AppId);
   } else if (u16PowerType == AMT_C_U16_PWR_PROXY_STATE_CHANGE_REQ){
      ILifeCycleConsumerProxy *_poILifeCycleConsumerProxy = dynamic_cast < ILifeCycleConsumerProxy* >( _cpoMain->getHandler("ILifeCycleConsumerProxy") );
      LCM_NULL_POINTER_CHECK(_poILifeCycleConsumerProxy);
      _poILifeCycleConsumerProxy->SendAppStateChangeRequest(u32AppId, u32PowerData1);
   }
}           //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

/**
  *  \brief trace the current and previous state of availabliltiy
  *
  *  \param [in] stateChange - state object passed by ASF
  *  \return none
  */
void DbusProxy::vTraceAvailability(const ServiceStateChange& stateChange){
   std::string  StateTransitionString("undefined->");
   ServiceState PrevState     = stateChange.getPreviousState();
   const char  *PrevStateName = ServiceState_Name(PrevState);

   if(PrevStateName != NULL){
      StateTransitionString = std::string(PrevStateName) + "->";
   } else {
      // nothing to do, already set to "undefined"
   }
   ServiceState CurrState     = stateChange.getCurrentState();
   const char  *CurrStateName = ServiceState_Name(CurrState);
   if(CurrStateName != NULL){
      StateTransitionString += std::string(CurrStateName);
   } else {
      StateTransitionString += "undefined";
   }
   ETG_TRACE_USR1( ( "state changed from %s", StateTransitionString.c_str() ) );
}           // DbusProxy::vTraceAvailability

/*! \fn
  *  \brief function to escape the unit names
  *    specified in
  *    copied from systemd sources at: systemd-217\src\shared\unit-name.c
  *    see also here: http://0pointer.de/blog/projects/instances.html
  *  \param UnitName original name of the unit
  *  \exception none.
  *  \return new escaped unitname
  */
std::string DbusProxy::unitNameEscape(const std::string UnitName){
   std::locale loc;
   std::string EscapedUnitName;

   for (std::string::const_iterator cit = UnitName.begin(); cit != UnitName.end(); ++cit){
      if (std::isalnum(* cit, loc) ){
         EscapedUnitName += * cit;
      } else {
         int               character = * cit;
         std::stringstream strstream;
         strstream << "_" << std::hex << character;
         EscapedUnitName += strstream.str( );
      }
   }
   return( EscapedUnitName );
}           // unitNameEscape

/**
  *  \brief extracts the applicationname out of the object path
  *
  *  \param [in] strObjectName object path given by registration message
  *  \return string containing the evaluated application name
  *  \todo rework this function to use std::string only, and no char* anymore.
  *        find C++11 stdlib function for replacement boost::split, boost::iterator_range, boost::iequals
  *
  *  \details this function assumes that the object path contains the application name is any of those two places
  *  1. /org/genivi/NodeStateManager/LifeCycleConsumer/<BOSCH naming convention>
  *    e.g. /org/genivi/NodeStateManager/LifeCycleConsumer/ExampleApplication
  *  2. /org/genivi/<assumed GENIVI naming convention>/LifeCycleConsumer
  *    e.g. /org/genivi/audiomanager/LifeCycleConsumer
  */
std::string DbusProxy::ExtractApplicationName(const std::string strObjectName){
   std::string LifeCycleConsumer = "LifeCycleConsumer";
   std::string Seperator         = "/";

   // if no string LifeCycleConsumer (case insensitive) is found just return the complete name
   typedef const boost::iterator_range < std::string::const_iterator > StringRange;
   StringRange range             = boost::algorithm::ifind_first(strObjectName, LifeCycleConsumer);
   if (range.empty() ){
      return( strObjectName );
   }

   // if no string / is found just return the complete name
   if (strObjectName.find(Seperator) == std::string::npos){
      return( strObjectName );
   }

   // return the first substring (split by /) that is not LifeCycleConsumer
   std::vector < std::string >                         strs;
   boost::split(strs, strObjectName, boost::is_any_of(Seperator) );
   std::vector < std::string >::const_reverse_iterator crit;
   for( crit = strs.rbegin();
        crit != strs.rend();
        ++crit){
      if ( !boost::iequals( ( * crit ), LifeCycleConsumer) ){
         if( ( * crit ).length() != 0){
            return ( * crit );
         }
      }
   }
   return( strObjectName );
}           // ExtractApplicationName

}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

