/****************************************************************************
  * Copyright (C) Robert Bosch Car Multimedia GmbH, 2013
  * This software is property of Robert Bosch GmbH. Unauthorized
  * duplication and disclosure to third parties is prohibited.
  ***************************************************************************/

/*!
  * \file     DbusManagerProxy.cpp
  * \brief    Implementation of the DbusManagerProxy interface
  *
  * \author   klaus-peter.kollai@de.bosch.com CM-AI/PJ-CB32
  *
  * \par Copyright:
  * (c) 2013-2015 Robert Bosch Car Multimedia GmbH
  ***************************************************************************/
// include for etg support lib
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"
#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_LCM_SERVER_LCMDBUS
#include "trcGenProj/Header/DbusManagerProxy.cpp.trc.h"
#endif

#include "IDbusProxy.h"
#include "IDbusServiceProxy.h"
#ifdef VARIANT_S_FTR_ENABLE_DBUS_PROPERTY
#include "IDbusPropertiesProxy.h"
#endif
#include "IDbusUnitProxy.h"
#include "ILcmRecoveryClientStub.h"
#include "I_lcm_ServiceLcmDbusIf.h"

#include "DbusManagerProxy.h"
#include <boost/algorithm/string.hpp>
//  boost::shared_ptr is used here cause by ASF interface.
namespace org {
namespace bosch {
namespace cm {
namespace lcm {

using namespace ::asf::core;
using namespace ::org::freedesktop::systemd1::Manager;

#define SPM_REG_KEY_SUPERVISION                 "SUPERVISION"
#define SPM_REG_KEY_SUPERVISION_SYSTEMD_FAILED_UNIT_WHITE_LIST "SYSTEMD_FAILED_UNIT_WHITE_LIST"

DEFINE_CLASS_LOGGER_AND_LEVEL("org/bosch/cm/lcm/DbusManagerProxy", DbusManagerProxy, Info);

/**
  *  \brief constructor of the class DBusProxy
  *
  *  \param [in] factory reference to the LCM-factory to get the references to other LCM classes
  *  \return class is created
  *
  *  \details constructs all stubs and permanent proxies. it checks if DBUS adress is defined
  */
DbusManagerProxy::DbusManagerProxy(lcm_tclAppMain *poMainAppl)
   : IDbusManagerProxy(poMainAppl),
   _JobMode("fail"),
   _managerProxy(ManagerProxy::createProxy( "DBusSystemdManagerPort", * this) ){
   ETG_TRACE_USR1( ( "DbusManagerProxy called" ) );

   _ServiceFailedEntryList.tServices.clear();
   _SystemdUnitFailedWhitelistSet.clear();
}

/**
  *  \brief Destructor of class
  *
  *  \return none
  *
  *  \details remove all dynamically created clients
  */
DbusManagerProxy::~DbusManagerProxy(){
   ETG_TRACE_USR1( ( "~DbusManagerProxy called" ) );
}

/**
  *  \brief start working
  *  \todo find C++11 stdlib function for replacement boost::split & boots::is_any_of
  *  \return
  *
  *  \details currently empty
  */
void DbusManagerProxy::vStartCommunication(){
   _poIDbusProxy = dynamic_cast < IDbusProxy* >( _cpoMain->getHandler("IDbusProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusProxy);

   tChar szSystemdFailedUnitWhiteList[2048] = "\0";

   if (!scd_bGetAppConfigurationString(CCA_C_U16_APP_SPM, // CCA AppID
                                       SPM_REG_KEY_SUPERVISION,
                                       SPM_REG_KEY_SUPERVISION_SYSTEMD_FAILED_UNIT_WHITE_LIST,
                                       szSystemdFailedUnitWhiteList, sizeof( szSystemdFailedUnitWhiteList ) ) ){
      ETG_TRACE_USR4( ( "vStartCommunication() -> No white list registry key found. No filtering of results." ) );
   } else {
      ETG_TRACE_USR4( ( "vStartCommunication() found whitelist in registry: %s", szSystemdFailedUnitWhiteList ) );
      boost::split(_SystemdUnitFailedWhitelistSet,
                   szSystemdFailedUnitWhiteList,
                   boost::is_any_of(",") );
      ETG_TRACE_USR4( ( "vStartCommunication(): Added %u services to failed white list", (tUInt)_SystemdUnitFailedWhitelistSet.size() ) );
   }
}           // vStartCommunication

/**
  *  \brief send the DBus message to systemd to get the list of units
  *
  *  \return void
  *
  *  \details get the list of currently known units to see if any of them is in a "wrong" state.
  *  Response is handled in onListUnitsResponse
  */
void DbusManagerProxy::vTriggerGetListUnit(){
   ETG_TRACE_USR1( ( "DbusManagerProxy::vTriggerGetListUnit" ) );
   _ServiceFailedEntryList.tServices.clear();
   _managerProxy->sendListUnitsRequest(* this);

   Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
   LCM_NULL_POINTER_CHECK(poServerRef);
   poServerRef->sendTriggerGetUnitListMResult();
}

/**
  *  \brief start a systemd unit
  *
  *  \param [in] cUnitName name of the unit
  *  \return none
  *
  *  \details sends the start request of the unit to systemd via manager interface and creates the unit and service proxies to interact with the newly started unit
  */
void DbusManagerProxy::vStartUnit(const char *cUnitName){

   std::string strUnitName = cUnitName;

   LCM_NULL_POINTER_CHECK(_poIDbusProxy);

   ETG_TRACE_USR1( ( "StartUnit %s", strUnitName.c_str() ) );
   _managerProxy->sendStartUnitRequest(* this, strUnitName, _JobMode);
   _managerProxy->sendGetUnitRequest(* this, strUnitName);
   ETG_TRACE_USR1( ( "add UnitProxy for %s", strUnitName.c_str() ) );

   std::string              strEscapedUnitName      = _poIDbusProxy->unitNameEscape(strUnitName);
   std::string              strObjectPath           = "/org/freedesktop/systemd1/unit/" + strEscapedUnitName;

   IDbusServiceProxy       *_poIDbusServiceProxy    = dynamic_cast < IDbusServiceProxy* >( _cpoMain->getHandler("IDbusServiceProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusServiceProxy);
   _poIDbusServiceProxy->vAddServiceProxy(strUnitName, strObjectPath);

   IDbusUnitProxy          *_poIDbusUnitProxy       = dynamic_cast < IDbusUnitProxy* >( _cpoMain->getHandler("IDbusUnitProxy") );
   LCM_NULL_POINTER_CHECK(_poIDbusUnitProxy);
   _poIDbusUnitProxy->vAddUnitProxy(strUnitName, strObjectPath);

   #ifdef VARIANT_S_FTR_ENABLE_DBUS_PROPERTY
      IDbusPropertiesProxy *_poIDbusPropertiesProxy = dynamic_cast < IDbusPropertiesProxy* >( _cpoMain->getHandler("IDbusPropertiesProxy") );
      LCM_NULL_POINTER_CHECK(_poIDbusPropertiesProxy);
      _poIDbusPropertiesProxy->vAddPropertiesProxy(strUnitName, strObjectPath);
   #endif
}           // StartUnit

/**
  *  \brief stops a systemd unit
  *
  *  \param [in] cUnitName name of the unit
  *  \return none
  *
  *  \details sends the stop request of the unit to systemd via manager interface
  */
void DbusManagerProxy::vStopUnit(const char *cUnitName){

   std::string             strUnitName               = cUnitName;

   ETG_TRACE_USR1( ( "StopUnit %s", strUnitName.c_str() ) );
   _managerProxy->sendStopUnitRequest(* this, strUnitName, _JobMode);
   ILcmRecoveryClientStub *_poILcmRecoveryClientStub = dynamic_cast < ILcmRecoveryClientStub* >( _cpoMain->getHandler("ILcmRecoveryClientStub") );
   LCM_NULL_POINTER_CHECK(_poILcmRecoveryClientStub);
   _poILcmRecoveryClientStub->vUnitStopped(strUnitName);
}           // vStopUnit
/**
  *  \brief restart a systemd unit
  *
  *  \param [in] cUnitName name of the unit
  *  \return none
  *
  *  \details sends the restart request of the unit to systemd via manager interface
  */
void DbusManagerProxy::vRestartUnit(const char *cUnitName){

   std::string             strUnitName               = cUnitName;

   ETG_TRACE_USR1( ( "RestartUnit %s", strUnitName.c_str() ) );
   _managerProxy->sendRestartUnitRequest(* this, strUnitName, _JobMode);
}// vRestartUnit

// ######################################
// ASF-Core implementations
// ######################################
#if 1       // allow editor to collapse this section
// ServiceAvailableIF implementation

/**
  *  \brief handles all service available indications by ASF
  *
  *  \param [in] proxy pointer to the proxy where the service is now available
  *  \param [in] stateChange contains the previous and new state of the service
  *  \return Return_Description
  *
  *  \details Details
  */
   void DbusManagerProxy::onAvailable(const boost::shared_ptr < Proxy >& proxy,
                                      const ServiceStateChange         & stateChange){
      ETG_TRACE_USR1( ( "onAvailable ..........." ) );
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "  - ManagerProxy is connected" ) );
         ETG_TRACE_USR1( ( "  - register attributes and events" ) );
         #ifdef VARIANT_S_FTR_ENABLE_DBUS_PROPERTY
            // activate the DbusProperty feature in systemd
            _asfManagerProxy._u32SubscribeRegisterId    = _managerProxy->sendSubscribeRequest(* this);
         #endif
         _asfManagerProxy._u32JobNewRegisterId          = (tU32)_managerProxy->sendJobNewRegister(* this);
         _asfManagerProxy._u32JobRemovedRegisterId      = (tU32)_managerProxy->sendJobRemovedRegister(* this);
         _asfManagerProxy._u32StartupFinishedRegisterId = (tU32)_managerProxy->sendStartupFinishedRegister(* this);
         _asfManagerProxy._u32VersionRegisterId         = (tU32)_managerProxy->sendVersionRegister(* this);
      }
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   /**
     *  \brief called if any service we are registered too is removed from DBus-Daemon
     *
     *  \param [in] proxy pointer to the proxy where the service state of the stub has been changed
     *  \param [in] stateChange structure containing the precvious and current service state
     *  \return void
     *
     *  \details Handles all actions to be taken when a service becomes unavailable.
     *  ATTENTION: This should never happen
     */
   void DbusManagerProxy::onUnavailable(const boost::shared_ptr < Proxy >& proxy,
                                        const ServiceStateChange         & stateChange){
      ETG_TRACE_USR1( ( "onUnavailable ..........." ) );
      LCM_NULL_POINTER_CHECK(_poIDbusProxy);
      _poIDbusProxy->vTraceAvailability(stateChange);

      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "managerProxy is disconnected" ) );
         _managerProxy->sendJobNewDeregister(_asfManagerProxy._u32JobNewRegisterId);
         _managerProxy->sendJobRemovedDeregister(_asfManagerProxy._u32JobRemovedRegisterId);
         _managerProxy->sendStartupFinishedDeregister(_asfManagerProxy._u32StartupFinishedRegisterId);
         _managerProxy->sendVersionDeregister(_asfManagerProxy._u32VersionRegisterId);
      }
   }        // onUnavailable

#endif      // #if 1 // allow editor to collapse this section

// ######################################
// ManagerProxy implementations
// ######################################
#if 1       // allow editor to collapse this section
// JobNewSignalCallbackIF implementation

   /**
     *  \brief a new job was successfully added to systemd
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] signal content of message from systemd
     *  \return void
     *
     *  \details when a new unit is started via StartUnit message a new job is created in systemd.
     *  This is signaled by this message.
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onJobNewSignal(const ::boost::shared_ptr < ManagerProxy >& proxy,
                                         const ::boost::shared_ptr < JobNewSignal >& signal){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onJobNewSignal: %s", signal->getJob().c_str() ) );
         ETG_TRACE_USR2( ( "Id                     : %d", signal->getId() ) );
         ETG_TRACE_USR2( ( "unit                   : %s", signal->getUnit().c_str() ) );
         LCM_NULL_POINTER_CHECK(_poIDbusProxy);
         _poIDbusProxy->vSetJobStarted(signal->getJob() );
         if (false == _poIDbusProxy->bSetServiceName(signal->getUnit(), signal->getJob() ) ){
            ETG_TRACE_USR1( ( "could not find job %s in JobList", signal->getUnit().c_str() ) );
         }
      }
   }

   /**
     *  \brief a new job was added to systemd but failed
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details when a new unit is started via StartUnit message a new job is created in systemd.
     *  This is signaled by this message.
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onJobNewError(const ::boost::shared_ptr < ManagerProxy >& proxy,
                                        const ::boost::shared_ptr < JobNewError > & error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onJobNewError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
//         _poIDbusProxy->vSetJobError(error->getJob());
      }
   }

// JobRemovedSignalCallbackIF implementation

   /**
     *  \brief a job of systemd has been removed
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] signal content of message from systemd
     *  \return void
     *
     *  \details when a new unit is started via StartUnit message a new job is created in systemd.
     *  If this job is finished this signal is send.
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     * if the job is finished now the PID is available, so the trigger is send to get it.
     */
   void DbusManagerProxy::onJobRemovedSignal(const boost::shared_ptr < ManagerProxy >      & proxy,
                                             const ::boost::shared_ptr < JobRemovedSignal >& signal){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onJobRemovedSignal for job: %s", signal->getJob().c_str() ) );
         ETG_TRACE_USR2( ( "Id                                 : %d", signal->getId() ) );
         ETG_TRACE_USR2( ( "unit                               : %s", signal->getUnit().c_str() ) );
         ETG_TRACE_USR2( ( "result                             : %s", signal->getResult().c_str() ) );
         // see http://www.freedesktop.org/wiki/Software/systemd/dbus/
         if( ( signal->getResult() == "timeout" )
             || ( signal->getResult() == "failed" )
             || ( signal->getResult() == "dependency" )
             ){
            ETG_TRACE_ERRMEM( ( "Error for job: %s", signal->getJob().c_str() ) );
            ETG_TRACE_ERRMEM( ( "Id    : %d", signal->getId() ) );
            ETG_TRACE_ERRMEM( ( "unit  : %s", signal->getUnit().c_str() ) );
            ETG_TRACE_ERRMEM( ( "result: %s", signal->getResult().c_str() ) );
         }
         LCM_NULL_POINTER_CHECK(_poIDbusProxy);
         _poIDbusProxy->vSetJobDone(signal->getJob() );
         IDbusServiceProxy *poServiceProxy = dynamic_cast < IDbusServiceProxy* >( _cpoMain->getHandler("IDbusServiceProxy") );
         LCM_NULL_POINTER_CHECK(poServiceProxy);
         poServiceProxy->vTriggerGetMainPid(_poIDbusProxy->strGetServiceNameByJobName(signal->getJob() ) );
      }
   }        // onJobRemovedSignal

   /**
     *  \brief a job was removed from systemd with an error
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details I have no idea when this could happen
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onJobRemovedError(const boost::shared_ptr < ManagerProxy >     & proxy,
                                            const ::boost::shared_ptr < JobRemovedError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onJobRemovedError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
//         _poIDbusProxy->vSetJobError(error->getJob());
      }
   }

// GetUnitCallbackIF implementation

   /**
     *  \brief GetUnit was send and this is the response
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] response content of message from systemd
     *  \return void
     *
     *  \details This is used to get all information about the new unit that was started by LCM
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onGetUnitResponse(const ::boost::shared_ptr < ManagerProxy >   & proxy,
                                            const ::boost::shared_ptr < GetUnitResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onGetUnitResponse: %s", response->getUnit().c_str() ) );
      }
   }

   /**
     *  \brief GetUnit was send and this is the response
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details The GetUnit call failed.
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onGetUnitError(const ::boost::shared_ptr < ManagerProxy >& proxy,
                                         const ::boost::shared_ptr < GetUnitError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onGetUnitError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }

// StartUnitCallbackIF implementation

   /**
     *  \brief StartUnit was send and this is the response
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] response content of message from systemd
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onStartUnitResponse(const ::boost::shared_ptr < ManagerProxy >     & proxy,
                                              const ::boost::shared_ptr < StartUnitResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onStartUnitResponse: %s", response->getJob().c_str() ) );
         LCM_NULL_POINTER_CHECK(_poIDbusProxy);
         _poIDbusProxy->vAddJob("unknown", response->getJob() );
         // results can be  done, canceled, timeout, failed, dependency, skipped - see src/core/job.c
      }
   }

   /**
     *  \brief StartUnit was send and somehow failed
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onStartUnitError(const ::boost::shared_ptr < ManagerProxy >  & proxy,
                                           const ::boost::shared_ptr < StartUnitError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERRMEM( ( "Received onStartUnitError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERRMEM( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERRMEM( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }

// StopUnitMethodCallbackIF implementation

   /**
     *  \brief StopUnit was send and this is the response
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] response content of message from systemd
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onStopUnitResponse(const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                             const ::boost::shared_ptr < StopUnitResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onStopUnitReply: %s", response->getJob().c_str() ) );
      }
   }

   /**
     *  \brief StopUnit was send and somehow failed
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onStopUnitError(const ::boost::shared_ptr < ManagerProxy > & proxy,
                                          const ::boost::shared_ptr < StopUnitError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERRMEM( ( "Received onStopUnitError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERRMEM( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERRMEM( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }
// RestartUnitMethodCallbackIF implementation

   /**
     *  \brief RestartUnit was send and this is the response
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] response content of message from systemd
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onRestartUnitResponse(const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                             const ::boost::shared_ptr < RestartUnitResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onRestartUnitReply: %s", response->getJob().c_str() ) );
      }
   }

   /**
     *  \brief RestartUnit was send and somehow failed
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] error content of message from systemd / DBusDaemon
     *  \return void
     *
     *  \details There are no really any usefull information except JobId
     *  see http://www.freedesktop.org/wiki/Software/systemd/dbus/
     */
   void DbusManagerProxy::onRestartUnitError(const ::boost::shared_ptr < ManagerProxy > & proxy,
                                          const ::boost::shared_ptr < RestartUnitError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERRMEM( ( "Received onRestartUnitError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERRMEM( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERRMEM( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }
// ListJobsMethodCallbackIF implementation

   /**
     *  \brief response to send
     *
     *  \param [in] proxy pointer to managerproxy
     *  \param [in] response list of jobs and their states
     *  \return Return_Description
     *
     *  \details can be used to find out which jobs are currently running and what state they are in
     */
   void DbusManagerProxy::onListJobsResponse(const ::boost::shared_ptr < ManagerProxy >    & proxy,
                                             const ::boost::shared_ptr < ListJobsResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onListJobsResponse" ) );
         ::std::vector < jobs >::const_iterator iterJobs;
         for(iterJobs = response->getJobs().begin();
             iterJobs != response->getJobs().end();
             ++iterJobs){
            jobs Job = * iterJobs;
            ETG_TRACE_USR1( ( "Found JobID: %d", Job.getJobId() ) );
         }
      }
   }        //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

   void DbusManagerProxy::onListJobsError(const ::boost::shared_ptr < ManagerProxy > & proxy,
                                          const ::boost::shared_ptr < ListJobsError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onListJobsError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }

// VersionCallbackIF
   void DbusManagerProxy::onVersionUpdate(const ::boost::shared_ptr < ManagerProxy > & proxy,
                                          const ::boost::shared_ptr < VersionUpdate >& update){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onVersionUpdate: %s", update->getVersion().c_str() ) );
      }
   }

   void DbusManagerProxy::onVersionError(const ::boost::shared_ptr < ManagerProxy >& proxy,
                                         const ::boost::shared_ptr < VersionError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onVersionError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }

   // ListUnitsCallbackIF

   /**
     *  \brief handles the response of sendListUnitsRequest. Gets the list of all systemd unit states
     *
     *  \param [in] proxy pointer to the used proxy -> should be managerProxy
     *  \param [in] response list of systemd units and it (sub)states
     *  \return void
     *  \details evaluates the response of the list of all units and their states, gets the sublist
     *      of units that are in "failed" state and compares it with
     *      the list of units from registry that are allowed to be in these states.
     *      the remaining units are saved in ErrMem to analyse why they failed
     */
   void DbusManagerProxy::onListUnitsResponse(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >     & proxy,
                                              const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ListUnitsResponse >& response){
      if(_managerProxy == proxy){
         ETG_TRACE_USR1( ( "Received onListUnitsResponse" ) );
         std::vector < units >::const_iterator iterUnits;
         for(iterUnits = response->getUnits().begin();
             iterUnits != response->getUnits().end();
             ++iterUnits){
            units Unit = * iterUnits;
            if( _SystemdUnitFailedWhitelistSet.find(Unit.getName() ) == _SystemdUnitFailedWhitelistSet.end() ){
               if( ( 0 == Unit.getLoadState().compare("not-found") )
                        // check for unwanted ActiveStates
                   || ( 0 == Unit.getActiveState().compare("activating") )
                   || ( 0 == Unit.getActiveState().compare("deactivating") )
                   || ( 0 == Unit.getActiveState().compare("failed") )
                   || ( 0 == Unit.getActiveState().compare("stopped") )

                        /* this seens to show too many units
                           // check for unwanted SubStates
                          || (0 == Unit.getSubState().compare("dead") )
                          || (0 == Unit.getSubState().compare("exited") )
                          */
                   ){
                  lcm_fi_tcl_ServiceElement ServiceFailedEntry;
                  ServiceFailedEntry.tUnitName    = Unit.getName().c_str();
                  ServiceFailedEntry.tDescription = Unit.getDescription().c_str();
                  ServiceFailedEntry.tLoadState   = Unit.getLoadState().c_str();
                  ServiceFailedEntry.tActiveState = Unit.getActiveState().c_str();
                  ServiceFailedEntry.tSubState    = Unit.getSubState().c_str();
                  ServiceFailedEntry.tFollower    = Unit.getFollower().c_str();
                  _ServiceFailedEntryList.tServices.push_back(ServiceFailedEntry);
                  ETG_TRACE_USR1( ( "Found (%u) failing/incorrect unit(s) : %s",
                                    (tUInt)_ServiceFailedEntryList.tServices.size(),
                                    Unit.getName().c_str() ) );
                  ETG_TRACE_USR1( ( "-> ActiveState : %s", Unit.getActiveState().c_str() ) );
                  ETG_TRACE_USR1( ( "-> LoadState   : %s", Unit.getLoadState().c_str() ) );
                  ETG_TRACE_USR1( ( "-> SubState    : %s", Unit.getSubState().c_str() ) );
               }
            }        // whitelist
            else {
               ETG_TRACE_USR1( ( "Whitelist handling: Ignoring failed/incorrect unit: %s", Unit.getName().c_str() ) );
            }
         }
         ETG_TRACE_USR1( ( "DbusManagerProxy::onListUnitsResponse: found %u failed units", (tUInt)_ServiceFailedEntryList.tServices.size() ) );
         // send acknowledge back that list was received
         Ilcm_tclServiceLcmDbusIf *poServerRef = dynamic_cast < Ilcm_tclServiceLcmDbusIf* >( _cpoMain->getHandler("Ilcm_tclServiceLcmDbusIf") );
         LCM_NULL_POINTER_CHECK(poServerRef);
         poServerRef->sendFailedServicesListStatus(_ServiceFailedEntryList);
      }
   }        // onListUnitsResponse

   void DbusManagerProxy::onListUnitsError(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >  & proxy,
                                           const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ListUnitsError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERR( ( "Received onListUnitsError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERR( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERR( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }

   // StartupFinishedCallbackIF

   /**
     *  \brief handles the spontaneous response of systemd when start-up is finished
     *
     *  \param [in] proxy pointer to the used proxy -> should be managerProxy
     *  \param [in] response of systemd with times
     *  \return void
     *
     *  \details Details
     */
   void DbusManagerProxy::onStartupFinishedSignal(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >         & proxy,
                                                  const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::StartupFinishedSignal >& signal){
      if(_managerProxy == proxy){
         ETG_TRACE_COMP( ( "Received onStartupFinishedSignal:\nstartup time:" ) );
         ETG_TRACE_COMP( ( "Firmware : %16u sec", (tUInt)signal->getFirmware() ) );
         ETG_TRACE_COMP( ( "Loader   : %16u sec", (tUInt)signal->getLoader() ) );
         ETG_TRACE_COMP( ( "Kernel   : %16u sec", (tUInt)signal->getKernel() ) );
         ETG_TRACE_COMP( ( "InitRd   : %16u sec", (tUInt)signal->getInitrd() ) );
         ETG_TRACE_COMP( ( "UserSpace: %16u sec", (tUInt)signal->getUserspace() ) );
         ETG_TRACE_COMP( ( "Total    : %16u sec", (tUInt)signal->getTotal() ) );
      }
   }        // onStartupFinishedSignal

   /**
     *  \brief Error on receiving StartupFinished???
     *
     *  \param [in] proxy pointer to the used proxy -> should be managerProxy
     *  \param [in] error of systemd with times
     *  \return void
     *
     *  \details I have no idea how this could happen???
     */
   void DbusManagerProxy::onStartupFinishedError(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >        & proxy,
                                                 const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::StartupFinishedError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERRMEM( ( "Received onStartupFinishedError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERRMEM( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERRMEM( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }        // onStartupFinishedError

   // SubscribeCallbackIF

   /**
     *  \brief handles the response to the subscribe call
     *
     *  \param [in] proxy pointer to the used proxy -> should be managerProxy
     *  \param [in] response of systemd
     *  \return void
     *
     *  \details Details
     */
   void DbusManagerProxy::onSubscribeResponse(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >     & proxy,
                                              const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::SubscribeResponse >& response){
      (void)response;
      if(_managerProxy == proxy){
         ETG_TRACE_COMP( ( "Received onSubscribeResponse." ) );
      }
   }        // onSubscribeResponse

   /**
     *  \brief Error on receiving Subscribe???
     *
     *  \param [in] proxy pointer to the used proxy -> should be managerProxy
     *  \param [in] error of systemd
     *  \return void
     *
     *  \details I have no idea how this could happen???
     */
   void DbusManagerProxy::onSubscribeError(const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::ManagerProxy >  & proxy,
                                           const ::boost::shared_ptr < ::org::freedesktop::systemd1::Manager::SubscribeError >& error){
      if(_managerProxy == proxy){
         ETG_TRACE_ERRMEM( ( "Received onSubscribeError" ) );
         if (error->hasName() && error->hasMessage() ){
            ETG_TRACE_ERRMEM( ( "%s", error->getName().c_str() ) );
            ETG_TRACE_ERRMEM( ( "%s", error->getMessage().c_str() ) );
         }
      }
   }        // onSubscribeError

#endif // #if 1 // allow editor to collapse this section
}
}
}
} // namespace org { namespace bosch { namespace cm { namespace lcm {

