/*!
  * \file spm_WorkerServer.cpp
  *  \brief
  *    Server to distribute messages within the FC SPM to support loose coupling. The worker server also distributes timer events to
  *               synchronous startup phases like early, startup, storage, cyclic.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 10.08.11  | TMS Fischer        | initial version
  ******
  */
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_WorkerServer.h"
#include "spm_CriticalSection.h"
#include "spm_ISuperVisionManager.h"
#include "spm_ISyncHandler.h"
#include "spm_factory.h"
#include "spm_IOsalProxy.h"
#include "spm_ISupervisionEnableSupervisor.h"

#if OSAL_CONF == OSAL_LINUX
   #include <limits.h>
   #include <unistd.h>
#endif
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_WorkerServer.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID   SPM_FILE_WORKERSERVER

/*!
   \todo remove this again when all projects have this properly defined in spm_Config_project.h
  */
#ifndef SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_HIGH
   #define SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_HIGH SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN + 10
#endif
#ifndef SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_CRTICICAL_HIGH
   #define SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_CRTICICAL_HIGH SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN + 20
#endif

#define SPM_MSGBOX_WORKER_MAX_MESSAGE_COUNT     120
#define SPM_MSGBOX_WORKER_MAX_MESSAGE_LENGTH    sizeof( TSpmWorkerMessage )

// length of supervision timeout, may be overridden
#define SPM_U32_TIMEOUT_SHUTDOWN_SUPERVISION    15000

#if SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN > 100
  #error Percentage must be below 100%
#endif

// -----------------------------------------------------------------------------
// class definition
// -----------------------------------------------------------------------------
spm_tclWorkerServer::spm_tclWorkerServer( const ISpmFactory& factory ) : ISpmWorkerServer( factory )
   , _hSupervisionTimer( OSAL_C_INVALID_HANDLE )
   , _hSpmWorkerQueue( OSAL_C_INVALID_HANDLE )
   , _u32ShutdownSupervision( SPM_U32_TIMEOUT_SHUTDOWN_SUPERVISION )
   , // default delay for supervision
   _hSemId( OSAL_C_INVALID_HANDLE )
   , _nStartupNumber( 0 )
   , // at that point in time ended startup processes
   _nCurrentStartupEnded( 0 )
   , // default value
   _nStartupTime( 0 )
   , _poclSupervisionManager( NULL )
   , _poclSyncHandler( NULL ){
   // create semaphore to protect clients access
   if ( OSAL_s32SemaphoreCreate( SPM_WORKER_SEM_NAME, &_hSemId, (tU32)1 ) != OSAL_OK) {
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! Could not create Semaphore: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

   if ( OSAL_s32TimerCreate( vSupervision, this, &_hSupervisionTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!! Could not create timer: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      // create msgbox
   }
   if ( OSAL_ERROR == OSAL_s32MessageQueueCreate(
           SPM_MSGQ_WORKER_SERVER,
           SPM_MSGBOX_WORKER_MAX_MESSAGE_COUNT,
           SPM_MSGBOX_WORKER_MAX_MESSAGE_LENGTH,
           OSAL_EN_READWRITE,
           &_hSpmWorkerQueue
           )
        ){
      _hSpmWorkerQueue = OSAL_C_INVALID_HANDLE;
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      ETG_TRACE_ERRMEM( ( "Error: OSAL_s32MessageQueueCreate() return with error: %s", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
   }
   _u32MaxMsgCount   = 0;
   _u32ErrorRetryCnt = 0;
}

spm_tclWorkerServer::~spm_tclWorkerServer( ){
   OSAL_s32SemaphoreClose( _hSemId );
   OSAL_s32SemaphoreDelete( SPM_WORKER_SEM_NAME );
   _poclSupervisionManager = NULL;
   _poclSyncHandler        = NULL;
}

tVoid spm_tclWorkerServer::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSyncHandler,        ISpmSyncHandler );

   if ( OSAL_s32MessageQueueNotify( _hSpmWorkerQueue, (OSAL_tpfCallback)vOnQueueCallback, this ) == OSAL_OK ){
   } else {
      ETG_TRACE_ERRMEM( ( "Error: OSAL_s32MessageQueueNotify() return with error: %s", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
   }
}

tVoid spm_tclWorkerServer::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  ******
  */
   vAddClient( this );
}

tVoid spm_tclWorkerServer::vStart( const std::string& strName,
                                   tU32               u32Prio,
                                   tU32               u32Stack ){
/*!
  * \fn
  *  \brief
  *   Starts a new thread for SPM Worker Server.
  *  \param[in] strName: name of thread.
  *  \param[in] u32Prio: priority of thread.
  *  \param[in] u32Stack: stack size of thread.
  ******
  */
      vAddClient( this );
   spm_tclActive::vStartThread( strName, u32Prio, u32Stack );
} // vStart

tVoid spm_tclWorkerServer::vOnQueueCallback( tVoid *pArg ){
   spm_tclWorkerServer *p = (spm_tclWorkerServer*)pArg;
   tU32                 u32MsgMax, u32MaxLen, u32MsgCount = 0;

                        SPM_NULL_POINTER_CHECK( p );
                        SPM_NULL_POINTER_CHECK( p->_poclSyncHandler );

   if ( OSAL_OK != OSAL_s32MessageQueueStatus( p->_hSpmWorkerQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
                        ETG_TRACE_ERRMEM( ( "ERROR OSAL_s32MessageQueueStatus() in spm_tclWorkerServer::vOnQueueCallback(Error=%s)", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
                        ETG_TRACE_ERRMEM( ( "Maximum used message count until now was: %u  CurrOsalTime=%u", (tUInt)p->_u32MaxMsgCount, (tUInt)OSAL_ClockGetElapsedTime( ) ) );
   }

                        ETG_TRACE_USR4( ( "spm_tclWorkerServer::vOnQueueCallback(): Context switch. MsgCount=%u (CurrMaxMsgCount=%u)", (tUInt)u32MsgCount, (tUInt)p->_u32MaxMsgCount ) );
   p->_poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_WORKER_TRIGGER );
} // vOnQueueCallback

tVoid spm_tclWorkerServer::main( ){ // main thread entry function
   TSpmWorkerMessage oWorkerMsg = { 0,0,0 };
   tU32              u32Prio    = 0;
   tS32              s32Ret     = 0;


   s32Ret = OSAL_s32MessageQueueWait( _hSpmWorkerQueue,
                                      (tU8*)&oWorkerMsg, /*pointer of the MsgHandle field*/
                                      sizeof( oWorkerMsg ),
                                      &u32Prio,
                                      OSAL_C_U32_INFINITE );

   if ( oWorkerMsg._clt ){
                        ETG_TRACE_USR4( ( "WorkerServer --> Received new event: %u, receiver: %u, name: '%s'",
                        ETG_ENUM( WORKER_EVENT_TYPE,    ( oWorkerMsg._msg & 0xffff ) ),
                        ETG_ENUM( WORKER_EVENT_TYPE_ID, ( oWorkerMsg._msg & 0xffff0000 ) ),
                        oWorkerMsg._clt->getName( )
                        ) );
   }

   if ( s32Ret != OSAL_ERROR ){
      spm_tclHandleSemaphore s( _hSemId );

      if ( bIsClientAvailable( oWorkerMsg._clt ) ){
         oWorkerMsg._clt->vHandleMessage( oWorkerMsg._msg, oWorkerMsg._parm );
      }
   }
} // main

tVoid spm_tclWorkerServer::vHandleMsgQueue( ){ // main thread entry function
   TSpmWorkerMessage oWorkerMsg;
   tU32              u32Prio, u32MsgMax, u32MaxLen, u32MsgCount = 0;
   tS32              s32Ret;

   SPM_NULL_POINTER_CHECK( _poclSyncHandler );

   #define MAX_RETRY_CNT 3
   do {
      s32Ret = OSAL_s32MessageQueueWait
               (
         _hSpmWorkerQueue,
         (tU8*)&oWorkerMsg, /*pointer of the MsgHandle field*/
         sizeof( oWorkerMsg ),
         &u32Prio,
         OSAL_C_TIMEOUT_NOBLOCKING
               );

      if ( s32Ret > 0 ){
         if ( oWorkerMsg._clt ){
            ETG_TRACE_USR4( ( "WorkerServer --> Received new event: %u, receiver: %u, name: '%s'",
                              ETG_ENUM( WORKER_EVENT_TYPE,    ( oWorkerMsg._msg & 0xffff ) ),
                              ETG_ENUM( WORKER_EVENT_TYPE_ID, ( oWorkerMsg._msg & 0xffff0000 ) ),
                              oWorkerMsg._clt->getName( )
                              ) );

            if ( s32Ret != OSAL_ERROR ){
               spm_tclHandleSemaphore s( _hSemId );

               if ( bIsClientAvailable( oWorkerMsg._clt ) ){
                  oWorkerMsg._clt->vHandleMessage( oWorkerMsg._msg, oWorkerMsg._parm );
               } else {
                  ETG_TRACE_USR4( ( "WorkerServer --> NO CLIENT AVAILABLE for received new event: %u, receiver: %u, name: '%s'",
                                    ETG_ENUM( WORKER_EVENT_TYPE,    ( oWorkerMsg._msg & 0xffff ) ),
                                    ETG_ENUM( WORKER_EVENT_TYPE_ID, ( oWorkerMsg._msg & 0xffff0000 ) ),
                                    oWorkerMsg._clt->getName( )
                                    ) );
               }
            }
         } else {
            ETG_TRACE_ERR( ( "WorkerServer Client is NULL" ) );
         }
      }
      if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSpmWorkerQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
            ETG_TRACE_ERRMEM( ( "ERROR OSAL_s32MessageQueueStatus() in spm_tclWorkerServer::vHandleMsgQueue(Error=%s)", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
            ETG_TRACE_ERRMEM( ( "Maximum used message count until now was: %u", (tUInt)_u32MaxMsgCount ) );

         if ( _u32ErrorRetryCnt < MAX_RETRY_CNT ){
            ETG_TRACE_ERRMEM( ( "Waiting for 500ms and sending SPM_CTX_SWITCH_WORKER_TRIGGER for retry. CurrOsalTime=%u ErrorRetryCnt=%u", (tUInt)OSAL_ClockGetElapsedTime( ), (tUInt)_u32ErrorRetryCnt ) );
            OSAL_s32ThreadWait( 500 );
            _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_WORKER_TRIGGER );
            _u32ErrorRetryCnt++;
         }
      } else {
         if ( _u32ErrorRetryCnt > 0 ){
            ETG_TRACE_ERRMEM( ( "OSAL_s32MessageQueueStatus() sucessful after error. Time=%u", (tUInt)OSAL_ClockGetElapsedTime( ) ) );
         }
         _u32ErrorRetryCnt = 0;
      }

   } while ( u32MsgCount != 0 );

} // main

tVoid spm_tclWorkerServer::vSetStartupTime( ){
/*!
  * \fn
  *  \brief This method sets the value of variable _nStartupTime to the total time taken for complete startup.
  *  \note
  *   This function is called by the StartupSystem after completing startup.
  ******
  */
   ++_nCurrentStartupEnded; // one startup process more ended now
   if ( _nCurrentStartupEnded >= _nStartupNumber ){
      _nStartupTime = OSAL_ClockGetElapsedTime( );
   }
} //vSetStartupTime

tVoid spm_tclWorkerServer::vAddClient( ISpmWorkerClient *pClient ){
   //spm_tclHandleSemaphore s( _hSemId );  --> commented out due lock when called in workerthread context

   _clients.insert( pClient );
}

tVoid spm_tclWorkerServer::vRemoveClient( ISpmWorkerClient *pClient ){
   //spm_tclHandleSemaphore                   s( _hSemId );  --> commented out due lock when called in workerthread context

   tBool                                    bFound = FALSE;

   std::set < ISpmWorkerClient* >::iterator it     = _clients.find( pClient );

   if ( it != _clients.end( ) ){
      _clients.erase( it );
      bFound = TRUE;
   }
   ETG_TRACE_USR4( ( "spm_tclWorkerServer::vRemoveClient: %u", (tUInt)_clients.size( ) ) );

   if ( FALSE == bFound ){
      ETG_TRACE_FATAL( ( "spm_tclWorkerServer::vRemoveClient: client '%s' not found.", pClient->getName() ) );
   }
   // check if the worker thread is still needed after removing this
   // client.
   bPostMessage( "ISpmWorkerServer", SPM_U32_WORKER_WRK_REMOVE_CLIENT );
} // spm_tclWorkerServer::vRemoveClient

tBool spm_tclWorkerServer::bIsClientAvailable( ISpmWorkerClient *pClient ) const {
   std::set < ISpmWorkerClient* >::const_iterator it;

   it = _clients.find( pClient );
   if ( it != _clients.end( ) ){
      return( TRUE );
   }
   return( FALSE );
}

tBool spm_tclWorkerServer::bPostMessageToWorker( tU32 u32Message,
                                                 tU32 u32Parm ){
   /* Stop retriggering of watchdog in any reset-case (not normal shutdown). As we dont can be sure
      that the system get stuck before the WORKER_SHUTDOWN message is processed from worker receive thread,
      disabling the wdt retriggering supervises the worker shutdown reset mechanism.
     */
   SPM_NULL_POINTER_CHECK_VAL( _poclSupervisionManager );

   if ( u32Message == SPM_U32_WORKER_SPM_SHUTDOWN ){
      if ( u32Parm != SPM_U32_SHUTDOWN_NORMAL ){
         _poclSupervisionManager->vDisableWdtRetriggering( );
         ETG_TRACE_USR4( ( "spm_tclWorkerServer::bPostMessageToWorker()-> Disabled WDT retriggering for SPM_U32_WORKER_SHUTDOWN supervision" ) );
      }
   }

   return( bPostMessage( "ISpmWorkerServer", u32Message, u32Parm ) );
}

/*!
  * \fn
  *  \brief
  *  Method is used to post messages to Clients.
  *  \param[in] strClientIfName: Client name.
  *  \param[in] u32Message: message to be posted Client.
  *  \param[in] u32Parm: parameters passed as a part of message.
  *  \param[out] tBool: if message has been posted successfully.
  ******
  */
tBool spm_tclWorkerServer::bPostMessage( const std::string& strClientIfName,
                                         tU32               u32Message,
                                         tU32               u32Parm ){
   TSpmWorkerMessage oWorkerMsg;

   oWorkerMsg._clt  = dynamic_cast < ISpmWorkerClient* >( _crfFactory.getSpmObjHandler( strClientIfName ) );
   oWorkerMsg._msg  = u32Message;
   oWorkerMsg._parm = u32Parm;

   if ( NULL == oWorkerMsg._clt ){
      ETG_TRACE_ERRMEM( ( "spm_tclWorkerServer::bPostMessage(): No object found for name: '%s'", strClientIfName.c_str( ) ) );
      NORMAL_M_ASSERT_ALWAYS( ); // to get the callstack of the calling function which requests a unkown workerclient
      return( FALSE );
   }
   return( bPostMessage( oWorkerMsg ) );
}

tBool spm_tclWorkerServer::bPostMessage( const TSpmWorkerMessage& msg ){
   tS32         s32Ret;

   static tBool fGenCallstack1 = TRUE;
   static tBool fGenCallstack2 = TRUE;

   s32Ret = OSAL_s32MessageQueuePost(
      _hSpmWorkerQueue,
      (tCU8*)&msg,
      sizeof( msg ),
      OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
   tU32 u32MsgMax, u32MaxLen, u32MsgCount = 0;
   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSpmWorkerQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERRMEM( ( "ERROR OSAL_s32MessageQueueStatus() in spm_tclWorkerServer::bPostMessage(Error=%s)", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
   }
   tU32 u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
   if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
   } else {
      if ( u32MsgCount > _u32MaxMsgCount ){
         _u32MaxMsgCount = u32MsgCount;
      }
   }

   /* Queue error level has reached create once a callstack:*/
   if ( u32FillLevel > SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_HIGH ){

      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclOsalProxy, ISpmOsalProxy );

      /* To avoid endless trigger here in case of permanent full queue, send error handling trigger only once:*/
      if ( fGenCallstack1 ){
         fGenCallstack1 = FALSE;
         ETG_TRACE_ERRMEM( ( "Triggering Callstack #1 at time=%u", (tUInt)OSAL_ClockGetElapsedTime( ) ) );
         #if OSAL_CONF == OSAL_LINUX
            char    exePath[PATH_MAX] = {0};
            strcpy( exePath, "procbase_out.out" );
            ssize_t count             = readlink( "/proc/self/exe", exePath, PATH_MAX );
            if ( count == 0 ){
               strcpy( exePath, "procbase_out.out" );
            }
            poclOsalProxy->bDumpProcessInfo( exePath, "OSAL", SPM_BL_OSALPROXY_IF_ENABLED );

         #endif

      }
      /* Check for second callstack generation trigger point:*/
      if ( u32FillLevel > SPM_ERROR_QUEUE_FILL_LEVEL_PERCENTAGE_CRTICICAL_HIGH ){
         if ( fGenCallstack2 ){
            fGenCallstack2 = FALSE;
            ETG_TRACE_ERRMEM( ( "Triggering Callstack #2 at time=%u", (tUInt)OSAL_ClockGetElapsedTime( ) ) );
            #if OSAL_CONF == OSAL_LINUX
               char    exePath[PATH_MAX] = {0};
               strcpy( exePath, "procbase_out.out" );
               ssize_t count             = readlink( "/proc/self/exe", exePath, PATH_MAX );
               if ( count == 0 ){
                  strcpy( exePath, "procbase_out.out");
               }
               poclOsalProxy->bDumpProcessInfo( exePath, "OSAL", SPM_BL_OSALPROXY_IF_ENABLED );

            #endif

         }
      }
   } /* Check if queue level is under the warning level and re-arm callstack trigger flags for next run: */
   else if ( u32FillLevel < SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
      if ( ( fGenCallstack1 == FALSE ) || ( fGenCallstack2 == FALSE ) ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error END detected !!!!!! Current QueueFull Level (%u) under %u percent  time=%u", (tUInt)u32FillLevel, (tUInt)SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN, (tUInt)OSAL_ClockGetElapsedTime( ) ) );
         fGenCallstack1 = TRUE; // re-arm for next queue check.
         fGenCallstack2 = TRUE;
      }

   }

   if ( s32Ret == OSAL_OK ){
      vOnQueueCallback( this );
      return( TRUE );
   }
   vOnMessageQueueError( );

   return( FALSE );
} // bPostMessage

tVoid spm_tclWorkerServer::vSynchronousCall( tU32   u32Message,
                                             tVoid *args ){
   std::set < ISpmWorkerClient* >::iterator it;

   for ( it = _clients.begin( ); it != _clients.end( ); ++it ){
      // only the first should handle the synchronous call
      // if the handler allows for multiple servers then it
      // returns FALSE.
      if ( ( * it )->bHandleSynchrounousCall( u32Message, args ) ){
         break;
      }
   }
}

tVoid spm_tclWorkerServer::vHandleMessage( tU32 u32Message,
                                           tU32 u32Parm ){
   // Terminate the application
   switch ( u32Message ){
      case SPM_U32_WORKER_WRK_SUPERVISION_TIMEOUT:
         // stop thread due to supervision timeout
         ETG_TRACE_FATAL( ( "Not all SPM threads are terminated within 15s. Current size: %u", (tU32)(_clients.size( ) )) );
            vSetTerminate( );
         break;

      case SPM_U32_WORKER_WRK_REMOVE_CLIENT:
         // stop thread if no client is connected
         if ( _clients.size( ) == 0 ){
            vSetTerminate( );
         }
         break;

      default:
         // broadcast message
         ETG_TRACE_USR4( ( "spm_tclWorkerServer::vHandleMessage(): Unknown message --> just broadcast: %u", ETG_ENUM( WORKER_EVENT_TYPE, u32Message ) ) );
         vSynchronousCall( u32Message, &u32Parm );
         break;
   } // switch
}    // vHandleMessage

tBool spm_tclWorkerServer::bHandleSynchrounousCall( tU32   /*u32Message*/,
                                                    tVoid */*args*/ ){
   // handle synchronous messages, used to support loose coupling.
   return( FALSE );
} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

tVoid spm_tclWorkerServer::vSupervision( tVoid *pArg ){
   spm_tclWorkerServer *poWorker = (spm_tclWorkerServer*)pArg;

   poWorker->bPostMessage( "ISpmWorkerServer", SPM_U32_WORKER_WRK_SUPERVISION_TIMEOUT );
}

tVoid spm_tclWorkerServer::vTerminate( ){
   ETG_TRACE_FATAL( ( "spm_tclWorkerServer: Maximum used message count until now was: %d  of %d", _u32MaxMsgCount, SPM_MSGBOX_WORKER_MAX_MESSAGE_COUNT ) );
   if ( OSAL_OK != OSAL_s32TimerSetTime( _hSupervisionTimer, SPM_U32_TIMEOUT_SHUTDOWN_SUPERVISION, 0 ) ) {
      ETG_TRACE_FATAL( ( "spm_tclWorkerServer: _hSupervisionTimer can not be set with return error %s", OSAL_coszErrorText( OSAL_u32ErrorCode( ) )) );
   }
}

tVoid spm_tclWorkerServer::vOnTerminate( ){
   ETG_TRACE_USR4( ( "Thread in class spm_tclWorkerServer terminated" ) );
}

tVoid spm_tclWorkerServer::vChangePriority( tU32 u32Priority ){
   spm_tclActive::vSetPriority( u32Priority );
}

tVoid spm_tclWorkerServer::vRestorePriority( ){
   spm_tclActive::vRestoreStartPriority( );
}

