/*!
  * \file spm_VersionInfo.cpp
  *  \brief
  *    Reads configuration data and writes it to the ffd if changed
  *
  *  \note
  *  \b PROJECT: NextGen \n
  *  \b SW-COMPONENT: FC SPM \n
  *  \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 11.08.11  | TMS Petroglou     | initial version
  ******
  */

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------

#include <iostream>
#include <iterator>
#include <sstream>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

#define SYSTEM_S_IMPORT_INTERFACE_FFD_DEF
#define SYSTEM_S_IMPORT_INTERFACE_KDS_DEF
#include "system_pif.h"

#include "spm_Config.h"

#include "spm_VersionInfo.h"
#include "spm_VersionInfoDp.h"
#include "spm_Registry_pathes.h"

#include "spm_IFactory.h"

#include "spm_ISystemPowerManager.h"

#include "spm_OsalProxyConfig.h"
#include "spm_GlobDefs.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSH
 #include "trcGenProj/Header/spm_VersionInfo.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID  SPM_FILE_VERSIONINFO

//to replace magic numbers assigned as string size, where ever possible.
#define VERSION_STRING_LEN     256

#define SPM_FFD_VERSION_ID EN_FFD_DATA_SET_VERSION

spm_tclVersionInfo::spm_tclVersionInfo( const ISpmFactory& factory ) : ISpmVersionInfo( factory )
   , _poclWorkerServer( NULL )
   , _u32BoardId( 0 )
   #ifdef VARIANT_S_FTR_ENABLE_CPLD
      , _u32CpldVersion( 0 )
   #endif
   , _bVersionChecked( FALSE )
   , _u32NextEmTraceCheck( SPM_START_CHECK_EMTRACE_VERS ){
}

spm_tclVersionInfo::~spm_tclVersionInfo( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );
   _poclWorkerServer = NULL;
}

tVoid spm_tclVersionInfo::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclVersionInfo::vStartCommunication( ){
      vUpdateVersionInfo( );
}

tBool spm_tclVersionInfo::bHandleSynchrounousCall( tU32 u32Message,
                                                   tVoid* ){ /*args*/
   /* In normal use cases, this leads only to a ffd read operation.
      Only after a sw-update, this can lead to a write access.*/
   if ( SPM_U32_WORKER_BROADCAST_STORAGE_DELAY == u32Message ){
      vUpdateVersionInfo( );
   }
   return( FALSE );
}

/*!
  * \fn
  *  \brief     VersionInfo like of Build, Customer version, SccAppInfo etc
  *             retrieved from datapool and KDS and updated in local strings
  *             for further processing.
  *
  *  \param[in] :  tVoid
  *
  *  \return: tVoid
  *
 *************************************************************************/
tVoid spm_tclVersionInfo::vUpdateVersionInfo( tVoid ){

   std::string                              strTmp( VERSION_STRING_LEN, 0 );

   dp_tclregVersionsBoardCfgBuildVersion    oBuildVer;

   oBuildVer.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strBuildVersion    = strTmp.c_str( );

   dp_tclregVersionsBoardCfgCustomerVersion oCustomerVer;
   strTmp.assign( VERSION_STRING_LEN, 0 );
   oCustomerVer.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strCustomerVersion = strTmp.c_str( );

   // see /ai_osal_common/components/osal/include/system/system_kds_def.h for KDS defines
   const tU8 SerialNumberLength = 8; // check PD data. Length of SerialNumber is 7 + 1 byte extra for some strange reason
   tLcmUString ustrSerialNumber( SerialNumberLength + 1, 0 );
   tU16        u16SerialNumberEntry   = M_KDS_ENTRY( KDS_TARGET_DEVICE, KDS_TYPE_SERIAL_NUMBER );

   if ( bReadKdsEntry( u16SerialNumberEntry, SerialNumberLength, &ustrSerialNumber ) ){
      std::stringstream result;
      std::copy(ustrSerialNumber.begin(), ustrSerialNumber.end(), std::ostream_iterator<unsigned char>(result, ""));
      _strEcuSerialNumber= result.str();
   } else {
      ETG_TRACE_USR4( ( "KDS read failed: SerialNumber") );
      _strEcuSerialNumber = "unknown";
   }

   const tU8 SparePartNumberLength = 11; // check PD data. Length of EcuSparePartNumber is 10 + 1 byte extra for some strange reason
   tLcmUString ustrSparePartNumber( SparePartNumberLength + 1, 0 );
   tU16        u16SparePartEntry   = M_KDS_ENTRY( KDS_TARGET_DEVICE, KDS_TYPE_TYP_NUMBER );

   if ( bReadKdsEntry( u16SparePartEntry, SparePartNumberLength, &ustrSparePartNumber ) ){
      std::stringstream result;
      std::copy(ustrSparePartNumber.begin(), ustrSparePartNumber.end(), std::ostream_iterator<unsigned char>(result, ""));
      _strEcuSparePartNumber= result.str();
   } else {
      ETG_TRACE_USR4( ( "KDS read failed: SparePartNumber") );
      _strEcuSparePartNumber = "unknown";
   }

   #ifdef VARIANT_S_FTR_ENABLE_INC
      dp_tclregVersionsBoardCfgSccAppVersion oSccAppInfo;
      strTmp.assign( VERSION_STRING_LEN, 0 );
      oSccAppInfo.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
      _strSccAppVersion = strTmp.c_str( );

      dp_tclregVersionsBoardCfgSccBlVersion  oSccBlInfo;
      strTmp.assign( VERSION_STRING_LEN, 0 );
      oSccBlInfo.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
      _strSccBlVersion  = strTmp.c_str( );

      dp_tclregVersionsBoardCfgSccBmVersion  oSccBmInfo;
      strTmp.assign( VERSION_STRING_LEN, 0 );
      oSccBmInfo.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
      _strSccBmVersion  = strTmp.c_str( );
   #endif

   dp_tclregVersionsBoardCfgName      oBuildName;
   strTmp.assign( VERSION_STRING_LEN, 0 );
   oBuildName.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strBoardName     = strTmp.c_str( );

   dp_tclregVersionsBoardCfgBuildType oBuildType;
   strTmp.assign( VERSION_STRING_LEN, 0 );
   oBuildType.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strBuildInfo     = "TYPE: '";
   _strBuildInfo    += strTmp.c_str( );

   dp_tclregVersionsBoardCfgBuildUsr oBuildUsr;
   strTmp.assign( VERSION_STRING_LEN, 0 );
   oBuildUsr.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strBuildInfo    += "', USER: '";
   _strBuildInfo    += strTmp.c_str( );

   #ifdef __DEBUG_BUILD__
      _strBuildInfo += "', 'DEBUG'";
   #else
      _strBuildInfo += "', 'RELEASE'";
   #endif

   dp_tclregVersionsBoardCfgBuildTime oBuildTime;
   strTmp.assign( VERSION_STRING_LEN, 0 );
   oBuildTime.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
   _strBuildInfo += "', TIME: '";
   _strBuildInfo += strTmp.c_str( );

      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): BuildVersion:       '%s'", _strBuildVersion.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): CustomerVersion:    '%s'", _strCustomerVersion.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): EcuSerialNumber:    '%s'", _strEcuSerialNumber.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): EcuSparePartNumber: '%s'", _strEcuSparePartNumber.c_str( ) ) );
   #ifdef VARIANT_S_FTR_ENABLE_INC
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): SccAppVersion:      '%s'", _strSccAppVersion.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): SccBmVersion:       '%s'", _strSccBmVersion.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): SccBlVersion:       '%s'", _strSccBlVersion.c_str( ) ) );
   #endif
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): BoardName:          '%s'", _strBoardName.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): Board ID:           '0x%04x'", _u32BoardId ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): Build Info:         '%s'", _strBuildInfo.c_str( ) ) );

   #ifdef VARIANT_S_FTR_ENABLE_CPLD
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): CPLD Version:       '%s'", _strCpldVersion.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclVersionInfo(): CPLD Version (num): 'V%02x'", _u32CpldVersion ) );
   #endif

} // vUpdateVersionInfo

/*!
  * \fn
  *  \brief     VersionInfo like of Build, Customer version, SccAppInfo etc
  *             is written into ErrMem.
  *             Just to make sure vUpdateVersionInfo is called again.
  *
  *  \param[in] :  tVoid
  *
  *  \return: tVoid
  *
 *************************************************************************/
tVoid spm_tclVersionInfo::vStoreVersionInfoInEmTrace( tVoid ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );

   vUpdateVersionInfo( );

   {
      std::string strEntry = "--------------------------------------------------------------------------------------------";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                           (const tU8*)strEntry.c_str( ),
                                           (tU16)strEntry.length( ) );
   }
   {
      std::string strEntry = "VERSIONINFO: BuildVersion:       '";
      strEntry += _strBuildVersion;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                            (const tU8*)strEntry.c_str( ),
                                            (tU16)strEntry.length( ) );
   }
   {
      std::string strEntry = "VERSIONINFO: BuildInfo:          '";
      strEntry += _strBuildInfo;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                           (const tU8*)strEntry.c_str( ),
                                           (tU16)strEntry.length( ) );
   }
   {
      std::string strEntry = "VERSIONINFO: CustomerVersion:    '";
      strEntry += _strCustomerVersion;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                            (const tU8*)strEntry.c_str( ),
                                            (tU16)strEntry.length( ) );
   }
   {
      std::string strEntry = "VERSIONINFO: EcuSerialNumber:    '";
      strEntry += _strEcuSerialNumber;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                            (const tU8*)strEntry.c_str( ),
                                            (tU16)strEntry.length( ) );
   }
   {
      std::string strEntry = "VERSIONINFO: EcuSparePartNumber: '";
      strEntry += _strEcuSparePartNumber;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                            (const tU8*)strEntry.c_str( ),
                                            (tU16)strEntry.length( ) );
   }
   #ifdef VARIANT_S_FTR_ENABLE_INC
      {
         std::string strEntry = "VERSIONINFO: SccAppVersion:      '";
         strEntry += _strSccAppVersion;
         strEntry += "'";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                              (const tU8*)strEntry.c_str( ),
                                              (tU16)strEntry.length( ) );
      }
      {
         std::string strEntry = "VERSIONINFO: SccBmVersion:       '";
         strEntry += _strSccBmVersion;
         strEntry += "'";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                              (const tU8*)strEntry.c_str( ),
                                              (tU16)strEntry.length( ) );
      }
      {
         std::string strEntry = "VERSIONINFO: SccBlVersion:       '";
         strEntry += _strSccBlVersion;
         strEntry += "'";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                              (const tU8*)strEntry.c_str( ),
                                              (tU16)strEntry.length( ) );
      }
   #endif // ifdef VARIANT_S_FTR_ENABLE_INC
   {
      std::string strEntry = "VERSIONINFO: BoardName:          '";
      strEntry += _strBoardName;
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                           (const tU8*)strEntry.c_str( ),
                                           (tU16)strEntry.length( ) );
   }
   {
      std::string                   strTmp( VERSION_STRING_LEN, 0 );
      dp_tclregVersionsAdrSwVersion oAdrSwVersion;
      oAdrSwVersion.u8GetData( &strTmp[0], (tU32)strTmp.length( ) );
      std::string strEntry = "VERSIONINFO: ADR SW:             '";
      strEntry += strTmp.c_str( );
      strEntry += "'";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                           (const tU8*)strEntry.c_str( ),
                                           (tU16)strEntry.length( ) );
   }

   #ifdef VARIANT_S_FTR_ENABLE_CPLD
      {
         std::string strEntry = "VERSIONINFO: CPLD Version:       '";
         strEntry += _strCpldVersion;
         strEntry += "'";
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                              (const tU8*)strEntry.c_str( ),
                                              (tU16)strEntry.length( ) );
      }
   #endif

   vStoreProjectVersionInfoInEmTrace( );

   {
      std::string strEntry = "--------------------------------------------------------------------------------------------";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),
                                           (const tU8*)strEntry.c_str( ),
                                           (tU16)strEntry.length( ) );
   }
} // vStoreVersionInfoInEmTrace

/*!
  * \fn
  *  \brief     VersionInfo is updated based on the contents read from
  *             FFD,Registry,File and KDS.
  *
  *  \param[in] :  tVoid
  *
  *  \return: tVoid
  *
 *************************************************************************/
tBool spm_tclVersionInfo::CheckAndUpdateVersionInfo( void ) const {
   tBool                    fIsEqual = FALSE;

   ETG_TRACE_USR4( ( "spm_tclVersionInfo::CheckAndUpdateVersionInfo" ) );
   OSAL_tsDevInfoSystemInfo tOsalSystemInfoFromFFD;
   OSAL_tsDevInfoSystemInfo tOsalSystemInfoFromRegistry;
   OSAL_pvMemorySet( (void*)&tOsalSystemInfoFromFFD,      0, sizeof( OSAL_tsDevInfoSystemInfo ) );
   OSAL_pvMemorySet( (void*)&tOsalSystemInfoFromRegistry, 0, sizeof( OSAL_tsDevInfoSystemInfo ) );

   /* Read from FFD. If nothing cant be read from ffd, the date is still set to zero*/
   tS32                     s32FFdOk  = ReadFromFFD( &tOsalSystemInfoFromFFD );
   tS32                     s32RegOk  = ReadFromRegistry( &tOsalSystemInfoFromRegistry );
   tS32                     s32FileOk = ReadFromFile( &tOsalSystemInfoFromRegistry );
   tS32                     s32KDSOk  = ReadFromKDS( &tOsalSystemInfoFromRegistry );


   if ( !s32FFdOk ){
      ETG_TRACE_USR4( ( "spm_tclVersionInfo - ReadFromFFD() not succesfull" ) );
   }
   if ( !s32RegOk ){
      ETG_TRACE_USR4( ( "spm_tclVersionInfo - ReadFromRegistry() not succesfull" ) );
   }
   if ( !s32FileOk ){
      ETG_TRACE_USR4( ( "spm_tclVersionInfo - ReadFromFile() not successful" ) );
   }
   if ( !s32KDSOk ){
      ETG_TRACE_USR4( ( "spm_tclVersionInfo - ReadFromKDS() not successful" ) );
      /* Difference between registry and ffd?*/
   }
   if ( 0 != OSAL_s32MemoryCompare( &tOsalSystemInfoFromFFD, &tOsalSystemInfoFromRegistry, sizeof( OSAL_tsDevInfoSystemInfo ) ) ){
      ETG_TRACE_USR4( ( "spm_tclVersionInfo -> WriteToFFD()" ) );
      WriteToFFD( &tOsalSystemInfoFromRegistry );
   } else {
      ETG_TRACE_USR4( ( "spm_tclVersionInfo ->NO WriteToFFD()" ) );
      fIsEqual = TRUE;
   }

   return( fIsEqual );
} // CheckAndUpdateVersionInfo

/*!
  * \fn
  *  \brief     Registry is read for version info. If not available the
  *             content is updated with 'REG_UNDEF'.
  *
  *  \param[in] :  tOsalSystemInfoFromRegistry: OSAL_tsDevInfoSystemInfo object.
  *
  *  \return: tS32: Return value depicts if registry read was successful
 *              or not
  *
 *************************************************************************/
tS32 spm_tclVersionInfo::ReadFromRegistry( OSAL_tsDevInfoSystemInfo *tOsalSystemInfoFromRegistry ) const {
   tS32          Vs32status    = OSAL_ERROR;

   reg_tclRegKey oReg;
   reg_tclRegKey oRegVersion;
   tBool         bReadFailed   = FALSE;

   std::string   strUndef      = "REG_UNDEF";
   tBool         bBoardIdFound = FALSE;

   if ( oReg.bOpen( SPM_REG_VERSIONS_BASE_PATH ) ){
      if ( !oReg.bQueryString( "BUILDVERSION_LABEL_NOR",
                               (tChar*)&tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NOR,
                               sizeof( tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NOR ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NOR, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NOR ) - 1 );
      }

      if ( !oReg.bQueryString( "BUILDVERSION_LABEL_NAND",
                               (tChar*)&tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NAND,
                               sizeof( tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NAND ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NAND, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strBUILDVERSION_LABEL_NAND ) - 1 );
      }

      if ( !oReg.bQueryString( "COMPANY_NAME",
                               (tChar*)&tOsalSystemInfoFromRegistry->strCOMPANY_NAME,
                               sizeof( tOsalSystemInfoFromRegistry->strCOMPANY_NAME ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strCOMPANY_NAME, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strCOMPANY_NAME ) - 1 );
      }

      if ( !oReg.bQueryString( "PRODUCT_NAME",
                               (tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_NAME,
                               sizeof( tOsalSystemInfoFromRegistry->strPRODUCT_NAME ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_NAME, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strPRODUCT_NAME ) - 1 );
      }

      //if (!oReg.bQueryString("PRODUCT_SERIAL_NUMBER",
      //(tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_SERIAL_NUMBER,
      //sizeof(tOsalSystemInfoFromRegistry->strPRODUCT_SERIAL_NUMBER)))
      //{
      //bReadFailed = TRUE;
      //OSAL_szStringNCopy((tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_SERIAL_NUMBER,strUndef,sizeof(tOsalSystemInfoFromRegistry->strPRODUCT_SERIAL_NUMBER));
      //};

      if ( !oReg.bQueryString( "PRODUCT_PART_NUMBER",
                               (tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_PART_NUMBER,
                               sizeof( tOsalSystemInfoFromRegistry->strPRODUCT_PART_NUMBER ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strPRODUCT_PART_NUMBER, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strPRODUCT_PART_NUMBER ) - 1 );
      }

      if ( !oReg.bQueryString( "SUPPLIER_ECU_SOFTWARE_VERSION_NUMBER",
                               (tChar*)&tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_SOFTWARE_VERSION_NUMBER,
                               sizeof( tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_SOFTWARE_VERSION_NUMBER ) ) ){
         bReadFailed = TRUE;
         OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_SOFTWARE_VERSION_NUMBER, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_SOFTWARE_VERSION_NUMBER ) - 1 );
      }

      //if (!oReg.bQueryString("SUPPLIER_ECU_HARDWARE_VERSION_NUMBER",
      //(tChar*)&tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER,
      //sizeof(tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER)))
      //{
      //bReadFailed = TRUE;
      //OSAL_szStringNCopy((tChar*)&tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER,strUndef,sizeof(tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER));
      //};

      if ( oRegVersion.bOpen( SPM_REG_VERSIONS_BOARDCFG_BASE_PATH ) ){
         std::string strBoardName( VERSION_STRING_LEN, 0 );

         if ( oRegVersion.bQueryString( "BOARD_NAME",
                                        &strBoardName[0],
                                        (tU32)strBoardName.length( )
                                        ) ){
            if ( strBoardName.length( ) > 0 ){
               tUInt  unBoardId = 0;
               char *ptrBid     = strstr( &strBoardName[0], "BID 0x" );
               if ( ptrBid ){
                  ptrBid += strlen( "BID 0x" );
                  if ( strlen( ptrBid ) >= 4 ){
                     if ( sscanf( ptrBid, "%X", &unBoardId ) ){
                        sprintf( tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER, "%4X", unBoardId );
                        bBoardIdFound = TRUE;
                        ETG_TRACE_USR4( ( "spm_tclVersionInfo(): Board ID: %4X", unBoardId ) );
                     }
                  }
               }
            }
         }
      }
      if ( !bBoardIdFound ){
         sprintf( tOsalSystemInfoFromRegistry->strSUPPLIER_ECU_HARDWARE_VERSION_NUMBER, "%4X", 0 );
         // if (!oReg.bQueryString("CONTROLLER_PART_NUMBER",
         // (tChar*)&tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER,
      }
      // sizeof(tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER)))
      // {bReadFailed = TRUE;
      // OSAL_szStringNCopy((tChar*)&tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER,strUndef,sizeof(tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER) - 1 );
      // };

      if ( !oReg.bQueryString( "CONTROLLER_SUPPLIER",
                               (tChar*)&tOsalSystemInfoFromRegistry->strCONTROLLER_SUPPLIER,
                               sizeof( tOsalSystemInfoFromRegistry->strCONTROLLER_SUPPLIER ) ) ){
         bReadFailed = TRUE;
            OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromRegistry->strCONTROLLER_SUPPLIER, strUndef.c_str( ), sizeof( tOsalSystemInfoFromRegistry->strCONTROLLER_SUPPLIER ) - 1 );
      }

      if ( !bReadFailed ){
         Vs32status = OSAL_OK;
      }
   }

   return( Vs32status );
} // ReadFromRegistry

/*!
  * \fn
  *  \brief    File is read for version info[on CONTROLLER_PART_NUMBER].
  *
  *  \param[in] :  tOsalSystemInfoFromRegistry: OSAL_tsDevInfoSystemInfo object.
  *
  *  \return: tS32: Return value depicts if file read was successful or not.
  *
 *************************************************************************/
tS32 spm_tclVersionInfo::ReadFromFile( OSAL_tsDevInfoSystemInfo *tOsalSystemInfoFromRegistry ) const {
   OSAL_tIODescriptor fd;
   tS32               Vs32status = OSAL_ERROR;

   fd = OSAL_IOOpen( SPM_REGISTRY_DYN_PATH_OSAL "/diag/EndModel_AlphaCode.dat", OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      std::string buffer( 2, 0 ); // this length is fix!!

      tS32        s32FileSize = OSALUTIL_s32FGetSize( fd );

      if ( (std::size_t)s32FileSize == buffer.length( ) ){

         tS32 s32Length = OSAL_s32IORead( fd, ( tPS8 ) & buffer[0], (tU32)buffer.length( ) );
         if ( (std::size_t)s32Length == buffer.length( ) ){
            std::string buffer2( 32, 0 );

            buffer2.assign( buffer.c_str( ), buffer.length( ) );
            OSAL_szStringNCopy( &tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER[0],
                                buffer2.c_str( ),
                                sizeof( tOsalSystemInfoFromRegistry->strCONTROLLER_PART_NUMBER ) );
            Vs32status = OSAL_OK;
         }
      }
      OSAL_s32IOClose( fd );
   }
   return( Vs32status );
} // ReadFromFile

/*!
  * \fn
  *  \brief    Initiates the read of KDS. If the read is successful then
  *            the PRODUCT_SERIAL_NUMBER is updated based on the Serial
  *            Number retrieved else updated as 'KDS_UNDEF'.
  *
  *  \param[in] :  tOsalSystemInfoFromKDS: OSAL_tsDevInfoSystemInfo object.
  *
  *  \return: tS32: Return value depicts if KDS read was successful or not.
  *
 *************************************************************************/
tS32 spm_tclVersionInfo::ReadFromKDS( OSAL_tsDevInfoSystemInfo *tOsalSystemInfoFromKDS ) const {
   #define KDS_DATA_LENGTH  14
   // #define M_KDS_ENTRY(Target,Type) ((tEntry)((((tMember)(Target))<<8) + ((tMember)(Type))))

   std::string strUndef   = "KDS_UNDEF";
   tS32        Vs32status = OSAL_ERROR;
   tLcmUString ustrSerialNumber( KDS_DATA_LENGTH + 1, 0 );
   tU16        u16Entry   = M_KDS_ENTRY( KDS_TARGET_DEVICE, KDS_TYPE_SERIAL_NUMBER );

   if ( bReadKdsEntry( u16Entry, KDS_DATA_LENGTH, &ustrSerialNumber ) ){
      ETG_TRACE_USR4( ( "KDS read ok. SerialNumber = %s", ustrSerialNumber.c_str( ) ) );
      OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromKDS->strPRODUCT_SERIAL_NUMBER, ustrSerialNumber.c_str( ), sizeof( tOsalSystemInfoFromKDS->strPRODUCT_SERIAL_NUMBER ) - 1 );
      Vs32status = OSAL_OK;
   } else {
      ETG_TRACE_USR4( ( "KDS read failed. SerialNumber = %s", strUndef.c_str( ) ) );
      OSAL_szStringNCopy( (tChar*)&tOsalSystemInfoFromKDS->strPRODUCT_SERIAL_NUMBER, strUndef.c_str( ), sizeof( tOsalSystemInfoFromKDS->strPRODUCT_SERIAL_NUMBER ) - 1 );
   }

   return( Vs32status );
} // ReadFromKDS

/*!
  * \fn
  *  \brief    Info is read from FFD.
  *
  *  \param[in] :  tOsalSystemInfoFromKDS: OSAL_tsDevInfoSystemInfo object.
  *
  *  \return: tS32: Return value depicts if FFD read was successful or not.
  *
 *************************************************************************/
tS32 spm_tclVersionInfo::ReadFromFFD( OSAL_tsDevInfoSystemInfo *tOsalSystemInfoFromFFD ) const {
   tS32                 Vs32status  = OSAL_ERROR;
   OSAL_tIODescriptor   VhDeviceFFD = OSAL_ERROR;
   OSAL_trFFDDeviceInfo VtsFFDDevInfo;

   /*open the device*/
   VhDeviceFFD = OSAL_IOOpen( OSAL_C_STRING_DEVICE_FFD, OSAL_EN_READONLY );
   /*check handle*/
   if ( VhDeviceFFD != OSAL_ERROR ){
      tS32 Vs32size      = sizeof( OSAL_tsDevInfoSystemInfo );
      tU8 *pu8FFDRAWdata = OSAL_NULL;
      pu8FFDRAWdata = (tU8*)OSAL_pvMemoryAllocate( (tU32)Vs32size );
      if ( pu8FFDRAWdata ){
         VtsFFDDevInfo.u8DataSet = SPM_FFD_VERSION_ID; // FFD_DATA_SET_VERSION
         VtsFFDDevInfo.pvArg     = (void*)pu8FFDRAWdata;
         Vs32status              = OSAL_s32IORead( VhDeviceFFD, ( tPS8 ) & VtsFFDDevInfo, Vs32size );
         if ( Vs32status != OSAL_ERROR ){
            Vs32status = OSAL_OK;
            memcpy( tOsalSystemInfoFromFFD, pu8FFDRAWdata, sizeof( OSAL_tsDevInfoSystemInfo ) );
            // tOsalSystemInfoFromFFD = (OSAL_tsDevInfoSystemInfo*)pu8FFDRAWdata;
         }
         OSAL_vMemoryFree( pu8FFDRAWdata );
      }

      /*close */
      (void)OSAL_s32IOClose( VhDeviceFFD );
   }
   return( Vs32status );
} // ReadFromFFD

/*
   Performs READ / MODIFY / WRITE action.
  */

/*!
  * \fn
  *  \brief    Info available in tOsalSystemInfoToFFD is copied into FFD.
  *
  *  \param[in] :  tOsalSystemInfoFromKDS: OSAL_tsDevInfoSystemInfo object.
  *
  *  \return: tS32: Return value depicts if FFD write was successful or not.
  *
 *************************************************************************/
tS32 spm_tclVersionInfo::WriteToFFD( OSAL_tsDevInfoSystemInfo *tOsalSystemInfoToFFD ) const {
   tS32                 Vs32status  = OSAL_ERROR;
   OSAL_tIODescriptor   VhDeviceFFD = OSAL_ERROR;
   OSAL_trFFDDeviceInfo VtsFFDDevInfo;

   /*open the device*/
   VhDeviceFFD = OSAL_IOOpen( OSAL_C_STRING_DEVICE_FFD, OSAL_EN_READWRITE );
   if ( VhDeviceFFD != OSAL_ERROR ){
      tS32 Vs32size      = sizeof( OSAL_tsDevInfoSystemInfo );
      tU8 *pu8FFDRAWdata = OSAL_NULL;
      pu8FFDRAWdata = (tU8*)OSAL_pvMemoryAllocate( (tU32)Vs32size );
      if ( pu8FFDRAWdata ){
         ETG_TRACE_USR4( ( "WriteToFFD() -> Try to WRITE..." ) );
         memcpy( pu8FFDRAWdata, tOsalSystemInfoToFFD, sizeof( OSAL_tsDevInfoSystemInfo ) );
         VtsFFDDevInfo.u8DataSet = SPM_FFD_VERSION_ID;
         VtsFFDDevInfo.pvArg     = (void*)pu8FFDRAWdata;
         // Vs32status = OSAL_s32IORead(VhDeviceFFD,(tPS8)&VtsFFDDevInfo,Vs32size);
         Vs32status              = OSAL_s32IOWrite( VhDeviceFFD, ( tPCS8 ) & VtsFFDDevInfo, Vs32size );
         if ( Vs32status != OSAL_ERROR ){
            ETG_TRACE_USR4( ( "WriteToFFD() -> Version WRITE OK" ) );
            Vs32status = OSAL_s32IOControl( VhDeviceFFD, OSAL_C_S32_IOCTRL_DEV_FFD_SAVENOW, ( intptr_t )&VtsFFDDevInfo );
            Vs32status = OSAL_OK;
            // VtsFFDDevInfo.u8DataSet=SPM_FFD_VERSION_ID;
            // VtsFFDDevInfo.pvArg=NULL;
            // Vs32status = OSAL_s32IOControl(VhDeviceFFD, OSAL_C_S32_IOCTRL_DEV_FFD_RELOAD,(tS32)&VtsFFDDevInfo);
         } else {
            ETG_TRACE_USR4( ( "WriteToFFD() -> Version WRITE ERROR" ) );
         }
            OSAL_vMemoryFree( pu8FFDRAWdata );
      }
      /*close */
      (void)OSAL_s32IOClose( VhDeviceFFD );
   }
   return( Vs32status );
} // WriteToFFD

/*!
  * \fn
  *  \brief    Build Version info is made available as traces.
  *
  *  \param[in] :  void
  *
  *  \return: void
  *
 *************************************************************************/
void spm_tclVersionInfo::TraceBuildversion( ){

   reg_tclRegKey oReg;

   std::string   strBUILDVERSION_LABEL( 40, 0 );
   std::string   strBUILDVERSION_BUILDUSER( 15, 0 );
   std::string   strBUILDVERSION_BUILDMACHINE( 20, 0 );
   std::string   strBUILDVERSION_TIMESTAMP( 15, 0 );
   std::string   strBUILDVERSION_BUILDTYPE( 4, 0 );
   std::string   strBUILDVERSION_CUSTVERSTRING( 40, 0 );
   std::string   output;

   if ( oReg.bOpen( SPM_REG_VERSIONS_BASE_PATH ) ){
      ETG_TRACE_FATAL( ( "\n ===================================================" ) );
      if ( oReg.bQueryString( "BUILDVERSION_LABEL",
                              &strBUILDVERSION_LABEL[0],
                              (tU32)strBUILDVERSION_LABEL.length( ) ) ){
         output += "\n   Label           = ";
         output += strBUILDVERSION_LABEL.c_str( );
      }
      if ( oReg.bQueryString( "BUILDVERSION_BUILDUSER",
                              &strBUILDVERSION_BUILDUSER[0],
                              (tU32)strBUILDVERSION_BUILDUSER.length( ) ) ){
         output += "\n   Build User      = ";
         output += strBUILDVERSION_BUILDUSER.c_str( );
      }
      if ( oReg.bQueryString( "BUILDVERSION_BUILDMACHINE",
                              &strBUILDVERSION_BUILDMACHINE[0],
                              (tU32)strBUILDVERSION_BUILDMACHINE.length( ) ) ){
         output += "\n   Build machine   = ";
         output += strBUILDVERSION_BUILDMACHINE.c_str( );
      }
      if ( oReg.bQueryString( "BUILDVERSION_TIMESTAMP",
                              &strBUILDVERSION_TIMESTAMP[0],
                              (tU32)strBUILDVERSION_TIMESTAMP.length( ) ) ){
         output += "\n   Timestamp       = ";
         output += strBUILDVERSION_TIMESTAMP.c_str( );
      }
      if ( oReg.bQueryString( "BUILDVERSION_BUILDTYPE",
                              &strBUILDVERSION_BUILDTYPE[0],
                              (tU32)strBUILDVERSION_BUILDTYPE.length( ) ) ){
         output += "\n   Build type      = ";
         output += strBUILDVERSION_BUILDTYPE.c_str( );
      }
      if ( oReg.bQueryString( "BUILDVERSION_CUSTVERSTRING",
                              &strBUILDVERSION_CUSTVERSTRING[0],
                              (tU32)strBUILDVERSION_CUSTVERSTRING.length( ) ) ){
         output += "\n   Customer string = ";
         output += strBUILDVERSION_CUSTVERSTRING.c_str( );
      }
      ETG_TRACE_FATAL( ( "%s", output.c_str( ) ) );
      ETG_TRACE_FATAL( ( "\n ===================================================" ) );

      etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM, ETG_LEVEL_FATAL ), 0, 0, SPM_ETG_FORMAT_VALUE_EN_T16, I_SPM_BUILDVERSION );
   }
   {
      dp_tclregVersionsBoardCfgName oBoardName;
      std::string                   strBoardN( VERSION_STRING_LEN, 0 );
      if ( DP_U8_ELEM_STATUS_INITVAL != oBoardName.u8GetData( &strBoardN[0], (tU32)strBoardN.length( ) ) ){
         ETG_TRACE_USR4( ( "%s", strBoardN.c_str( ) ) );
      }
   }
} // TraceBuildversion

/*!
  * \fn
  *  \brief    Read KDS data.
  *
  *  \param[in] :  u32Entry       : Entry to be read from KDS.
  *  \param[in] :  u32datalength  : Length of data that needs to be read.
  *  \param[in] :  ustrKDSReadData: Data read is stored into
  *
  *  \return: tBool : KDS read successful or not
  *
 *************************************************************************/
tBool spm_tclVersionInfo::bReadKdsEntry( tU32         u32Entry,
                                         tU32         u32datalength,
                                         tLcmUString *ustrKDSReadData ) const {
   tBool              bReadKds       = FALSE;
   tS32               s32ReturnValue = OSAL_ERROR;
   OSAL_tIODescriptor tIoKdsHandle   = OSAL_ERROR;
   tsKDSEntry         sKDSEntryData;

   tIoKdsHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE );

   if ( OSAL_ERROR != tIoKdsHandle ){
      if ( u32datalength > KDS_MAX_ENTRY_LENGTH ){
         u32datalength = KDS_MAX_ENTRY_LENGTH;
         // Read Data from KDS
      }
      sKDSEntryData.u16Entry        = (tU16)u32Entry;
      sKDSEntryData.u16EntryLength  = (tU16)u32datalength;   // KDS_MAX_ENTRY_LENGTH
      sKDSEntryData.u16EntryFlags   = M_KDS_ENTRY_FLAG_NONE; // M_KDS_ENTRY_FLAG_WRITE_PROTECTED
      sKDSEntryData.au8EntryData[0] = 0;

      if ( OSAL_ERROR != ( s32ReturnValue = OSAL_s32IORead( tIoKdsHandle, ( tPS8 ) & sKDSEntryData, ( sizeof( sKDSEntryData ) ) ) ) ){
         bReadKds = TRUE;
         ustrKDSReadData->assign( sKDSEntryData.au8EntryData, u32datalength - 1 );
      } else {
         s32ReturnValue = OSAL_u32ErrorCode( );
         ETG_TRACE_USR4( ( "kds read not ok. Error = %d (%s)"
                           , s32ReturnValue
                           , OSAL_coszErrorText( s32ReturnValue ) ) );
      }

      s32ReturnValue = OSAL_s32IOClose( tIoKdsHandle );
      // Close KDS
      if ( OSAL_ERROR == s32ReturnValue ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Closing KDS value failed !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   } else {
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Opening KDS device failed !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   return( bReadKds );
} // bReadKdsEntry

/*!
  * \fn
  *  \brief    Write KDS data.
  *
  *  \param[in] :  key             : Entry value.
  *  \param[in] :  ustrKDSWriteData:Data to be written into KDS.
  *  \param[in] :  length          : Length of the data that is written into KDS
  *
  *  \return: tBool : KDS write successful or not.
  *
 *************************************************************************/
tBool spm_tclVersionInfo::bWriteKdsEntry( tU16        key,
                                          tLcmUString ustrKDSWriteData,
                                          tU16        length ) const {

   OSAL_tIODescriptor tIoKdsHandle = OSAL_ERROR;
   tsKDSEntry         sKDSEntryData;
   tBool              bReadKds     = FALSE;

   tIoKdsHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READWRITE );

   if ( OSAL_ERROR != tIoKdsHandle ){
      if ( length > KDS_MAX_ENTRY_LENGTH ){
         length = KDS_MAX_ENTRY_LENGTH;
      }
      if ( OSAL_OK == OSAL_s32IOControl( tIoKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_ENABLE, ( intptr_t )TRUE ) ){

         sKDSEntryData.u16Entry       = key;
         sKDSEntryData.u16EntryLength = length;
         sKDSEntryData.u16EntryFlags  = M_KDS_ENTRY_FLAG_NONE;
         memcpy( sKDSEntryData.au8EntryData, ustrKDSWriteData.c_str( ), length );

         if ( OSAL_s32IOWrite( tIoKdsHandle, ( tPCS8 ) & sKDSEntryData, ( sizeof( sKDSEntryData ) ) ) != OSAL_ERROR ){
            bReadKds = TRUE;
         } else {
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
         }
         if ( OSAL_OK != OSAL_s32IOControl( tIoKdsHandle, OSAL_C_S32_IOCTRL_KDS_WRITE_BACK, ( intptr_t )TRUE ) ){
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
            // Close KDS
         }
         if ( OSAL_ERROR == OSAL_s32IOClose( tIoKdsHandle ) ){
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
         }
      } else {
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
      }
   } else {
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!" ) );
   }
   return( bReadKds );
} // bWriteKdsEntry

/*!
  * \fn
  *  \brief    Function checks if 'errmemflag.dnl' exists in FFS else
  *            creates it. Updates the version info in Errmem.
  *
  *  \param[in] : void
  *
  *  \return: tVoid
  *
 *************************************************************************/
tVoid spm_tclVersionInfo::vUpdateErrmemVersion( ){
   TVersionStorage tEmTraceStore      = { 0,{0},{0},{0} };
   tBool           bStore             = FALSE;
   tU32            u32StartTimeEmRead = OSAL_ClockGetElapsedTime( );

   if ( _u32NextEmTraceCheck < u32StartTimeEmRead ){
      dp_tclSpmDpInternDataEmTraceVers oEmVers;
      oEmVers >> tEmTraceStore;

      ETG_TRACE_USR4( ( "SPM : EMTRACE check: Is 'errmemflag.dnl' file in FFS?." ) );
      OSAL_tIODescriptor               fd = OSAL_IOOpen( "/dev/root/var/opt/bosch/persistent/errmemflag.dnl", OSAL_EN_READONLY );
      if ( fd == OSAL_ERROR ){
            ETG_TRACE_USR1( ( "SPM : EMTRACE check: No 'errmemflag.dnl' file found -> store version info." ) );
         fd = OSAL_IOCreate( "/dev/root/var/opt/bosch/persistent/errmemflag.dnl", OSAL_EN_READWRITE );
         if ( fd != OSAL_ERROR ){
            ETG_TRACE_USR1( ( "SPM : EMTRACE check: 'errmemflag.dnl' created." ) );
            bStore = TRUE;
            if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
            }
         } else {
            ETG_TRACE_USR1( ( "SPM : EMTRACE check: Could not create 'errmemflag.dnl' file." ) );
         }
      } else {
         (tVoid)OSAL_s32IOClose( fd );

         if ( !_bVersionChecked ){
            std::string                           strVersionReg( VERSION_STRING_LEN, 0 );
            _bVersionChecked = TRUE;

            dp_tclregVersionsBoardCfgBuildVersion oBuildVer;
            oBuildVer.u8GetData( &strVersionReg[0], (tU32)strVersionReg.length( ) );

            ETG_TRACE_USR4( ( "SPM : EMTRACE check: compare main versions: '%80s' '%80s'.", tEmTraceStore.strMainVers, strVersionReg.c_str( ) ) );
            if ( 0 != OSAL_s32StringCompare( tEmTraceStore.strMainVers, strVersionReg.c_str( ) ) ){
               bStore = TRUE;
               memset( tEmTraceStore.strMainVers, 0, SPM_DP_MAIN_VERS_SIZE );
               OSAL_szStringNCopy( tEmTraceStore.strMainVers, strVersionReg.c_str( ), SPM_DP_MAIN_VERS_SIZE - 1 );
            }


            strVersionReg.assign( VERSION_STRING_LEN, 0 );
            dp_tclregVersionsAdrSwVersion oAdrVer;
            oAdrVer.u8GetData( &strVersionReg[0], (tU32)strVersionReg.length( ) );

            ETG_TRACE_USR4( ( "SPM : EMTRACE check: compare ADR versions: '%80s' '%80s'.", tEmTraceStore.strArdVers, strVersionReg.c_str( ) ) );
            if ( 0 != OSAL_s32StringCompare( tEmTraceStore.strArdVers, strVersionReg.c_str( ) ) ){
               bStore = TRUE;
               memset( tEmTraceStore.strArdVers, 0, SPM_DP_ADR_VERS_SIZE - 1 );
               OSAL_szStringNCopy( tEmTraceStore.strArdVers, strVersionReg.c_str( ), SPM_DP_ADR_VERS_SIZE - 1 );
            }

            #ifdef VARIANT_S_FTR_ENABLE_CPLD

               strVersionReg.assign( VERSION_STRING_LEN, 0 );
               dp_tclregVersionsCpldVer oCpldVer;
               oCpldVer.u8GetData( &strVersionReg[0], (tU32)strVersionReg.length( ) );

               ETG_TRACE_USR4( ( "SPM : EMTRACE check: compare CPLD versions: '%20s' '%20s'.", tEmTraceStore.strCpldVers, strVersionReg ) );
               if ( 0 != OSAL_s32MemoryCompare( tEmTraceStore.strCpldVers, strVersionReg.c_str( ), 8 ) ){ // compare only the first 8 char --> version number of CPLD
                  bStore = TRUE;
                  memset( tEmTraceStore.strCpldVers, 0, SPM_DP_CPLD_VERS_SIZE );
                  OSAL_szStringNCopy( tEmTraceStore.strCpldVers, strVersionReg.c_str( ), SPM_DP_CPLD_VERS_SIZE - 1 );
               }
            #endif
         }
      }
      if ( bStore ){
         std::string strVersionReg( VERSION_STRING_LEN, 0 );

         vStoreVersionInfoInEmTrace( );
         SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
         poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

         oEmVers << tEmTraceStore;

         ETG_TRACE_USR1( ( "SPM : EMTRACE version '%80s' stored with index %d.", strVersionReg.c_str( ), tEmTraceStore.u32EmTraceCount ) );
      }

      tU32 u32EndCheck = OSAL_ClockGetElapsedTime( );
      _u32NextEmTraceCheck = u32EndCheck + 10000;

      ETG_TRACE_USR4( ( "SPM : EMTRACE version check duration %dms (next check at TS %dms).", ( u32EndCheck - u32StartTimeEmRead ), _u32NextEmTraceCheck ) );
   }

   vAdditionalCheck( );
} // vUpdateErrmemVersion

/*!
  * \fn
  *  \brief    could be overwritten by project to add additional infos to errmem.
  *
  *  \param[in] : void
  *
  *  \return: tVoid
  *
 *************************************************************************/
tVoid spm_tclVersionInfo::vStoreProjectVersionInfoInEmTrace( ){

}

