/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SuperVisionManager.h
  * @brief        This is header file contains functions for SuperVisionManager.
  * @copyright    (C) 2012 - 2016 Robert Bosch Engineering and Business Solutions Limited.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include <set>

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------
#include "spm_ISuperVisionManager.h"

class ISpmSubStateClient;
class ISpmSystemPowerManager;
class ISpmWorkerServer;
class ISpmOsalProxy;
class ISpmLocalAppManager;
class ISpmSupervisionClient;

/*!
  *  \class spm_tclSupervisionManager
  *  \brief SupervisionManager is responsible monitoring CCA heartbeat and also Software and hardware Watchdogs(if enabled).
          It keeps the counters for CCA and Heartbeat ticking which are reset by CCAMessageHandler as long as SPM messagebox is active.
  */

class spm_tclSupervisionManager : public ISpmSupervisionManager
{
protected:
tU32                                _u32MakeTrackerRunning;
tU32                                _u32WaitInterval;
tBool                               _bEnableTriggerHwWdt;
tBool                               _bTimeTrackerRunning;
tBool                               _bWdtGetpEnabled;

ISpmSystemPowerManager             *_poclSystemPowerManager;
ISpmSubStateClient                 *_poclSubStateHandler;
ISpmWorkerServer                   *_poclWorkerServer;
ISpmOsalProxy                      *_poclOsalProxy;
ISpmLocalAppManager                *_poclLocalAppManager;

static tVoid vHeartBeatTimerCallback( tVoid *pArg );

OSAL_tTimerHandle                   _hHeartBeatTimer;

tU32                                _u32PreWarnTime;
tU32                                _u32CCAHeartBeatCount;
tU32                                _u32WdtCCAHeartBeatTime;

tU32                                _u32LastSupervisionCall;

tU32                                _u32TimeStampCCAHeartBeatTimerRetriggerd;
tU32                                _u32TimeStampCCAHeartBeatTimerChecked;

std::set < ISpmSupervisionClient* > tClientHdlList;

protected:
virtual tVoid vCheckCCAHeartBeat( );

virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

public:
spm_tclSupervisionManager( const ISpmFactory& factory,
                           tU32               u32WaitInterval );
virtual ~spm_tclSupervisionManager( );

virtual tVoid main( );

virtual tVoid vOnStart( );

virtual tVoid vOnTerminate( );

virtual tVoid vSetPrewarnTime( tU32 u32PreWarnTime ){
   _u32PreWarnTime = u32PreWarnTime;
}

virtual tVoid vSetCCAHeartBeatInterval( tU32 u32WdtCCAHeartBeatTime ){
   _u32WdtCCAHeartBeatTime = u32WdtCCAHeartBeatTime;
}

virtual tVoid vSetGetpEnabled( tBool bEnabled = TRUE ){ _bWdtGetpEnabled = bEnabled; }

// trace handling
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual tVoid vTerminate( ) const {}

virtual tVoid vEnableRetriggerHwWdt( tBool bEnableTrigger ){
   _bEnableTriggerHwWdt = bEnableTrigger;
}

virtual tVoid vStartUpFinished( );

virtual tVoid vTraceInfo( );

virtual tVoid vWriteStringToErrmem( std::string pszString );

virtual tVoid vTriggerCCAHeartBeat( ){
   _u32CCAHeartBeatCount                    = 0;
   _u32TimeStampCCAHeartBeatTimerRetriggerd = OSAL_ClockGetElapsedTime( );
}

virtual tBool bIsCCAHeartBeatWarnTime( ) const {
   if ( ( _u32WdtCCAHeartBeatTime > 0 ) && ( _u32CCAHeartBeatCount == _u32WdtCCAHeartBeatTime ) ){
      return( TRUE );
   }
   return( FALSE );
}

virtual tVoid vNewCCAHeartBeatInterval( ){ ++_u32CCAHeartBeatCount; }

virtual tVoid vDisableWdtRetriggering( tVoid );

virtual tVoid vAddSupervisionClient( ISpmSupervisionClient *client );

virtual tVoid vRemoveSupervisionClient( ISpmSupervisionClient *client );

protected:
// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 /*u32Message*/,
                              tU32 /*u32Parm = 0*/ ){} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used
virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );

virtual const tChar*getName( ) const { return( "spm_tclSupervisionManager" ); }
};

