/*!
  * \file spm_ProcessSupervision.cpp
  *  \brief
  *    Implementation of Process Supervision on SPM Started Processes and Additional Set of Processes.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 | 14.03.2013 | Petroglou Johannis (BSOT/ENG), Bipin Krishnan(RBEI/ECG4) | initial Version \n
  * 1.1 | 28.03.2013 | Bipin Krishnan(RBEI/ECG4)                                | Signal Handler for SIGCHLD to supervise SPM Started Processes.
  ******
  */
#include "spm_Config.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"


#include <map>
#include <list>
#include <string>
#include <stdlib.h>
#include <sstream>

#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <errno.h>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_ProcessSupervision.cpp.trc.h"
#endif

#include "spm_GlobDefs.h"
#include "spm_Config.h"
#include "spm_ProcessSupervision.h"
#include "spm_StringTokenizer.h"
#include "spm_ProcessDatabase.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationErrorHandler.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_rootdaemon.h"
#include "spm_OsLinux.h"

#include "dirent.h"
#include "spm_IFactory.h"
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define SPM_REG_KEY_SUPERVISION                 "SUPERVISION"
#define SPM_REG_KEY_SUPERVISION_EXTERNAL_PROCESS_LIST "EXTERNAL_PROCESS_LIST"
#define SPM_REG_KEY_SUPERVISION_WRITE_TO_ERRMEM       "PROC_SV_WRITE_TO_ERRMEM"

spm_tclProcessSupervision*spm_tclProcessSupervision::_pMyStaticRef     = NULL;
ISpmWorkerServer         *spm_tclProcessSupervision::_poclWorkerServer = NULL;

spm_tclProcessSupervision::spm_tclProcessSupervision( const ISpmFactory& factory ) : ISpmProcessSupervision( factory ),
   _poProcessDatabase( NULL ),
   _poclSystemPowerManager( NULL ),
   _hProcessSupervisionSem( OSAL_C_INVALID_HANDLE ),
   _hPrcSupervisionStartSem( OSAL_C_INVALID_HANDLE ),
   hDevice( OSAL_ERROR ),
   u32Data( 0 ),
   _bProcSupervisionWriteToErrmem( TRUE ),
   _bStopProcessSupervision( TRUE ),
   _bProcSuperVisionStopped( FALSE ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in] factory: spm factory object.
  *  \todo
  *    Integration of SIGNAL HANDLER(SIGCHLD) for Process Supervision.\n
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */
   _pMyStaticRef = this;

   if ( OSAL_s32SemaphoreCreate( "LcmProcSv", &_hProcessSupervisionSem, (tU32)1 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessSupervision !!!!!! Error detected !!!!!! cannot create Semaphore _hProcessSupervisionSem: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   if ( OSAL_s32SemaphoreCreate( "PrcSvStart", &_hPrcSupervisionStartSem, (tU32)0 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessSupervision !!!!!! Error detected !!!!!! cannot create Semaphore _hPrcSupervisionStartSem: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclProcessSupervision::~spm_tclProcessSupervision( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  ******
  */
   if ( _hProcessSupervisionSem != OSAL_C_INVALID_HANDLE ){
      OSAL_s32SemaphoreClose( _hProcessSupervisionSem );
      OSAL_s32SemaphoreDelete( "LcmProcSv" );
   }

   if ( _hPrcSupervisionStartSem != OSAL_C_INVALID_HANDLE ){
      OSAL_s32SemaphoreClose( _hPrcSupervisionStartSem );
      OSAL_s32SemaphoreDelete( "PrcSvStart" );
   }

   if ( _poclWorkerServer != NULL ){
      _poclWorkerServer->vRemoveClient( this );
   }
   _poclWorkerServer       = NULL;
   _poProcessDatabase      = NULL;
   _poclSystemPowerManager = NULL;

   if ( hDevice != OSAL_ERROR ){
      (void)OSAL_s32IOClose( hDevice );
   }
   _pMyStaticRef           = NULL;


}

tVoid spm_tclProcessSupervision::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclProcessSupervision.
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer, ISpmWorkerServer );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poProcessDatabase, spm_tclProcessDatabase, ISpmProcessDatabase );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
}

tVoid spm_tclProcessSupervision::vStartCommunication( ){
}

tVoid spm_tclProcessSupervision::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  *   This method adds the instance(this) of spm_tclProcessSupervision as the client of spm_tclWorkerServer.
  ******
  */
   _poclWorkerServer->vAddClient( this );
   //PRMIMPL(OSAL PRM Implementation to get the notifications for the child processes getting killed)
   OSAL_trNotifyDataSystem rNotifyData;
   hDevice = OSAL_IOOpen( OSAL_C_STRING_DEVICE_PRM, OSAL_EN_READWRITE );
   if ( hDevice != OSAL_ERROR ){
      ETG_TRACE_USR4( ( "spm_tclProcessSupervision::spm_tclProcessSupervision OSAL_IOOpen Success" ) );
      rNotifyData.u16AppID  = CCA_C_U16_APP_SPM;
      rNotifyData.pCallback = ( OSALCALLBACKFUNC )spm_tclProcessSupervision::vNotifyHandler;
      rNotifyData.pu32Data  = &u32Data;

      if ( OSAL_s32IOControl( hDevice,
                              OSAL_C_S32_IOCTRL_PRM_REG_SYSTEM_STAT_INFO,
                              ( intptr_t )&rNotifyData ) == OSAL_ERROR ){
         ETG_TRACE_FATAL( ( "PRM OSAL_C_S32_IOCTRL_PRM_REG_SYSTEM_STAT_INFO failed" ) );
      }
   }
} // vStartCommunication

tVoid spm_tclProcessSupervision::vCheckForExternalProcessesToSupervise( ){
/*!
  * \fn
  *  \brief
  *     Reads the process list(only additional processes) from Registry and update the map of Additional Processes.
  *  \todo
  *     Testing Process Supervision on Additional Set of Processes.
  ******
  */
   std::string szProcessList( 2048, 0 );

   if ( !scd_bGetAppConfigurationValue( CCA_C_U16_APP_SPM,                       // CCA AppID
                                        SPM_REG_KEY_SUPERVISION,                 // Key name
                                        SPM_REG_KEY_SUPERVISION_WRITE_TO_ERRMEM, // Value name
                                        &_bProcSupervisionWriteToErrmem )
        ){
      _bProcSupervisionWriteToErrmem = FALSE;
      /* Get the path for the script file to execute (stored in regkey "SCRIPT_REGREADY") */
   }
   if ( !scd_bGetAppConfigurationString( CCA_C_U16_APP_SPM, // CCA AppID
                                         SPM_REG_KEY_SUPERVISION,
                                         SPM_REG_KEY_SUPERVISION_EXTERNAL_PROCESS_LIST,
                                         &szProcessList[0], (tU32)szProcessList.length( ) ) ){
      ETG_TRACE_USR4( ( "vCheckForExternalProcessesToSupervise() -> No registry key found. Exit." ) );
   } else {
      ETG_TRACE_USR4( ( "vCheckForExternalProcessesToSupervise() Found processlist in registry: %s", szProcessList.c_str( ) ) );
      spm_tclStringTokenizer st( szProcessList, ", ", FALSE );
      while ( st.bHasMoreTokens( ) ){
         std::string s = st.oNextToken( );
         if ( s == " " ){
            continue;
         }
         ETG_TRACE_USR4( ( "spm_tclProcessSupervision: Adding additional process %s ", s.c_str( ) ) );
         if ( bAddAdditionalProzessToSupervise( s ) == FALSE ){
            std::string strError;

            strError = "External started process not found after startup-finished! process=" + s;
            ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
            _poclWorkerServer->bPostMessage( "ISpmProcessSupervision", SPM_U32_WORKER_PRC_SIGCHLD, 0 );
         }
      }
   }
} // vCheckForExternalProcessesToSupervise

tBool spm_tclProcessSupervision::bAddAdditionalProzessToSupervise( const std::string& strProcessName ){
/*!
  * \fn
  *  \brief
  *     Adds a process in the map of Additional Processes.
  *  \param[in]  strProcessName:  Name of the process to supervise.
  *
  *  \return     TRUE if successfully adds the process to supervision.
  *
  *  \todo
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */
   tBool                 bFound = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK_VAL( _poProcessDatabase );

   TProcessConfiguration processConfig;
   TPidVector            myPidList;
   TPidVector::iterator  it;

   tInt                  iPidCnt               = iGetProcessIdList( strProcessName, myPidList );
   OSAL_tMSecond         tMUnitListRequestTime = { 0 };

   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclAppErrHandler, ISpmApplicationErrorHandler );

   if ( iPidCnt == 0 ){
      std::string strError;
      ETG_TRACE_ERR( ( "bAddAdditionalProzessToSupervise(): process %s not found!", strProcessName.c_str( ) ) );

      // OSALUTIL_s32SaveNPrintFormat(strError, sizeof(strError), "Add. proc. missing info #1: ShutdownTrigger received=%d ShutdownFinishedAckSend=%d", _bShutdownTriggerReceived,
      // _bShutdownFinishedAckSend);
      // ETG_TRACE_FATAL(("%s", strError.c_str()));
      // vWriteErrmem(U16_M_ERRMEM_SPM_ERROR(SPM_U8_ERRMEM_TYPE_STRING), (tU8*)strError, (tU16)strError.length());
      // OSALUTIL_s32SaveNPrintFormat(strError, sizeof(strError), "Add. proc. missing info #2:ShutdownReverseTrigger received=%d ShutdownReversedFinishedAckSend=%d.",
      // _bShutdownReversedTriggerReceived, _bShutdownReversedAckSendRunning);
      // ETG_TRACE_FATAL(("%s", strError.c_str()));
      // vWriteErrmem(U16_M_ERRMEM_SPM_ERROR(SPM_U8_ERRMEM_TYPE_STRING), (tU8*)strError, (tU16)strError.length());

      strError = "Looks like a additional process was never started (process is missing after startup, no handle available):";
      ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclStartupInvest, ISpmStartupInvestigationServer );
      poclStartupInvest->vDump( );
      {
         strError = "Triggering unit fail list for process missing failure (Time=" + std::to_string( OSAL_ClockGetElapsedTime( ) ) + ")! (processname=" + strProcessName;
         ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

         poclAppErrHandler->vSetWriteEmptyUnitFailedList( true );
         poclAppErrHandler->vGetUnitFailedList( );
         tMUnitListRequestTime = OSAL_ClockGetElapsedTime( );
      }

      SPM_ROOTDAEMON_CALLER_rPerformRootOp(
         "lcm",
         GET_CURRENT_PROCESSES,
         "" );

      OSAL_s32ThreadWait( 1000 );
      if ( !_bProcSuperVisionStopped ){
         tBool bUnitResultListReceived = FALSE;
         /* Wait up to 3s for dbus unit list result*/
         int   iWaitCnt                = 30;
         while ( iWaitCnt > 0 ){
            if ( poclAppErrHandler->fUnitFailedListResultReceived( ) == TRUE ){
               bUnitResultListReceived = TRUE;
               break; // End while loop. Unitlist was received and written to errmem
            } else {
               ETG_TRACE_USR4( ( "Waiting for UNIT-FAILED-LIST result... (RetryCnt=%d, Time=%d)", iWaitCnt, OSAL_ClockGetElapsedTime( ) ) );
            }
            iWaitCnt--;
            OSAL_s32ThreadWait( 100 );
         }

         if ( bUnitResultListReceived == FALSE ){
            strError = "SPM: spm_tclProcessSupervision: No failed unit service list was received! (Triggered=" + std::to_string( tMUnitListRequestTime ) + ", CurrTime=" + std::to_string( OSAL_ClockGetElapsedTime( ) );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
            poclAppErrHandler->vSetWriteEmptyUnitFailedList( true );
         }
         if ( _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_PROCESS_SUPERVISION ) == FALSE ){

            SPM_GET_IF_REFERENCE_USE_VAR_VAL( _poclSystemPowerManager, ISpmSystemPowerManager );

            strError = "SPM: spm_tclProcessSupervision: Supervision reset request cant be posted. Using direct function call instead.";
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
            _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION );
            OSAL_s32ThreadWait( 3000 );
            strError = "SPM: spm_tclProcessSupervision: Supervision reset request via direct funtion call has no effect, using FATAL_ASSERT now.";
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
            FATAL_M_ASSERT_ALWAYS( );
         }
      } else {
         strError = "SPM: spm_tclProcessSupervision: Supervision was stopped.";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
      }
   } else {
      bFound = TRUE;
      for ( it = myPidList.begin( ); it != myPidList.end( ); ++it ){
         ETG_TRACE_USR4( ( "Name=%s", strProcessName.c_str( ) ) );
         ETG_TRACE_USR4( ( "PID=%u", * it ) );

         processConfig.handle          = * it;
         processConfig.instance        = 0;
         processConfig.strProcessName  = strProcessName;
         processConfig.ProcPrio        = 0;
         processConfig.Affinity        = 0;
         processConfig.AffinityEnd     = 0;
         processConfig.NiceLevel       = 0;
         processConfig.NiceLevelEnd    = 0;
         processConfig.CGroupMemLevel1 = 0;
         processConfig.CGroupMemLevel2 = 0;
         processConfig.startTimeStamp  = 0;
         processConfig.isMissing       = FALSE;

         // _processDB->_oAdditionalProcesses[*it] = processConfig;
         _poProcessDatabase->bUpdateMapAdditionalProc( processConfig );
      }
   }

   return( bFound );
} // bAddAdditionalProzessToSupervise

tVoid spm_tclProcessSupervision::vRememberAdditionalProzessToSupervise( const std::string& strProcessName,
                                                                        tU32               u32SupervisionMode ){
/*!
  * \fn
  *  \brief
  *     vRememberAdditionalProzessToSupervise
  *  \param[in]    strProcessName name of process
  *  \todo
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */

   TProcessSupervisionConfiguration svCfg;

   SPM_NULL_POINTER_CHECK( _poProcessDatabase );

   svCfg.supervisionMode = u32SupervisionMode;
   svCfg.procName        = strProcessName;
   // _poProcessDatabase->vecGetExternalProcesses().push_back(svCfg);

   _poProcessDatabase->bUpdateExternalProcess( svCfg );
} // vRememberAdditionalProzessToSupervise

tVoid spm_tclProcessSupervision::vActivateAdditionalProcessSupervision( tVoid ){
/*!
  * \fn
  *  \brief
  *     Copys the list of external started processes to supervise from the standby list
  *              to the real/active on which the supervision is using.
  *  \todo
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poProcessDatabase );

   std::vector < TProcessSupervisionConfiguration >::const_iterator it;
   // for (it = (_processDB->_vecExternalProcessesList).begin(); it != (_processDB->_vecExternalProcessesList).end(); ++it)
   for ( it = ( _poProcessDatabase->vecGetExternalProcesses( ) ).begin( ); it != ( _poProcessDatabase->vecGetExternalProcesses( ) ).end( ); ++it ){
      bAddAdditionalProzessToSupervise( ( * it ).procName );
   }
} // vActivateAdditionalProcessSupervision

tBool spm_tclProcessSupervision::bCheckIfProcessAlive( pid_t tPid ) const

/*!
  * \fn
  *  \brief
  *     Check the linux /proc filesystem if a entry with this given pid
  *              exists.
  *  \param[in]
  *     tPid: Process ID of the process to check.
  ******
  */
{
   std::string       name;
   OSAL_trIOCtrlDir *hDir = NULL;

   name = "/proc/" + std::to_string( tPid );

   hDir = OSALUTIL_prOpenDir( name.c_str( ) );
   if ( hDir == OSAL_NULL ){
      // ETG_TRACE_ERR(("Process %s with PID=%d is missing!",pidName,tPid));
      return( FALSE );
   }
   OSALUTIL_s32CloseDir( hDir );
   return( TRUE );
} // bCheckIfProcessAlive

tInt spm_tclProcessSupervision::iGetProcessIdList( const std::string& strPidName,
                                                   TPidVector       & procPidList ) const

/*!
  * \fn
  *  \brief
  *     gets a list of handle for the the given task/process name
  *  \param[in] pidName: Process Name.
  *  \param[out] procPidList: List of Handle(Process ID list).
  *  \return number of processes found
  *  \note
  *     This function searches the linux /proc filesystem for the give process name,
  *     and returns the found handle(s) via procPidList.
  ******
  */
{
   tInt        iPidCnt = 0;

   std::string strFilename;
   std::string strBuffer;
   std::string strName( READ_BUF_SIZE, 0 );

   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poOsLinux, ISpmOsLinux );

   if ( !strPidName.empty( ) ){
      OSAL_trIOCtrlDir    *hDir         = NULL;
      OSAL_trIOCtrlDirent *nextDirEntry = NULL;

      hDir = OSALUTIL_prOpenDir( ROOT_PATH_PROC );

      if ( hDir == OSAL_NULL ){
         ETG_TRACE_ERR( ( "spm_tclProcessSupervision: Cannot open /proc" ) );
         return( 0 );
      }

      nextDirEntry = OSALUTIL_prReadDir( hDir );

      while ( nextDirEntry != OSAL_NULL ){
         // Skip path outside /proc
         if ( 0 == OSAL_s32StringCompare( nextDirEntry->s8Name, OUTSIDE_ROOT_PATH_PROC ) ){
            // Get next dir enty
            nextDirEntry = OSALUTIL_prReadDir( hDir );
            continue;
         }

         // Break here is this is not a number
         if ( !OSAL_s32IsDigit( nextDirEntry->s8Name[0] ) ){
            // Get next dir enty
            nextDirEntry = OSALUTIL_prReadDir( hDir );
            continue;
         }

         strFilename = "/proc/" + std::string( (const char*)nextDirEntry->s8Name ) + "/status";

         if ( poOsLinux->s32ReadFileContent( strFilename, strBuffer, READ_BUF_SIZE - 1 ) <= 0 ){
            // Get next dir enty
            nextDirEntry = OSALUTIL_prReadDir( hDir );
            continue;
         }

         OSAL_s32ScanFormat( strBuffer.c_str( ), PATH_PROC_NAME, &strName[0] );

         if ( !OSAL_s32StringNCompare( strName.c_str( ), strPidName.c_str( ), 15 ) ){
            procPidList.push_back( (pid_t)( strtol( (const char*)nextDirEntry->s8Name, NULL, 0 ) ) );
            iPidCnt++;
         }
         // Get next dir enty
         nextDirEntry = OSALUTIL_prReadDir( hDir );
      } // while
      OSALUTIL_s32CloseDir( hDir );
   }

   return( iPidCnt );
} // iGetProcessIdList

tVoid spm_tclProcessSupervision::vDumpOOMInfo( ) const

/*!
  * \fn
  *  \brief
  *     vDumpOOMInfo
  ******
  */
{
   std::string strError;
   CmdData     rCmdData = { 0, 0, 0, 0, 0 };

   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
   strError = "OOM-TEST: ------------START------------";
   _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

   rCmdData = SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", GET_OOM_INFO, "" );

   if ( atoi( rCmdData.message ) < 0 ){
      strError = (std::string)"OOM-TEST: SPM_ROOTDAEMON_CALLER_rPerformRootOp(GET_OOM_INFO) failed with error = '" + rCmdData.message + "'";
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
   }

   strError = "OOM-TEST: ------------END------------";
   _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
} // vDumpOOMInfo

tBool spm_tclProcessSupervision::bDoRAMtest( tS32 lNumberOfBytes,
                                             tS32 lStride ) const

/*!
  * \fn
  *  \brief
  *     bDoRAMtest
  *
  *  \param[in] lNumberOfBytes: Number of bytes.
  *  \param[in] lStride:        stride size.
  ******
  */
{
   std::string strError;
   tBool       bSuccess = FALSE;
   tChar      *pBuffer  = 0;

   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );

   strError = "RAMtest start: Try to allocate " + std::to_string( lNumberOfBytes ) + " bytes...";

   // ETG_TRACE_FATAL(("%s", strError));
   _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

   // if (!(pBuffer = (char*)malloc (lNumberOfBytes * sizeof(char))))
   pBuffer = (tChar*)malloc( lNumberOfBytes * sizeof( tChar ) );
   if ( 0 == pBuffer ){
      strError = "RAMtest error: Allocation of " + std::to_string( lNumberOfBytes ) + " bytes failed!";
      // ETG_TRACE_FATAL(("%s", strError));
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
   } else {
      strError = "RAMtest action: Allocated " + std::to_string( lNumberOfBytes ) + " bytes. Starting write access (stride=" + std::to_string( lStride ) + ") to allocated memory.";
      // ETG_TRACE_FATAL(("%s", strError));
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
      for ( long long i = 0; i < lNumberOfBytes; i += lStride ){
         pBuffer[i] = 'B';
      }
      std::stringstream strStream;
      strStream << std::hex << (intptr_t)pBuffer;
      strError = "RAMtest end: Write access to ptr=0x" + strStream.str( ) + ", size=" + std::to_string( lNumberOfBytes ) + ", stride=" + std::to_string( lStride ) + " was sucessfull. ";
      // pBuffer should be casted to void* to print address otherwise content of string will be printed.
      free( pBuffer );

      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

      bSuccess = TRUE;
   }

   return( bSuccess );
} // bDoRAMtest

tBool spm_tclProcessSupervision::bGetProcessHandleByName( const std::string& strProcessName,
                                                          tU32              *pu32handle ){
/*!
  * \fn
  *  \brief
  *     Returns the handle (pid) of the given processname.
  *     This function searches the internal list of (spm) started process for the
  *     given process name and returns the handle we have got from OSAL ProcessSpawn().
  *
  *  \param[in] strProcessName: Process Name.
  *  \param[out] pu32handle: Handle(pid) of the process.
  *  \return TRUE if successfully gets the process handle
  ******
  */
   TMapStartedProcesses::const_iterator it;
   tBool                                bHandleFound = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poProcessDatabase );

   if ( !strProcessName.empty( ) ){

      // for (it = (_processDB->_oStartedProcesses).begin(); it != (_processDB->_oStartedProcesses).end(); ++it)
      for ( it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ); ++it ){
         TProcessConfiguration processConfig = ( * it ).second;
         if ( processConfig.strProcessName == strProcessName ){
            ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bGetProcessHandleByName() found id=%d for processname %s", processConfig.handle, processConfig.strProcessName.c_str( ) ) );
            * pu32handle = processConfig.handle;
            bHandleFound = TRUE;
            break;
         }
      }
   }
   return( bHandleFound );
} // bGetProcessHandleByName

tVoid spm_tclProcessSupervision::vDumpStartedProcessList( tBool bToErrmem,
                                                          tBool bToTrace ){
/*!
  * \fn
  *  \brief
  *    Prints out the list of (spm) started linux processes.
  *
  *  \param[in] bToErrmem: Write output to error memory.
  *  \param[in] bToTrace: Write output to TTFIS-Trace.
  ******
  */
   TMapStartedProcesses::const_iterator it;
   std::string                          strError;
   tInt                                 iCnt = 0;

   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK( _poProcessDatabase );

   strError = "--- List of started linux processes ---";
   if ( bToErrmem ){
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
   }
   if ( bToTrace ){
      ETG_TRACE_USR4( ( "%s", strError.c_str( ) ) );
   }
   // for (it = (_processDB->_oStartedProcesses).begin(); it != (_processDB->_oStartedProcesses.end()); ++it)
   for ( it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ); ++it ){
      TProcessConfiguration processConfig = ( * it ).second;
      tS32                  status        = 0;
      pid_t                 w             = waitpid( processConfig.handle, &status, WNOHANG );
      if ( w == - 1 ){
         processConfig.isMissing = TRUE;
      }
      strError = "#" + std::to_string( iCnt ) + ": Process=" + processConfig.strProcessName + " pid=" + std::to_string( processConfig.handle ) + " started(ms)=" + std::to_string( processConfig.startTimeStamp ) + " isMissing=" + std::to_string( (int)processConfig.isMissing ) + " supervisionActive=" + std::to_string( (int)processConfig.bSupervisionActive );
      if ( bToTrace ){
         ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
      }
      if ( bToErrmem ){
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
      }
      iCnt++;
   }
} // vDumpStartedProcessList

tVoid spm_tclProcessSupervision::vSetSupervisionForAllProcesses( tBool bEnableSupervision ){
/*!
  * \fn
  *  \brief
  *    Changes the supervision active flag for all the processes(SPM started processes and Additional Processes).
  *
  *  \param[in] bEnableSupervision: New supervision active flag (TRUE/FALSE).
  ******
  */
   TMapStartedProcesses::iterator it;

   SPM_NULL_POINTER_CHECK( _poProcessDatabase );

   // for (it = (_processDB->_oStartedProcesses).begin(); it != (_processDB->_oStartedProcesses).end(); ++it)
   for ( it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      processConfig.bSupervisionActive = bEnableSupervision;
   }

   // for (it = (_processDB->_oAdditionalProcesses).begin(); it != (_processDB->_oAdditionalProcesses).end(); ++it)
   for ( it = ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      processConfig.bSupervisionActive = bEnableSupervision;
   }

} // vSetSupervisionForAllProcesses

tBool spm_tclProcessSupervision::bChangeSupervisionForProcess( const std::string& strProcessName,
                                                               tBool              bSupervisionActive ){
/*!
  * \fn
  *  \brief
  *    Changes the supervision active flag for the given process.
  *
  *  \param[in] strProcessName: name of the process.
  *  \param[in] bSupervisionActive: New supervision active flag (TRUE/FALSE).
  *  \return TRUE if successfully changes supervision of the process.
  ******
  */
   tBool                          bFoundProcess = FALSE;

   TMapStartedProcesses::iterator it;

   SPM_NULL_POINTER_CHECK_VAL( _poProcessDatabase );

   for ( it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( processConfig.strProcessName == strProcessName ){
         bFoundProcess                    = TRUE;
         processConfig.bSupervisionActive = bSupervisionActive;
      }
   }

   for ( it = ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( processConfig.strProcessName == strProcessName ){
         bFoundProcess                    = TRUE;
         processConfig.bSupervisionActive = bSupervisionActive;
      }
   }

   return( bFoundProcess );
} // bChangeSupervisionForProcess

tBool spm_tclProcessSupervision::bRemoveProcessFromSupervisionList( const std::string& strProcessName ){
/*!
  * \fn
  *  \brief
  *    Removes a given process from Process Supervision list.
  *
  *  \param[in] strProcessName: name of the process.
  *  \return TRUE if successfully removes the process from supervision.
  ******
  */
   tBool bFoundProcess = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poProcessDatabase );

   bFoundProcess = _poProcessDatabase->bRemoveMapProc( strProcessName );

   return( bFoundProcess );
}

tVoid spm_tclProcessSupervision::vHandleMessage( tU32 u32Message,
                                                 tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    Handles the Message from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm: process id.
  *  \note
  *     If Message is SPM_U32_WORKER_PRC_SIGCHLD, this method checks whether an SPM started process(an SPM child process) is missing.
  *     If it is missing, it updates the error memory with neccessary process information.\n
  *  \todo
  *    Integration of SIGNAL HANDLER(SIGCHLD) for Process Supervision.
  ******
  */
   (tVoid)u32Message;
   (tVoid)u32Parm;

   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK( _poProcessDatabase );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   switch ( u32Message ){
      case SPM_U32_WORKER_PRC_SIGCHLD:
      {
         std::string strFirstMissingProName;
         pid_t       pidMissingProc = u32Parm;

         if ( OSAL_s32SemaphoreWait( _hProcessSupervisionSem, OSAL_C_U32_INFINITE ) != OSAL_OK ){
            tU32 u32ErrorReason = OSAL_u32ErrorCode( );
            ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessSupervision !!!!!! Error detected !!!!!! cannot wait for Semaphore _hProcessSupervisionSem: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
         }

         TMapStartedProcesses::iterator it;
         tBool                          bFoundMissingProcess = FALSE;

         it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).find( pidMissingProc );
         if ( it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ) ){
            TProcessConfiguration& processConfig = ( * it ).second;
            if ( !processConfig.isMissing && ( processConfig.bSupervisionActive == TRUE ) ){    // Process allready marked as missing?
               std::string strError;
               strError               = "PRM: Process missing! pid=" + std::to_string( processConfig.handle ) + " Process=" + processConfig.strProcessName;
               ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
               _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

               bFoundMissingProcess   = TRUE;
               strFirstMissingProName = processConfig.strProcessName;
            }
         } // end of if map.end


         for ( it = ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).end( ); ++it ){
            TProcessConfiguration& processConfig = ( * it ).second;
            if ( processConfig.handle == pidMissingProc ){
               if ( !processConfig.isMissing && ( processConfig.bSupervisionActive == TRUE ) ){    // Process allready marked as missing?
                  std::string strError;
                  strError = "PRM: External Process missing! pid=" + std::to_string( processConfig.handle ) + " Process=" + processConfig.strProcessName;

                  ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
                  if ( !bFoundMissingProcess ){
                     bFoundMissingProcess   = TRUE;
                     strFirstMissingProName = processConfig.strProcessName;
                  }

               }
            }
         }

         if ( bFoundMissingProcess && !_bProcSuperVisionStopped ){
            std::string strError;
            strError = "PRM: Addional process information:";
            ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

            SPM_ROOTDAEMON_CALLER_rPerformRootOp(
               "lcm",
               GET_CURRENT_PROCESSES,
               "" );

            strError = "Posting process supervision reset request (caused by missing " + strFirstMissingProName + " process)";
            ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

            if ( _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_PROCESS_SUPERVISION ) == FALSE ){

               strError = "SPM: spm_tclProcessSupervision: Supervision reset request cant be posted. Using direct function call instead.";
               _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
               _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION );
               OSAL_s32ThreadWait( 3000 );
               strError = "SPM: spm_tclProcessSupervision: Supervision reset request via direct function call has no effect, using FATAL_ASSERT now.";
               _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ), SPM_WRITE_DIRECTLY );
               FATAL_M_ASSERT_ALWAYS( );
            }
         }

         OSAL_s32SemaphorePost( _hProcessSupervisionSem );

         break;
      }

      default:
         break;
   } // switch
}    // vHandleMessage

tBool spm_tclProcessSupervision::bHandleSynchrounousCall( tU32   u32Message,
                                                          tVoid *args ){
/*!
  * \fn
  *  \brief
  *   spm_tclWorkerServer invokes this method for notifying messages.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] args: not used.
  *  \return always return FALSE.
  *  \note
  *  if Message is SPM_U32_WORKER_BROADCAST_STARTUP_DELAY, this method activates Process Supervision Thread and updates the map of additional processes from external list
  *  and from registry.
  *  \todo
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */
   (tVoid)args;
   switch ( u32Message ){

      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
      {
         // startup has ended
         // restore all nice levels and binding to a specific CPU
         // GEN3-TO IMPLEMENT/PORT FROM GEN2-SPMSLV: vSetStartUpFinishedProcessConfiguration(instance);
         // activate the entry thread and let them running
         if ( _bStopProcessSupervision ){
            _bStopProcessSupervision = FALSE;
            /* Release sempahore so that process supervision tclActiveThread will run now*/
            OSAL_s32SemaphorePost( _hPrcSupervisionStartSem );


            // tS32 s32Error = OSAL_s32ThreadActivate(_hThreadProcessSupervision);
            // if (s32Error != OSAL_OK)
            // {
            // ETG_TRACE_FATAL(("Error activating vProcessSupervisionThread!"));
            // }
         }
         break;
      }

      default:
      {
         // nothing to do
         break;
      }
   } // switch

   return( FALSE );
} // bHandleSynchrounousCall

tVoid spm_tclProcessSupervision::vMainMonitor( ){
/*!
  * \fn
  *  \brief
  *    Method to supervise SPM started processes and additional processes.
  *  \param
  *  If any process is missing, it updates the error memory with needed process information.
  *  \todo
  *    Integration of SIGNAL HANDLER(SIGCHLD) for Process Supervision.\n
  *    Testing Process Supervision on Additional Set of Processes.
  ******
  */
   std::string strFirstMissingProName;
   std::string strFirstMissingExternalProName;
   tS32        FirstMissingHdl = 0;
   tS32        FirstMissingExternalHdl = 0;
   pid_t       w;
   tS32        status = 0;

   SPM_NULL_POINTER_CHECK( _poProcessDatabase );
   SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );

   tBool       bFoundMissingProcess, bFoundMissingExternalProcess;

   ETG_TRACE_USR4( ( "vProcessSupervisionThread() wait(2000ms)..." ) );

   OSAL_s32ThreadWait( 2000 );
   // Semaphore to sync with current running shutdown. If shutdown trigger comes in,
   // the sempahore is obtained and this supervisionthread is then blocked here until
   // powersupply is swiched off, or a reverse-shutdown trigger (back to on) comes in,
   // which also release the sempahore.
   // ETG_TRACE_USR4(("vProcessSupervisionThread() obtaining sempahore..."));
   if ( OSAL_s32SemaphoreWait( _hProcessSupervisionSem, OSAL_C_U32_INFINITE ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessSupervision !!!!!! Error detected !!!!!! cannot wait for Semaphore _hProcessSupervisionSem: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   // ETG_TRACE_USR4(("vProcessSupervisionThread() got sempahore..."));

   TMapStartedProcesses::iterator it;

   bFoundMissingProcess = FALSE;

   for ( it = ( _poProcessDatabase->mapGetStartedProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetStartedProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( !processConfig.isMissing && ( processConfig.bSupervisionActive == TRUE ) ){    // Process allready marked as missing?
         w = waitpid( processConfig.handle, &status, WNOHANG );
         int waitPidErr = errno;
         if ( w == - 1 ){
            if ( ( waitPidErr != EINTR ) ){
               std::string strError;
               strError = "Process missing! pid=" + std::to_string( processConfig.handle ) + " Process=" + processConfig.strProcessName + " (errno=" + std::to_string( waitPidErr ) + "=" + strerror( waitPidErr ) + ")";

               ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
               _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
               if ( !bFoundMissingProcess ){
                  bFoundMissingProcess   = TRUE;
                  strFirstMissingProName = processConfig.strProcessName;
                  FirstMissingHdl        = processConfig.handle;
               }
               processConfig.isMissing = TRUE;
            } else {
               std::string strError;
               strError = "ERR: waitpid(pid=" + std::to_string( processConfig.handle ) + ",..,WNOHANG) returns with errorcode EINTR for process=" + processConfig.strProcessName + " (errno=" + std::to_string( waitPidErr ) + "=" + strerror( waitPidErr ) + ")";
               ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
               if ( _bProcSupervisionWriteToErrmem ){
                  _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
               }
            }
         }
      }
   }

   bFoundMissingExternalProcess = FALSE;
   for ( it = ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).begin( ); it != ( _poProcessDatabase->mapGetAdditionalProcesses( ) ).end( ); ++it ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( !processConfig.isMissing && ( processConfig.bSupervisionActive == TRUE ) ){    // Process allready marked as missing?
         // tBool bIsProcessAlive = fCheckIfProcessAlive(processConfig.strProcessName, processConfig.handle);
         tBool bIsProcessAlive = bCheckIfProcessAlive( processConfig.handle );
         if ( bIsProcessAlive == FALSE ){
            std::string strError;
            strError = "External started process missing! pid=" + std::to_string( processConfig.handle ) + " Process=" + processConfig.strProcessName;

            ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );
            if ( !bFoundMissingExternalProcess ){
               bFoundMissingExternalProcess   = TRUE;
               strFirstMissingExternalProName = processConfig.strProcessName;
               FirstMissingExternalHdl        = processConfig.handle;
            }

            processConfig.isMissing = TRUE;
         }
      }
   }

   OSAL_s32SemaphorePost( _hProcessSupervisionSem );

   if ( bFoundMissingProcess || bFoundMissingExternalProcess ){
      std::string strError;
      // OSALUTIL_s32SaveNPrintFormat(strError, sizeof(strError), "Add. proc. missing info #1: ShutdownTrigger received=%d ShutdownFinishedAckSend=%d",
      // poThis->_bShutdownTriggerReceived,
      // poThis->_bShutdownFinishedAckSend);
      // ETG_TRACE_FATAL(("%s", strError));
      // poThis->vWriteErrmem(U16_M_ERRMEM_SPM_ERROR(SPM_U8_ERRMEM_TYPE_STRING), (tU8*)strError, OSAL_u32StringLength(strError));
      // OSALUTIL_s32SaveNPrintFormat(strError, sizeof(strError), "Add. proc. missing info #2:ShutdownReverseTrigger received=%d ShutdownReversedFinishedAckSend=%d.",
      // poThis->_bShutdownReversedTriggerReceived, poThis->_bShutdownReversedAckSendRunning);
      // ETG_TRACE_FATAL(("%s", strError));
      // poThis->vWriteErrmem(U16_M_ERRMEM_SPM_ERROR(SPM_U8_ERRMEM_TYPE_STRING), (tU8*)strError, OSAL_u32StringLength(strError));
      strError = "Addional process information:";
      ETG_TRACE_FATAL( ( "%s", strError.c_str( ) ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strError.c_str( ), (tU16)strError.length( ) );

      SPM_ROOTDAEMON_CALLER_rPerformRootOp(
         "lcm",
         GET_CURRENT_PROCESSES,
         "" );

      OSAL_s32ThreadWait( 1000 );
      if ( bFoundMissingProcess ){
         ETG_TRACE_USR4( ( "spm_tclProcessSupervision::vMainMonitor - LCM started process with ID %d missing (%s)", FirstMissingHdl, strFirstMissingProName.c_str( ) ) );
         _poclWorkerServer->bPostMessage( "ISpmProcessSupervision", SPM_U32_WORKER_PRC_SIGCHLD, FirstMissingHdl );
      } else if ( bFoundMissingExternalProcess ){
         ETG_TRACE_USR4( ( "spm_tclProcessSupervision::vMainMonitor - External process with ID %d missing", FirstMissingExternalHdl ) );
         _poclWorkerServer->bPostMessage( "ISpmProcessSupervision", SPM_U32_WORKER_PRC_SIGCHLD, FirstMissingExternalHdl );
      }
   }

   //
   OSAL_s32ThreadWait( 3000 );
   // ETG_TRACE_USR4(("vProcessSupervisionThread() released sempahore"));
} // vMainMonitor

tVoid spm_tclProcessSupervision::main( ){
/*!
  * \fn
  *  \brief
  *    Entry Point of Process Supervision Thread.
  ******
  */
   ETG_TRACE_USR4( ( "vProcessSupervisionThread() started. MyPid=%d MyParentPid=%d", getpid( ), getppid( ) ) );
   ETG_TRACE_USR4( ( "Waiting for semaphore to run process supervision thread" ) );
   if ( OSAL_s32SemaphoreWait( _hPrcSupervisionStartSem, OSAL_C_U32_INFINITE ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessSupervision !!!!!! Error detected !!!!!! cannot wait for Semaphore _hPrcSupervisionStartSem: error 0x%08X (%s)",
                          (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   /* Start the supervision for the processes, given via process spawn calls from master spm*/
   vActivateAdditionalProcessSupervision( );
   /* Check now also the registry key for more additional processes to supervise */
   vCheckForExternalProcessesToSupervise( );

   ETG_TRACE_USR4( ( "Starting process supervision handling NOW!" ) );
   while ( _bStopProcessSupervision == FALSE ){
      vMainMonitor( );
   }
   ETG_TRACE_USR4( ( "vProcessSupervisionThread() exit." ) );

   vSetTerminate( );
} // vProcessSupervisionThread

// ex. case SPM_SLV_CMD_PROCESS_SUPERVISION_STOP:
tBool spm_tclProcessSupervision::bSupervisionStop( tVoid ){
/*!
  * \fn
  *  \brief
  *      Stops supervision, but waits for current cycle to be done before leaving functions
  ******
  */
   tBool bStoppedSucessfull = TRUE;

   ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bSupervisionStop()" ) );
   if ( !_bProcSuperVisionStopped ){
      /* Wait for trigger that current supervision check is finished.*/
      if ( OSAL_s32SemaphoreWait( _hProcessSupervisionSem, 15000 ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "spm_tclProcessSupervision::bSupervisionStop(): Cant get sempahore after timeout of 15s." ) );
         bStoppedSucessfull = FALSE;
      } else {
         ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bSupervisionStop(): Got sempahore" ) );
         _bProcSuperVisionStopped = TRUE; // Flag that we got a shutdown trigger and obtained the semaphore
      }
   } else {
      ETG_TRACE_USR4( ( "spm_tclProcessSupervision: Already stopped. Nothing to do here." ) );
   }
   return( bStoppedSucessfull );
} // bSupervisionStop

tVoid spm_tclProcessSupervision::vSupervisionStart( tVoid ){
/*!
  * \fn
  *  \brief
  *      cancels current stop of supervision
  ******
  */
   ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bSupervisionStart()" ) );
   if ( _bProcSuperVisionStopped ){
      _bProcSuperVisionStopped = FALSE;
      ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bSupervisionStart: Releasing sempahore" ) );
      OSAL_s32SemaphorePost( _hProcessSupervisionSem );
   } else {
      ETG_TRACE_USR4( ( "spm_tclProcessSupervision::bSupervisionStart received, but supervision is allready active." ) );
   }
}

// ex. SPM_SLV_CMD_PROCESS_SUPERVION_CHANGE
tBool spm_tclProcessSupervision::bSupervisionChange( const std::string& strProcessName,
                                                     tBool              bSupervisionActive ){
/*!
  * \fn
  *  \brief
  *      toggles between SupervisionStopped and SupervisionStarted for a single process
  *  \param[in]  strProcessName:     process name.
  *  \param[in]  bSupervisionActive: New supervision active flag (TRUE/FALSE).
  *  \return   TRUE if successfully changes supervision otherwise FALSE.
  ******
  */
   tBool bChangedSuccessfull = TRUE;

   ETG_TRACE_USR4( ( "spm_tclProcessSupervision::vSupervisionChange: Received change request for process %s.", strProcessName.c_str( ) ) );
   ETG_TRACE_USR4( ( "NewActiveFlag=%d", bSupervisionActive ) );
   if ( !_bProcSuperVisionStopped ){
      if ( OSAL_s32SemaphoreWait( _hProcessSupervisionSem, OSAL_C_U32_INFINITE ) == OSAL_OK ){
         ETG_TRACE_USR4( ( "Got sempahore, searching now for process %s...", strProcessName.c_str( ) ) );
         if ( bChangeSupervisionForProcess( strProcessName, bSupervisionActive ) ){
            ETG_TRACE_USR4( ( "Found process %s", strProcessName.c_str( ) ) );
         } else {
            ETG_TRACE_USR4( ( "Process %s NOT FOUND!", strProcessName.c_str( ) ) );
         }
         OSAL_s32SemaphorePost( _hProcessSupervisionSem );
         ETG_TRACE_USR4( ( "supervision sempahore released." ) );
      }
   } else {
      ETG_TRACE_USR4( ( "Supervision allready stopped. Searching now for process %s...", strProcessName.c_str( ) ) );
      if ( bChangeSupervisionForProcess( strProcessName, bSupervisionActive ) ){
         ETG_TRACE_USR4( ( "Found process %s", strProcessName.c_str( ) ) );
      } else {
         ETG_TRACE_USR4( ( "Process %s NOT FOUND!", strProcessName.c_str( ) ) );
      }
   }
   return( bChangedSuccessfull );
} // bSupervisionChange

tBool spm_tclProcessSupervision::bSupervisionChangeAll( tBool bSupervisionActive ){
/*!
  * \fn
  *  \brief
  *      toggles between SupervisionStopped and SupervisionStarted for all processes
  *  \param[in]  bSupervisionActive: New supervision active flag (TRUE/FALSE).
  ******
  */
   ETG_TRACE_USR4( ( " spm_tclProcessSupervision::bSupervisionChangeAll" ) );

   /* Check if process supervision is currently switched to OFF (e.g. because new shutdown was triggered
      and unmount scripts are running).
      If supervision is running, we have to wait for the _hProcessSupervisionSem semaphore.
      If no supervision is running, it is save to change the supervision flags for the processes
      without obtaining the semaphore before.
     */
   if ( !_bProcSuperVisionStopped ){
      if ( OSAL_s32SemaphoreWait( _hProcessSupervisionSem, OSAL_C_U32_INFINITE ) == OSAL_OK ){
         ETG_TRACE_USR4( ( "Got Semaphore. Changing now now supervision for all processes to %d", bSupervisionActive ) );
         vSetSupervisionForAllProcesses( bSupervisionActive );
         OSAL_s32SemaphorePost( _hProcessSupervisionSem );
         ETG_TRACE_USR4( ( "supervision sempahore released." ) );
      }
   } else {
      ETG_TRACE_USR4( ( "Supervision allready stopped. Changing now now supervision for all processes to %d", bSupervisionActive ) );
      vSetSupervisionForAllProcesses( bSupervisionActive );

   }

   return( TRUE );
} // bSupervisionChangeAll

// PRMIMPL(PRM Implementation for SIGCHLD Interface)
tVoid spm_tclProcessSupervision::vNotifyHandler( const tU32 *pu32Param ){
/*!
  * \fn
  *  \brief
  *    Notification Handler for SIGCHLD. This handler is notified by OSAL when a child process(spm started process) ends.
  *     It sends a message(SIGCHLD message) to WorkerServer.
  *
  *  \param[in] pu32Param: Process ID
  ******
  */


   if ( _poclWorkerServer ){
      if ( pu32Param ){
         ETG_TRACE_USR4( ( "spm_tclProcessSupervision::vNotifyHandler - Process with ID %d terminated", * pu32Param ) );
         _poclWorkerServer->bPostMessage( "ISpmProcessSupervision", SPM_U32_WORKER_PRC_SIGCHLD, * pu32Param );
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclProcessSupervision::vNotifyHandler() called with invalid _poclWorkerServer!" ) );
   }
}

