/*!
  * \file spm_SysStateSupervisor.cpp
  *  \brief
  *    A software watchdog base implementation
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author               | Modification
  * 23.09.13  | CM-AI/PJVW32 Kollai  | initial version
  ******
  */
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SysStateSupervisor.h"

// interfaces class definitions
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IWorkerClient.h"
#include "spm_ISuperVisionManager.h"

#include "spm_IFactory.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SysStateSupervisor.cpp.trc.h"
#endif

#include "spm_trace.h"


#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

#if OSAL_CONF == OSAL_LINUX
   #include <limits.h>
   #include <unistd.h>
#endif


spm_tclSysStateSupervisor::spm_tclSysStateSupervisor( const ISpmFactory& factory )
   : ISpmSupervisionClient( factory ),
   _bTimeTrackerRunning( FALSE ),
   _hHeartBeatTimer( 0 ),
   _u32PreWarnTime( 2 ),
   _u32HeartBeatCount( 0 ),
   _u32WdtHeartBeatTime( 0 ),
   _u32TimeStampHeartBeatTimerStarted( 0 ),
   _u32TimeStampHeartBeatTimerCallbackCalled( 0 ),
   _u32TimeStampHeartBeatTimerChecked( 0 ),
   _u32TimeStampHeartBeatTimerCallbackExited( 0 ),
   _u32TimeStampHeartBeatTimerRetriggerd( 0 ),
   _poclOsalProxy( NULL ),
   _poclSubStateHandler( NULL ),
   _poclWorkerServer( NULL ),
   _poclSystemPowerManager( NULL ),
   _poclSupervisionManager( NULL ){
   _bTriggeringEnabled  = TRUE;

   dp_tclSpmDpConfigWdtHeartbeatResetTime oHeartRstTime;
   _u32WdtHeartBeatTime = oHeartRstTime.tGetData( );

   if ( OSAL_s32TimerCreate( vHeartBeatTimerCallback, this, &_hHeartBeatTimer ) != OSAL_OK ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   } else {
      if ( OSAL_s32TimerSetTime( _hHeartBeatTimer, 0, 0 ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
   }
}

spm_tclSysStateSupervisor::~spm_tclSysStateSupervisor( ){
   if ( OSAL_s32TimerSetTime( _hHeartBeatTimer, 0, 0 ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   if ( OSAL_s32TimerDelete( _hHeartBeatTimer ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }

   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vRemoveSupervisionClient( this );

   _poclOsalProxy          = NULL;
   _poclWorkerServer       = NULL;
   _poclSubStateHandler    = NULL;
   _poclSystemPowerManager = NULL;
   _poclSupervisionManager = NULL;
}

tVoid spm_tclSysStateSupervisor::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,          ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
} // vGetReferences

tVoid spm_tclSysStateSupervisor::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   _poclSupervisionManager->vAddSupervisionClient( this );
   // set cyclic timer for 1s HeartBeat
   if ( OSAL_s32TimerSetTime( _hHeartBeatTimer, SPM_U32_SUPERVISION_HEARTBEAT_INTERVAL, SPM_U32_SUPERVISION_HEARTBEAT_INTERVAL ) != OSAL_OK ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   } else {
      _u32TimeStampHeartBeatTimerStarted = OSAL_ClockGetElapsedTime( );
   }
}

tVoid spm_tclSysStateSupervisor::vHeartBeatTimerCallback( tVoid *pArg ){
   spm_tclSysStateSupervisor *poSupervisor = (spm_tclSysStateSupervisor*)pArg;

   poSupervisor->_u32TimeStampHeartBeatTimerCallbackCalled = OSAL_ClockGetElapsedTime( );
   poSupervisor->_poclWorkerServer->bPostMessage( "ISpmSubStateClient", SPM_U32_WORKER_SSH_HEARTBEAT );
   poSupervisor->_u32TimeStampHeartBeatTimerCallbackExited = OSAL_ClockGetElapsedTime( );
}

tVoid spm_tclSysStateSupervisor::vCheckSupervisionState( ){
   tU8         u8Dummy = 0;

   std::string szStr;
   std::string szStrHeartBeat;

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   // check heartbeat -> warn level
   if ( bIsHeartBeatPreWarnTime( ) ){
      // in 2sec wdt timeout will be reported -> write ready threads in errmem
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy,                      sizeof( u8Dummy ),              FALSE );

      szStr          = "SPM: SW Watchdog Supervision: HeartBeat missing for " + std::to_string( _u32HeartBeatCount ) + "s!!!!";
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)szStr.c_str( ),          (tU16)szStr.length( ),          SPM_WRITE_DIRECTLY );

      szStrHeartBeat = "TimerStarted=" + std::to_string( _u32TimeStampHeartBeatTimerStarted ) + " TimerCBcalled=" + std::to_string( _u32TimeStampHeartBeatTimerCallbackCalled ) + " TimerCBexited=" + std::to_string( _u32TimeStampHeartBeatTimerCallbackExited ) + " HeartBeatCntRetriggered=" + std::to_string( _u32TimeStampHeartBeatTimerRetriggerd ) + " LastCheckOfHeartBeat=" + std::to_string( _u32TimeStampHeartBeatTimerChecked ) + " CurrentTimeStamp=" + std::to_string( OSAL_ClockGetElapsedTime( ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)szStrHeartBeat.c_str( ), (tU16)szStrHeartBeat.length( ), SPM_WRITE_DIRECTLY );

      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy,                      sizeof( u8Dummy ),              FALSE );
      _poclOsalProxy->bDumpThreadInfoReady( SPM_S32_THREAD_PRIO_LOW, SPM_BL_OSALPROXY_IF_ENABLED );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );


      TRACE_SPM_ERROR_MY_THREAD( _poclSystemPowerManager );
      if ( !_bTimeTrackerRunning ){
         #if OSAL_CONF == OSAL_LINUX
            std::string exePath( PATH_MAX, 0 );
            ssize_t     count = readlink( "/proc/self/exe", &exePath[0], PATH_MAX );
            if ( count == 0 ){
               exePath = "procbase_out.out";
            }
            szStr = "SPM: SW Watchdog Supervision: HeartBeat missing (count=" + std::to_string( _u32HeartBeatCount ) + "): Triggering " + exePath + " callstack generation";
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
            _poclOsalProxy->bDumpProcessInfo( exePath, "OSAL", SPM_BL_OSALPROXY_IF_ENABLED );

         #endif

         szStr                = "SPM: Heartbeat supervision: Starting TT " + std::to_string( _u32HeartBeatCount ) + " ...";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
         _poclOsalProxy->bDumpTimeTrace( );
         _bTimeTrackerRunning = TRUE;
      }
   }

   // check heartbeat -> error
   if ( bIsHeartBeatWarnTime( ) ){
      // ETG_TRACE_USR4(("HeartBeat missing for 20s -> entry errmem"));

      // entry in errormemory
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy,                      sizeof( u8Dummy ),              FALSE );

      szStr          = "SPM: SW Watchdog Supervision: HeartBeat missing for " + std::to_string( _u32HeartBeatCount ) + "s!!!!";
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)szStr.c_str( ),          (tU16)szStr.length( ),          SPM_WRITE_DIRECTLY );

      szStrHeartBeat = "TimerStarted=" + std::to_string( _u32TimeStampHeartBeatTimerStarted ) + " TimerCBcalled=" + std::to_string( _u32TimeStampHeartBeatTimerCallbackCalled ) + " TimerCBexited=" + std::to_string( _u32TimeStampHeartBeatTimerCallbackExited ) + " HeartBeatCntRetriggered=" + std::to_string( _u32TimeStampHeartBeatTimerRetriggerd ) + " LastCheckOfHeartBeat=" + std::to_string( _u32TimeStampHeartBeatTimerChecked ) + " CurrentTimeStamp=" + std::to_string( OSAL_ClockGetElapsedTime( ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ),                (const tU8*)szStrHeartBeat.c_str( ), (tU16)szStrHeartBeat.length( ), SPM_WRITE_DIRECTLY );

      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy,                      sizeof( u8Dummy ),              FALSE );
      _poclOsalProxy->bDumpThreadInfoReady( SPM_S32_THREAD_PRIO_LOW, SPM_BL_OSALPROXY_IF_ENABLED );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );

      TRACE_SPM_ERROR_MY_THREAD( _poclSystemPowerManager );
      // \todo: make this via message from ISpmWorkerServer
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
      if ( SPM_PRJ_SYSTEM_STATE_DOWNLOAD == poclSystemStateManager->u32GetSystemState( ) ){
         // no reset in SYSTEM_STATE_DOWNLOAD
         szStr              = "SPM: SW Watchdog Supervision: NO HeartBeat reset because we are in Download mode!!!!";
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
         _u32HeartBeatCount = 0;
      } else {
         // sw watchdog thread has not trigger once in last 20 sec -> reset
         if ( _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION ) == FALSE ){
            szStr = "SPM: SW Watchdog Supervision: HEARTBEAT supervision reset request cant be posted. Using direct function call instead.";
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
            _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION );
            OSAL_s32ThreadWait( SPM_ERRMEM_WRITE_DELAY_MS );
            szStr = "SPM: SW Watchdog Supervision: HEARTBEAT supervision reset request via direct funtion call has no effect, using FATAL_ASSERT now.";
            _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)szStr.c_str( ), (tU16)szStr.length( ), SPM_WRITE_DIRECTLY );
            FATAL_M_ASSERT_ALWAYS( );
         }
         // DONT perform the below line. Above reset trigger take some time (12s linux shutdown),
         // so we reach the FAILURE macro to early (raises assert)
         // TRACE_SPM_FAILURE_MY_THREAD(_poclSystemPowerManager);

      }
   }

   vNewHeartBeatInterval( );
   _u32TimeStampHeartBeatTimerChecked = OSAL_ClockGetElapsedTime( );
} // vCheckHeartBeat

tVoid spm_tclSysStateSupervisor::vTriggerSupervisionState( ){
   if ( _bTriggeringEnabled ){
      _u32HeartBeatCount                    = 0;
      _u32TimeStampHeartBeatTimerRetriggerd = OSAL_ClockGetElapsedTime( );
   }
}

tVoid spm_tclSysStateSupervisor::vTraceInfo( ){

   ETG_TRACE_FATAL( ( "----- SUPERVISOR: SYSTEM STATE -----" ) );
   ETG_TRACE_FATAL( ( "HeartBeatCount = %d, HeartBeatTime = %d, PreWarnTime = %d", _u32HeartBeatCount, _u32WdtHeartBeatTime, _u32PreWarnTime ) );
   ETG_TRACE_FATAL( ( "bIsHeartBeatWarnTime = %d, bIsHeartBeatPreWarnTime = %d", bIsHeartBeatWarnTime( ), bIsHeartBeatPreWarnTime( ) ) );

}

