/*!
  * \file spm_StartupSupervisor.cpp
  *  \brief
  *    A software implementation to supervise project specific information
  *               The Startup state follows the below transitions in case of successful startup
  *               SPM_U8_STARTUP_STATE_INIT----->SPM_U8_STARTUP_STATE_SPM_STARTED---->
  *               ------->SPM_U8_STARTUP_STATE_SYSTEM_STARTED-----> SPM_U8_STARTUP_STATE_SYSTEM_UP
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author               | Modification
  * 21.01.14  | CM-AI/EPB2 Kollai    | initial version
  * 16.04.14  | Basavaraj Nagar      | Doxygen Addition
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_StartupSupervisor.h"

// interfaces class definitions
#include "spm_IWakeupHandler.h"
#include "spm_IWorkerClient.h"
#include "spm_IOsalProxy.h"
#include "spm_IOsLinux.h"
#include "spm_ISubStateClient.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IStartupSystem.h"
#include "spm_ISuperVisionManager.h"
#include "spm_ILateServiceHandler.h"
#include "spm_IApplicationErrorHandler.h"
#include "spm_ISystemLoadStatistics.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_StartupSupervisor.cpp.trc.h"
#endif

#include "spm_trace.h"

#define SPM_U32_DMVERITY_TIMEOUT_SEC    (tU32)120

spm_tclStartupSupervisor::spm_tclStartupSupervisor( const ISpmFactory& factory )
   : ISpmStartupSupervisor( factory )
   , _poclWakeupHandler( NULL )
   , _poclWorkerServer( NULL )
   , _poclStartupTimeStatistic( NULL )
   , _u8StartUpState( SPM_U8_STARTUP_STATE_INIT )
   , _bStartMinConfig( FALSE ){
   //Override _bSupervisionEnabled in abstract class ISpmSupervisionClient.
   _bSupervisionEnabled = TRUE;

   dp_tclSpmDpConfigStartupSystemUp         oUp;

   oUp >> _u32SystemCheckUp;

   dp_tclSpmDpConfigStartupSystemRunning    oRunning;
   oRunning >> _u32SystemCheckRunning;


}

spm_tclStartupSupervisor::~spm_tclStartupSupervisor( ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionManager, ISpmSupervisionManager );
   poclSupervisionManager->vRemoveSupervisionClient( this );

   _poclStartupTimeStatistic = NULL;
   _poclWorkerServer  = NULL;
   _poclWakeupHandler = NULL;
}

tVoid spm_tclStartupSupervisor::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    Perform a dependency Injection here.
  *    The dependencies injected are WorkerServer and WakeupHandler
  ******
  */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWakeupHandler, ISpmWakeupHandler );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,  ISpmWorkerServer );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclStartupTimeStatistic, spm_tclStartupTimeStatistics, ISpmSystemLoadStatistics );
}

tVoid spm_tclStartupSupervisor::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *        The Supervision Concept follows a sort of Subject-Observer Pattern.
  *        All observers(Supervision Clients) are required to register themselves at the Subject(SupervisionManager)
  *        The interface exposed for registration is vAddSupervisionClient(ISpmsupervisonclient*)
  ******
  */
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionManager, ISpmSupervisionManager );
   poclSupervisionManager->vAddSupervisionClient( this );

}

tVoid spm_tclStartupSupervisor::vCheckSupervisionState( ){
/*!
  * \fn
  *  \brief
  *     This method performs multiple actions:
  *      1) Checks if system is started(SPM_U8_STARTUP_STATE_SYSTEM_STARTED) and stable for more than configured time
  *         (dp_tclSpmDpConfigStartupSystemRunning), and if true updates _u8StartUpState attribute to SPM_U8_STARTUP_STATE_SYSTEM_UP
  *      2) Updates the object attribute _u8StartUpState to SPM_U8_STARTUP_STATE_SYSTEM_UP,
  *         indicating that all applications are spawned successfuly
  *      3) Checks if system is up and stable for more than configured time(dp_tclSpmDpConfigStartupSystemUp),
  *         and then post worker message SPM_U32_WORKER_BROADCAST_STARTUP_DELAY,
  *         so that LAM and CCASrvHDler can check for missing applications or services
  *      4) This method is called periodically by the Supervision Manager(Supervision Client Interface)
  *      5) Update the start up time statistic data.
  *
  ******
  */

   tU32 u32CurrentTime = OSAL_ClockGetElapsedTime( );

   SPM_GET_IF_REFERENCE_NEW_VAR( poclStartup, ISpmStartupSystem );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   if ( ( _u8StartUpState == SPM_U8_STARTUP_STATE_SYSTEM_STARTED ) ){
      _u8StartUpState = SPM_U8_STARTUP_STATE_SYSTEM_STARTUPCONTROLLER;
      poclStartup->vOnSyncPointReached( SPM_STR_SYNC_POINT_LCM_STARTUP_READY );

      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): Trigger now StartupController to start 'LateTarget' --> state: %u (clear reset counter now)", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );

      SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poDbusTmp, spm_ISpmLateServiceHandler);
      if ( poDbusTmp ){
         if ( !poDbusTmp->bStartupReady( ) ){
             ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupSupervisor::vCheckSupervisionState() !!!!!! Error detected !!!!!! poDbusTmp->bStartupReady() returns FALSE" ) );
         }
      }
   }


   if ( ( ( _u8StartUpState == SPM_U8_STARTUP_STATE_SYSTEM_UP ) || ( _u8StartUpState == SPM_U8_STARTUP_STATE_SYSTEM_STARTUPCONTROLLER ) ) && ( u32CurrentTime > ( _u32SystemCheckRunning * 1000 ) ) ){
      _u8StartUpState = SPM_U8_STARTUP_STATE_SYSTEM_RUNNING;
      poclStartup->vOnSyncPointReached( SPM_STR_SYNC_POINT_LCM_SYSTEM_RUNNING );
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): System is now running --> state: %u (clear reset counter now)", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
      dp_tclSpmDpInternDataContinuousResets oContinuousResetCounter;
      oContinuousResetCounter << 0;
      {
         SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
         poclOsalProxy->bStartupProcessed( SPM_U8_OSALPROXY_STARTUPSTATE_RUNNING );
      }
      {
         SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
         spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;
         tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SYSTEM_RUNNING;
         poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
      }
   }

   if ( ( ( _u8StartUpState == SPM_U8_STARTUP_STATE_SYSTEM_STARTED ) || ( _u8StartUpState == SPM_U8_STARTUP_STATE_SYSTEM_STARTUPCONTROLLER ) ) && ( u32CurrentTime > ( _u32SystemCheckUp * 1000 ) ) ){
      _u8StartUpState = SPM_U8_STARTUP_STATE_SYSTEM_UP;
      poclStartup->vOnSyncPointReached( SPM_STR_SYNC_POINT_LCM_SYSTEM_UP );
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): System is now up --> state: %u", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );

      {
         SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
         SPM_NULL_POINTER_CHECK( _poclWakeupHandler );
         poclOsalProxy->bStartupProcessed( );
         _poclWakeupHandler->vSetResetCounter( 0 );
      }
      {
         SPM_GET_IF_REFERENCE_NEW_VAR( poclAppErrHandler, ISpmApplicationErrorHandler );
         poclAppErrHandler->vGetUnitFailedList( );
      }
      {
         SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
         spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;
         tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SYSTEM_UP;
         poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
      }
      {
         tUInt nStartupTime = _poclWorkerServer->nGetStartupTime();
         _poclStartupTimeStatistic->bUpdateLoading( (tU32) nStartupTime );
      }
      _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_BROADCAST_STARTUP_DELAY, 1 );
   }

   if ( _u8StartUpState != SPM_U8_STARTUP_STATE_SYSTEM_UP ){
      poclStartup->vOnCheckStartup( );
   }

   #ifdef SPM_ENABLE_DM_VERITY_CHECK
   //check if DM Verity scan is ready
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSubState, ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsLx, ISpmOsLinux );
   if (poclSubState->bIsTriggerSet(SPM_U32_DMVERITY_CHECK)) {
       tBool bSutdownNow = TRUE;
       ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Check active: current time: %dms", u32CurrentTime) );

       SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
       if (u32CurrentTime > (SPM_U32_DMVERITY_TIMEOUT_SEC * 1000)) {
           // shutdown with DM-Verity CHECK_FAILED
           ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Check failed due to TIMEOUT") );
           poclOsalProxy->bSetDmVerityResult( DEV_WUP_C_U8_DM_VERITY_CHECK_RESULT_FAILED );
       } else if (poclOsLx->bIsFileInFs("/tmp/filesystem_integrity_check_fail")) {
           ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Check FAILED!") );
           poclOsalProxy->bSetDmVerityResult( DEV_WUP_C_U8_DM_VERITY_CHECK_RESULT_FAILED );
       } else if (poclOsLx->bIsFileInFs("/tmp/filesystem_integrity_check_passed")) {
           // check content of file
           std::string strBuf;
           tBool bPassed = FALSE;

           if ( 0 <= poclOsLx->s32ReadFileContent( "/tmp/filesystem_integrity_check_passed", strBuf, 255 ) ){
               if (strBuf.find("CheckPassed12893478") != std::string::npos) {
                   ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Content of file is valid --> send PASSED!") );
                   bPassed = TRUE;
               }
           }
           if (bPassed) {
               ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Check PASSED!") );
               poclOsalProxy->bSetDmVerityResult( DEV_WUP_C_U8_DM_VERITY_CHECK_RESULT_PASSED );
           } else {
               ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vCheckSupervisionState(): DM Verity: Check FAILED (due to file content)!") );
               poclOsalProxy->bSetDmVerityResult( DEV_WUP_C_U8_DM_VERITY_CHECK_RESULT_FAILED );
           }
      } else {
           bSutdownNow = FALSE;
       }
       if (bSutdownNow) {
           if ( !poclOsalProxy->bShutdownSystem( SPM_C_S32_SYSTEMINIT_SWITCH_OFF ) ){
               ETG_TRACE_ERRMEM( ( "ERROR using DEV_SPM  SPM_C_S32_SYSTEMINIT_SWITCH_OFF " ) );
           }
       }

   }
   #endif


} // vCheckSupervisionState

tVoid spm_tclStartupSupervisor::vTriggerSupervisionState( ){
}

tVoid spm_tclStartupSupervisor::vStartupSpmProcessed( ){
/*!
  * \fn
  *  \brief
  *     This method performs multiple actions:
  *      1) Updates the object attribute _u8StartUpState to SPM_U8_STARTUP_STATE_SPM_STARTED,
  *         indicating that all SPM internal threads are up and  running(poststart)
  *      2) Checks for continous resets and increments it in case it is another reset, or clears it
  *      3) If the reset counter surpasses the maximum number of resets allowed(can be configured), then, Default Start Target is switched to minimum Target
  *
  ******
  */

   SPM_NULL_POINTER_CHECK( _poclWakeupHandler );

   if ( _u8StartUpState != SPM_U8_STARTUP_STATE_INIT ){
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Invalid state: %u", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
   }
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): New startup state: SPM_STARTED (old -> %u)", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
   _u8StartUpState = SPM_U8_STARTUP_STATE_SPM_STARTED;


   dp_tclSpmDpInternDataShutdownCounter  oShutdownCounter;
   dp_tclSpmDpInternDataContinuousResets oContinuousResetCounter;

   // check if last shutdow was reset
   if ( _poclWakeupHandler->bIsReset( ) ){

      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Reset detected. System was not switched off correctly." ) );
      // increment reset counter;
      oShutdownCounter.vSetData( oShutdownCounter.tGetData( ) + 1 );
      oContinuousResetCounter.vSetData( oContinuousResetCounter.tGetData( ) + 1 );

   } else {

      ETG_TRACE_USR1( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Last shutdown was no reset." ) );
      // last shutdown was no reset
      oShutdownCounter.vSetData( 0 );
      oContinuousResetCounter.vSetData( 0 );
   }

   // masrk this start as invalid
   dp_tclSpmDpInternDataShutdownConfirmed                  oShutdownConfirmedSet;
   oShutdownConfirmedSet.vSetData( FALSE );


   // check if min target has to be started
   dp_tclSpmDpConfigMaxResetCount2StartMinTargetLcm        oMaxCountLcm;
   dp_tclSpmDpConfigMaxResetCount2StartMinTargetBootloader oMaxCountBootloader;
   dp_tclSpmDpInternDataContinuousResets                   oRstCountLcm;
   tU32                                                    u32MaxCountLcm        = oMaxCountLcm.tGetData( );
   tU32                                                    u32MaxCountBootloader = oMaxCountBootloader.tGetData( );
   tU32                                                    u32RstCountLcm        = oRstCountLcm.tGetData( );
   tU32                                                    u32RstCountBootloader = _poclWakeupHandler->u8GetResetCounter( );
   ETG_TRACE_USR4( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Continous reset count: %d, max. count: %d", u32RstCountLcm, u32MaxCountLcm ) );
   if ( ( u32MaxCountLcm != 0 )
        && ( u32MaxCountBootloader != 0 )
        && ( ( u32RstCountLcm >= u32MaxCountLcm )
             || ( u32RstCountBootloader >= u32MaxCountBootloader )
             )
        ){
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Start only minimal set of Applications (Continous reset count: %d, max. count: %d).", u32RstCountLcm, u32MaxCountLcm ) );
      _bStartMinConfig = TRUE;
   }
} // vStartupSpmProcessed

tVoid spm_tclStartupSupervisor::vStartupSystemProcessed( ){
/*!
  * \fn
  *  \brief
  *     This method updates the object attribute _u8StartUpState to SPM_U8_STARTUP_STATE_SYSTEM_STARTED,
  *     indicating all the processes in the system are atarted successfully, called by StartupSystem
  *
  ******
  */
   if ( _u8StartUpState != SPM_U8_STARTUP_STATE_SPM_STARTED ){
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): Invalid state: %u", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
   }
      ETG_TRACE_FATAL( ( "spm_tclStartupSupervisor::vStartupSpmProcessed(): New startup state: SYSTEM_STARTED (old -> %u)", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
   _u8StartUpState = SPM_U8_STARTUP_STATE_SYSTEM_STARTED;

   {
      SPM_GET_IF_REFERENCE_NEW_VAR( poclCcaServiceHandler, ISpmCcaServiceServer );
      spm_corefi_tclMsgRunlevelStateStatus tRunLevelState;
      tRunLevelState.eRunlevel.enType = spm_fi_tcl_SPM_e32_RUNLEVEL::FI_EN_SPM_U32_RUNLEVEL_SYSTEM_STARTED;
      poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_RUNLEVELSTATE, &tRunLevelState );
   }
} // vStartupSystemProcessed

tVoid spm_tclStartupSupervisor::vClearEmergencyCounter( ){
/*!
  * \fn
  *  \brief
  *        This method clears the Continuous reset count and ShutdownCounter
  ******
  */

   ETG_TRACE_USR1( ( "spm_tclStartupSupervisor::vClearEmergencyCounter()!!!!." ) );

   // last shutdown was no reset
   dp_tclSpmDpInternDataShutdownCounter  oShutdownCounter;
   dp_tclSpmDpInternDataContinuousResets oContinuousResetCounter;

   oShutdownCounter.vSetData( 0 );
   oContinuousResetCounter.vSetData( 0 );
}

tVoid spm_tclStartupSupervisor::vTraceInfo( ){
/*!
  * \fn
  *  \brief
  *     This method is used to trace the Startup State Attribute
  ******
  */
   ETG_TRACE_FATAL( ( "----- SUPERVISOR: STARTUPSTATE  -----" ) );
   ETG_TRACE_FATAL( ( "Current startup state: %u", ETG_ENUM( SPM_START_STATE, _u8StartUpState ) ) );
}

