/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_LxProcInfo.h
  * @brief        This is header file contains functions for LxStatusInfo (thread or process related information),
  *               LxThreadStatusInfo (thread related information and implements the functionality to calculate the CPU usage of a thread),
  *               LxCpuInfo(CPU related information).
  * @copyright    (C) 2014 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#define LX_PROC_INFO_CONF_PROC_NAME_BUFFER_LENGTH  16

/*!
  *  \class
  *  \brief This class implements the functionality to gather thread or process related information from the /proc file system.
  */
class spm_tclLxStatusInfo
{
private:
struct TProcStat
{
   // tChar state;                       // %c
   // tInt ppid;                         // %d
   // tInt pgrp;                         // %d
   // tInt session;                      // %d
   // tInt tty_nr;                       // %d
   // tInt tpgid;                        // %d
   // tULong flags;                      // %lu
   // tULong minflt;                     // %lu
   // tULong cminflt;                    // %lu
   // tULong majflt;                     // %lu
   // tULong cmajflt;                    // %lu
   tULong utime;                      // %lu
   tULong stime;                      // %lu
   // tLong cutime;                      // %ld
   // tLong cstime;                      // %ld
   tLong priority;                    // %ld
   // tLong nice;                        // %ld
   // tLong num_threads;                 // %ld
   // tLong itrealvalue;                 // %ld
   // tULong starttime;                  // %lu
   // tULong vsize;                      // %lu
   // tLong rss;                         // %ld
   // tULong rlim;                       // %lu
   // tULong startcode;                  // %lu
   // tULong endcode;                    // %lu
   // tULong startstack;                 // %lu
   // tULong kstkesp;                    // %lu
   // tULong kstkeip;                    // %lu
   // tULong signal;                     // %lu
   // tULong blocked;                    // %lu
   // tULong sigignore;                  // %lu
   // tULong sigcatch;                   // %lu
   // tULong wchan;                      // %lu
   // tULong nswap;                      // %lu
   // tULong cnswap;                     // %lu
   // tInt exit_signal;                  // %d
   // tInt processor;                    // %d
   // tULong rt_priority;                // %lu
   // tULong policy;                     // %lu
   // tULongLong delayacct_blkio_ticks;  // %llu
};

TProcStat _rProcStat;
tChar     _szName[LX_PROC_INFO_CONF_PROC_NAME_BUFFER_LENGTH];

public:
spm_tclLxStatusInfo( );
virtual ~spm_tclLxStatusInfo( ){}

const tChar* szGetName( )    const { return( _szName );             }
tLong               lGetPriority( ) const { return( _rProcStat.priority ); }
tULong              ulGetUTime( )   const { return( _rProcStat.utime );    }
tULong              ulGetSTime( )   const { return( _rProcStat.stime );    }

virtual tBool bReadStatusFromProcFileSystem( int nProcId,
                                             int nThreadId );

virtual tBool bReadNameFromProcFileSystem( int nProcId,
                                           int nThreadId );

};

/*!
  *  \class
  *  \brief This class stores thread related information and implements the functionality to calculate the CPU usage of a thread.
  */
class spm_tclLxThreadStatusInfo : public spm_tclLxStatusInfo
{
private:
tInt          _nParentProcID;
tInt          _nThreadID;
std::string   _strParentProcName;
std::string   _strThreadName;
tBool         _bHighCpuUsageDetected;
tBool         _bIsSupervisionThread;
tBool         _bCalculated;
tBool         _bOverallTicksRead;
tU32          _u32CpuUsage;
tULong        _ulOverallTicks;
tULong        _ulPeriodOverallTicks;
tLong         _lRawPriority;
OSAL_tMSecond _msHighUsageStarttime;
OSAL_tMSecond _msHighUsageDuration;

public:
spm_tclLxThreadStatusInfo( );
spm_tclLxThreadStatusInfo( tInt        nThreadID,
                           tInt        nParentProcID,
                           std::string strParentProcName );

tInt          nGetParentProcID( )        const { return( _nParentProcID );         }
std::string   strGetParentProcName( )    const { return( _strParentProcName );     }
tInt          nGetThreadID( )            const { return( _nThreadID );             }
std::string   strGetThreadName( )        const { return( _strThreadName );         }
tU32          u32GetCpuUsage( )          const { return( _u32CpuUsage );           }
tLong         ulGetPeriodOverallTicks( ) const { return( _ulPeriodOverallTicks );  }
tLong         lGetRawPriority( )         const { return( _lRawPriority );          }
OSAL_tMSecond msGetHighUsageDuration( )  const { return( _msHighUsageDuration );   }
tBool         bIsSupervisionThread( )    const { return( _bIsSupervisionThread );  }
tBool         bIsHighCpuUsageDetected( ) const { return( _bHighCpuUsageDetected ); }
tBool         bIsCalculated( )           const { return( _bCalculated );           }

tVoid         vSetHighCpuUsageDetected( tBool bHighCpuUsageDetected ){ _bHighCpuUsageDetected = bHighCpuUsageDetected; }

tVoid         vCalculateCpuUsage( tULong ulAllCpuCoresPeriodOverallTicks );

tVoid         vUpdateHighUsageDuration( );

tVoid         vResetHighUsageDuration( );

tVoid         vResetCalculationData( );

};

/*!
  *  \class
  *  \brief This class stores process related information and implements the functionality to calculate the CPU usage of a process.
  */
class spm_tclLxProcessStatusInfo : public spm_tclLxStatusInfo
{
private:
tInt        _nProcID;
std::string _strProcName;
tBool       _bHighCpuUsagePending;
tBool       _bCalculated;
tBool       _bOverallTicksRead;
tU32        _u32CpuUsage;
tULong      _ulOverallTicks;
tULong      _ulPeriodOverallTicks;

public:
spm_tclLxProcessStatusInfo( );
spm_tclLxProcessStatusInfo( tInt nProcID );
tInt        nGetProcID( )               const { return( _nProcID );               }
std::string strGetProcName( )           const { return( _strProcName );           }
tU32        u32GetCpuUsage( )           const { return( _u32CpuUsage );           }
tLong       ulGetPeriodOverallTicks( )  const { return( _ulPeriodOverallTicks );  }
tBool       bIsHighCpuUsagePending( )   const { return( _bHighCpuUsagePending );  }
tBool       bIsCalculated( )            const { return( _bCalculated );           }
tVoid       vSetHighCpuUsagePending( tBool bHighCpuUsagePending ){ _bHighCpuUsagePending = bHighCpuUsagePending; }
tVoid       vCalculateCpuUsage( tULong ulAllCpuCoresPeriodOverallTicks );

};

/*!
  *  \class
  *  \brief This class implements the functionality to gather CPU related information from the /proc file system and calculates the CPU usage for each core.
  */
class spm_tclLxCpuInfo
{
private:
struct TCpuUsage
{
   tU32 u32CpuUsage;
   tU32 u32CpuUsageSystem;
   tU32 u32CpuUsageUser;

   tULong ulCpuOverallTicks;
   tULong ulCpuIdleTicks;
   tULong ulCpuUserTicks;
   tULong ulCpuSystemTicks;
   tULong ulCpuRemainingTicks;

   tULong ulCpuPeriodOverallTicks;
   tULong ulCpuPeriodIdleTicks;
   tULong ulCpuPeriodUserTicks;
   tULong ulCpuPeriodSystemTicks;
   tULong ulCpuPeriodRemainingTicks;
};
std::map < std::string, TCpuUsage > _mapCpuUsageMap;
// Switch between private and public to be able to access method u32DetectNumberOfCpus() for unit tests.
#ifndef LCM_UNIT_TESTS
   tU32                             _u32NumberOfCpus;
   tU32                             u32DetectNumberOfCpus( tVoid );

public:
#else

public:
   tU32 _u32NumberOfCpus;
   tU32 u32DetectNumberOfCpus( tVoid );
#endif

spm_tclLxCpuInfo( );

tU32 u32GetNumberOfCpus( tVoid ) const { return( _u32NumberOfCpus ); }

tU32 u32GetCpuUsage( const std::string& strCpuName );

tULong ulUpdateFromProcFileSystem( tVoid );
};

