/*!
  * \file spm_LxProcInfo.cpp
  *  \brief
  *    Access the Linux /proc file system to gather informations about :
  *
  *    - CPU cores => number of cores, times spend in the various operation states
  *    - processes => name, priority, times spend in the various operation states
  *    - threads   => name, priority, times spend in the various operation states
  *
  *    Also calculate the CPU usage for a process/thread on the basis of the collected information from the /proc file system.
  *
  *    See the following link for a description of the /proc file system : http://www.linuxhowtos.org/manpages/5/proc.htm
  *
  *  \b PROJECT: NextGen
  *  \b SW-COMPONENT: FC SPM
  *  \b COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
  *  \version
  *    Date      | Author              | Modification
  *    09.03.16  | CM-AI/ESO3   Kalms  | Overall rework due to new requirements.
  *    02.01.14  | CM-AI/PJCB32 Kollai | initial version
  ******
  */

#include <boost/algorithm/string.hpp>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_LxProcInfo.h"

// interfaces class definitions

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_LxProcInfo.cpp.trc.h"
#endif

#include "spm_trace.h"

#define SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE  512
#define SPM_LX_PROC_INFO_CPU_STAT_FILE_BUFFER_SIZE        2048

spm_tclLxStatusInfo::spm_tclLxStatusInfo( ){
   OSAL_pvMemorySet( &_rProcStat, 0, sizeof( _rProcStat ) );
   _szName[0] = '\0';
} // spm_tclLxStatusInfo

tBool spm_tclLxStatusInfo::bReadNameFromProcFileSystem( tInt nProcId,
                                                        tInt nThreadId ){
/*!
  * \fn
  *  \brief
  *        Read the name of a process or thread from the /proc file system files '/proc/#procid/comm' or '/proc/#procid/task/#threadid/comm'.
  *
  *  \param [in]  tInt : ID of the process or parent-process.
  *  \param [in]  tInt : ID of the thread. Is allowed to be 0 if the process statistics shall be read.
  *
  *  \return tBool : TRUE  = Success
  *                  FALSE = Failure
  ******
  */
   OSAL_tIODescriptor hIODescriptor = OSAL_ERROR;
   tS32               s32FileSize   = 0;
   std::string        strPath;

   if ( nProcId == 0 ){
      return( FALSE );
   }

   if ( nThreadId == 0 ){
      strPath = "/dev/root/proc/" + std::to_string( nProcId ) + "/comm";
   } else {
      strPath = "/dev/root/proc/" + std::to_string( nProcId ) + "/task/" + std::to_string( nThreadId ) + "/comm";
   }

   hIODescriptor = OSAL_IOOpen( strPath.c_str( ), OSAL_EN_READONLY );
   if ( hIODescriptor != OSAL_ERROR ){
      s32FileSize = OSAL_s32IORead( hIODescriptor, (tPS8)_szName, LX_PROC_INFO_CONF_PROC_NAME_BUFFER_LENGTH );
      (tVoid)OSAL_s32IOClose( hIODescriptor );
      if ( s32FileSize != OSAL_ERROR ){
         _szName[s32FileSize - 1] = '\0';
         return( TRUE );
      }
   }

   return( FALSE );
} // bReadNameFromProcFileSystem

tBool spm_tclLxStatusInfo::bReadStatusFromProcFileSystem( tInt nProcId,
                                                          tInt nThreadId ){
/*!
  * \fn
  *  \brief
  *        Read process or thread related statistics from the /proc file system files '/proc/#procid/stat' or '/proc/#procid/task/#threadid/stat'.
  *	   Currently the followling items are collected :
  *
  *          - utime     = Amount of time that this process/thread has been scheduled in user mode.
  *          - stime     = Amount of time that this process/thread has been scheduled in kernel mode.
  *          - priority  = Priority of the process/thread.
  *                        For processes running a real-time scheduling policy, this is the negated scheduling priority, minus one;
  *                        that is, a number in the range -2 to -100, corresponding to real-time priorities 1 to 99. For processes running under
  *                        a non-real-time scheduling policy, this is the raw nice value as represented in the kernel. The kernel stores nice
  *                        values as numbers in the range 0 (high) to 39 (low), corresponding to the user-visible nice range of -20 to 19.
  *
  *  \param [in]  tInt : ID of the process or parent-process.
  *  \param [in]  tInt : ID of the thread. Is allowed to be 0 if the process statistics shall be read.
  *
  *  \return tBool : TRUE  = Success
  *                  FALSE = Failure
  ******
  */

   OSAL_tIODescriptor hIODescriptor = OSAL_ERROR;
   tS32               s32FileSize   = 0;
   tChar             *pcFileBuffer  = NULL;
   tChar              szFileBuffer[SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE];
   std::string        strPath;
   tInt               i;

   if ( nProcId == 0 ){
      ETG_TRACE_FATAL( ( "spm_tclLxStatusInfo::bReadStatusFromProcFileSystem() => Passed wrong parameter nProcId = 0" ) );
      return( FALSE );
   }

   if ( nThreadId == 0 ){
      strPath = "/dev/root/proc/" + std::to_string( nProcId ) + "/stat";
   } else {
      strPath = "/dev/root/proc/" + std::to_string( nProcId ) + "/task/" + std::to_string( nThreadId ) + "/stat";
   }

   hIODescriptor = OSAL_IOOpen( strPath.c_str( ), OSAL_EN_READONLY );
   if ( hIODescriptor != OSAL_ERROR ){
      s32FileSize                   = OSAL_s32IORead( hIODescriptor, (tPS8)szFileBuffer, SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE );
      (tVoid)OSAL_s32IOClose( hIODescriptor );
      if ( ( s32FileSize == OSAL_ERROR ) || ( s32FileSize == SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE ) ){
         ETG_TRACE_FATAL( ( "spm_tclLxStatusInfo::bReadStatusFromProcFileSystem() => OSAL_s32IORead() for file '%32s' failed with s32FileSize = %d", strPath.c_str( ), s32FileSize ) );
         return( FALSE );
      }
      szFileBuffer[s32FileSize - 1] = '\0';
      // Skip the first two attributes 'pid' and 'comm' and set the position to the 'state' attribute.
      // This is necessary because the with '(' and ')' encapsulated 'comm' attribute might contain
      // white-space characters which would corrupt the following sscanf() operation.
      for ( i = 0; i < SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE; i++ ){
         if ( szFileBuffer[i] == '\0' ){
            break;
         }
         if ( ( szFileBuffer[i] == ')' ) && ( i < ( SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE - 2 ) ) ){
            pcFileBuffer = &szFileBuffer[i + 2];
            break;
         }
      }

      if ( pcFileBuffer != NULL ){
         if ( 3 == sscanf( pcFileBuffer
                                                  //  01   02  03  04  05  06  07   08   09   10   11   12  13  14   15   16  17   18   19   20   21   22   23   24   25   26   27
                                                  //   28   29   30   31   32   33   34   35   36  37  38   39   40
                           , "%*c %*d %*d %*d %*d %*d %*u  %*u  %*u  %*u  %*u  %lu %lu %*d  %*d  %ld %*d  %*d  %*d  %*u  %*u  %*d  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*u  %*d %*d %*u  %*u  %*u"
                                                  // ,"%*c %*d %*d %*d %*d %*d %*lu %*lu %*lu %*lu %*lu %lu %lu %*ld %*ld %ld %*ld %*ld %*ld %*lu %*lu %*ld %*lu %*lu %*lu %*lu %*lu
                                                  // %*lu %*lu %*lu %*lu %*lu %*lu %*lu %*lu %*d %*d %*lu %*lu %*llu"
                                                  // Removed length modifiers 'l' and 'll' from all with '*' eliminated arguments due to compiler warning.
                                                  // Be sure to again add the length modifiers 'l' and 'll' if one of the eliminated arguments is activated.
                                                  // ,&_rProcStat.state                 /*  1 */
                                                  // ,&_rProcStat.ppid                  /*  2 */
                                                  // ,&_rProcStat.pgrp                  /*  3 */
                                                  // ,&_rProcStat.session               /*  4 */
                                                  // ,&_rProcStat.tty_nr                /*  5 */
                                                  // ,&_rProcStat.tpgid                 /*  6 */
                                                  // ,&_rProcStat.flags                 /*  7 */
                                                  // ,&_rProcStat.minflt                /*  8 */
                                                  // ,&_rProcStat.cminflt               /*  9 */
                                                  // ,&_rProcStat.majflt                /* 10 */
                                                  // ,&_rProcStat.cmajflt               /* 11 */
                           , &_rProcStat.utime,   /* 12 */
                           &_rProcStat.stime      /* 13 */
                                                  // ,&_rProcStat.cutime                /* 14 */
                                                  // ,&_rProcStat.cstime                /* 15 */
                           , &_rProcStat.priority /* 16 */
                                                  // ,&_rProcStat.nice                  /* 17 */
                                                  // ,&_rProcStat.num_threads           /* 18 */
                                                  // ,&_rProcStat.itrealvalue           /* 19 */
                                                  // ,&_rProcStat.starttime             /* 20 */
                                                  // ,&_rProcStat.vsize                 /* 21 */
                                                  // ,&_rProcStat.rss                   /* 22 */
                                                  // ,&_rProcStat.rlim                  /* 23 */
                                                  // ,&_rProcStat.startcode             /* 24 */
                                                  // ,&_rProcStat.endcode               /* 25 */
                                                  // ,&_rProcStat.startstack            /* 26 */
                                                  // ,&_rProcStat.kstkesp               /* 27 */
                                                  // ,&_rProcStat.kstkeip               /* 28 */
                                                  // ,&_rProcStat.signal                /* 29 */
                                                  // ,&_rProcStat.blocked               /* 30 */
                                                  // ,&_rProcStat.sigignore             /* 31 */
                                                  // ,&_rProcStat.sigcatch              /* 32 */
                                                  // ,&_rProcStat.wchan                 /* 33 */
                                                  // ,&_rProcStat.nswap                 /* 34 */
                                                  // ,&_rProcStat.cnswap                /* 35 */
                                                  // ,&_rProcStat.exit_signal           /* 36 */
                                                  // ,&_rProcStat.processor             /* 37 */
                                                  // ,&_rProcStat.rt_priority           /* 38 */
                                                  // ,&_rProcStat.policy                /* 39 */
                                                  // ,&_rProcStat.delayacct_blkio_ticks /* 40 */
                           ) ){
            return( TRUE );
         }
      } else {
         ETG_TRACE_FATAL( ( "spm_tclLxStatusInfo::bReadStatusFromProcFileSystem() => Info file '%32s' doesn't contain a closing round bracket char ')'", strPath.c_str( ) ) );
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclLxStatusInfo::bReadStatusFromProcFileSystem() => OSAL_IOOpen() for '%32s' failed", strPath.c_str( ) ) );
   }

   return( FALSE );
} // bReadStatusFromProcFileSystem

spm_tclLxThreadStatusInfo::spm_tclLxThreadStatusInfo( ) : spm_tclLxStatusInfo( ),
   _nParentProcID( 0 ),
   _nThreadID( 0 ),
   _bHighCpuUsageDetected( FALSE ),
   _bIsSupervisionThread( FALSE ),
   _bCalculated( FALSE ),
   _bOverallTicksRead( FALSE ),
   _u32CpuUsage( 0 ),
   _ulOverallTicks( 0 ),
   _ulPeriodOverallTicks( 0 ),
   _lRawPriority( 20 ),
   _msHighUsageStarttime( 0 ),
   _msHighUsageDuration( 0 ){
} // spm_tclLxThreadStatusInfo

spm_tclLxThreadStatusInfo::spm_tclLxThreadStatusInfo( tInt        nThreadID,
                                                      tInt        nParentProcID,
                                                      std::string strParentProcName ) : spm_tclLxStatusInfo( ),
   _nParentProcID( nParentProcID ),
   _nThreadID( nThreadID ),
   _strParentProcName( strParentProcName ),
   _bHighCpuUsageDetected( FALSE ),
   _bIsSupervisionThread( FALSE ),
   _bCalculated( FALSE ),
   _bOverallTicksRead( FALSE ),
   _u32CpuUsage( 0 ),
   _ulOverallTicks( 0 ),
   _ulPeriodOverallTicks( 0 ),
   _lRawPriority( 20 ),
   _msHighUsageStarttime( 0 ),
   _msHighUsageDuration( 0 ){
   spm_tclLxStatusInfo oLxStatusInfo;

   if ( oLxStatusInfo.bReadNameFromProcFileSystem( _nParentProcID, _nThreadID ) == TRUE ){
      _strThreadName = oLxStatusInfo.szGetName( );
   }

   if ( OSAL_s32StringCompare( _strThreadName.c_str( ), SPM_HW_WATCHDOGTHREAD ) == 0 ){
      _bIsSupervisionThread = TRUE;
   }
} // spm_tclLxThreadStatusInfo

tVoid spm_tclLxThreadStatusInfo::vCalculateCpuUsage( tULong ulAllCpuCoresPeriodOverallTicks ){
/*!
  * \fn
  *  \brief
  *        Read the thread statistics from the /proc file system file '/proc/#procid/task/#threadid/stat'.
  *        Calculate the CPU usage of a certain thread by comparing the relative amount of the thread
  *        related ticks againt the relative amount of possible ticks over all CPU cores.
  *        Also get and store the information about the priority of the thread.
  *
  *  \param [in]  tU32: Relative number of possible ticks over all CPU cores which are counted within the
  *                     period of the last two measurements.
  ******
  */
   spm_tclLxStatusInfo oLxThreadStatusInfo;

   if ( oLxThreadStatusInfo.bReadStatusFromProcFileSystem( _nParentProcID, _nThreadID ) == TRUE ){
      _lRawPriority      = oLxThreadStatusInfo.lGetPriority( );

      if ( _bOverallTicksRead == TRUE ){
         _ulPeriodOverallTicks = ( oLxThreadStatusInfo.ulGetUTime( ) + oLxThreadStatusInfo.ulGetSTime( ) ) - _ulOverallTicks;
      }

      _ulOverallTicks    = oLxThreadStatusInfo.ulGetUTime( ) + oLxThreadStatusInfo.ulGetSTime( );
      _bOverallTicksRead = TRUE;

      if ( ulAllCpuCoresPeriodOverallTicks > 0 ){
         _u32CpuUsage = (tU32)( ( _ulPeriodOverallTicks * 100 ) / ulAllCpuCoresPeriodOverallTicks );
         _bCalculated = TRUE;
      }
   }
} // vCalculateCpuUsage

tVoid spm_tclLxThreadStatusInfo::vUpdateHighUsageDuration( ){
/*!
  * \fn
  *  \brief
  *        Update the high usage duration of a supervised thread.
  ******
  */
   if ( _msHighUsageStarttime == 0 ){
      _msHighUsageStarttime = OSAL_ClockGetElapsedTime( );
   } else {
      _msHighUsageDuration = OSAL_ClockGetElapsedTime( ) - _msHighUsageStarttime;
   }
} // vUpdateHighUsageDuration

tVoid spm_tclLxThreadStatusInfo::vResetHighUsageDuration( ){
/*!
  * \fn
  *  \brief
  *        Reset the high usage duration of a supervised thread.
  ******
  */
   _msHighUsageStarttime = 0;
   _msHighUsageDuration  = 0;
} // vResetHighUsageDuration

tVoid spm_tclLxThreadStatusInfo::vResetCalculationData( ){
/*!
  * \fn
  *  \brief
  *        Reset all CPU usage calculation relevant data to force a re-calculation
  *        with two new measurements via the vCalculateCpuUsage() method.
  *
  ******
  */
   _bOverallTicksRead    = FALSE;
   _bCalculated          = FALSE;
   _ulOverallTicks       = 0;
   _ulPeriodOverallTicks = 0;
   _u32CpuUsage          = 0;
} // vResetCalculationData

spm_tclLxProcessStatusInfo::spm_tclLxProcessStatusInfo( ) : spm_tclLxStatusInfo( ),
   _nProcID( 0 ),
   _bHighCpuUsagePending( FALSE ),
   _bCalculated( FALSE ),
   _bOverallTicksRead( FALSE ),
   _u32CpuUsage( 0 ),
   _ulOverallTicks( 0 ),
   _ulPeriodOverallTicks( 0 ){
} // spm_tclLxProcessStatusInfo

spm_tclLxProcessStatusInfo::spm_tclLxProcessStatusInfo( tInt nProcID ) : spm_tclLxStatusInfo( ),
   _nProcID( nProcID ),
   _bHighCpuUsagePending( FALSE ),
   _bCalculated( FALSE ),
   _bOverallTicksRead( FALSE ),
   _u32CpuUsage( 0 ),
   _ulOverallTicks( 0 ),
   _ulPeriodOverallTicks( 0 ){
   spm_tclLxStatusInfo oLxStatusInfo;

   if ( oLxStatusInfo.bReadNameFromProcFileSystem( nProcID, 0 ) == TRUE ){
      _strProcName = oLxStatusInfo.szGetName( );
   }
} // spm_tclLxThreadStatusInfo

tVoid spm_tclLxProcessStatusInfo::vCalculateCpuUsage( tULong ulAllCpuCoresPeriodOverallTicks ){
/*!
  * \fn
  *  \brief
  *        Read the thread statistics from the /proc file system file '/proc/#procid/stat'.
  *        Calculate the CPU usage of a certain process by comparing the relative amount of the process
  *        related ticks againt the relative amount of possible ticks over all CPU cores.
  *
  *  \param [in]  tU32: Relative number of possible ticks over all CPU cores which are counted within the
  *                     period of the last two measurements.
  ******
  */
   spm_tclLxStatusInfo oLxProcessStatusInfo;

   if ( oLxProcessStatusInfo.bReadStatusFromProcFileSystem( _nProcID, 0 ) == TRUE ){

      if ( _bOverallTicksRead == TRUE ){
         _ulPeriodOverallTicks = ( oLxProcessStatusInfo.ulGetUTime( ) + oLxProcessStatusInfo.ulGetSTime( ) ) - _ulOverallTicks;
      }

      _ulOverallTicks    = oLxProcessStatusInfo.ulGetUTime( ) + oLxProcessStatusInfo.ulGetSTime( );
      _bOverallTicksRead = TRUE;

      if ( ulAllCpuCoresPeriodOverallTicks > 0 ){
         _u32CpuUsage = (tU32)( ( _ulPeriodOverallTicks * 100 ) / ulAllCpuCoresPeriodOverallTicks );
         _bCalculated = TRUE;
      }
   }
} // vCalculateCpuUsage

spm_tclLxCpuInfo::spm_tclLxCpuInfo( ){
   _u32NumberOfCpus = u32DetectNumberOfCpus( );
} // spm_tclLxCpuInfo

tU32 spm_tclLxCpuInfo::u32DetectNumberOfCpus( tVoid ){
/*!
  * \fn
  *  \brief
  *        This method detects the number of CPUs via the /proc file system file '/proc/stat'
  *        and initializes the CPU usage map with default values.
  *
  *  \return tU32 : Number of detected CPU cores or 0 on error.
  ******
  */
   tU32               u32NumberOfCpus = 0;
   std::size_t        unPosition      = std::string::npos;
   OSAL_tIODescriptor fd              = OSAL_IOOpen( "/dev/root/proc/stat", OSAL_EN_READONLY );

   if ( fd == OSAL_ERROR ){
      return( 0 );
   }

   std::string strProcCat( SPM_LX_PROC_INFO_CPU_STAT_FILE_BUFFER_SIZE, 0 );
   tS32        s32ReadLen = OSAL_s32IORead( fd, ( tPS8 ) & strProcCat[0], (tU32)SPM_LX_PROC_INFO_CPU_STAT_FILE_BUFFER_SIZE );
   (tVoid)OSAL_s32IOClose( fd );

   if ( ( s32ReadLen == OSAL_ERROR ) || ( s32ReadLen == SPM_LX_PROC_INFO_CPU_STAT_FILE_BUFFER_SIZE ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclLxCpuInfo::u32DetectNumberOfCpus() => Failed to determine number of CPUs. Failed to access file /proc/stat." ) );
      return( 0 );
   }

   unPosition = strProcCat.find( "cpu" );
   while ( unPosition != std::string::npos ){
      // The /proc/stat file shows each CPU core in the form "cpu# ".
      // Under the assumption that there are never more than 99 CPU cores (two decimal places) a width of 5 characters for the CPU name (e.g. "cpu99") is sufficient.
      // For CPU names with only one decimal place like "cpu1 " the 5th space character is trimmed.
      std::string strCpuName = strProcCat.substr( unPosition, 5 );
      // \todo: find C++11 function replace for boost::trim
      boost::trim( strCpuName );

      TCpuUsage   tUsage     = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
      _mapCpuUsageMap[strCpuName] = tUsage;

      u32NumberOfCpus++;

      if ( strProcCat.size( ) > ( unPosition + 5 ) ){ // Don't grab beyond end of buffer
         strProcCat = strProcCat.substr( unPosition + 5 );
         unPosition = strProcCat.find( "cpu" );
      }
   }

   if ( u32NumberOfCpus == 0 ){
      ETG_TRACE_ERRMEM( ( "spm_tclLxCpuInfo::u32DetectNumberOfCpus() => Failed to determine number of CPUs. Didn't find pattern 'cpu' in file /proc/stat." ) );
      return ( 0 );
   }

   return( u32NumberOfCpus - 1 ); // Substract 1 because one entry in the /proc/stats file stands for the overall CPU statistics and doesn't count as an own CPU core.
} // u32DetectNumberOfCpus

tU32 spm_tclLxCpuInfo::u32GetCpuUsage( const std::string& strCpuName ){
/*!
  * \fn
  *  \brief
  *         This method is used to get the current CPU usage of a certain CPU core.
  *
  *  \param [in] strCpuName : Name of the CPU core to get the CPU usage from.
  *
  *  \return tU32 : CPU usage of the selected CPU core or 0 on error.
  ******
  */
   std::map < std::string, TCpuUsage >::iterator it = _mapCpuUsageMap.find( strCpuName );

   if ( it != _mapCpuUsageMap.end( ) ){
      return( it->second.u32CpuUsage );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclLxCpuInfo::u32GetCpuUsage() => CPU usage map doesn't contain an entry for '%s'. Return CPU usage of 0.", strCpuName.c_str( ) ) );
   }

   return( 0 );
} // u32GetCpuUsage

tULong spm_tclLxCpuInfo::ulUpdateFromProcFileSystem( tVoid ){
/*!
  * \fn
  *  \brief
  *        Read the CPU core related tick statistics from the /proc file system file '/proc/stat'.
  *        Calculate the CPU usage over all CPUs and for each CPU seperately by comparing the relative
  *        amount of possible ticks against the relative amount of counted ticks. It is distinguished
  *        between :
  *
  *          - CPU usage overall                  (overall ticks vs. non-idle ticks)
  *          - CPU usage caused from user space   (overall ticks vs. user ticks)
  *          - CPU usage caused from system space (overall ticks vs. system ticks).
  *
  *  \return tULong : Relative number of overall ticks of all CPU cores which are counted within the
  *                   period of the last two measurements. This value represents the maximum possible
  *                   computing power all CPU cores together are able to offer within a measurement
  *                   period.
  ******
  */
   tULong             ulAllCpuCoresPeriodOverallTicks = 0;

   OSAL_tIODescriptor fd                              = OSAL_IOOpen( "/dev/root/proc/stat", OSAL_EN_READONLY );

   if ( fd != OSAL_ERROR ){
      std::string strProcCat( SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE, 0 );
      tChar      *strStat    = NULL;
      tS32        s32ReadLen = OSAL_s32IORead( fd, ( tPS8 ) & strProcCat[0], (tU32)SPM_LX_PROC_INFO_PROC_TASK_STAT_FILE_BUFFER_SIZE );
      (tVoid)OSAL_s32IOClose( fd );

      if ( ( s32ReadLen != OSAL_ERROR ) && ( s32ReadLen != SPM_LX_PROC_INFO_CPU_STAT_FILE_BUFFER_SIZE ) ){
         std::map < std::string, TCpuUsage >::iterator it;
         // check the name with
         for ( it = _mapCpuUsageMap.begin( ); it != _mapCpuUsageMap.end( ); ++it ){
            const tChar *strCpuName = it->first.c_str( );
            strStat = OSAL_ps8StringSubString( &strProcCat[0], strCpuName );
            if ( strStat != NULL ){
               tULong ulUser, ulNice, ulSystem, ulIdle, ulIOWait, ulIRQ, ulSoftIRQ, ulSteal, ulGuest, ulGuestNice;
               strStat = strStat + it->first.size( ) + 1; // skip "cpu(n) " to start with load information
               if ( 10 == sscanf( strStat, "%lu %lu %lu %lu %lu %lu %lu %lu %lu %lu", &ulUser, &ulNice, &ulSystem, &ulIdle, &ulIOWait, &ulIRQ, &ulSoftIRQ, &ulSteal, &ulGuest, &ulGuestNice ) ){

                  tULong ulCpuOverallTicks = ulUser + ulNice + ulSystem + ulIdle + ulIOWait + ulIRQ + ulSoftIRQ + ulSteal + ulGuest + ulGuestNice;

                  if ( it->second.ulCpuOverallTicks != 0 ){
                     it->second.ulCpuPeriodOverallTicks = ulCpuOverallTicks - it->second.ulCpuOverallTicks;
                  }

                  if ( it->second.ulCpuIdleTicks != 0 ){
                     it->second.ulCpuPeriodIdleTicks = ulIdle - it->second.ulCpuIdleTicks;
                  }

                  if ( it->second.ulCpuUserTicks != 0 ){
                     it->second.ulCpuPeriodUserTicks = ( ulUser + ulNice ) - it->second.ulCpuUserTicks;
                  }

                  if ( it->second.ulCpuSystemTicks != 0 ){
                     it->second.ulCpuPeriodSystemTicks = ulSystem - it->second.ulCpuSystemTicks;
                  }

                  if ( it->second.ulCpuRemainingTicks != 0 ){
                     it->second.ulCpuPeriodRemainingTicks = ( ulIOWait + ulIRQ + ulSoftIRQ + ulSteal + ulGuest + ulGuestNice ) - it->second.ulCpuRemainingTicks;
                  }

                  it->second.ulCpuOverallTicks   = ulCpuOverallTicks;
                  it->second.ulCpuIdleTicks      = ulIdle;
                  it->second.ulCpuUserTicks      = ulUser + ulNice;
                  it->second.ulCpuSystemTicks    = ulSystem;
                  it->second.ulCpuRemainingTicks = ulIOWait + ulIRQ + ulSoftIRQ + ulSteal + ulGuest + ulGuestNice;

                  if ( it->second.ulCpuPeriodOverallTicks != 0 ){
                     it->second.u32CpuUsage       = (tU32)( ( ( it->second.ulCpuPeriodOverallTicks - it->second.ulCpuPeriodIdleTicks ) * 100 ) / it->second.ulCpuPeriodOverallTicks );
                     it->second.u32CpuUsageUser   = (tU32)( ( it->second.ulCpuPeriodUserTicks * 100 ) / it->second.ulCpuPeriodOverallTicks );
                     it->second.u32CpuUsageSystem = (tU32)( ( it->second.ulCpuPeriodSystemTicks * 100 ) / it->second.ulCpuPeriodOverallTicks );

                     ETG_TRACE_USR3( ( "CPU usage supervision => %-4s usage=%u (user=%u, system=%u) | period ticks => all=%u user+nice=%u system=%u idle=%u remaining=%u",
                                       strCpuName,
                                       it->second.u32CpuUsage,
                                       it->second.u32CpuUsageUser,
                                       it->second.u32CpuUsageSystem,
                                       (tUInt)it->second.ulCpuPeriodOverallTicks,
                                       (tUInt)it->second.ulCpuPeriodUserTicks,
                                       (tUInt)it->second.ulCpuPeriodSystemTicks,
                                       (tUInt)it->second.ulCpuPeriodIdleTicks,
                                       (tUInt)it->second.ulCpuPeriodRemainingTicks ) );
                  }

                  if ( 0 == OSAL_s32StringCompare( "cpu", strCpuName ) ){
                     ulAllCpuCoresPeriodOverallTicks = it->second.ulCpuPeriodOverallTicks;
                  }
               }
            }
         }
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclLxCpuInfo::ulUpdateFromProcFileSystem() => Failed to determine number of CPUs. Failed to access file /proc/stat." ) );
      }
   }
   return( ulAllCpuCoresPeriodOverallTicks );
} // u32UpdateFromProcFileSystem

