/*!
  * \file spm_IgnitionOffSupervisor.cpp
  *  \brief
  *    A software watchdog base implementation
     This watchdog will be started if the Clamp 15 Status is set to OFF. After t_standby +
     t_mmi_off1 + t_mmi_standby_pwr_save + 2 min the Watchdog will switch of the System. If
     the customer presses a Hardkey the Watchdog will reset its timer. If the clamp status switches
     to clamp 15 = ON the watchdog will be stopped.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2013 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author               | Modification
  * 23.09.13  | CM-AI/PJVW32 Kollai  | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_IgnitionOffSupervisor.h"

// interfaces class definitions
#include "spm_ISubStateClient.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ISuperVisionManager.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_IgnitionOffSupervisor.cpp.trc.h"
#endif

#include "spm_trace.h"

spm_tclIgnitionOffSupervisor::spm_tclIgnitionOffSupervisor( const ISpmFactory& factory )
   : ISpmSupervisionClient( factory )
   , _poclSubStateHandler( NULL )
   , _poclSystemPowerManager( NULL ){
   _u32WdtClamp15Time = 0;

   dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
   oCurIgnCount >> _u32Clamp15OffCount;
}

spm_tclIgnitionOffSupervisor::~spm_tclIgnitionOffSupervisor( ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionManager, ISpmSupervisionManager );
   poclSupervisionManager->vRemoveSupervisionClient( this );

   _poclSubStateHandler    = NULL;
   _poclSystemPowerManager = NULL;
}

tVoid spm_tclIgnitionOffSupervisor::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,    ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
}

tVoid spm_tclIgnitionOffSupervisor::vStartCommunication( ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclSupervisionManager, ISpmSupervisionManager );
   poclSupervisionManager->vAddSupervisionClient( this );
}

tVoid spm_tclIgnitionOffSupervisor::vStartLateCommunication( ){
   vUpdateTime( );
}

tVoid spm_tclIgnitionOffSupervisor::vUpdateTime( ){

   tU32 u32TmpTime = 0;

   {                                            // t_standby
      dp_tclSpmDpConfigIgnitionOffTime oIgnOff; // in minutes
      oIgnOff >> u32TmpTime;
      if ( u32TmpTime == 0 ){
         // ignition off timer is off -> set to 31d
         u32TmpTime = 44640;
      }
   }

   _u32WdtClamp15Time = ( u32TmpTime * 60 );
}

tVoid spm_tclIgnitionOffSupervisor::vSetCount( tU32 u32Count ){
   _u32Clamp15OffCount = 0;

   dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
   oCurIgnCount << _u32Clamp15OffCount;

   _u32WdtClamp15Time  = u32Count;
}

tVoid spm_tclIgnitionOffSupervisor::vCheckSupervisionState( ){

      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   // The watchdog is only active if there is no SWDL and no diagnostic session.

   if ( !_bStopSupervision ){
      _u32Clamp15OffCount++;

      dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
      oCurIgnCount << _u32Clamp15OffCount;

      if ( _u32Clamp15OffCount == _u32WdtClamp15Time ){
         // trigger hour logic
         _poclSubStateHandler->vSetSubStateType( SPM_U32_IGNITION_LOGIC, TRUE );
         _u32Clamp15OffCount = 0;
      }
   }
} // vCheckSupervisionState

tVoid spm_tclIgnitionOffSupervisor::vTriggerSupervisionState( ){
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   if ( _bTriggeringEnabled ){
      vUpdateTime( );
      _u32Clamp15OffCount = 0;

      dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
      oCurIgnCount << _u32Clamp15OffCount;
   }
}

tVoid spm_tclIgnitionOffSupervisor::vStartSupervision( ){

   ETG_TRACE_USR4( ( "spm_tclIgnitionOffSupervisor::vStartSupervision()!!!!" ) );
   _bStopSupervision   = FALSE;
   _u32Clamp15OffCount = 0;

   dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
   oCurIgnCount << _u32Clamp15OffCount;

}

tVoid spm_tclIgnitionOffSupervisor::vStopSupervision( ){
   ETG_TRACE_USR4( ( "spm_tclIgnitionOffSupervisor::vStopSupervision()!!!!" ) );
   _bStopSupervision   = TRUE;
   _u32Clamp15OffCount = 0;

   dp_tclSpmDpInternDataIgnitionOffCount oCurIgnCount;
   oCurIgnCount << _u32Clamp15OffCount;
}

tVoid spm_tclIgnitionOffSupervisor::vTraceInfo( ){
   ETG_TRACE_FATAL( ( "----- SUPERVISOR: IGNITION OFF (HOUR) LOGIC -----" ) );
   ETG_TRACE_FATAL( ( "IgnitionOffCount = %d sec, IgnitionOffWdt = %d sec", _u32Clamp15OffCount, _u32WdtClamp15Time ) );
   ETG_TRACE_FATAL( ( "Remaining Ignition Off time: %d sec", _u32WdtClamp15Time - _u32Clamp15OffCount ) );
}

