/*!
  * \file spm_SubStateHandlerUpdate.cpp
  *  \brief
  *    Implementation of Sub State and Trigger Handling.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *   Date    | Author            | Modification
  * 06.12.04  | CM-DI/ESA2 Kollai | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "OsalConf.h"
#include "osal_public.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// FI-zone starts
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SubStateHandler.h"

// spm class definitions w/o interface

// interfaces class definitions
#include "spm_ICcaServiceServer.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_ISuperVisionClient.h"
#include "spm_ISystemStateManager.h"
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationConfiguration.h"
#include "spm_ICcaServer.h"
#include "spm_IOsalProxy.h"

#ifdef SPM_FEATURE_ENABLE_SUBSTATE_DEBOUNCE
#include "spm_ISubStateDebounce.h"
#endif

#include "spm_factory.h"

// spm helper
#include "spm_PowerOnCount.h"
#include "spm_SubStatesStatistics.h"
#include "spm_TransportmodeStatistics.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_SubStateHandlerUpdate.cpp.trc.h"
#endif

#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_SUBSTATEHANDLERUPDATE

tBool spm_tclSubStateHandler::bIsOneTriggerSet( TTriggerMsg *pTriggerMsg ) const
/*!
  * \fn
  *  \brief
  *    Checks if at least one trigger is present(set) or not in the updatetrigger array of passed trigger message
  *
  *  \param[in] pTriggerMsg: TriggerMsg to be checked for
  *  \return true or false
  *
  */
{
   tBool bSuccess = FALSE;

   // check if at least one trigger is set
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      if ( pTriggerMsg->u32TriggerUpdate[i] != 0 ){
         bSuccess = TRUE;
         break;
      }
   }
   return( bSuccess );
}

tVoid spm_tclSubStateHandler::vRemoveAllTrigger( TTriggerMsg *pTriggerMsg,
                                                 tBool        bDataFlags,
                                                 tBool        bUpdateFlags ){
/*!
  * \fn
  *  \brief
  *    Remove(clear) all the triggers present in the passed Triggermessage
  *
  *  \param[in, out] u32TriggerMsg: Trigger Message to be cleared
  *  \param[in] bDataFlags: set it to true if Trigger Data is to be cleared
  *  \param[in] bUpdateFlags: set it to true if UpdateArray is to be cleared
  *  \return none
  *
  */
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      if ( bDataFlags ){
         pTriggerMsg->u32Trigger[i] = 0;
      }
      if ( bUpdateFlags ){
         pTriggerMsg->u32TriggerUpdate[i] = 0;
      }
   }
}

tBool spm_tclSubStateHandler::bIsTriggerUpdateSet( tU32 u32TriggerType ){
/*!
  * \fn
  *  \brief
  *    Checks if trigger is present(set) or not in the updatetrigger array of centrally maintained trigger message
  *
  *  \param[in] u32TriggerType: Trigger whose status is to be checked
  *  \return true or false
  *
  */
   tBool bRet      = FALSE;
   tU8   u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8   u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   if ( _tTrigger.u32TriggerUpdate[u8Index] & ( 1 << u8BitCode ) ){
      bRet = TRUE;
   }
   return( bRet );
}

tBool spm_tclSubStateHandler::bIsTriggerSet( tU32 u32TriggerType ){
/*!
  * \fn
  *  \brief
  *    Checks if trigger is present(set) or not in the trigger array of centrally maintained trigger message
  *
  *  \param[in] u32triggerType: Trigger to be checked
  *  \return true or false
  *
  */
   tBool bRet      = FALSE;
   tU8   u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8   u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   if ( _tTrigger.u32Trigger[u8Index] & ( 1 << u8BitCode ) ){
      bRet = TRUE;
   }
   return( bRet );
}

tBool spm_tclSubStateHandler::bIsUpdateFlag( TTriggerMsg *pTriggerMsg,
                                             tU32         u32TriggerType ){
/*!
  * \fn
  *  \brief
  *    If trigger is present(set) in the updateTrigger of passed Tiggermsg, then, return true or else return false
  *
  *  \param[in] pTriggerMsg: TriggerMsg on which masking has to be carried out
  *  \param[in] u32triggerType: Trigger to be checked
  *  \return true or false
  *
  */
   tBool bRet      = FALSE;
   tU8   u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8   u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   if ( pTriggerMsg->u32TriggerUpdate[u8Index] & ( 1 << u8BitCode ) ){
      bRet = TRUE;
   }
   return( bRet );
}

tVoid spm_tclSubStateHandler::vMaskTrigger( TTriggerMsg *pTriggerMsg,
                                            tU32        *pu32MaskArray,
                                            tU8          u8NumberOfElements ) const
/*!
  * \fn
  *  \brief
  *    Uses the pu32MaskArray as a mask and masks(clear) every other trigger in the trigger message
  *
  *  \param[in] pTriggerMsg: TriggerMsg on which masking has to be carried out
  *  \param[in] pu32MaskArray: Array of triggers to be masked
  *  \param[in] u8NumberOfElements: number of elements in the trigger array[use macro ELEMENTE]
  *  \return none
  *
  */
{
   tU32       *pu32Desc     = pu32MaskArray;
   TTriggerMsg tTriggerMask = TTriggerMsg( );

   while ( pu32Desc < ( pu32MaskArray + u8NumberOfElements ) ){
      tU32 u32ElementNb = * pu32Desc++;
      tU8  u8Index      = (tU8)( u32ElementNb / ( 8 * sizeof( tU32 ) ) );
      tU8  u8BitCode    = (tU8)( u32ElementNb % ( 8 * sizeof( tU32 ) ) );

      tTriggerMask.u32Trigger[u8Index] |= ( 1 << u8BitCode );
      tTriggerMask.u32TriggerUpdate[u8Index] |= ( 1 << u8BitCode );
   }

   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      pTriggerMsg->u32Trigger[i] &= tTriggerMask.u32Trigger[i];
      pTriggerMsg->u32TriggerUpdate[i] &= tTriggerMask.u32TriggerUpdate[i];
   }
} // vMaskTrigger

tBool spm_tclSubStateHandler::bIsDataFlag( TTriggerMsg *pTriggerMsg,
                                           tU32         u32TriggerType ){
/*!
  * \fn
  *  \brief
  *    checks if trigger is present(set) or not in the trigger array of trigger message
  *  \param[in] pTriggerMsg: TriggerMsg to be checked for
  *  \param[in] u32TriggerType: Trigger whose status is to be checked in the passed triggerMsg
  *  \return true if trigger is present else false
  *
  */
   tBool bRet      = FALSE;
   tU8   u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8   u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   if ( pTriggerMsg->u32Trigger[u8Index] & ( 1 << u8BitCode ) ){
      bRet = TRUE;
   }
   return( bRet );
}

tBool spm_tclSubStateHandler::bIsOneDataFlag( TTriggerMsg *pTriggerMsg,
                                              tU32        *pu32TriggerType,
                                              tU8          u8NbTypes )  const
/*!
  * \fn
  *  \brief
  *    Check if atleast one of the trigger type passed in the trigger array exists or not in the trigger array of Trigger message passed.
  *  \param[in] pTriggerMsg: TriggerMsg to be checked for
  *  \param[in] pu32TriggerType: Array of Triggers to be checked for
  *  \param[in] u8NbTypes: number of elements in the trigger array[use macro ELEMENTE]
  *  \return true or false
  *
  */
{
   for ( tU8 i = 0; i < u8NbTypes; i++ ){
      if ( bIsDataFlag( pTriggerMsg, * pu32TriggerType++ ) ){
         return( TRUE );
      }
   }
   return( FALSE );
}

tBool spm_tclSubStateHandler::bIsOneUpdateFlag( TTriggerMsg *pTriggerMsg,
                                                tU32        *pu32TriggerType,
                                                tU8          u8NbTypes )  const
/*!
  * \fn
  *  \brief
  *    Check if any one of the trigger type passed in the trigger array exists or not in the update trigger array of Trigger message passed.
  *  \param[in] pTriggerMsg: TriggerMsg to be checked for
  *  \param[in] pu32TriggerType: Array of Triggers to be checked for
  *  \param[in] u8NbTypes: number of elements in the trigger array[use macro ELEMENTE]
  *  \return true or false
  *
  */
{
   for ( tU8 i = 0; i < u8NbTypes; i++ ){
      if ( bIsUpdateFlag( pTriggerMsg, * pu32TriggerType++ ) ){
         return( TRUE );
      }
   }
   return( FALSE );
}

tVoid spm_tclSubStateHandler::vRemoveTrigger( TTriggerMsg *pTriggerMsg,
                                              tU32         u32TriggerType ){
/*!
  * \fn
  *  \brief
  *    Remove(clear) only the specified(u32triggerType) in the Triggermsg
  *  \param[in,out] pTriggerMsg: TriggerMsg to be checked for
  *  \param[in] u32TriggerType: Trigger to be cleared from the message
  *  \return none
  *
  */
   tU8 u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8 u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   pTriggerMsg->u32Trigger[u8Index]       &= ~( 1 << u8BitCode );
   pTriggerMsg->u32TriggerUpdate[u8Index] &= ~( 1 << u8BitCode );

   return;
}

tVoid spm_tclSubStateHandler::vRemoveAllFlags4AllreadySetTrigger( TTriggerMsg *pTriggerMsg ){
/*!
  * \fn
  *  \brief
  *    Remove(clear)  those flags(trigger updates) from the update array of Triggermsg,
  *    that are already present in the system(central trigger message maitained by substatehandler)
  *  \param[in,out] pTriggerMsg: TriggerMsg to be checked for
  *  \return none
  *
  */
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      for ( tU8 u8BitCode = 0; u8BitCode < 8; u8BitCode++ ){
         if ( ( ( pTriggerMsg->u32TriggerUpdate[i] & ( 1 << u8BitCode ) ) != 0 )
              && ( ( pTriggerMsg->u32Trigger[i] & ( 1 << u8BitCode ) ) != 0 )
              && ( ( _tTrigger.u32Trigger[i] & ( 1 << u8BitCode ) ) != 0 )
              ){
            pTriggerMsg->u32TriggerUpdate[i] &= ~( 1 << u8BitCode );

         }
      }
   }
   return;
} // vRemoveAllFlags4AllreadySetTrigger

tVoid spm_tclSubStateHandler::vMergeTrigger( TTriggerMsg *pTriggerDstMsg,
                                             TTriggerMsg *pTriggerSrcMsg,
                                             tBool        bIgnoreDataFlags,
                                             tBool        bIgnoreUpdateFlags ){
/*!
  * \fn
  *  \brief
  *    Merge all the multiple trigger and trigger updates from source message to destination message
  *  \param[in] pTriggerDstMsg: target for merging of triggers
  *  \param[in] pTriggerSrcMsg: Source Trigger Message
  *  \param[in] bIgnoreDataFlags: Switch for enable/disable of merging of trigger array
  *  \param[in] bIgnoreUpdateFlags: Switch for enable/disable of merging Update Trigger array
  *  \return none
  *
  */
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      if ( !bIgnoreUpdateFlags ){
         pTriggerDstMsg->u32TriggerUpdate[i] |= pTriggerSrcMsg->u32TriggerUpdate[i];
      }
      if ( !bIgnoreDataFlags ){
         pTriggerDstMsg->u32Trigger[i] &= ~pTriggerSrcMsg->u32TriggerUpdate[i];
         pTriggerDstMsg->u32Trigger[i] |= pTriggerSrcMsg->u32Trigger[i];
      }
   }
}

tVoid spm_tclSubStateHandler::vCopyTrigger( TTriggerMsg *pTriggerDstMsg,
                                            TTriggerMsg *pTriggerSrcMsg ){
/*!
  * \fn
  *  \brief
  *    Copy the whole array of triggers(byte by byte copy) from dource message to Destination Message
  *  \param[in] pTriggerSrcMsg: Source message of triggers
  *  \param[in] pTriggerDstMsg: Destination message of triggers
  *  \return none
  *
  */
   for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
      pTriggerDstMsg->u32Trigger[i]       = pTriggerSrcMsg->u32Trigger[i];
      pTriggerDstMsg->u32TriggerUpdate[i] = pTriggerSrcMsg->u32TriggerUpdate[i];
   }
}

tVoid spm_tclSubStateHandler::vSetTrigger( tU32  u32TriggerType,
                                           tBool bUpdate ){
/*!
  * \fn
  *  \brief
  *    Set the particular trigger in central trigger array maintained by substate handler
  *
  *  \param[in] u32triggerType: Trigger to be updated
  *  \param[in] bUpdate: If true, even the update array of trigger message is updated with this trigger
  *  \return none
  *
  */
   tU8 u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8 u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   _tTrigger.u32Trigger[u8Index] |= ( 1 << u8BitCode );
   if ( bUpdate ){
      _tTrigger.u32TriggerUpdate[u8Index] |= ( 1 << u8BitCode );
   }

   vAddSubStateInSet( u32TriggerType );
}

tVoid spm_tclSubStateHandler::vClearTrigger( tU32  u32TriggerType,
                                             tBool bUpdate ){
/*!
  * \fn
  *  \brief
  *    Clear the particular trigger in central trigger array maintained by substate handler
  *
  *  \param[in] u32triggerType: Trigger to be updated
  *  \param[in] bUpdate: If true, the trigger Update array is updated with this trigger. Note that the Trigger information
  *   is cleared from Trigger array and not from Trigger Update array.
  *  \return none
  *
  */
   tU8 u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8 u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   _tTrigger.u32Trigger[u8Index] &= ~( 1 << u8BitCode );
   if ( bUpdate ){
      _tTrigger.u32TriggerUpdate[u8Index] |= ( 1 << u8BitCode );
   }

   vRemoveSubStateInSet( u32TriggerType );
}

tVoid spm_tclSubStateHandler::vPrjSubStateType( tU32  /*u32TriggerType*/,
                                                tBool /*bTriggerState*/ ){
/*!
  * \fn
  *  \brief
  *    Project Specific Sub-State logic can be handled here.
  *    This method should be overridden by SubstateHandlerConfig and is given a chance to execute when vsetsubstatetype() is called
  *    Say, Fastshutdown could be implemented here.
  *
  *  \param[in] u32triggerType: Trigger to be handled
  *  \param[in] btriggerState: State of the trigger(True or False)
  *  \return none
  *
  */
   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vPrjSubStateType(): Should be overloaded by class 'spm_tclSubStateHandlerConfig'." ) );
} //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

tVoid spm_tclSubStateHandler::vSetSubStateType( tU32  u32TriggerType,
                                                tBool bTriggerState,
                                                tBool bDebounced ){
   /*!
     * \fn
     *  \brief
     *    This function is used set different types of sub states to ON/OFF. OnOffTrigger method message
     *    is created with the trigger information and the message is posted to the CCA message handler.
     *
     *  \param[in] u32TriggerType: Trigger to be handled
     *  \param[in] bTriggerState: State of the trigger(True or False)
     *  \return none
     *
     */

   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): Substate %u, TriggerState: %u, Debounced: %u", ETG_ENUM( SPM_SUBSTATETYPE, u32TriggerType ), ETG_ENUM( SPM_BOOL_STATE, bTriggerState ), ETG_ENUM( SPM_BOOL_STATE, bDebounced ) ) );
   #ifdef SPM_ENABLE_DM_VERITY_CHECK
       if (bIsTriggerSet(SPM_U32_DMVERITY_CHECK)) {
           ETG_TRACE_FATAL( ( "spm_tclSubStateHandler::vSetSubStateType()-->DM_VERITY check is running." ) );
           spm_tclSubStateHandler::vTraceTrigger(SSH_vCurrentTrigger, &_tTriggerWakeup);
           if (bTriggerState && bIsDataFlag(&_tTriggerWakeup, u32TriggerType) ){
               SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );
               // reboot immediately because trigger is a wakeup
               ETG_TRACE_FATAL( ( "spm_tclSubStateHandler::vSetSubStateType(): !!!!!!!!!!! REBOOT --> immediately because trigger is a wakeup !!!!!!!" ) );
               _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_LATE_WAKEKUP );
           }
       }
   #endif

      #ifdef SPM_FEATURE_ENABLE_SUBSTATE_DEBOUNCE
         tBool bSendMsg = TRUE;
         if ( bDebounced ){
            SPM_GET_IF_REFERENCE_NEW_VAR( poSubStateDebounce, ISpmSubStateDebounce );
            if ( TRUE == poSubStateDebounce->bUpdateSubState( u32TriggerType, bTriggerState ) ){
               bSendMsg = FALSE;
            }
         }
         if ( bSendMsg ){
            #else
               (tVoid)bDebounced;
               {
            #endif
			#ifndef LCM_UNIT_TESTS
			//Create Object..
            spm_corefi_tclMsgOnOffTriggerMethodStart oStartData;

            oStartData.eSwitchOnTrigger.enType = (spm_fi_tcl_SPM_e32_SubStateType::tenType)u32TriggerType;
            oStartData.bState                  = bTriggerState;

            fi_tclVisitorMessage                     oSpmMsg( oStartData );
            oSpmMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                                      0,
                                      AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                      0,
                                      0,
                                      0,
                                      0x100,
                                      SPM_COREFI_C_U16_ONOFFTRIGGER,
                                      AMT_C_U8_CCAMSG_OPCODE_METHODSTART );

            SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
            if ( _poclCcaMsgHandler->bPostMessage( &oSpmMsg ) == FALSE ){ // If posting of OnOffStateChange message failed, trigger reset now.
               ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vSetSubStateType(): Cant post SPM_COREFI_C_U16_ONOFFTRIGGER message!" ) );
               FATAL_M_ASSERT_ALWAYS( );
            }
			#endif   // ifndef LCM_UNIT_TESTS
         }
} // vSetSubStateType

tVoid spm_tclSubStateHandler::vTriggerSubStateType( tU32  u32TriggerType,
                                                    tBool bTriggerState,
                                                    tBool bSetTriggerOnly ){
/*!
  * \fn
  *  \brief
  *    Generic  Sub-State logic is  handled here.This is the entry point for any sub-state in the system
  *    Only after  sub-states are initially handled here(updation of substatestatistics,CCA properties,) the message
  *    is posted to the message queue of SubstateHandler Thread, which unblocks and then updates SystemStateManager of the new substate
  *    provided system is not experiencing low voltage conditions
  *
  *  \param[in] u32TriggerType: Trigger to be handled
  *  \param[in] bTriggerState: State of the trigger(True or False)
  *  \param[in] bSetTriggerOnly: if true, then only update the centrally maintained\
  *         trigger message (no message will be posted to the message queue of Substatehandler)
  *  \return none
  *
  */
   TTriggerMsg       tMsg       = TTriggerMsg( );

   #ifndef LCM_UNIT_TESTS
      OSAL_tThreadID myThreadId = OSAL_ThreadWhoAmI( );
      if ( myThreadId != _idt ){
         std::string strTmpThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 );
         (tVoid)bGetTaskName( myThreadId, myThreadId, &strTmpThreadName[0] );

         ETG_TRACE_FATAL( ( "spm_tclSubStateHandler::vTriggerSubStateType(): Wrong thread context for calling database: Thread: (tid=%d) '%s' .",
                            myThreadId,
                            strTmpThreadName.c_str( ) ) );

         NORMAL_M_ASSERT_ALWAYS( );
      }
   #endif

   // get position of current trigger in OnReason array
   tU8 u8Index   = (tU8)( u32TriggerType / ( 8 * sizeof( tU32 ) ) );
   tU8 u8BitCode = (tU8)( u32TriggerType % ( 8 * sizeof( tU32 ) ) );

   tMsg.u32TriggerUpdate[u8Index] = 1 << u8BitCode;
   if ( bTriggerState ){
      tMsg.u32Trigger[u8Index] = 1 << u8BitCode;
      // type specific implememtaion (for all generic project independent trigger)
   }
   if ( bTriggerState ){
      vAddSubStateInSet( u32TriggerType );
   } else {
      vRemoveSubStateInSet( u32TriggerType );
   }

   SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
   if ( poclSystemStateManager->bPointOfNoReturnReached( ) ){

      SPM_NULL_POINTER_CHECK( _poclSystemPowerManager );

      //check for changes to active
      spm_tclSubStateHandler::vTraceTrigger( SSH_vWakeupTrigger, &_tTriggerWakeup );
      if ( bIsDataFlag( &_tTriggerWakeup, u32TriggerType ) ){
         if ( bTriggerState ){
            #ifdef SPM_SUBSTATE_WAKEUP_RESTART_ACTIVE
               tU32 au32RestartTrigger[] = { SPM_SUBSTATE_WAKEUP_RESTART_ACTIVE };
               tU8  i                    = 0;
               tU8  u8ArrayCount         = ELEMENTE( au32RestartTrigger );
               for ( i = 0; i < u8ArrayCount; i++ ){
                  if ( au32RestartTrigger[i] == u32TriggerType ){
                     // reboot immediately because trigger is a wakeup
                     ETG_TRACE_FATAL( ( "spm_tclSubStateHandler::vTriggerSubStateType(): !!!!!!!!!!! REBOOT --> immediately because trigger is a wakeup !!!!!!!" ) );
                     _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_LATE_WAKEKUP );
                  }
               }
            #endif

         } else {

            #ifdef SPM_SUBSTATE_WAKEUP_RESTART_INACTIVE
               //check for changes to inactive
               tU32 au32RestartTrigger[] = { SPM_SUBSTATE_WAKEUP_RESTART_INACTIVE };
               tU8  i                    = 0;
               tU8  u8ArrayCount         = ELEMENTE( au32RestartTrigger );

               for ( i = 0; i < u8ArrayCount; i++ ){
                  if ( au32RestartTrigger[i] == u32TriggerType ){
                     // reboot immediately because trigger is a wakeup
                     ETG_TRACE_FATAL( ( "spm_tclSubStateHandler::vTriggerSubStateType(): !!!!!!!!!!! REBOOT --> immediately because trigger is a wakeup !!!!!!!" ) );
                     _poclSystemPowerManager->vShutdownSystem( SPM_U32_SHUTDOWN_LATE_WAKEKUP );
                  }
               }
            #endif
         }
      }
   }

   {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poTransportmodeStatistics, spm_tclTransportmodeStatistics, ISpmStatistics );
      if ( u32TriggerType == SPM_U32_TRANSPORTMODE ){
         if ( bTriggerState ){
            poTransportmodeStatistics->vSetNewData( SPM_U32_TRANSPORT_MODE_ON );
         }
      }
   }
   {
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poSubStateStatistics, spm_tclSubStateStatistics, ISpmStatistics );
         if ( bTriggerState ){
            if ( u32TriggerType == SPM_U32_ON_TIPPER ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_TIPPER );
            } else if ( u32TriggerType == SPM_U32_DOOROPEN ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_DOOROPEN );
            } else if ( u32TriggerType == SPM_U32_IGNITION ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_IGNITION );
            } else if ( u32TriggerType == SPM_U32_S_CONTACT ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_SCONT );
            } else if ( u32TriggerType == SPM_U32_PHONE ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_PHONE );
            } else if ( u32TriggerType == SPM_U32_DOWNLOAD ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_DOWNLOAD );
            } else if ( u32TriggerType == SPM_U32_DIAGNOSIS ){
               poSubStateStatistics->vSetNewData( SPM_U32_STATISTIC_DIAG );
            }
         }
      }
   }

   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( pPowerOn, spm_tclPowerOnCount, ISpmPowerOnStatistics );
      pPowerOn->vCheckPwrOnTime( );
   }

   if ( u32TriggerType == SPM_U32_IGNITION ){
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSupervisionClientIgnOff, spm_tclIgnitionOffSupervisor, ISpmSupervisionClient );

      if ( bTriggerState ){
         // stop hour logic
         vClearTrigger( SPM_U32_IGNITION_LOGIC, TRUE );
         poclSupervisionClientIgnOff->vStopSupervision( );
      } else {
         poclSupervisionClientIgnOff->vStartSupervision( );
      }
   }

   // forward eject via CCA
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );
   if ( u32TriggerType == SPM_U32_EJECT ){
      spm_corefi_tclMsgMediaEjectStateStatus oEjectStatus;
      if ( bTriggerState ){
         oEjectStatus.MediaEjectState.enType = spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE::FI_EN_SPM_U32_EJECT_PRESSED;
         _u32StartUpReasons                 |= SPM_BC0_EJECT;
      } else {
         oEjectStatus.MediaEjectState.enType = spm_fi_tcl_SPM_e32_MEDIA_EJECT_STATE::FI_EN_SPM_U32_EJECT_RELEASED;
      }
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_MEDIAEJECTSTATE, &oEjectStatus );

      _oOnceTriggeredSubStates[u32TriggerType] = OSAL_ClockGetElapsedTime();
      vUpdateHistoryTriggerOnInterface();
   }

   // forward TIPPER via CCA
   if ( u32TriggerType == SPM_U32_ON_TIPPER ){
      spm_corefi_tclMsgOnTipperStateStatus oTipperStatus;
      if ( bTriggerState ){
         oTipperStatus.OnTipperState.enType = spm_fi_tcl_SPM_e32_KEY_STATE::FI_EN_SPM_U32_PRESSED;
      } else {
         oTipperStatus.OnTipperState.enType = spm_fi_tcl_SPM_e32_KEY_STATE::FI_EN_SPM_U32_RELEASED;
      }
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_ONTIPPERSTATE, &oTipperStatus );

      _oOnceTriggeredSubStates[u32TriggerType] = OSAL_ClockGetElapsedTime();
      vUpdateHistoryTriggerOnInterface();
   }

   // forward PHONE_MUTE via CCA
   if ( u32TriggerType == SPM_U32_PHONE_MUTE ){
      spm_corefi_tclMsgPhoneMuteStateStatus oPhoneStatus;
      if ( bTriggerState ){
         oPhoneStatus.PhoneMuteState.enType = spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE::FI_EN_SPM_U32_PHONEMUTE_ACTIVE;
      } else {
         oPhoneStatus.PhoneMuteState.enType = spm_fi_tcl_SPM_e32_PHONE_MUTE_STATE::FI_EN_SPM_U32_PHONEMUTE_INACTIVE;
      }
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_PHONEMUTESTATE, &oPhoneStatus );

      _oOnceTriggeredSubStates[u32TriggerType] = OSAL_ClockGetElapsedTime();
      vUpdateHistoryTriggerOnInterface();
   }

   // forward CD Insert via CCA
   if ( ( u32TriggerType == SPM_U32_CD_INSERT ) && ( !bTriggerState ) ){
      vClearTrigger( SPM_U32_CD_INSERT_CLAMPS );
      spm_corefi_tclMsgInsertStateStatus oInsertStatus;
      oInsertStatus.InsertState.enType = spm_fi_tcl_SPM_e32_CD_INSERT_STATE::FI_EN_SPM_U32_CD_INSERT_INACTIVE;
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_INSERTSTATE, &oInsertStatus );
   }

   if ( u32TriggerType == SPM_U32_CD_INSERT_CLAMPS ){
      spm_corefi_tclMsgInsertStateStatus oInsertStatus;
      if ( bTriggerState ){
         oInsertStatus.InsertState.enType = spm_fi_tcl_SPM_e32_CD_INSERT_STATE::FI_EN_SPM_U32_CD_INSERT_ACTIVE;
      } else {
         oInsertStatus.InsertState.enType = spm_fi_tcl_SPM_e32_CD_INSERT_STATE::FI_EN_SPM_U32_CD_INSERT_INACTIVE;
      }
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_INSERTSTATE, &oInsertStatus );
   }

#ifdef SPM_ENABLE_DM_VERITY_CHECK
   if ( (u32TriggerType == SPM_U32_EMERGENCY_OFF)  || (u32TriggerType == SPM_U32_FAST_SHUTDOWN)){
       if ( bTriggerState ){
           if (_poclOsalProxy != NULL) _poclOsalProxy->bSetFastShutdown(DEV_WUP_C_U8_FAST_SHUTDOWN_ACTIVE);
       }
   }
#endif

   #ifdef SPM_DIAG_FAST_SHUTDOWN_APPS_IGNORE
      if ( u32TriggerType == SPM_U32_DIAG_FAST_SHUTDOWN ){
         if ( bTriggerState ){
            SPM_GET_IF_REFERENCE_NEW_VAR( poAppCfg, ISpmApplicationConfiguration );
            tU16 au16AppIgnore[] = { SPM_DIAG_FAST_SHUTDOWN_APPS_IGNORE };
            tU8  u8ArrayCount    = ELEMENTE( au16AppIgnore );

            for ( tU8 i = 0; i < u8ArrayCount; i++ ){
               ETG_TRACE_ERR( ( "spm_tclSubStateHandler::vTriggerSubStateType(): No state change for application %u if fast shutdown is triggered by diagnosis!", ETG_ENUM( ail_u16AppId, au16AppIgnore[i] ) ) );
               tU32 u32CurAppState = poAppCfg->u32GetNewAppState( au16AppIgnore[i] );

               poAppCfg->vSetValue( au16AppIgnore[i], spm_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_PREPARE_SHUTDOWN, u32CurAppState );
            }
         }
      }
   #endif // ifdef SPM_DIAG_FAST_SHUTDOWN_APPS_IGNORE

   if ( u32TriggerType == SPM_U32_STATE_DEBOUNCE ){
      if ( bTriggerState ){
         if ( OSAL_s32TimerSetTime( _hDeboounceTimer, SPM_U32_DEBOUNCE_TIME, 0 ) != OSAL_OK ){
            ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vTriggerSubStateType(): !!!!!! Timer _hDeboounceTimer could not get started !!!!!!" ) );
         }
      }
   }

   SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );
   if ( u32TriggerType == SPM_U32_OFF_LASTSTATE_CYCLE ){
      _poclGlobalApplicationManager->vForceIntermediateState( SPM_SYSTEM_PREPARE_SHUTDOWN, TRUE );
      // and now check for project specific trigger
   }
   vPrjSubStateType( u32TriggerType, bTriggerState );
   if ( bSetTriggerOnly ){
      vMergeTrigger( &_tTrigger, &tMsg );
   } else if ( !bPostTriggerMessage( &tMsg ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vTriggerSubStateType(): bPostTriggerMessage() has failed, current state: " ) );
      vTraceTrigger( SSH_vCurrentTrigger, &tMsg, ETG_LEVEL_ERRMEM );
   }
} // vSetSubStateType

tVoid spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger( tU32 u32WakeupConfigValue ){
/*!
  * \fn
  *  \brief
  *    Map the dev_wup wakeup config mask with the substate trigger and Set the particular triggers in wakeup trigger array maintained by substate handler
  *
  *  \param[in] u32WakeupConfigValue: wake up config value to be updated in wakeup trigger array.
  *  \return none
  *
  */
         ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger() wakeup config value %d", u32WakeupConfigValue ) );

   for ( tU32 i = 0; i < ( 8 * sizeof( tU32 ) ); i++ ){
      tU32  u32Temp             = (tU32)1 << i;
      tU32  u32SubStateTrigger  = 0;
      tBool bSetWakeupTrigger   = TRUE;
      tU32  u32WakeupConfigMask = u32WakeupConfigValue & u32Temp;
      if ( 0 != u32WakeupConfigMask ){
         ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger(): WakeupConfigMask '%u'", ETG_ENUM( DEV_WUP_u32WakeupConfigMask, i ) ) );
         switch ( u32WakeupConfigMask ){
            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_ON_TIPPER:
               u32SubStateTrigger = SPM_U32_ON_TIPPER;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN:
               u32SubStateTrigger = SPM_U32_NETWORK_ACTIVITY;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_IGN_PIN:
               u32SubStateTrigger = SPM_U32_IGNITION;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_TEL_MUTE:
               u32SubStateTrigger = SPM_U32_PHONE_MUTE;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CD_INSERT_DETECTED:
               u32SubStateTrigger = SPM_U32_CD_INSERT;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CD_EJECT_DETECTED:
               u32SubStateTrigger = SPM_U32_EJECT;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_S_CONTACT_WAKEUP:
               u32SubStateTrigger = SPM_U32_S_CONTACT;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CELLNETWORK_WAKEUP:
               u32SubStateTrigger = SPM_U32_CELLNETWORK;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN2_WAKEUP:
            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN3_WAKEUP:
            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_CAN4_WAKEUP:
               u32SubStateTrigger = SPM_U32_CAN_WAKEUP;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_LIN_WAKEUP:
               u32SubStateTrigger = SPM_U32_LIN;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_EXTERNAL_PIN_WAKEUP:
               u32SubStateTrigger = SPM_U32_EXTERNAL_GPIO;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_DEBUG_WAKEUP:
               bSetWakeupTrigger = FALSE;
               break;

            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_MOST:
            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_RTC_WAKEUP:
            case DEV_WUP_C_U32_WAKEUP_REASONS_MASK_ODOMETER_WAKEUP:
               bSetWakeupTrigger = FALSE;
               ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger(): No Mapping Substate Trigger for WakeupConfigMask '%u'", ETG_ENUM( DEV_WUP_u32WakeupConfigMask, (tUInt)i ) ) );
               break;

            default:
               bSetWakeupTrigger = FALSE;
               ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger(): Unknown WakeupConfigMask. (bit '%u' is set in WakeupConfigValue %u).", ETG_ENUM( DEV_WUP_u32WakeupConfigMask, (tUInt)i ), (tUInt)u32WakeupConfigValue ) );
               break;
         } // switch

         if ( bSetWakeupTrigger ){
            ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vMapAndUpdateWakeupTrigger(): u32SubStateTrigger %u", ETG_ENUM( spmTriggerTypes, (tUInt)u32SubStateTrigger ) ) );
            tU8 u8Index   = (tU8)( u32SubStateTrigger / ( 8 * sizeof( tU32 ) ) );
            tU8 u8BitCode = (tU8)( u32SubStateTrigger % ( 8 * sizeof( tU32 ) ) );

            _tTriggerWakeup.u32Trigger[u8Index] |= ( 1 << u8BitCode );
         }
      }
   }
} // vMapAndUpdateWakeupTrigger

tVoid spm_tclSubStateHandler::vTraceTrigger( tU8          u8TraceType,
                                             TTriggerMsg *pTrigger,
                                             tU16         u16Level ){
/*!
  * \fn
  *  \brief
  *    This method is used to trace triggers present in the passed trigger message.
  *
  *
  *  \param[in] u8TraceType: SSH_vCurrentTrigger is to be used for tracing centrally maintained triggers in SubstateHandler
  *                          SSM_vHistoryTrigger is to be used for tracing Substate Statistics(Engineering History)
  *                          SSH_vRestoredTrigger is to be used for tracing Persistently stored Triggers
  *                          SSH_vWakeupTrigger is to be used for tracing wakeup Triggers which are configured.
  *  \param[in] pTrigger:    Trigger message to be traced
  *  \param[in] u16Level:    Trace level (by default trace level is TR_LEVEL_USER_1)
  *  \return none
  *
  */
   if ( pTrigger != NULL ){
      #ifndef LCM_UNIT_TESTS
         if ( spm_tclTrace::spm_bIsTraceActive( SPM_TRACE_CLASS_SPM_SSH, u16Level ) ){
            SPM_ETG_CREATE_FORMAT_STRING
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 1 )
               SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 2 )
               SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T8,  3 )
            #if SPM_TRIGGER_ARRAY_SIZE > 0
               etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM_SSH, u16Level ), 0, 0, u32Format, I_SPM_SSH_04, pTrigger->u32TriggerUpdate[0], pTrigger->u32Trigger[0], u8TraceType );
            #endif
            #if SPM_TRIGGER_ARRAY_SIZE > 1
               etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM_SSH, u16Level ), 0, 0, u32Format, I_SPM_SSH_05, pTrigger->u32TriggerUpdate[1], pTrigger->u32Trigger[1], u8TraceType );
            #endif
            #if SPM_TRIGGER_ARRAY_SIZE > 2
               etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM_SSH, u16Level ), 0, 0, u32Format, I_SPM_SSH_06, pTrigger->u32TriggerUpdate[2], pTrigger->u32Trigger[2], u8TraceType );
            #endif
            #if SPM_TRIGGER_ARRAY_SIZE > 3
               etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM_SSH, u16Level ), 0, 0, u32Format, I_SPM_SSH_07, pTrigger->u32TriggerUpdate[3], pTrigger->u32Trigger[3], u8TraceType );
            #endif
            #if SPM_TRIGGER_ARRAY_SIZE > 4
               etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM_SSH, u16Level ), 0, 0, u32Format, I_SPM_SSH_08, pTrigger->u32TriggerUpdate[4], pTrigger->u32Trigger[4], u8TraceType );
            #endif
         }
      #endif // ifndef LCM_UNIT_TESTS
   }
   if ( u16Level == (etg_tU16)(ETG_LEVEL_ERRMEM) ){
       vTraceTrigger( u8TraceType, pTrigger, ETG_LEVEL_FATAL );
   }
} // vTraceTrigger

// EOF

