/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_SubStateHandler.h
  * @brief        This is header file contains implementation for the sub state and trigger handling.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <set>
#include <map>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_FITypes.h"
#include "spm_ISubStateClient.h"

class ISpmSupervisionClient;
class ISpmGlobalApplicationManager;
class ISpmCcaServiceServer;
class ISpmCcaServer;
class ISpmApplicationDatabase;
class ISpmOsalProxy;
class ISpmSystemPowerManager;
class ISpmWorkerServer;
class ISpmSyncHandler;


// TRCgen enum tenSpmFastShutDownReasons {
#define SPM_U32_FAST_SHUTDOWN_AUS_HMI   0x00000001
#define SPM_U32_FAST_SHUTDOWN_EMERGENCY 0x00000002
#define SPM_U32_FAST_SHUTDOWN_TRANSPORT 0x00000004
#define SPM_U32_FAST_SHUTDOWN_DEVICE    0x00000008
#define SPM_U32_FAST_SHUTDOWN_USER      0x00000010
#define SPM_U32_FAST_SHUTDOWN_FACTORY   0x00000020
#define SPM_U32_FAST_SHUTDOWN_DIAG      0x00000040
// TRCgen };

// TRCgen enum tenSpmTraceTrigger {
#define SSH_vNewTrigger              0
#define SPM_vChannelMgr              1
#define SSH_vCurrentTrigger          2
#define SSM_vHistoryTrigger          3
#define SSH_vRestoredTrigger         4
#define SSH_vWakeupTrigger           5

// TRCgen };

// substate ALIAS
#define SPM_U32_S_CONTACT SPM_U32_ACCESSORY

#define SPM_SUBSTATE_IGNITION_OFF_TIME_SECOND_REF      ( 10 )
#define SPM_SUBSTATE_IGNITION_OFF_TIME_MINUTE_REF      ( 60 / SPM_SUBSTATE_IGNITION_OFF_TIME_SECOND_REF )
#define SPM_SUBSTATE_IGNITION_OFF_TIME_BASE            ( 1 * SPM_SUBSTATE_IGNITION_OFF_TIME_SECOND_REF * 1000 )

/*!
  *  \class spm_tclSubStateHandler
  *  \brief This class implements the sub state and trigger handling.
  *
  *         Any substate change always gets notified first to SubStateHandler and SubStateHandler
  *         then decides, whether or not to inform SystemStateManager of the new SubState Trigger.
  *
  *        This class acts as an interface for the other sub components like spm_tclCriticalVoltageManagerIntern,
  *        spm_tclOsalProxy, spm_tclSupplyManagementServiceHandler, spm_tclThermalManagementServiceHandler etc in FC SPM.
  *        These sub components in FC SPM can trigger the state change by interacting with SubstateHandler.
  */
class spm_tclSubStateHandler : public ISpmSubStateClient
{
// new implementation for SubState/Trigger handling

public:

// virtual tVoid vNewTrigger(tU32 u32TriggerType, tBool bTriggerState);

virtual tVoid vRemoveAllTrigger( TTriggerMsg *pTriggerMsg,
                                 tBool        bDataFlags = TRUE,
                                 tBool        bUpdateFlags = TRUE );

static tVoid vMergeTrigger( TTriggerMsg *pTriggerDstMsg,
                            TTriggerMsg *pTriggerSrcMsg,
                            tBool        bIgnoreDataFlags = FALSE,
                            tBool        bIgnoreUpdateFlags = FALSE );

virtual tVoid vCopyTrigger( TTriggerMsg *pTriggerDstMsg,
                            TTriggerMsg *pTriggerSrcMsg );

static tBool bIsDataFlag( TTriggerMsg *pTriggerMsg,
                          tU32         u32TriggerType );

static tBool bIsUpdateFlag( TTriggerMsg *pTriggerMsg,
                            tU32         u32TriggerType );

static tVoid vTraceTrigger( tU8          u8TraceType,
                            TTriggerMsg *pTrigger = NULL,
                            tU16         u16Level = TR_LEVEL_USER_1 );

virtual tBool bIsTriggerSet( tU32 u32TriggerType );

virtual tBool bIsTriggerUpdateSet( tU32 u32TriggerType );

protected:
virtual tVoid vAddSubStateInSet( tU32 u32SubState );

virtual tVoid vRemoveSubStateInSet( tU32 u32SubState );

virtual tVoid vUpdateSubStatesOnInterface( );
virtual tVoid vUpdateHistoryTriggerOnInterface( );

tBool bPostTriggerMessage( TTriggerMsg *pTriggerMsg );

static tVoid dwSubStateHandlerProc( tPVoid pvArg );

tBool bIsOneTriggerSet( TTriggerMsg *pTriggerMsg )  const;

tBool bIsOneDataFlag( TTriggerMsg *pTriggerMsg,
                      tU32        *pu32TriggerType,
                      tU8          u8NbTypes ) const;

tBool bIsOneUpdateFlag( TTriggerMsg *pTriggerMsg,
                        tU32        *pu32TriggerType,
                        tU8          u8NbTypes ) const;

virtual tVoid vRemoveTrigger( TTriggerMsg *pTriggerMsg,
                              tU32         u32TriggerType );

virtual tVoid vRemoveAllFlags4AllreadySetTrigger( TTriggerMsg *pTriggerMsg );

tVoid vMaskTrigger( TTriggerMsg *pTriggerMsg,
                    tU32        *pu32MaskArray,
                    tU8          u8NumberOfElements ) const;

tVoid vSetTrigger( tU32  u32TriggerType,
                   tBool bUpdate = FALSE );

tVoid vClearTrigger( tU32  u32TriggerType,
                     tBool bUpdate = FALSE );

virtual tU32 u32GetIgnitionCycleCount( ){ return( _u32IgnitionCycleCount ); }

TTriggerMsg _tTrigger;
TTriggerMsg _tTriggerWhileUnderVolt;
TTriggerMsg _tTriggerWhileDebouncing;

TTriggerMsg _tTriggerWakeup;

// ******************************************************

protected:
#ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   tU32 _u32SubStateLockMask;
   tU32              _u32InternSubStateLockMask;

   // tVoid vDebounceStateChange();
   static tVoid vDebounceStateChangeTimerCallback( tVoid *pArg );

   OSAL_tTimerHandle _hDeboounceTimer;
#endif // ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE

#ifdef SPM_FEATURE_ENABLE_LONG_PRESS_RESET_TIMER
   tVoid  vLongPressReset( );

   static tVoid vLongPressResetTimerCallback( tVoid *pArg );

   OSAL_tTimerHandle _hLongPressResetTimer;
#endif

tBool _bSubStateChange;

// Systemstate: some apps are interested in this substate
tU32                          _u32SubStateForceReset;

tU32                          _u32FastShutdownReason;

// remember trigger reasons
tU32                          _u32StartUpReasons;

static tVoid vEjectLongPressDetect( tVoid *pArg );

OSAL_tTimerHandle             _hEjectLongPressTimer;

tBool                         _bInitDevCall;

tBool                         _bRequestPhone;

tBool                         _bSContact;

tBool                         _bHwIgnition;

tU32                          _u32LastCpuSum;
tU32                          _u32LastIdle;

ISpmSupervisionClient        *_poclSupervisionManager;
ISpmGlobalApplicationManager *_poclGlobalApplicationManager;
ISpmCcaServiceServer         *_poclCcaServiceHandler;
ISpmCcaServer                *_poclCcaMsgHandler;
ISpmSystemPowerManager       *_poclSystemPowerManager;
ISpmWorkerServer             *_poclWorkerServer;
ISpmOsalProxy                *_poclOsalProxy;
ISpmSyncHandler              *_poclSyncHandler;

tU16                          _u16AppId;

tU32                          _u32IgnitionCycleCount;

std::set < tU32 >             _oActiveSubStates;
std::map < tU32, tU32 >       _oOnceTriggeredSubStates;

public:
spm_tclSubStateHandler( const ISpmFactory& factory,
                        tU32               u32AppId );
virtual ~spm_tclSubStateHandler( );

// trace handling
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

virtual TTriggerMsg tGetTrigger( ){ return( _tTrigger ); }

virtual tVoid vTriggerSubStateType( tU32  u32TriggerType,
                                    tBool bTriggerState,
                                    tBool bSetTriggerOnly = FALSE );

virtual tVoid vSetSubStateType( tU32  u32SubStateType,
                                tBool bSubStateData,
                                tBool bDebounced = TRUE );

virtual tVoid vPrjSubStateType( tU32  u32SubStateType,
                                tBool bSubStateData );

virtual tU32  u32GetStartupOnReason( tBool bResetState = FALSE ){
   tU32 u32Ret = 0;

   if ( _u32StartUpReasons != 0xffffffff ){
      u32Ret = _u32StartUpReasons;
      if ( bResetState ){
         _u32StartUpReasons = 0xffffffff;
      }
   }
   return( u32Ret );
}

// tU32  u32GetSubstateUpdate();

tBool bIsHwIgnActive( ) const {
   return( _bHwIgnition );
}

virtual tBool bSetWakeupConfig( tU32 u32WakeupConfig );

virtual tVoid vProcessSystemState( tU32 u32NewSystemState );

virtual tBool bIsFastShutdownActive( ) const {
   return( ( _u32FastShutdownReason == 0 ) ? FALSE : TRUE );
}

// start the cca input thread
virtual tVoid vStart( const std::string& strName,
                      tU32               u32Prio,
                      tU32               u32Stack );

// implementation of spm_tclActive
virtual tVoid vOnStart( );

virtual tVoid vOnTerminate( );

virtual tVoid main( );

virtual tVoid vHandleMsgQueue( );

protected:
// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args );

virtual const tChar*getName( ) const {
   return( "spm_tclSubStateHandler" );
}

virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

static tVoid vOnQueueCallback( tVoid *pArg );

virtual tVoid vMapAndUpdateWakeupTrigger( tU32 u32WakeupConfigValue );

private:
virtual tBool bCheckForFastShutdown( tU32 /*u32FastShutdownReason*/ ){ return( FALSE ); } //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

OSAL_tMQueueHandle _hSubStateQueue;
tBool              _bDataToStore;

OSAL_tThreadID     _idt;

tBool _bIsPrepareShutdownTmlTest;

public:
virtual tVoid vTerminate( );

virtual tVoid vSendSubStatePropertyChange( );

};

