/*!
  * \file spm_SubStateHandler.cpp
  *  \brief
  *    Implementation of Sub State and Trigger Handling. Any substate change always gets notified first to SubStateHandler and SubStateHandler
  *    then decides, whether or not to inform SystemStateManager of the new SubState Trigger.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *   20.01.12  | CM-AI/PJVW32 Kollai  | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_SubStateHandler.h"
// spm helper
#include "spm_CriticalSection.h"

// spm class definitions w/o interface

// interfaces class definitions
#include "spm_ISuperVisionClient.h"
#include "spm_IOsalProxy.h"
#include "spm_ISystemStateManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ISystemPowerManager.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IGlobalApplicationManager.h"
#include "spm_IWakeupHandler.h"
#include "spm_ISyncHandler.h"
#include "spm_ICcaServer.h"

#include "spm_IFactory.h"

// spm helper


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSH
#include "trcGenProj/Header/spm_SubStateHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_SUBSTATEHANDLER

spm_tclSubStateHandler::spm_tclSubStateHandler( const ISpmFactory& factory,
                                                tU32               u32AppId )
   : ISpmSubStateClient( factory )
   #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   , _hDeboounceTimer(OSAL_C_INVALID_HANDLE)
   #endif
   #ifdef SPM_FEATURE_ENABLE_LONG_PRESS_RESET_TIMER
   , _hLongPressResetTimer(OSAL_C_INVALID_HANDLE)
   #endif
   , _bSubStateChange( FALSE )
   , _u32StartUpReasons( 0 )
   , _hEjectLongPressTimer(OSAL_C_INVALID_HANDLE)
   , _bInitDevCall( TRUE )
   , _bRequestPhone( FALSE )
   , _bSContact( FALSE )
   , _u32LastCpuSum( 0 )
   , _u32LastIdle( 0 )
   , _poclSupervisionManager( NULL )
   , _poclGlobalApplicationManager( NULL )
   , _poclCcaServiceHandler( NULL )
   , _poclCcaMsgHandler( NULL )
   , _poclSystemPowerManager( NULL )
   , _poclWorkerServer( NULL )
   , _poclOsalProxy( NULL )
   , _poclSyncHandler( NULL )
   , _u16AppId( (tU16)u32AppId )
   , _u32IgnitionCycleCount(0)
   , _hSubStateQueue( OSAL_C_INVALID_HANDLE )
   , _bDataToStore( TRUE )
   , _idt( - 1 )
   , _bIsPrepareShutdownTmlTest( FALSE ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in] factory: spm factory object.
  *  \param[in] u32AppId:  FC SPM Application Id.
  ******
  */

   /* create semaphore to protect list access */
   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)vEjectLongPressDetect, ( tPVoid ) this,
                             &_hEjectLongPressTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::spm_tclSubStateHandler: !!!!!! Timer vEjectLongPressDetect could not be created: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
      _u32SubStateLockMask       = 0;
      _u32InternSubStateLockMask = 0;

      if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)vDebounceStateChangeTimerCallback, ( tPVoid ) this,
                                &_hDeboounceTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::spm_tclSubStateHandler: !!!!!! Timer vDebounceStateChangeTimerCallback could not be created: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif
   // hardwired ignition signal is connected at the copro
   // check first if HW signal should be used
   _bHwIgnition = FALSE;

   #ifdef SPM_FEATURE_ENABLE_LONG_PRESS_RESET_TIMER
      if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)vLongPressResetTimerCallback, ( tPVoid ) this,
                                &_hLongPressResetTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::spm_tclSubStateHandler: !!!!!! Timer vLongPressResetTimerCallback could not be created: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif

   _idt                   = OSAL_ThreadWhoAmI( );

   _oActiveSubStates.clear( );
   _oOnceTriggeredSubStates.clear( );

   _u32SubStateForceReset = 0;
   _u32FastShutdownReason = 0;

   // create msgbox
   if ( OSAL_ERROR == OSAL_s32MessageQueueCreate( SPM_MSGBOX_NAME,
                                                  SPM_MSGBOX_MAX_MESSAGE_COUNT,
                                                  sizeof( TTriggerMsg ), // sizeof(TSubStateMsg),
                                                  OSAL_EN_READWRITE, &_hSubStateQueue )
        ){
      _hSubStateQueue = OSAL_C_INVALID_HANDLE;
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::spm_tclSubStateHandler: !!!!!! OSAL_s32MessageQueueCreate failed: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

}

spm_tclSubStateHandler::~spm_tclSubStateHandler( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  ******
  */

   SPM_NULL_POINTER_CHECK( _poclOsalProxy );

   if ( OSAL_s32TimerDelete( _hEjectLongPressTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
      if ( OSAL_s32TimerSetTime( _hDeboounceTimer, 0, 0 ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
      if ( OSAL_s32TimerDelete( _hDeboounceTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif // ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE

   #ifdef SPM_FEATURE_ENABLE_IGNITION_TIMEOUT
      if ( OSAL_s32TimerSetTime( _hIgnitionTimeOutTimer, 0, 0 ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
      if ( OSAL_s32TimerDelete( _hIgnitionTimeOutTimer ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   #endif // ifdef SPM_FEATURE_ENABLE_IGNITION_TIMEOUT

   _poclGlobalApplicationManager = NULL;
   _poclCcaServiceHandler        = NULL;
   _poclCcaMsgHandler            = NULL;
   _poclSystemPowerManager       = NULL;
   _poclSupervisionManager       = NULL;
   _poclWorkerServer             = NULL;
   _poclOsalProxy                = NULL;
   _poclSyncHandler              = NULL;
}

tVoid spm_tclSubStateHandler::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclSubStateHandler.
  ******
  */
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSupervisionManager, spm_tclSysStateSupervisor, ISpmSupervisionClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclGlobalApplicationManager, ISpmGlobalApplicationManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,        ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager,       ISpmSystemPowerManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,             ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,                ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaMsgHandler,            ISpmCcaServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSyncHandler,              ISpmSyncHandler );

   vRemoveAllTrigger( &_tTrigger );
   vRemoveAllTrigger( &_tTriggerWhileUnderVolt );
   vRemoveAllTrigger( &_tTriggerWhileDebouncing );
   vRemoveAllTrigger( &_tTriggerWakeup );
} // vGetReferences

tVoid spm_tclSubStateHandler::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.
  ******
  */
   spm_tclSubStateHandler::bSetWakeupConfig( SPM_U32_WAKEUPCONFIG_DEFAULT );

   SPM_GET_IF_REFERENCE_NEW_VAR(poclWakeupHandler, ISpmWakeupHandler);



   tU32               u32WakeupReason   = poclWakeupHandler->u32GetWakeupReason( );
   ETG_TRACE_USR3( ( "spm_tclSubStateHandler::vStartCommunication(): WakeupReason '%u'.",
                     ETG_ENUM( SPM_WAKEUP_REASON, u32WakeupReason ) ) );

   if ( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EJECT == u32WakeupReason ){
      //wakeup was eject --> simulate eject update
      vTriggerSubStateType( SPM_U32_EJECT, TRUE );
      vTriggerSubStateType( SPM_U32_EJECT, FALSE );
   }


   if ( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON != u32WakeupReason ){
      // restore stored trigger if no PowerOn reset is detected
      dp_tclSpmDpInternDataTriggerState oStoredTrigger;
      TTriggerMsg                       tTrigger               = oStoredTrigger.tGetData( );

      vTraceTrigger( SSH_vRestoredTrigger, &tTrigger );

      tU32                              au32TriggerToRestore[] = { SPM_U32_RESTORE_TRIGGER_TYPES };
      vMaskTrigger( &tTrigger, au32TriggerToRestore, ELEMENTE( au32TriggerToRestore ) );

      vMergeTrigger( &_tTrigger, &tTrigger );

      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSupervisionClientIgnOff, spm_tclIgnitionOffSupervisor, ISpmSupervisionClient );

         if ( bIsTriggerSet( SPM_U32_IGNITION ) ){
            // stop hour logic
            vClearTrigger( SPM_U32_IGNITION_LOGIC, TRUE );
            poclSupervisionClientIgnOff->vStopSupervision( );
         } else {
            poclSupervisionClientIgnOff->vStartSupervision( );
         }
      }
   }

   vTraceTrigger( SSH_vCurrentTrigger, &_tTrigger );

   #ifdef SPM_FEATURE_ENABLE_IGN_AUTOMATIC_ON
      if ( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON == u32WakeupReason ){
         vSetTrigger( SPM_U32_AUTOMATIC );
      }
   #endif

   #ifdef SPM_FEATURE_ENABLE_IGN_AUTOMATIC_OFF
      if ( spm_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON == u32WakeupReason ){
         vClearTrigger( SPM_U32_AUTOMATIC );
      }
   #endif

   #ifdef SPM_ENABLE_DM_VERITY_CHECK
      SPM_NULL_POINTER_CHECK(_poclOsalProxy);
       if ( _poclOsalProxy->u8GetOperationalState( ) == DEV_WUP_C_U8_OPERATIONAL_STATE_DMVERITY_CHECK ){
           vSetTrigger( SPM_U32_DMVERITY_CHECK );
       }
   #endif

   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );

   if ( OSAL_s32MessageQueueNotify( _hSubStateQueue, (OSAL_tpfCallback)vOnQueueCallback, this ) == OSAL_OK ){
   } else {
   }

} // vStartCommunication

tVoid spm_tclSubStateHandler::vOnQueueCallback( tVoid *pArg ){
   spm_tclSubStateHandler *p = (spm_tclSubStateHandler*)pArg;
   tU32                    u32MsgMax, u32MaxLen, u32MsgCount = 0;

   SPM_NULL_POINTER_CHECK( p );
   SPM_NULL_POINTER_CHECK( p->_poclSyncHandler );

   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vOnQueueCallback(): Context switch." ) );

   if ( OSAL_OK != OSAL_s32MessageQueueStatus( p->_hSubStateQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vOnQueueCallback: !!!!!! OSAL_s32MessageQueueStatus failed !!!!!!" ) );
   }

   if ( u32MsgCount > 0 ){
      ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vOnQueueCallback(): Context switch: %d.", u32MsgCount ) );
      p->_poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_SUBSTATE_TRIGGER );
   } else {
      ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vOnQueueCallback(): No message in queue." ) );
   }
} // vOnQueueCallback

tVoid spm_tclSubStateHandler::vEjectLongPressDetect( tVoid *pArg ){
   spm_tclSubStateHandler *poSubStateHandler = (spm_tclSubStateHandler*)pArg;

      SPM_NULL_POINTER_CHECK( poSubStateHandler );
      SPM_NULL_POINTER_CHECK( poSubStateHandler->_poclWorkerServer );

   poSubStateHandler->_poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_SPM_SHUTDOWN, SPM_U32_SHUTDOWN_AFTER_LONG_PRESS );
}

#ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
   tVoid spm_tclSubStateHandler::vDebounceStateChangeTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *  \version
  *    1.0   - Initial
  ******
  */
      spm_tclSubStateHandler *poSubStateHandler = (spm_tclSubStateHandler*)pArg;

      SPM_NULL_POINTER_CHECK( poSubStateHandler );

      poSubStateHandler->vSetSubStateType( SPM_U32_STATE_DEBOUNCE, FALSE );
   }

#endif // ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE

tVoid spm_tclSubStateHandler::vProcessSystemState( tU32 u32NewSystemState ){
/*!
  * \fn
  *  \brief
  *   This method processes new system state detected in SystemStateManager.
  *
  *  \param[in] u32NewSystemState: new system state detected.
  ******
  */
   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vProcessSystemState(): u32NewSystemState '%u'.",
                     ETG_ENUM( SPM_SYSTEM_STATES, u32NewSystemState ) ) );

   if ( u32NewSystemState != SPM_SYSTEM_INVALID ){
      vRemoveTrigger( &_tTrigger, SPM_U32_STATE_TO );
      // _u32SystemState = u32NewSystemState;

      SPM_NULL_POINTER_CHECK( _poclWorkerServer );
      _poclWorkerServer->bPostMessageToWorker( SPM_U32_WORKER_BROADCAST_SYSSTATE_CHANGED, u32NewSystemState );
   }

   tBool bLowVoltBlockingActive = FALSE;
   #ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
      bLowVoltBlockingActive = bIsTriggerSet( SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER );
   #endif

   if ( ( ( !bLowVoltBlockingActive )
        || ( u32NewSystemState == SPM_SYSTEM_OFF )
        || ( u32NewSystemState == SPM_SYSTEM_SUSPEND )
        || ( u32NewSystemState == SPM_SYSTEM_PREPARE_SHUTDOWN )
         ) && (_bIsPrepareShutdownTmlTest == FALSE)) {

      SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );
      _poclGlobalApplicationManager->vTriggerAppStateChange( &_tTrigger );
      vRemoveAllFlags4AllreadySetTrigger( &_tTrigger );
   }

   tU32 au32TriggerToStore[] = { SPM_U32_STORE_TRIGGER_TYPES };
   if ( bIsOneUpdateFlag( &_tTrigger, au32TriggerToStore, ELEMENTE( au32TriggerToStore ) ) ){
      // remember Trigger
      TTriggerMsg                       tCurTrigger = TTriggerMsg( );
      vCopyTrigger( &tCurTrigger, &_tTrigger );

      vMaskTrigger( &tCurTrigger, au32TriggerToStore, ELEMENTE( au32TriggerToStore ) );

      dp_tclSpmDpInternDataTriggerState oTrigger;
      // clear all Update flags before storing
      for ( tU8 i = 0; i < SPM_TRIGGER_ARRAY_SIZE; i++ ){
         tCurTrigger.u32TriggerUpdate[i] = 0;
      }
      oTrigger.vSetData( tCurTrigger );
   }

} // vProcessSystemState

tVoid spm_tclSubStateHandler::vStart( const std::string& strName,
                                      tU32               u32Prio,
                                      tU32               u32Stack ){
/*!
  * \fn
  *  \brief
  *   This method checks the wakeup reason and starts thread in spm_tclSubStateHandler to receive new substate messages.
  *
  *  \param[in] strName: name of thread.
  *  \param[in] u32Prio: priority of thread.
  *  \param[in] u32Stack: stack size of thread.
  ******
  */
   spm_tclActive::vStartThread( strName, u32Prio, u32Stack );
} // vStart

tVoid spm_tclSubStateHandler::vOnStart( ){
/*!
  * \fn
  *  \brief Method invoked before the start(main) of the thread of SubStateHandler.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
}

tVoid spm_tclSubStateHandler::vOnTerminate( ){
/*!
  * \fn
  *  \brief Method invoked just before we stop the thread of SubStateHandler.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
}

tVoid spm_tclSubStateHandler::vHandleMsgQueue( ){
/*!
  * \fn
  *  \brief main thread function to receive new substate messages
  ******
  */
   tU32        u32Prio, u32MsgMax, u32MaxLen, u32MsgCount = 0;
   TTriggerMsg tTriggerMsg         = TTriggerMsg( );
   tS32        s32Ret;
   tU32        u32ConsumedMsgCount = 0;
   tBool       bHeartbeatOnly      = TRUE;

   SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
   SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );

   do {

      s32Ret = OSAL_s32MessageQueueWait( _hSubStateQueue, (tU8*)&tTriggerMsg, sizeof( TTriggerMsg ), &u32Prio, OSAL_C_TIMEOUT_NOBLOCKING );

      if ( s32Ret > 0 ){
         u32ConsumedMsgCount++;
         // check for heartbeat --> substate should be set by supervision monitor
         if ( bIsUpdateFlag( &tTriggerMsg, SPM_U32_HEARTBEAT ) ){
            // increment heartbeat counter
            _poclSupervisionManager->vTriggerSupervisionState( );
            vRemoveTrigger( &tTriggerMsg, SPM_U32_HEARTBEAT );
            ETG_TRACE_USR3( ( "spm_tclSubStateHandler::main(): Heartbeat is triggered." ) );
         }

         // TRACE_SPM_INFO_STRING("spm_tclSubStateHandlerXBase-dwThreadProc");
         if ( !bIsTerminated( ) && ( s32Ret != OSAL_ERROR ) ){
            vRemoveAllFlags4AllreadySetTrigger( &tTriggerMsg );
            vMergeTrigger( &_tTrigger, &tTriggerMsg );

            if ( bIsOneTriggerSet( &_tTrigger ) ){
               // trace trigger info
               vTraceTrigger( SSH_vNewTrigger,     &tTriggerMsg );
               vTraceTrigger( SSH_vCurrentTrigger, &_tTrigger );

               bHeartbeatOnly = FALSE;

               {
                  if ( bIsUpdateFlag( &_tTrigger, SPM_U32_EJECT ) || bIsUpdateFlag( &_tTrigger, SPM_U32_ON_TIPPER ) ){

                     tBool bEject  = bIsDataFlag( &_tTrigger, SPM_U32_EJECT );
                     tBool bTipper = bIsDataFlag( &_tTrigger, SPM_U32_ON_TIPPER );
                     ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): Tipper/Eject (%d,%d)", bTipper, bEject ) );

                     if ( bEject && bTipper ){
                        dp_tclSpmDpConfigLongPressEject oEject;
                        // start long press detction
                        if ( OSAL_s32TimerSetTime( _hEjectLongPressTimer, oEject.tGetData( ) * 1000, 0 ) != OSAL_OK ){
                           ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vHandleMsgQueue: !!!!!! Timer _hEjectLongPressTimer could not get started !!!!!!" ) );
                        }
                     } else {
                        // stop long press detction
                        if ( OSAL_s32TimerSetTime( _hEjectLongPressTimer, 0, 0 ) != OSAL_OK ){
                           ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vHandleMsgQueue: !!!!!! Timer _hEjectLongPressTimer could not get started !!!!!!" ) );
                        }
                     }
                  }
               }
               {
                  // check for fast shutdown reason
                  tU32 au32FastShutdownTrigger[] = { SPM_U32_FASTSHUTDOWN_TYPES };
                  if ( bIsOneDataFlag( &_tTrigger, au32FastShutdownTrigger, ELEMENTE( au32FastShutdownTrigger ) ) ){
                     if ( !bIsDataFlag( &_tTrigger, SPM_U32_FAST_SHUTDOWN ) ){
                        ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): One fastshutdown reason is active." ) );
                        vSetSubStateType( SPM_U32_FAST_SHUTDOWN, TRUE );
                     }
                  } else {
                     if ( bIsDataFlag( &_tTrigger, SPM_U32_FAST_SHUTDOWN ) ){
                        ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): At least no data flag is set -> stop fastshutdown." ) );
                        vSetSubStateType( SPM_U32_FAST_SHUTDOWN, FALSE );
                     }
                  }
               }

               #ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
                  if ( bIsUpdateFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER )
                       && !bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) ){

                     // low voltage end detected
                     vMergeTrigger( &_tTrigger, &_tTriggerWhileUnderVolt, TRUE, FALSE );
                  }
               #endif

               #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
                  if ( bIsUpdateFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE )
                       && !bIsDataFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE ) ){
                     // debouncing finished -> trigger now all missed updates
                     vMergeTrigger( &_tTrigger, &_tTriggerWhileDebouncing, TRUE, FALSE );

                  }
               #endif

               // trigger new substate only if no undervoltage is active
               // substate emergency off is needed for shutdown for enduring undervoltage

               #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
                  if ( !bIsDataFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE ) ){
               #endif

               #ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
                  tU32 au32Trigger[] = {
                     SPM_U32_SHUTDOWN_TYPES_EN_LOW_VOLT, SPM_U32_SYNC_SHUTDOWN_ACK
                  }

                  ;

                  if ( !bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER )
                       || ( bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) && bIsOneUpdateFlag( &_tTrigger, au32Trigger, ELEMENTE( au32Trigger ) ) )
                       ){
                     SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
                     poclSystemStateManager->vNewSubState( &_tTrigger );
                  } else if ( bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) ){
                     // remember some events in undervoltage to trigger a transition at the end of the undervoltage condition

                     // add phone trigger
                     // ######### \todo #######

                     // remove all update trigger
                     vRemoveAllTrigger( &_tTrigger, FALSE );
                  }
               #else // ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
                  SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
                  poclSystemStateManager->vNewSubState( &_tTrigger );
               #endif // ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER

               #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
            } else {
               vMergeTrigger( &_tTriggerWhileDebouncing, &_tTrigger, TRUE );

               // remove all update trigger
               vRemoveAllTrigger( &_tTrigger, FALSE );
            }
               #endif

               // and now inform clients about change
               vSendSubStatePropertyChange( );
            }
         }
         vRemoveAllTrigger( &_tTrigger, FALSE );

         if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSubStateQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
            ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::vHandleMsgQueue !!!!!! OSAL_s32MessageQueueStatus failed !!!!!!" ) );
         }
         if ( ( u32MsgCount == 0 ) && !bHeartbeatOnly ){
            // no more message in queue -> all current substate updates consumed
            _bSubStateChange = FALSE;

            // trigger LAM
            _poclGlobalApplicationManager->vAllSubStatesProcessed( );

         }
      }

   } while ( u32MsgCount != 0 );

   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vHandleMsgQueue(): consumed messages: %d.", u32ConsumedMsgCount ) );

} // main

tVoid spm_tclSubStateHandler::main( ){
/*!
  * \fn
  *  \brief main thread function to receive new substate messages
  ******
  */
   tU32 u32Prio            = 0;
   tU32 u32MsgMax          = 0;
   tU32 u32MaxLen          = 0;
   tU32 u32MsgCount        = 0;
   TTriggerMsg tTriggerMsg = TTriggerMsg( );
   tS32 s32Ret;

      SPM_NULL_POINTER_CHECK( _poclSupervisionManager );
      SPM_NULL_POINTER_CHECK( _poclGlobalApplicationManager );

   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSubStateQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   if ( u32MsgCount == 0 ){
      // no more message in queue -> all current substate updates consumed
      _bSubStateChange = FALSE;

      // trigger LAM
      _poclGlobalApplicationManager->vAllSubStatesProcessed( );

   }

   s32Ret = OSAL_s32MessageQueueWait( _hSubStateQueue, (tU8*)&tTriggerMsg, sizeof( TTriggerMsg ), &u32Prio, OSAL_C_U32_INFINITE );

   // check for heartbeat --> substate should be set by supervision monitor
   if ( bIsUpdateFlag( &tTriggerMsg, SPM_U32_HEARTBEAT ) ){
      // increment heartbeat counter
      _poclSupervisionManager->vTriggerSupervisionState( );
      vRemoveTrigger( &tTriggerMsg, SPM_U32_HEARTBEAT );
      ETG_TRACE_USR3( ( "spm_tclSubStateHandler::main(): Heartbeat is triggered." ) );
   }

   // TRACE_SPM_INFO_STRING("spm_tclSubStateHandlerXBase-dwThreadProc");
   if ( !bIsTerminated( ) && ( s32Ret != OSAL_ERROR ) ){
      vRemoveAllFlags4AllreadySetTrigger( &tTriggerMsg );
      vMergeTrigger( &_tTrigger, &tTriggerMsg );

      if ( bIsOneTriggerSet( &_tTrigger ) ){
         // trace trigger info
         vTraceTrigger( SSH_vNewTrigger,     &tTriggerMsg );
         vTraceTrigger( SSH_vCurrentTrigger, &_tTrigger );

         {
            if ( bIsUpdateFlag( &_tTrigger, SPM_U32_EJECT ) || bIsUpdateFlag( &_tTrigger, SPM_U32_ON_TIPPER ) ){

               tBool bEject  = bIsDataFlag( &_tTrigger, SPM_U32_EJECT );
               tBool bTipper = bIsDataFlag( &_tTrigger, SPM_U32_ON_TIPPER );
               ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): Tipper/Eject (%d,%d)", bTipper, bEject ) );

               if ( bEject && bTipper ){
                  dp_tclSpmDpConfigLongPressEject oEject;
                  // start long press detction
                  if ( OSAL_s32TimerSetTime( _hEjectLongPressTimer, oEject.tGetData( ) * 1000, 0 ) != OSAL_OK ){
                     ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
                  }
               } else {
                  // stop long press detction
                  if ( OSAL_s32TimerSetTime( _hEjectLongPressTimer, 0, 0 ) != OSAL_OK ){
                     ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
                  }
               }
            }
         }
         {
            // check for fast shutdown reason
            tU32 au32FastShutdownTrigger[] = { SPM_U32_FASTSHUTDOWN_TYPES };
            if ( bIsOneDataFlag( &_tTrigger, au32FastShutdownTrigger, ELEMENTE( au32FastShutdownTrigger ) ) ){
               if ( !bIsDataFlag( &_tTrigger, SPM_U32_FAST_SHUTDOWN ) ){
                  ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): One fastshutdown reason is active." ) );
                  vTriggerSubStateType( SPM_U32_FAST_SHUTDOWN, TRUE );
               }
            } else {
               if ( bIsDataFlag( &_tTrigger, SPM_U32_FAST_SHUTDOWN ) ){
                  ETG_TRACE_USR4( ( "spm_tclSubStateHandler::vSetSubStateType(): At least no data flag is set -> stop fastshutdown." ) );
                  vTriggerSubStateType( SPM_U32_FAST_SHUTDOWN, FALSE );
               }
            }
         }

         #ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
            if ( bIsUpdateFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER )
                 && !bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) ){

               // low voltage end detected
               vMergeTrigger( &_tTrigger, &_tTriggerWhileUnderVolt, TRUE, FALSE );
            }
         #endif

         #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
            if ( bIsUpdateFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE )
                 && !bIsDataFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE ) ){
               // debouncing finished -> trigger now all missed updates
               vMergeTrigger( &_tTrigger, &_tTriggerWhileDebouncing, TRUE, FALSE );

            }
         #endif

         // trigger new substate only if no undervoltage is active
         // substate emergency off is needed for shutdown for enduring undervoltage

         #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
            if ( !bIsDataFlag( &_tTrigger, SPM_U32_STATE_DEBOUNCE ) ){
         #endif

         #ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
            tU32 au32Trigger[] = {
               SPM_U32_SHUTDOWN_TYPES_EN_LOW_VOLT, SPM_U32_SYNC_SHUTDOWN_ACK
            }

            ;

            if ( !bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER )
                 || ( bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) && bIsOneDataFlag( &_tTrigger, au32Trigger, ELEMENTE( au32Trigger ) ) )
                 ){
               SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
               poclSystemStateManager->vNewSubState( &_tTrigger );
            } else if ( bIsDataFlag( &_tTrigger, SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER ) ){
               // remember some events in undervoltage to trigger a transition at the end of the undervoltage condition

               // add phone trigger
               // ######### \todo #######

               // remove all update trigger
               vRemoveAllTrigger( &_tTrigger, FALSE );
            }
         #else // ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER
            SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemStateManager, ISpmSystemStateManager );
            poclSystemStateManager->vNewSubState( &_tTrigger );
         #endif // ifdef SPM_U32_LOW_VOLTAGE_BLOCKING_TRIGGER

         #ifdef SPM_FEATURE_ENABLE_DEBOUNCE_STATE_CHANGE
      } else {
         vMergeTrigger( &_tTriggerWhileDebouncing, &_tTrigger, TRUE );

         // remove all update trigger
         vRemoveAllTrigger( &_tTrigger, FALSE );
      }
         #endif

         // and now inform clients about change
         vSendSubStatePropertyChange( );
      }
   }
   vRemoveAllTrigger( &_tTrigger, FALSE );
} // main

tBool spm_tclSubStateHandler::bPostTriggerMessage( TTriggerMsg *pTriggerMsg ){
/*!
  * \fn
  *  \brief Method to post trigger message to Message Queue read by main thread of SubStateHandler.
  *
  *  \param[in] pTriggerMsg: Trigger Message.
  ******
  */
   tBool bSuccess = FALSE;

   // check if at least one trigger is set
   bSuccess = bIsOneTriggerSet( pTriggerMsg );

   if ( bSuccess ){
      tS32 s32Ret = OSAL_s32MessageQueuePost( _hSubStateQueue, (tCU8*)pTriggerMsg, sizeof( TTriggerMsg ), OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
      if ( s32Ret == OSAL_OK ){
         vOnQueueCallback( this );
         bSuccess = TRUE;
      } else {
         bSuccess = FALSE;
         /* impossible to put back the message */
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::bPostTriggerMessage() has failed: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
         TRACE_SPM_FAILURE_THREAD_INFO;
      }
      tU32 u32MsgMax, u32MaxLen, u32MsgCount = 0;
      if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSubStateQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      }
      tU32 u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
      if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
         std::string strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue _hSubStateQueue (" + std::to_string(u32FillLevel) + "%) exceeded maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN) + "%";
         ETG_TRACE_ERRMEM( ( "%s", strError.c_str( ) ) );
      }
   }
   return( bSuccess );
} // bPostTriggerMessage

tVoid spm_tclSubStateHandler::vTerminate( ){
/*!
  * \fn
  *  \brief Method to stop internal main thread of SubStateHandler.
  ******
  */
   vSetTerminate( );

   // trigger thread to terminate
      vTriggerSubStateType( spm_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_HEARTBEAT, FALSE );

}

tVoid spm_tclSubStateHandler::vHandleMessage( tU32 u32Message,
                                              tU32 u32Parm ){
/*!
  * \fn
  *  \brief
  *    Handles the Message from WorkerServer.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] u32Parm:
  ******
  */
   (tVoid)u32Parm;

   if ( SPM_U32_WORKER_SSH_HEARTBEAT == u32Message ){
      // if (FALSE == bPostSubStateMessage(0,0, SPM_U32_INTERNAL_SUBSTATE_HEARTBEAT, 0)) {
      // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
      // }
      vTriggerSubStateType( SPM_U32_HEARTBEAT, FALSE );
   }
} // vHandleMessage

tBool spm_tclSubStateHandler::bHandleSynchrounousCall( tU32   u32Message,
                                                       tVoid *args ){
/*!
  * \fn
  *  \brief
  *   spm_tclWorkerServer invokes this method for notifying messages.
  *
  *  \param[in] u32Message: Message Identifier.
  *  \param[in] args: not used.
  ******
  */
   (tVoid)args;

   switch ( u32Message ){
      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
      {
         // if (!bPostSubStateMessage(0, 0, SPM_U32_INTERNAL_SUBSTATE_STARTUP, 0))
         {
            // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
         }
      }
      break;

      default:
         // nothing to do
         break;
   }
   return( FALSE );
} // bHandleSynchrounousCall

tVoid spm_tclSubStateHandler::vSendSubStatePropertyChange( ){
/*!
  * \fn
  *  \brief This method updates the CCA clients about change in substate properties through _poclCcaServiceHandler.
  ******
  */

   // update substate property
   if ( _poclCcaServiceHandler ){
      spm_corefi_tclMsgSysPwrModeStatus tPwrKeyState;
      tPwrKeyState.SysPwrMode.enType = spm_fi_tcl_SPM_e32_SYSPWRMODE::FI_EN_SPM_U32_SYSPWRMODE_OFF;

      spm_corefi_tclMsgSubStatesStatus tSubState;
      tSubState.HwSubState           = 0;
      tSubState.InternSubState       = 0;
      tSubState.SubState             = 0;

      if ( bIsTriggerSet( SPM_U32_ACCESSORY ) ){
         tSubState.SubState            |= SPM_U32_SUBSTATE_TYPE_S_CONTACT;
         tPwrKeyState.SysPwrMode.enType = spm_fi_tcl_SPM_e32_SYSPWRMODE::FI_EN_SPM_U32_SYSPWRMODE_ACCESSORY;
      }
      if ( bIsTriggerSet( SPM_U32_IGNITION ) ){
         tSubState.SubState            |= SPM_U32_SUBSTATE_TYPE_IGNITION;
         tPwrKeyState.SysPwrMode.enType = spm_fi_tcl_SPM_e32_SYSPWRMODE::FI_EN_SPM_U32_SYSPWRMODE_RUN;
      }
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_SUBSTATES, &tSubState );
      _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_SYSPWRMODE, &tPwrKeyState );
   }

   if ( bIsTriggerUpdateSet( SPM_U32_IGNITION ) ){
      SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSupervisionClientIgnOff, spm_tclIgnitionOffSupervisor, ISpmSupervisionClient );

      if ( bIsTriggerSet( SPM_U32_IGNITION ) ){
         // stop hour logic
         vClearTrigger( SPM_U32_IGNITION_LOGIC, TRUE );
         poclSupervisionClientIgnOff->vStopSupervision( );
      } else {
         poclSupervisionClientIgnOff->vStartSupervision( );
      }
   }
} // vSendSubStatePropertyChange

tBool spm_tclSubStateHandler::bSetWakeupConfig( tU32 u32WakeupConfig ){
/*!
  * \fn
  *  \brief method to set certain wakeup config and also updates the wakeup trigger array maintained by substate handler
  *  \param [in] u32WakeupConfig: Bitfield of Wakeups to be enabled - see e.g.
  *              configuration for SPM_U32_WAKEUPCONFIG_DEFAULT in spm_Config_project.h
  *  \return tBool: True if successfuly set.
  *
  *  \details This method is used to configure certain wake-up reasons to be considered for the initial wake-up of the system.
  ******
  */
   tBool bReturn = FALSE;

   ETG_TRACE_USR4( ( "spm_tclSubStateHandler::bSetWakeupConfig() wakeup config value %d", ETG_ENUM( SPM_WAKEUP_CONFIG, u32WakeupConfig ) ) );
   SPM_NULL_POINTER_CHECK_VAL( _poclOsalProxy );

   //set the wakeup config value
   bReturn = _poclOsalProxy->bSetWakeupConfig( u32WakeupConfig );

   if ( bReturn ){
      memset( (tChar*)&_tTriggerWakeup, 0, sizeof( TTriggerMsg ) );
      //map and update the trigger wakeup array
      vMapAndUpdateWakeupTrigger( u32WakeupConfig );
      vTraceTrigger( SSH_vWakeupTrigger, &_tTriggerWakeup );
   } else {
      ETG_TRACE_ERRMEM( ( "spm_tclSubStateHandler::bSetWakeupConfig() ERROR : wakeup config %d not set", ETG_ENUM( SPM_WAKEUP_CONFIG, u32WakeupConfig ) ) );
   }
   return( bReturn );
} // bSetWakeupConfig

tVoid spm_tclSubStateHandler::vAddSubStateInSet( tU32 u32SubState ){
   if ( _oActiveSubStates.find( u32SubState ) == _oActiveSubStates.end( ) ){
      _oActiveSubStates.insert( u32SubState );
      vUpdateSubStatesOnInterface( );
   }
}

tVoid spm_tclSubStateHandler::vRemoveSubStateInSet( tU32 u32SubState ){
   if ( _oActiveSubStates.find( u32SubState ) != _oActiveSubStates.end( ) ){
      _oActiveSubStates.erase( _oActiveSubStates.find( u32SubState ) );
      vUpdateSubStatesOnInterface( );
   }
}

tVoid spm_tclSubStateHandler::vUpdateSubStatesOnInterface( ){

   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

   std::set < tU32 >::iterator iter;
   spm_corefi_tclMsgActiveOnOffTriggerStatus oActiveTrigger;

   oActiveTrigger.tOnOffTriggerList.clear( );

   for ( iter = _oActiveSubStates.begin( ); iter != _oActiveSubStates.end( ); iter++ ){
      //add substate to property
      spm_fi_tcl_SPM_e32_SubStateType oSubState;
      oSubState.enType = (spm_fi_tcl_SPM_e32_SubStateType::tenType)* iter;
      oActiveTrigger.tOnOffTriggerList.push_back( oSubState );
   }
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_ACTIVEONOFFTRIGGER, &oActiveTrigger );

} // vUpdateSubStatesOnInterface

tVoid spm_tclSubStateHandler::vUpdateHistoryTriggerOnInterface( ){

   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

   std::map < tU32,tU32 >::iterator iter;
   spm_corefi_tclMsgHistoryTriggerStatus oTrigger;

   oTrigger.tTriggeredStates.clear( );

   for ( iter = _oOnceTriggeredSubStates.begin( ); iter != _oOnceTriggeredSubStates.end( ); iter++ ){
      //add substate to property
      spm_fi_tcl_SPM_HistoryTrigger oSubState;
      oSubState.eTriggerType.enType = (spm_fi_tcl_SPM_e32_SubStateType::tenType) iter->first;
      oSubState.u32Timestamp        = iter->second;

      oTrigger.tTriggeredStates.push_back( oSubState );
      ETG_TRACE_USR1( ( "spm_tclSubStateHandler::vUpdateHistoryTriggerOnInterface(): SubState '%d (%dms)' in history", ETG_ENUM( spmTriggerTypes, (tU32) oSubState.eTriggerType.enType), oSubState.u32Timestamp ) );
   }
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_HISTORYTRIGGER, &oTrigger );

} // vUpdateSubStatesOnInterface

// EOF

