/*!
  * \file spm_StartupSystemVariant.cpp
  *  \brief
  *        Determine current configuration by evaluating all variants and consistently switch off Processes, Components and
  *        Services from the original Process configuration.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 24.03.11  | TMS Fischer       |
  ******
  */

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_IStartupCommon.h"
#include "spm_IRegistry.h"
#include "spm_IFactory.h"

#include "spm_StartupSystemVariant.h"
#include "spm_StringTokenizer.h"
#include "spm_Registry_pathes.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_StartupSystemVariant.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------


spm_tclStartupSystemVariant::spm_tclStartupSystemVariant( const ISpmFactory& factory ) : ISpmStartupSystemVariant( factory ){
   /*!
     * \fn
     *  \brief
     *    Constructor
     *
     *  \param[in] factory: spm factory object.
     ******
     */
}

spm_tclStartupSystemVariant::~spm_tclStartupSystemVariant( ){
/*!
  * \fn
  *  \brief
  *    destructor has to delete all entries dynamically allocated in
  *    oEvaluateString or oEvaluateNumber. This is done in the destructors
  *    of _var._switches automatically. You can not see them here
  *
  *  \param
  ******
  */
}

tVoid spm_tclStartupSystemVariant::vAddVariable( const std::string& strName,
                                                 tU32               u32Value ){
   ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::vAddVariable(%d,%s)", u32Value, strName.c_str( ) ) );
   // TRACE_SPM_INFO_STRING("vAddVariable");
   // TRACE_SPM_INFO_STRING(strName);
   // TRACE_SPM_INFO(u32Value);
   _variables[strName] = TSpmVariantVariable( u32Value );
   ETG_TRACE_USR4( ( "size=%d", _variables.size( ) ) );
}

tVoid spm_tclStartupSystemVariant::vAddVariable( const std::string& strName,
                                                 const std::string& strValue ){
   // TRACE_SPM_INFO_STRING("vAddVariable");
   // TRACE_SPM_INFO_STRING(strName);
   // TRACE_SPM_INFO_STRING(strValue.c_str());
   _variables[strName] = TSpmVariantVariable( strValue );
}

TSpmVariantValue*spm_tclStartupSystemVariant::oEvaluateString( const std::string& str ){
   /*!
     * \fn
     *  \brief
     *    Parses input string and Creates new TSpmVariantValue .
     *
     *  \param[in] str: input string.
     *  \return  TSpmVariantValue pointer.
     *
     ******
     */
   spm_tclStringTokenizer st( str, "\x5C[,", TRUE ); // backslash

   TSpmVariantValue      *pVal   = new TSpmVariantValue( );

   SPM_NULL_POINTER_CHECK_VAL( pVal );

   bool                   bFirst = true;
   std::string            strTmp;

   while ( st.bHasMoreTokens( ) ){
      std::string strToken = st.oNextToken( );

      if ( strToken == "\x5C" ){  // backslash
         if ( bFirst ){
            bFirst = false;
         } else {
            strTmp += strToken;
         }
      } else if ( strToken == "[" ){
         strTmp += strToken;
      } else if ( strToken == "," ){
         // nothing to do, separator
      } else {
         strTmp += strToken;
         ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::oEvaluateString(): GenerateSingleValue '%s'", strTmp.c_str( ) ) );
         pVal->_items.push_back( new TSpmVariantSingleValue( strTmp ) );

         // clear the string for the next iteration
         strTmp.clear( );
      }
   }
   return( pVal );
} // oEvaluateString

TSpmVariantValue*spm_tclStartupSystemVariant::oEvaluateNumber( const std::string& strNum ){
   /*!
     * \fn
     *  \brief
         Parses input string and Creates new TSpmVariantValue .
       state = 0 start, expect start symbol
      state = 1 expect single number
      state = 2 expect comma
      state = 3 expect comma or minus
      state = 4 expect single number for range
     *
     *  \param[in] strNum: input string to be parsed.
     *  \return
     *
     ******
     */
   spm_tclStringTokenizer st( strNum, "[,-] ", TRUE );

   bool                   bNumberAvail   = false;
   tU32                   u32State       = 0;
   tU32                   u32FirstValue  = 0;
   tU32                   u32SecondValue = 0; // for Ranges
   TSpmVariantValue      *pVal           = new TSpmVariantValue( );

   SPM_NULL_POINTER_CHECK_VAL( pVal );

   while ( st.bHasMoreTokens( ) ){
      std::string strToken = st.oNextToken( );

      if ( strToken == "[" ){
         // startsymbol found
         if ( u32State != 0 ){
            break;
         }
         u32State = 1; // valid startsymbol found, expect number
      } else if ( strToken == " " ){
         // ignore whitespace
         continue;
      } else if ( strToken == "]" ){
         if ( bNumberAvail ){
                        ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::oEvaluateNumber(): GenerateSingleValue '%d'", u32FirstValue ) );
            pVal->_items.push_back( new TSpmVariantSingleValue( u32FirstValue ) );
         }
         // end of expression found, ignore the rest
         break;
      } else if ( strToken == "," ){
         if ( u32State != 2 ){
            break;
         }
                        ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::oEvaluateNumber(): GenerateSingleValue '%d'", u32FirstValue ) );

         pVal->_items.push_back( new TSpmVariantSingleValue( u32FirstValue ) );
         u32State = 1;
      } else if ( strToken == "-" ){
         u32State = 4; // expect single number for range
      } else {
         // number found
         if ( u32State == 1 ){
            // evaluate a single number
            u32FirstValue = OSAL_s32AsciiToS32( strToken.c_str( ) );
            bNumberAvail  = true; // store this number with next terminal symbol
         } else if ( u32State == 4 ){
            // evaluate upper limit for range
            u32SecondValue = OSAL_s32AsciiToS32( strToken.c_str( ) );
                        ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::oEvaluateNumber(): Generate value range '%d - %d'", u32FirstValue, u32SecondValue ) );
            pVal->_items.push_back( new TSpmVariantRange( u32FirstValue, u32SecondValue ) );
            bNumberAvail   = false;
         } else {
            // not valid, stop
            break;
         }
         u32State = 2;
      }

   }
   return( pVal );
} // oEvaluateNumber

TSpmVariantValue*spm_tclStartupSystemVariant::oEvaluateValue( const std::string& strVal ){
   if ( strVal[0] != '[' ){  // escape character found?
      return( oEvaluateString( strVal ) );
   }
   return( oEvaluateNumber( strVal ) );
}

tVoid spm_tclStartupSystemVariant::vEvaluateProcessVariant( ){
   OSAL_tIODescriptor fd;

   std::string        strBaseName;
   tS32               as32Data[256];  // 1kB(32Bit*256) as temporary buffer

   // TRACE_SPM_INFO_STRING("vEvaluateProcessVariant");

   strBaseName = (std::string)SPM_REG_PROCESS_VARIANT_BASE_PATH;

   // first read "PATH", "KEY" and "TYPE" out of registry
   fd          = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      // HKEY_LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/VARIANT available
      // read in the following entries:
      // VARxx means a definition of a variable
      //
      //
      OSAL_trIOCtrlDir      rDir = { 0,0,0 };
      OSAL_trIOCtrlRegistry rReg;


      rDir.fd        = fd;
      rDir.s32Cookie = 0;

      // get all defined variables
      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDir ) != OSAL_ERROR ){
         // now read in all variable definitions
         // they all have to be like this: VARxx xx is numerical
         rReg.pcos8Name = rDir.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );
         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){
            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               tUInt unVarNr;

               if ( OSAL_s32ScanFormat( (tChar*)rDir.dirent.s8Name, SPM_STARTUP_FORMAT_VARIABLE, &unVarNr ) == 1 ){
                  // set an entry for a variable.
                  // TRACE_SPM_INFO(u32VarNr);
                  // TRACE_SPM_INFO_STRING(rDir.dirent.s8Name);
                        ETG_TRACE_USR4( ( "u32VarNr=%u, s8Name=%s", unVarNr, (tChar*)rDir.dirent.s8Name ) );
                  std::map < std::string, TSpmVariantVariable >::iterator it;

                  // find the variable in the map
                  if ( ( it = _variables.find( (tChar*)as32Data ) ) != _variables.end( ) ){
                        ETG_TRACE_USR4( ( "u32VarNr=%u, s8Name=%s", unVarNr, as32Data ) );
                     _var._variables[unVarNr] = it->second;
                  }
               }
            }
         }
      }

      // the variables are not needed any longer, all variables are now in _var._variables with a number
      // as key
      //_variables.clear();

      rDir.fd        = fd;
      rDir.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGREADDIR, ( intptr_t )&rDir ) != OSAL_ERROR ){
         std::string           strVarDirName;
         OSAL_tIODescriptor    fdvar;
         OSAL_trIOCtrlDir      rDirVar = { 0,0,0 };
         OSAL_trIOCtrlRegistry rRegVar;

         // new variant directory found
                        ETG_TRACE_USR4( ( "OSAL_C_S32_IOCTRL_REGREADDIR %s", strVarDirName.c_str( ) ) );

         strVarDirName = strBaseName + (tString)rDir.dirent.s8Name;

         // TRACE_SPM_INFO_STRING(strVarDirName);
                        ETG_TRACE_USR4( ( "strVarDirName %s", strVarDirName.c_str( ) ) );
         fdvar         = OSAL_IOOpen( strVarDirName.c_str( ), OSAL_EN_READONLY );

         if ( fdvar != OSAL_ERROR ){
            rDirVar.fd        = fdvar;
            rDirVar.s32Cookie = 0;

            while ( OSAL_s32IOControl( fdvar, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirVar ) != OSAL_ERROR ){
               rRegVar.pcos8Name = rDirVar.dirent.s8Name;
               rRegVar.ps8Value  = (tU8*)as32Data;
               rRegVar.u32Size   = sizeof( as32Data );

               if ( OSAL_s32IOControl( fdvar, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rRegVar ) != OSAL_ERROR ){
                  if ( rRegVar.s32Type == OSAL_C_S32_VALUE_STRING ){
                     tUInt unNumber;

                     if ( OSAL_s32ScanFormat( (tChar*)rDirVar.dirent.s8Name, SPM_STARTUP_FORMAT_VALUE, &unNumber ) == 1 ){
                        // VALxx means a definition of a value to be compared with corresponding variable
                        // TRACE_SPM_INFO(u32Number);
                        // TRACE_SPM_INFO_STRING(rDirVar.dirent.s8Name);
                        // TRACE_SPM_INFO_STRING((tString)as32Data);
                        ETG_TRACE_USR4( ( "a)unNumber=%d s8Name=%s ", unNumber, rDirVar.dirent.s8Name ) );
                        ETG_TRACE_USR4( ( "as32Data=%s ", as32Data ) );
                        _var._switch[(tString)rDir.dirent.s8Name]._values[unNumber] = oEvaluateValue( (tChar*)as32Data );
                     } else if ( OSAL_s32ScanFormat( (tChar*)rDirVar.dirent.s8Name, SPM_STARTUP_FORMAT_EXPRESSION, &unNumber ) == 1 ){
                        // EXPRxx means an expression with variable expressions, EXPRESSIONS are ordered
                        // TRACE_SPM_INFO(u32Number);
                        // TRACE_SPM_INFO_STRING(rDirVar.dirent.s8Name);
                        // TRACE_SPM_INFO_STRING((tString)as32Data);
                        ETG_TRACE_USR4( ( "b)unNumber=%d s8Name=%s ", unNumber, rDirVar.dirent.s8Name ) );
                        ETG_TRACE_USR4( ( "as32Data=%s ", as32Data ) );
                        _var._switch[(tString)rDir.dirent.s8Name]._expressions[unNumber] = (tChar*)as32Data;
                     } else if ( OSAL_s32StringCompare( (tString)rDirVar.dirent.s8Name, SPM_STARTUP_KEY_CONF_EXCLUDE ) == 0 ){
                        // EXCLUDE found. That means that a list of processes, components or services is available
                        // TRACE_SPM_INFO(166);
                        // TRACE_SPM_INFO_STRING(rDirVar.dirent.s8Name);
                        // TRACE_SPM_INFO_STRING((tString)as32Data);
                        ETG_TRACE_USR4( ( "166 s8Name=%s ", rDirVar.dirent.s8Name ) );
                        ETG_TRACE_USR4( ( "as32Data=%s ", as32Data ) );
                        _var._switch[(tString)rDir.dirent.s8Name]._exclude = (tChar*)as32Data;
                     }
                  }
               }
            }
            OSAL_s32IOClose( fdvar );
         }
      }
            OSAL_s32IOClose( fd );
   }
} // vEvaluateProcessVariant

tVoid spm_tclStartupSystemVariant::vEvaluateProcessVariantV2( ){
   OSAL_tIODescriptor fd;

   std::string        strBaseName;
   tS32               as32Data[256];  // 1kB(32Bit*256) as temporary buffer

   // TRACE_SPM_INFO_STRING("vEvaluateProcessVariant");

   strBaseName = (std::string)SPM_REG_PROCESS_VARIANT_BASE_PATH;

   // first read "PATH", "KEY" and "TYPE" out of registry
   fd          = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      // HKEY_LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/VARIANT available
      // read in the following entries:
      OSAL_trIOCtrlDir rDir = { 0,0,0 };
      rDir.fd        = fd;
      rDir.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGREADDIR, ( intptr_t )&rDir ) != OSAL_ERROR ){
         // and iterate through all possible VARIANT subfolder
         // --> HKEY_LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/VARIANT/1
         // --> HKEY_LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/VARIANT/2
         // --> ...

         std::string           strVarDirName;
         OSAL_tIODescriptor    fdvar;
         OSAL_trIOCtrlDir      rDirVar = { 0,0,0 };
         OSAL_trIOCtrlRegistry rRegVar;

         // new variant directory found
         strVarDirName = strBaseName + (tString)rDir.dirent.s8Name;

         ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): strVarDirName %s", strVarDirName.c_str( ) ) );
         fdvar         = OSAL_IOOpen( strVarDirName.c_str( ), OSAL_EN_READONLY );

         if ( fdvar != OSAL_ERROR ){
            rDirVar.fd        = fdvar;
            rDirVar.s32Cookie = 0;

            tBool             bVariantValid             = FALSE;
            TSpmVariantValue *pGetVariantExcludeList    = OSAL_NULL;
            TSpmVariantValue *pGetVariantIncludeList    = OSAL_NULL;
            TSpmVariantValue *pGetVariantAppExcludeList = OSAL_NULL;
            TSpmVariantValue *pGetVariantDeactivateList = OSAL_NULL;

            while ( OSAL_s32IOControl( fdvar, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirVar ) != OSAL_ERROR ){
               // and now evaluate each VARIANT

               rRegVar.pcos8Name = rDirVar.dirent.s8Name;
               rRegVar.ps8Value  = (tU8*)as32Data;
               rRegVar.u32Size   = sizeof( as32Data );

               if ( OSAL_s32IOControl( fdvar, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rRegVar ) != OSAL_ERROR ){
                  if ( rRegVar.s32Type == OSAL_C_S32_VALUE_STRING ){

                     ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): strVarDirName '%40s':'%s'", (tChar*)rDirVar.dirent.s8Name, (tChar*)as32Data ) );
                     if ( OSAL_s32StringCompare( (tString)rDirVar.dirent.s8Name, "PROC_EXCLUDE" ) == 0 ){
                        pGetVariantExcludeList = oEvaluateValue( (tChar*)as32Data );
                     } else if ( OSAL_s32StringCompare( (tString)rDirVar.dirent.s8Name, "PROC_INCLUDE" ) == 0 ){
                        pGetVariantIncludeList = oEvaluateValue( (tChar*)as32Data );
                     } else if ( OSAL_s32StringCompare( (tString)rDirVar.dirent.s8Name, "CCAAPP_EXCLUDE" ) == 0 ){
                        pGetVariantAppExcludeList = oEvaluateValue( (tChar*)as32Data );
                     } else if ( OSAL_s32StringCompare( (tString)rDirVar.dirent.s8Name, "CCASRV_DEACTIVATE" ) == 0 ){
                        pGetVariantDeactivateList = oEvaluateValue( (tChar*)as32Data );
                     } else {
                        std::map < std::string, TSpmVariantVariable >::const_iterator item = _variables.find( (tChar*)rDirVar.dirent.s8Name );
                        if ( item != _variables.end( ) ){
                           ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): Find an entry for this VARIANT with value: '%d'", item->second._numVal ) );
                           TSpmVariantValue                                   *pGetVariantValue = oEvaluateValue( (tChar*)as32Data );
                           std::vector < ISpmVariantCompare* >::const_iterator it;
                           for ( it = pGetVariantValue->_items.begin( ); !bVariantValid && it != pGetVariantValue->_items.end( ); ++it ){
                              if ( ( (TSpmVariantSingleValue*)* it )->_type == SPM_STARTUP_VARIANT_VALUE_NUMERICAL ){
                                 if ( item->second._numVal == ( (TSpmVariantSingleValue*)* it )->_numVal ){
                                    ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): Variant is valid '%d'", ( (TSpmVariantSingleValue*)* it )->_numVal ) );
                                    bVariantValid = TRUE;
                                 }
                              }
                           }
                           delete pGetVariantValue;
                        }
                     }
                  }
               }

            }
            OSAL_s32IOClose( fdvar );


            if ( bVariantValid ){
               if ( pGetVariantExcludeList != OSAL_NULL ){
                  //exclude list
                  std::vector < ISpmVariantCompare* >::const_iterator it;
                  for ( it = pGetVariantExcludeList->_items.begin( ); it != pGetVariantExcludeList->_items.end( ); ++it ){
                     if ( ( (TSpmVariantSingleValue*)* it )->_type == SPM_STARTUP_VARIANT_VALUE_STRING ){
                        ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): Process to exclude is '%s'", ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) ) );
                        _setDisabledProcesses.insert( ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) );
                     }
                  }
                  delete pGetVariantExcludeList;
                  pGetVariantExcludeList = OSAL_NULL;
               }

               if ( pGetVariantDeactivateList != OSAL_NULL ){
                  //deactivated services list
                  std::vector < ISpmVariantCompare* >::const_iterator it;
                  for ( it = pGetVariantDeactivateList->_items.begin( ); it != pGetVariantDeactivateList->_items.end( ); ++it ){
                     if ( ( (TSpmVariantSingleValue*)* it )->_type == SPM_STARTUP_VARIANT_VALUE_STRING ){
                        ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): CCA service to deactivate is '%s'", ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) ) );
                        _setDisabledServices.insert( ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) );
                     }
                  }
                  delete pGetVariantDeactivateList;
                  pGetVariantDeactivateList = OSAL_NULL;
               }
               if ( pGetVariantAppExcludeList != OSAL_NULL ){
                  //deactivated services list
                  std::vector < ISpmVariantCompare* >::const_iterator it;
                  for ( it = pGetVariantAppExcludeList->_items.begin( ); it != pGetVariantAppExcludeList->_items.end( ); ++it ){
                     if ( ( (TSpmVariantSingleValue*)* it )->_type == SPM_STARTUP_VARIANT_VALUE_STRING ){
                        ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): CCA APP to exclude is '%s'", ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) ) );
                        _setDisabledComponents.insert( ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) );
                     }
                  }
                  delete pGetVariantAppExcludeList;
                  pGetVariantAppExcludeList = OSAL_NULL;
               }
            } else { // variant is not set in this scope --> check for include list
               if ( pGetVariantIncludeList != OSAL_NULL ){
                  //exclude list
                  std::vector < ISpmVariantCompare* >::const_iterator it;
                  for ( it = pGetVariantIncludeList->_items.begin( ); it != pGetVariantIncludeList->_items.end( ); ++it ){
                     if ( ( (TSpmVariantSingleValue*)* it )->_type == SPM_STARTUP_VARIANT_VALUE_STRING ){
                        ETG_TRACE_USR1( ( "spm_tclStartupSystemVariant::vEvaluateProcessVariantV2(): Process to exclude is '%s'", ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) ) );
                        _setDisabledProcesses.insert( ( (TSpmVariantSingleValue*)* it )->_strVal.c_str( ) );
                     }
                  }
                  delete pGetVariantIncludeList;
                  pGetVariantIncludeList = OSAL_NULL;
               }
            }
            if (pGetVariantExcludeList != OSAL_NULL) {
               delete pGetVariantExcludeList;
               pGetVariantExcludeList = OSAL_NULL;
            }
            if (pGetVariantDeactivateList != OSAL_NULL) {
               delete pGetVariantDeactivateList;
               pGetVariantDeactivateList = OSAL_NULL;
            }
            if (pGetVariantAppExcludeList != OSAL_NULL) {
               delete pGetVariantAppExcludeList;
               pGetVariantAppExcludeList = OSAL_NULL;
            }
            if (pGetVariantIncludeList != OSAL_NULL) {
               delete pGetVariantIncludeList;
               pGetVariantIncludeList = OSAL_NULL;
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vEvaluateProcessVariant

tBool spm_tclStartupSystemVariant::bEvaluateVariantExpression( const std::map < tU32, TSpmVariantValue* >& values,
                                                               const std::string& strExpression ){
   // the expression has the form
   // VARxx
   // !VARxx
   // VARxx & VARxy
   // the evaluation goes from left to right, parentheses are not allowed
   // if you need something like VARxx && (VARxy || VARxz)
   // then write two expressions
   // VARxx & VARxy
   // VARxx & VARxz
   // these are allways ored, the basics come from the "Disjunctive normal form"
   // of "Boolean algebra".
   //
   // state = 0: VARxx, ! are allowed
   // state = 1: & is allowed

                     ETG_TRACE_USR4( ( "bEvaluateVariantExpression" ) );
   spm_tclStringTokenizer st( strExpression, "!& ", TRUE );
   tU32                   u32State = 0;
   tBool                  bNot     = FALSE;

   while ( st.bHasMoreTokens( ) ){
      std::string strToken = st.oNextToken( );
                     ETG_TRACE_USR4( ( "bNot=%d u32State=%d s.c_str()=%s", bNot, u32State, strToken.c_str( ) ) );
      // TRACE_SPM_INFO_STRING(strToken.c_str());
      // TRACE_SPM_INFO(bNot);
      // TRACE_SPM_INFO(u32State);

      if ( strToken == "!" ){
         if ( u32State == 0 ){
            // several ! are allowed in sequence
            bNot = !bNot;
         } else {
            // ! not allowed here
            return( FALSE );
         }
      } else if ( strToken == "&" ){
         if ( u32State == 1 ){
            u32State = 0;
         }
      } else if ( strToken == " " ){
         // eliminate whitespace
         continue;
      } else {
         if ( u32State == 0 ){
            // evaluate a variable
            tUInt unVarNr;

            // TRACE_SPM_INFO_STRING("Evaluate Variable");
                     ETG_TRACE_USR4( ( "Evaluate Variable" ) );
            if ( OSAL_s32ScanFormat( (const tChar*)strToken.c_str( ), SPM_STARTUP_FORMAT_VARIABLE, &unVarNr ) == 1 ){
               std::map < tU32, TSpmVariantVariable >::const_iterator itvar;
               std::map < tU32, TSpmVariantValue* >::const_iterator   itval;

               itvar = _var._variables.find( unVarNr );
               itval = values.find( unVarNr );

               // valid variable found
               if ( ( itvar != _var._variables.end( ) )
                    && ( itval != values.end( ) ) ){
                     ETG_TRACE_USR4( ( "Variable and Value found" ) );
                  // variable and corresponding value exist
                  if ( ( itval->second )->operator ==( itvar->second ) ){
                     ETG_TRACE_USR4( ( "OP returns true (bNot=%d)", bNot ) );
                     // TRACE_SPM_INFO(bNot);
                     if ( bNot ){
                        return( FALSE );
                     }
                  } else {
                     // TRACE_SPM_INFO_STRING("OP returns false");
                     // TRACE_SPM_INFO(bNot);
                     ETG_TRACE_USR4( ( "OP returns false (bNot=%d)", bNot ) );
                     if ( !bNot ){
                        return( FALSE );
                     }
                  }
               } else {
                  // variable or according value are missing
                  return( FALSE );
               }
               u32State = 1; // allow only & as next symbol
            } else {
               // wrong variable format in expression found
               return( FALSE );
            }
         } else {
            // expression not valid
            return( FALSE );
         }
      }
   }

   return( TRUE );
} // bEvaluateVariantExpression

tVoid spm_tclStartupSystemVariant::vDeactivate( const std::string& strExcl ){
   spm_tclStringTokenizer st( strExcl, "/", FALSE );

   std::string            strProcess;
   std::string            strComponent;
   std::string            strService;
   tU32                   u32State = 0;

                     ETG_TRACE_USR4( ( "spm_tclStartupSystemVariant::vDeactivate" ) );
   // TRACE_SPM_INFO_STRING("vDeactivate");

   while ( st.bHasMoreTokens( ) ){
      switch ( u32State ){
         case 0:
            strProcess   = st.oNextToken( );
            u32State     = 1;
            break;

         case 1:
            strComponent = st.oNextToken( );
            u32State     = 2;
            break;

         case 2:
            strService   = st.oNextToken( );
            u32State     = 3;
            break;

         default:
            st.oNextToken( );
            break;
      } // switch
   }

   // do not evaluate services in the moment. These are not needed in the current project
   // if needed then extend evaluation of token and add an array with services to be switched off.

   switch ( u32State ){
      case 1:
         _setDisabledProcesses.insert( strProcess );
                     ETG_TRACE_USR4( ( "_setDisabledProcesses.insert(%s)", strProcess.c_str( ) ) );
         // TRACE_SPM_INFO_STRING("disable process");
         // TRACE_SPM_INFO_STRING(process.c_str());
         break;

      case 2:
         _setDisabledComponents.insert( strProcess + "/" + strComponent );
                     ETG_TRACE_USR4( ( "_setDisabledComponents.insert(%s)", strProcess.c_str( ) ) );
         // TRACE_SPM_INFO_STRING("disable component");
         // TRACE_SPM_INFO_STRING(strProcess.c_str());
         // TRACE_SPM_INFO_STRING(strComponent.c_str());
         break;

      default:
         _setDisabledServices.insert( strProcess + "/" + strComponent + "/" SPM_STR_VARIANT_SERVICES_PATH + strService );
         break;
         // used later for deleting entries in the registry
   } // switch
}    // vDeactivate

tVoid spm_tclStartupSystemVariant::vDeactivateList( const std::string& strExcl ){
   // fill the local variables for processes and components
   spm_tclStringTokenizer st( strExcl, ", ", FALSE );

                     ETG_TRACE_USR4( ( "vDeactivateList=%s", strExcl.c_str( ) ) );
   while ( st.bHasMoreTokens( ) ){
                     vDeactivate( st.oNextToken( ) );
   }
}

tVoid spm_tclStartupSystemVariant::vCalculateValidConfiguration( ){
   // determine which processes, components and services are not in the configuration
   // processes are switched off by changing internal data structures, components and
   // services are switched off by deleting the corresponding registry entries.
   // Whereas process switch off is a single task to do because spm starts further
   // processes, components and services are switched off any time before a new process
   // gets started because just before that a new registry has been evaluated.

   // first calculate the cases in the switches which are valid.

   std::map < std::string, TSpmVariantSwitch >::iterator it;

            ETG_TRACE_USR4( ( "vCalculateValidConfiguration size=%d", _var._switch.size( ) ) );
   // TRACE_SPM_INFO_STRING("vCalculateValidConfiguration");
   // TRACE_SPM_INFO(_var._switch.size());

   // iterate over all switches
   for ( it = _var._switch.begin( ); it != _var._switch.end( ); ++it ){
      std::map < tU32, std::string >::iterator it2;

      // iterate over all expressions found in the switch, these are all ored together
      for ( it2 = it->second._expressions.begin( ); it2 != it->second._expressions.end( ); ++it2 ){
         if ( bEvaluateVariantExpression( it->second._values, it2->second ) ){
            ETG_TRACE_USR4( ( "bEvaluateVariantExpression() found TRUE expression, calling vDeactivateList" ) );
            // found an expression that evaluates to true. Get the exclude list
            // and manipulate internal data and build registry paths.
            //
            vDeactivateList( it->second._exclude );
            break; // goto next switch and check if it yields "TRUE" to deactivate other processes
         }
      }
   }
} // vCalculateValidConfiguration

tVoid spm_tclStartupSystemVariant::vUpdateRegistry( ){
   // here the already calculated paths to the registry are applied to purify
   // the registry from components and services which are not part of the
   // current configuration.
   SPM_GET_IF_REFERENCE_NEW_VAR( poclRegistry, ISpmRegistry );

   // TRACE_SPM_INFO_STRING("vUpdateRegistry");
   // TRACE_SPM_INFO(_setDisabledComponents.size());
   std::set < std::string >::const_iterator it;

   // a component can be disabled by deleting the key in its component registry path
   // example:
   // [HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\PROCESS\BASE\DEMO_SERVER]
   // "APPID"=dword:00000819
   // only this APPID entry is deleted to disable the component
   for ( it = _setDisabledComponents.begin( ); it != _setDisabledComponents.end( ); ++it ){
      poclRegistry->vRemoveEntry( ( * it ), SPM_STR_VARIANT_DELETE_COMPONENT );
   }
   for ( it = _setDisabledServices.begin( ); it != _setDisabledServices.end( ); ++it ){
      // add here deleting of services
      // example:
      // [HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\PROCESS\BASE\DEMO_SERVER\SERVICES\DEMO_SERVER]
      // "SERVICEID"=dword:0530
      // only this SERVICEID entry is deleted to disable the service of a component
      poclRegistry->vRemoveEntry( ( * it ), SPM_STR_VARIANT_DELETE_SERVICE );
   }
} // vUpdateRegistry

tVoid spm_tclStartupSystemVariant::vTrace( ){
   std::map < std::string, TSpmVariantVariable >::iterator it;
   std::map < tU32, TSpmVariantVariable >::iterator        itVar;

         ETG_TRACE_USR4( ( "Values of variant variables: %d", _variables.size( ) ) );
   for ( it = _variables.begin( ); it != _variables.end( ); ++it ){
      if ( it->second._type == SPM_STARTUP_VARIANT_VALUE_STRING ){
         ETG_TRACE_USR4( ( "Name: %20s, String: %s",
                           it->first.c_str( ),
                           it->second._strVal.c_str( ) ) );
      } else {
         ETG_TRACE_USR4( ( "Name: %20s, Number: %d",
                           it->first.c_str( ),
                           it->second._numVal ) );
      }
   }

         ETG_TRACE_USR4( ( "Values of var._variant variables: %d", _var._variables.size( ) ) );
   for ( itVar = _var._variables.begin( ); itVar != _var._variables.end( ); ++itVar ){
      if ( itVar->second._type == SPM_STARTUP_VARIANT_VALUE_STRING ){
         ETG_TRACE_USR4( ( "No: %d, String: %s",
                           itVar->first,
                           itVar->second._strVal.c_str( ) ) );
      } else {
         ETG_TRACE_USR4( ( "No: %d, Number: %d",
                           itVar->first,
                           itVar->second._numVal ) );
      }
   }

} // vTrace

