/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_StartupSystem.h
  * @brief        This is header file contains functions for StartupSystem (start-up configuration).
  *               also functions for regarding registry read or update and process stop or start.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <algorithm>
#include <map>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_IStartupSystem.h"

class ISpmStartupInvestigationServer;
class ISpmWorkerServer;
class ISpmOsLinux;
class ISpmStartupCommon;
class ISpmStartupSupervisor;
class ISpmSyncHandler;
class ISpmRegistry;

struct TSpmStartupSystemItem
{
   TSpmStartupSystemItem( ){ _s32GroupNr = - 1; }  // no group as default

   std::string _process;
   tS32 _s32GroupNr;
};


typedef std::map < tU32, TSpmStartupSystemItem > TMapProcStartOrder;
#define SPM_STARTUP_SYSTEM_MAX_STR_LEN 16

/*!
  *  \class spm_tclStartupSystem
  *  \brief This is the base class for configuration of system startup and process spawn.
  *
  *         The detailed startup of one process has to be implemented in the derived class.
  *
  *         spm_tclFactory creates an instance of this class. spm_tclStartupLocal extends this class
  *         to provide specific implementation of process spawn.
  *
  *         This class provides startup investigation information to spm_tclStartupInvestigation.
  *         This class posts messages to WorkerServer when a process is stopped and when startup is completed.
  *          It provides software block information to spm_tclStartupCommon while adding a software block.
  */
class spm_tclStartupSystem : public ISpmStartupSystem
{
public:
spm_tclStartupSystem( const ISpmFactory& factory,
                      const std::string& strConfigPath,
                      const std::string& strInstance,
                      tBool              bReadFirstRegistry );
virtual ~spm_tclStartupSystem( );

virtual tVoid vSystemStart( );

// Implementation of ISpmSwBlockClient
virtual void vOnSwBlockConnected( const TSpmSwBlockSet& sw );

virtual void vOnSwBlockUp( const TSpmSwBlockSet& sw );

virtual void vOnSwBlockLoaded( const TSpmSwBlockSet& sw );

virtual void vOnSyncPointReached( const std::string& strSyncPointName );

virtual void vOnCheckStartup( );


virtual void vOnSwBlockForced( const TSpmSwBlockSet& sw );

// optionally use a group evaluator object
tVoid vSetGroupEvaluator( ISpmStartupSystemGroupServer *grp ){
   _poclGrp = grp;
}

// is this a startup process for local components?
virtual tBool bIsLocal( ) const { return( TRUE ); }
virtual tVoid vSetSwBlockServer( ISpmApplicationDatabase */*pDatabase*/,
                                 ISpmCcaSupplierServer   */*pSupplier*/,
                                 ISpmSwBlockServer       */*pServer*/ ){ } //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual tU32  u32CalculateStartupRegistry(tU32 u32StartupConfig);

protected:
static tVoid vStartProcessTimerCallback( tVoid *pArg );

OSAL_tTimerHandle _hStartProcessTimer;

virtual tBool bStartProcess( const std::string& strProcessname );

virtual tBool bStopProcess( const std::string& strProcessname );

virtual tBool bKillProcess( TProcConfiguration& oProcConf );

virtual tBool bForceProcessStopSyncronous( const std::string& strProcessName,
                                           std::string      & strForcedSwBlockNameToWaitFor );

virtual tBool bWaitForSwBlockRemoval( tU32               u32SyncMode,
                                      const std::string& strSwBlockName );

virtual tVoid vRemoveSwBlockAndKillProcessSyncronous( );

// Pure abstract here
virtual tBool bProcessSpawn( TProcConfiguration& _oConf )     = 0;

virtual tVoid vRemoveProcess( TProcConfiguration& oProcConf ) = 0;

virtual tVoid vEvaluateProcessStartOrder( );

virtual tBool bAlternativeProcConfig( tChar *strProcessName, tBool bValidationOnly = FALSE );
virtual tBool bAlternativeProcConfig( const TProcConfiguration& oProcConf, tChar *strProcessName, tBool bValidationOnly = FALSE );
virtual tBool bCheckStartCondition( TProcConfiguration& oProcConf, tChar *strProcessName);

virtual tVoid vLoadSelectorRegistries( tChar *strProcessName);

virtual tVoid onPreProcessStart( OSAL_trProcessAttribute *processAttributes ) const;

virtual tVoid onPostProcessStart( OSAL_trProcessAttribute *processAttributes ) const;

virtual tVoid vSetStartUpFinishedProcessConfiguration( tU32 instance );

virtual tBool bPrepareStart( );

virtual tVoid vWaitForSync( tU32 u32Sync );

virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

virtual tVoid vProjectSpecificSetup( tCString strConfigPath,
                                     tCString strInstance );

virtual tBool isPartOf( const std::string& strProcessName );

virtual tBool bForceProcessStop( const std::string& strProcessName );

virtual std::string strGetListOfBlocks( const TSpmSwBlockSet *poBlock );

virtual tBool bGetNextProcess2Start( TProcConfiguration& roProc2StartConfig,
                                     tU32              & ru32SyncMode,
                                     TSpmSwBlockSet    & roWaitingBlocks );

virtual tU32  u32GetSyncBlock( TProcConfiguration& _oConf,
                               tU32                u32HighPrioSync,
                               TSpmSwBlockSet    & roWaitingBlocks );

virtual tVoid vTraceSwBlock( const std::string& strName,
                             TSpmSwBlockSet     oSwBlocks );

virtual tBool bStartMyProcess( TProcConfiguration& _oConf,
                               tU32                u32SyncMode );

virtual tVoid vCheckForProcessToStart( );

virtual TOperationResult eCheckNextOperation(std::string strCommad);
virtual tBool bGetCondition(std::string strCommad, tBool bStartOnError);
virtual tBool bGetStartCondition(std::string strCommad, tBool bStartOnError = TRUE);

TSpmSwBlockSet     _oNextProcWaitingBlocks;
TSpmSwBlockSet     _oNextProcWaitingSyncBlocks;

TProcConfiguration _oNextProc2StartConfig;
tU32               _u32NextProcSyncMode;

TMapProcStartOrder _oProcStartOrder;
TMapProcStartOrder _oRemainingProc2Start;

OSAL_tSemHandle    _hSemHandle;
OSAL_tEventHandle  _hSyncEvent;
const std::string  _regConfPath;
const std::string  _strInstance;
std::string        _syncName;
std::string        _semName;
TSpmSwBlockSet     _oLoadedBlocks;
TSpmSwBlockSet     _oUpBlocks;
TSpmSwBlockSet     _oConnectedBlocks;
TSpmSwBlockSet     _oWaitingBlocks;
TSpmSwBlockSet     _oPredecessorSwBlocks;
TSpmSwBlockSet     _oAllSwBlocks;
TSpmSwBlockSet     _oSyncPoints;

tU32               _startWaitTime;
tU32               _syncWaitTimeout;
tBool              _bKeepWaitingAfterTimeout;
tBool              _bIgnoreSyncBlocks;


/* SW-Block lists, needed to force hole sw-block into a state (and get the acknowledge
   when the sw-block has reached the forced state). Needed for e.g. NAVCOPY downloads usecase
   which is executed in SPM NORMAL state (and not in DOWNLOAD state)*/
TSpmSwBlockSet                     _oForcedSwBlocks;
TSpmSwBlockSet                     _oWaitingToBeForcedSwBlocks;
TMapProcessConfiguration::iterator posProcToKill;

tU32                               _u32CurrentSyncState;

tU32                               _u32DelayBeforeStartNextProc;

ISpmStartupCommon                 *_poclStartupCommon;
ISpmStartupSystemGroupServer      *_poclGrp;
ISpmStartupInvestigationServer    *_poclStartupInvest;
ISpmWorkerServer                  *_poclWorkerServer;
ISpmOsLinux                       *_poSpmOsLinux;
ISpmStartupSupervisor             *_poclStartupSupervisor;
ISpmSyncHandler                   *_poclSyncHandler;
ISpmRegistry                      *_poclRegistry;

std::string                        _context;
tU32                               _u32Instance;
tBool                              _bStartUpFinished;

virtual tU32 u32GetSupervisionMode( tU32 ProcId ){
   (tVoid)ProcId;
   return( 0 );
}

virtual std::string strGetProcName( tU32 u32ProcId ){
   (tVoid)u32ProcId;
   return( "" );
}

#ifdef LCM_UNIT_TESTS

public:
   tVoid vSetWaitingBlocks( TSpmSwBlockSet waitingBlocks ){ _oWaitingBlocks = waitingBlocks; }
   tVoid vSetWaitingToBeForcedBlocks( TSpmSwBlockSet waitingBlocks ){ _oWaitingToBeForcedSwBlocks = waitingBlocks; }
   tVoid vSetSyncMode( tU32 syncState ){ _u32CurrentSyncState = syncState; }
#endif
};

