/*!
  * \file spm_StartupSystem.cpp
  *  \brief
  *    The base class for a startup of several processes. In the
  *               derived class the detailed startup of one process has to
  *               be implemented.
  *
  *  \b PROJECT: NextGen \n
  *  \b SW-COMPONENT: FC SPM \n
  *  \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \version
  * 1.0 |  24.01.11  | TMS Fischer       | initial version \n
  * 1.1 |  27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#include <sstream>
#include <iomanip>

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_StartupSystem.h"

// spm class definitions w/o interface
#include "spm_StartupCommon.h"
#include "spm_SoftwareBlock.h"

// interfaces class definitions
#include "spm_ISystemPowerManager.h"
#include "spm_IOsLinux.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_IApplicationDatabase.h"
#include "spm_IStartupSupervisor.h"
#include "spm_IProcessSupervision.h"
#include "spm_ISyncHandler.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IRegistry.h"
#include "spm_ILocalAppManager.h"
#include "spm_ILateServiceHandler.h"

#include "spm_factory.h"
#include "spm_rootdaemon.h"

// spm helper
#include "spm_CriticalSection.h"
#include "spm_OsLinux.h"
#include "spm_Registry_pathes.h"

#include <dirent.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <errno.h>
// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
    #include "trcGenProj/Header/spm_StartupSystem.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// //#define SPM_TRACE_FILE_ID   SPM_FILE_STARTUPSYSTEM

// trace startup processes
// #define __SPM_TRACE_PREDECESSOR

// /////////////////////////////////////////////////////////////////////////////
// keys used in registry
//
#define SPM_STARTUP_FOLD_ORDER                           "/ORDER"
#define SPM_STARTUP_FORMAT_PROCESS_STARTUP_ORDER         "PROC%02u"
#define SPM_STARTUP_FORMAT_PROCESS_STARTUP_ORDER_EXTENDED "PROC%03u"
#define SPM_STARTUP_PROCEXTENDED_STR_LENGTH              7  // "PROCxxx"


// ------------------------------------------------------------------------------
// constructor spm_tclStartupSystem
spm_tclStartupSystem::spm_tclStartupSystem( const ISpmFactory& factory,
                                            const std::string& strConfigPath,
                                            const std::string& strInstance,
                                            tBool                 bReadFirstRegistry )
   : ISpmStartupSystem( factory )
   , _hStartProcessTimer(OSAL_C_INVALID_HANDLE)
   , _u32NextProcSyncMode( SPM_U32_SYNC_NONE )
   , _hSemHandle(OSAL_C_INVALID_HANDLE)
   , _hSyncEvent(OSAL_C_INVALID_HANDLE)
   , _regConfPath( strConfigPath )
   , _strInstance( strInstance )
   , _startWaitTime( 0 )
   , _syncWaitTimeout( SPM_U32_SYNC_WAITING_TIMEOUT )
   , _bKeepWaitingAfterTimeout( TRUE )
   , _bIgnoreSyncBlocks( FALSE )
   , _u32CurrentSyncState( SPM_U32_SYNC_NONE )
   , _u32DelayBeforeStartNextProc( 0 )
   , _poclStartupCommon( NULL )
   , _poclGrp( NULL )
   , _poclStartupInvest( NULL )
   , _poclWorkerServer( NULL )
   , _poSpmOsLinux( NULL )
   , _poclStartupSupervisor( NULL )
   , _poclSyncHandler( NULL )
   , _poclRegistry( NULL )
   , _context( strConfigPath )
   , _bStartUpFinished( FALSE ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in] factory: spm factory object.
  *  \param[in] strConfigPath: config path
  *  \param[in] strInstance: instance number
  *  \param[in] bReadFirstRegistry: Flag deciding registry read.
  ******
  */
   (void)bReadFirstRegistry;
   _u32Instance = atoi( strInstance.c_str( ) );
   _syncName    = "SpmSEv" + strInstance;
   _semName     = "SpmSSm" + strInstance;

   if ( OSAL_ERROR == OSAL_s32EventCreate( _syncName.c_str( ), &_hSyncEvent ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupSystem !!!!!! Error detected !!!!!! cannot create Event: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( _semName.c_str( ), &_hSemHandle, (tU32)1 ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupSystem !!!!!! Error detected !!!!!! cannot create Semaphore: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
   _context += strInstance;

   if ( OSAL_s32TimerCreate( (OSAL_tpfCallback)vStartProcessTimerCallback, ( tPVoid ) this, &_hStartProcessTimer ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupSystem !!!!!! Error detected , cannot create timer for vStartProcessTimerCallback!!!!!!: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

}

spm_tclStartupSystem::~spm_tclStartupSystem( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param
  ******
  */
   if ( OSAL_s32TimerSetTime( _hStartProcessTimer, 0, 0 ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   if ( OSAL_s32TimerDelete( _hStartProcessTimer ) != OSAL_OK ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }

   OSAL_s32EventClose( _hSyncEvent );
   OSAL_s32EventDelete( _syncName.c_str( ) );

   OSAL_s32SemaphoreClose( _hSemHandle );
   OSAL_s32SemaphoreDelete( _semName.c_str( ) );

   _poclStartupCommon     = NULL;
   _poclGrp               = NULL;
   _poclStartupInvest     = NULL;
   _poclWorkerServer      = NULL;
   _poSpmOsLinux          = NULL;
   _poclStartupSupervisor = NULL;
   _poclSyncHandler       = NULL;
   _poclRegistry          = NULL;
}

tVoid spm_tclStartupSystem::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclStartupSystem.
  *
  *  \param none
  ******
  */
// get all needed references now -> all SPM objects are now available
   vProjectSpecificSetup( _regConfPath.c_str( ), _strInstance.c_str( ) );

   SPM_GET_IF_REFERENCE_USE_VAR( _poclStartupCommon,     ISpmStartupCommon );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,      ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclStartupInvest,     ISpmStartupInvestigationServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poSpmOsLinux,          ISpmOsLinux );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclStartupSupervisor, ISpmStartupSupervisor );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclRegistry,          ISpmRegistry );


      SPM_GET_IF_REFERENCE_USE_VAR( _poclSyncHandler, ISpmSyncHandler );

} // vGetReferences

tVoid spm_tclStartupSystem::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.

  *  \param none
  ******
  */
}

tVoid spm_tclStartupSystem::vSystemStart( ){
   SPM_NULL_POINTER_CHECK( _poclStartupSupervisor );
   SPM_NULL_POINTER_CHECK( _poclStartupCommon );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   bPrepareStart( );
   _poclStartupCommon->vAddClient( this );

   ETG_TRACE_USR1( ( "spm_tclStartupSystem::main(): Startup process \"%s\" begins to work in parallel...", _regConfPath.c_str( ) ) );

   vEvaluateProcessStartOrder( ); // evaluate the process startup configuration from registry

   _oRemainingProc2Start = _oProcStartOrder;

   // start further processes
   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poDbusTmp, spm_ISpmLateServiceHandler);

   if ( poDbusTmp ){
      poDbusTmp->vStartupActivated( );
   }

   vCheckForProcessToStart( );
} // vSystemStart

tVoid spm_tclStartupSystem::vEvaluateProcessStartOrder( ){
/*!
  * \fn
  *  \brief
  *   Method to evaluate the start order of processes.

  *  \param none
  ******
  */
// try to open binary config file
   OSAL_tIODescriptor fd = OSAL_IOOpen( "/dev/ffs/registry/procorder.bin", OSAL_EN_READONLY );

   if ( fd != OSAL_ERROR ){
      tS32 s32Result = OSALUTIL_s32FGetSize( fd );
      if ( s32Result > 0 ){
         tU8 *pu8FileBuffer = new tU8[(tU32)s32Result];
         if ( pu8FileBuffer ){
            tS32   s32ReadLen = OSAL_s32IORead( fd, (tPS8)pu8FileBuffer, (tU32)s32Result );
            tChar *pCurPos    = (tChar*)pu8FileBuffer;
            tChar *pEndPos    = (tChar*)pu8FileBuffer + s32ReadLen;
            tU32   u32ProcNb  = 0;
            while ( pCurPos < pEndPos ){
               // get instance
               tU8    u8Instance     = * pCurPos++;
               tChar *strProcessName = pCurPos;
               pCurPos += strlen( strProcessName ) + 1; // 0-termination

               ETG_TRACE_USR4( ( "vEvaluateProcessStartOrder(): Instance: %d, Proc: '%s'.", u8Instance, strProcessName ) );
               if ( u8Instance == (tU8)_u32Instance ){
                  ETG_TRACE_USR4( ( "vEvaluateProcessStartOrder(): Add entry to config!" ) );
                  _oProcStartOrder[u32ProcNb]._process = strProcessName;
                  u32ProcNb                           += 1;
               }
            }
         }
         delete[] pu8FileBuffer;
      }
      if ( OSAL_s32IOClose( fd ) != OSAL_OK ){
         // TRACE_ERROR_OSAL(TRC::FnReadFfsData);
      }

   } else {
      OSAL_tIODescriptor fdReg;
      OSAL_trIOCtrlDir   rDirProc  = { 0,0,0 };
      std::string        strBaseName;

      strBaseName = SPM_REG_PROCESS_BASE_PATH;
      strBaseName = strBaseName + _regConfPath + SPM_STARTUP_FOLD_ORDER;

      fdReg       = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );

      if ( fdReg != OSAL_ERROR ){

         /* First, recursively show all the subkeys */
         rDirProc.fd        = fdReg;
         rDirProc.s32Cookie = 0;

         while ( OSAL_s32IOControl( fdReg, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirProc ) != OSAL_ERROR ){
            OSAL_trIOCtrlRegistryValue rReadVal = { {0},0,{0 }};
            rReadVal.s32Type = OSAL_C_S32_VALUE_STRING;
            OSAL_szStringNCopy( rReadVal.s8Name, rDirProc.dirent.s8Name, sizeof( rReadVal.s8Name ) - 1 );

            if ( OSAL_s32IOControl( fdReg, OSAL_C_S32_IOCTRL_REGREADVALUE, ( intptr_t )&rReadVal ) != OSAL_ERROR ){

               if ( rReadVal.s32Type == OSAL_C_S32_VALUE_STRING ){
                  tUInt unProcNb;
                  if (strlen((tChar*)rDirProc.dirent.s8Name) < SPM_STARTUP_PROCEXTENDED_STR_LENGTH) {
                     if ( OSAL_s32ScanFormat( (tChar*)rDirProc.dirent.s8Name, SPM_STARTUP_FORMAT_PROCESS_STARTUP_ORDER, &unProcNb ) == 1 ){
                        // _oProcStartOrder[unProcNb]._process = (tChar*)as32Data;
                        _oProcStartOrder[unProcNb]._process = (tChar*)rReadVal.s8Value;
                     }
                     #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING_GRP
                     else if ( _poclGrp != NULL ){
                          // is a group evaluator available?
                          if ( _poclGrp->bIsGroup( (tChar*)rDirProc.dirent.s8Name, unProcNb ) ){
                             // activate the group evaluation
                             _oProcStartOrder[unProcNb]._s32GroupNr = _poclGrp->s32AddGroup( (tChar*)rReadVal.s8Value );
                          }
                     }
                     #endif
                  } else {
                      if ( OSAL_s32ScanFormat( (tChar*)rDirProc.dirent.s8Name, SPM_STARTUP_FORMAT_PROCESS_STARTUP_ORDER_EXTENDED, &unProcNb ) == 1 ){
                         // _oProcStartOrder[unProcNb]._process = (tChar*)as32Data;
                         _oProcStartOrder[unProcNb]._process = (tChar*)rReadVal.s8Value;
                      }
                      #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING_GRP
                      else if ( _poclGrp != NULL ){
                         // is a group evaluator available?
                         if ( _poclGrp->bIsGroup( (tChar*)rDirProc.dirent.s8Name, unProcNb ) ){
                            // activate the group evaluation
                            _oProcStartOrder[unProcNb]._s32GroupNr = _poclGrp->s32AddGroup( (tChar*)rReadVal.s8Value );
                         }
                     }
                     #endif
                  }
               } else {
                  ETG_TRACE_FATAL( ( "VAL unkown: '%30s': unknown type %d", (const tChar*)rDirProc.dirent.s8Name, rReadVal.s32Type ) );
               }
            }
         }
      }
   }

} // vEvaluateProcessStartOrder

void spm_tclStartupSystem::vWaitForSync( tU32 u32Sync ){
/*!
  * \fn
  *  \brief
  *   Method to synchronize the startup of different software blocks.
  *
  *  \param[in] u32Sync: synchronization mode
  ******
  */
   if ( ( _hSyncEvent != OSAL_C_INVALID_HANDLE )
        && ( _oWaitingBlocks.size( ) > 0 )
        && ( u32Sync != SPM_U32_SYNC_NONE ) ){ // event available
      // Need here an inner block so that the life cycle of the spm_tclHandleSemaphore object
      // ends before entering the OSAL_s32EventWait function.
      {
         // Synchronize with concurrent access from vOnSw...
         spm_tclHandleSemaphore hdl( _hSemHandle );

         spm_tclCompareSwBlocks cmp;


         #ifdef __SPM_TRACE_PREDECESSOR
            {
               std::string                    strBuffer( "vWaitForSync[Predecessor]: " );
               TSpmSwBlockSet::const_iterator it;

               for ( it = _oWaitingBlocks.begin( ); it != _oWaitingBlocks.end( ); ++it ){
                  strBuffer += * it + " ";
               }
               ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
            }
         #endif
         if ( u32Sync == SPM_U32_SYNC_UP ){
            if ( std::includes( _oUpBlocks.begin( ),
                                _oUpBlocks.end( ),
                                _oWaitingBlocks.begin( ),
                                _oWaitingBlocks.end( ),
                                cmp ) ){
               // we are already up
               OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_UP ), OSAL_EN_EVENTMASK_AND );
               return;
            }
         } else if ( u32Sync == SPM_U32_SYNC_CONNECTED ){
            if ( std::includes( _oConnectedBlocks.begin( ),
                                _oConnectedBlocks.end( ),
                                _oWaitingBlocks.begin( ),
                                _oWaitingBlocks.end( ),
                                cmp ) ){    // we are already connected
               OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_CONNECTED ), OSAL_EN_EVENTMASK_AND );
               return;
            }
         } else if ( u32Sync == SPM_U32_SYNC_LOADED ){
            if ( std::includes( _oLoadedBlocks.begin( ),
                                _oLoadedBlocks.end( ),
                                _oWaitingBlocks.begin( ),
                                _oWaitingBlocks.end( ),
                                cmp ) ){    // we are already connected
               OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_LOADED ), OSAL_EN_EVENTMASK_AND );
               return;
            }
         } else if ( u32Sync == SPM_U32_SYNC_FORCE ){
            if ( std::includes( _oForcedSwBlocks.begin( ),
                                _oForcedSwBlocks.end( ),
                                _oWaitingToBeForcedSwBlocks.begin( ),
                                _oWaitingToBeForcedSwBlocks.end( ),
                                cmp ) ){  // blocks allready in forced state
               OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_FORCE ), OSAL_EN_EVENTMASK_AND );
               return;
            }
         } else if ( u32Sync == SPM_U32_SYNC_POINT ){
            if ( std::includes( _oSyncPoints.begin( ),
                                _oSyncPoints.end( ),
                                _oWaitingBlocks.begin( ),
                                _oWaitingBlocks.end( ),
                                cmp ) ){    // we are already connected
               OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_POINT ), OSAL_EN_EVENTMASK_AND );
               return;
            }
         }
         // need to be inside critical section because the vOnSwBlock... functions check the waiting status
         _u32CurrentSyncState = u32Sync;
      }

      OSAL_tEventMask hWaitEv   = u32Sync;
      OSAL_tEventMask hResultEv = 0;

      _startWaitTime = OSAL_ClockGetElapsedTime( );

      // wait for sw block to be available
      (void)OSAL_s32EventWait( // here synchronization with application states
               _hSyncEvent,    // needed. Either look for initialized or startup
               hWaitEv,        // state (NORMAL, PAUSE).
               OSAL_EN_EVENTMASK_OR,
               OSAL_C_TIMEOUT_FOREVER,
               &hResultEv );

      tU32 endWaitTime = OSAL_ClockGetElapsedTime( );

      ETG_TRACE_USR4( ( "vWaitForSync(): Waited %dms for sync [%d]", ( endWaitTime - _startWaitTime ), endWaitTime ) );

      _startWaitTime = 0;

      {
         // Synchronize with concurrent access from vOnSw...
         spm_tclHandleSemaphore hdl( _hSemHandle );

         // first set sync state to NONE so that vOnSwBlock... functions to not enter OSAL_s32EventPost
         _u32CurrentSyncState = SPM_U32_SYNC_NONE;

         // clear always all event flags because now we're in idle state
         OSAL_s32EventPost( _hSyncEvent, ~( SPM_U32_SYNC_CONNECTED | SPM_U32_SYNC_UP | SPM_U32_SYNC_LOADED | SPM_U32_SYNC_FORCE ), OSAL_EN_EVENTMASK_AND );
      }
   }
} // vWaitForSync

void spm_tclStartupSystem::vOnSwBlockLoaded( const TSpmSwBlockSet& SwBlockSet ){
/*!
  * \fn
  *  \brief This method is invoked when a software block is loaded (aka spawned)
  *         If waiting blocks are subset of software block, this method posts an event - SPM_U32_SYNC_LOADED.
  *
  *  \param[in] sw: set of software blocks in state loaded.
  ******
  */
// Synchronize with concurrent access from vWaitForSync
   spm_tclHandleSemaphore hdl( _hSemHandle );

   _oLoadedBlocks = SwBlockSet;
   #ifdef __SPM_TRACE_PREDECESSOR
      {
         std::string                    strBuffer( "vOnSwBlockLoaded[Predecessor]: " );
         TSpmSwBlockSet::const_iterator it;

         for ( it = _oLoadedBlocks.begin( ); it != _oLoadedBlocks.end( ); ++it ){
            strBuffer += * it + " ";
         }
         ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
      }
   #endif

   if ( _u32CurrentSyncState == SPM_U32_SYNC_LOADED ){
      spm_tclCompareSwBlocks cmp;

      // Is _oWaitingBlocks a subset of sw?
      if ( std::includes( _oLoadedBlocks.begin( ),
                          _oLoadedBlocks.end( ),
                          _oWaitingBlocks.begin( ),
                          _oWaitingBlocks.end( ),
                          cmp ) ){
         OSAL_tEventMask hPSEvRequest = SPM_U32_SYNC_LOADED;

         OSAL_s32EventPost( _hSyncEvent, hPSEvRequest, OSAL_EN_EVENTMASK_OR );
      }
   }

      SPM_NULL_POINTER_CHECK( _poclSyncHandler );
      _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

} // vOnSwBlockLoaded

void spm_tclStartupSystem::vOnSwBlockConnected( const TSpmSwBlockSet& SwBlockSet ){
/*!
  * \fn
  *  \brief   This method is invoked when all the applications within a software block are connected.
  *           If waiting blocks are subset of software block, this method posts an event - SPM_U32_SYNC_CONNECTED.
  *  \param[in] sw: software block.
  ******
  */
// Synchronize with concurrent access from vWaitForSync
   spm_tclHandleSemaphore hdl( _hSemHandle );

   _oConnectedBlocks = SwBlockSet;
   #ifdef __SPM_TRACE_PREDECESSOR
      {
         std::string                    strBuffer( "vOnSwBlockConnected[Predecessor]: " );
         TSpmSwBlockSet::const_iterator it;

         for ( it = _oConnectedBlocks.begin( ); it != _oConnectedBlocks.end( ); ++it ){
            strBuffer += * it + " ";
         }
         ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
      }
   #endif

   if ( _u32CurrentSyncState == SPM_U32_SYNC_CONNECTED ){
      spm_tclCompareSwBlocks cmp;

      // Is _oWaitingBlocks a subset of sw?
      if ( std::includes( _oConnectedBlocks.begin( ),
                          _oConnectedBlocks.end( ),
                          _oWaitingBlocks.begin( ),
                          _oWaitingBlocks.end( ),
                          cmp ) ){
         OSAL_tEventMask hPSEvRequest = SPM_U32_SYNC_CONNECTED;

         OSAL_s32EventPost( _hSyncEvent, hPSEvRequest, OSAL_EN_EVENTMASK_OR );
      }
   }

      SPM_NULL_POINTER_CHECK( _poclSyncHandler );
      _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

} // vOnSwBlockConnected

void spm_tclStartupSystem::vOnSwBlockForced( const TSpmSwBlockSet& SwBlockSet ){
/*!
  * \fn
  *  \brief This method is invoked when a whole software block is UP and vSetForceBlockEnabled was called.
  *         If waiting blocks are subset of software block, this method posts an event - SPM_U32_SYNC_FORCE.
  *
  *  \param[in] SwBlockSet: software block.
  ******
  */
// Synchronize with concurrent access from vWaitForSync
   spm_tclHandleSemaphore hdl( _hSemHandle );

   _oForcedSwBlocks = SwBlockSet;

   if ( _u32CurrentSyncState == SPM_U32_SYNC_FORCE ){
      spm_tclCompareSwBlocks cmp;
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::vOnSwBlockForced() _u32CurrentSyncState == SPM_U32_SYNC_FORCE" ) );
      // Is _oWaitingBlocks a subset of sw?
      if ( std::includes( _oForcedSwBlocks.begin( ),
                          _oForcedSwBlocks.end( ),
                          _oWaitingToBeForcedSwBlocks.begin( ),
                          _oWaitingToBeForcedSwBlocks.end( ),
                          cmp ) ){
         ETG_TRACE_USR4( ( "spm_tclStartupSystem::vOnSwBlockForced() Sending SPM_U32_SYNC_FORCE event!" ) );
         OSAL_tEventMask hPSEvRequest = SPM_U32_SYNC_FORCE;
         OSAL_s32EventPost( _hSyncEvent, hPSEvRequest, OSAL_EN_EVENTMASK_OR );

      }
   } else {
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::vOnSwBlockForced() _u32CurrentSyncState != SPM_U32_SYNC_FORCE" ) );
   }

      SPM_NULL_POINTER_CHECK( _poclSyncHandler );
      _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );
} // vOnSwBlockForced

void spm_tclStartupSystem::vOnSwBlockUp( const TSpmSwBlockSet& SwBlockSet ){
/*!
  * \fn
  *  \brief This method is invoked when a whole software block is UP.
  *         If waiting blocks are subset of software block, this method posts an event - SPM_U32_SYNC_UP.
  *
  *  \param[in] SwBlockSet: software block.
  ******
  */
// Synchronize with concurrent access from vWaitForSync
   spm_tclHandleSemaphore hdl( _hSemHandle );

   _oUpBlocks = SwBlockSet;
   #ifdef __SPM_TRACE_PREDECESSOR
      {
         std::string                    strBuffer( "vOnSwBlockUp[Predecessor]: " );
         TSpmSwBlockSet::const_iterator it;

         for ( it = _oUpBlocks.begin( ); it != _oUpBlocks.end( ); ++it ){
            strBuffer += * it + " ";
         }
         ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
      }
   #endif

   if ( _u32CurrentSyncState == SPM_U32_SYNC_UP ){
      spm_tclCompareSwBlocks cmp;

      // Is _oWaitingBlocks a subset of sw?
      if ( std::includes( _oUpBlocks.begin( ),
                          _oUpBlocks.end( ),
                          _oWaitingBlocks.begin( ),
                          _oWaitingBlocks.end( ),
                          cmp ) ){
         OSAL_tEventMask hPSEvRequest = SPM_U32_SYNC_UP;

         OSAL_s32EventPost( _hSyncEvent, hPSEvRequest, OSAL_EN_EVENTMASK_OR );
      }
   }

      SPM_NULL_POINTER_CHECK( _poclSyncHandler );
      _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

} // vOnSwBlockUp

void spm_tclStartupSystem::vOnSyncPointReached( const std::string& strSyncPointName ){
   _oSyncPoints.insert( strSyncPointName );

   ETG_TRACE_USR1( ( "spm_tclStartupSystem::vOnSyncPointReached(): Current sync: %d, SyncPoint added '%s'", _u32CurrentSyncState, strSyncPointName.c_str( ) ) );

   if ( _u32CurrentSyncState == SPM_U32_SYNC_POINT ){
      spm_tclCompareSwBlocks cmp;

      // Is _oWaitingBlocks a subset of sw?
      if ( std::includes( _oSyncPoints.begin( ),
                          _oSyncPoints.end( ),
                          _oWaitingBlocks.begin( ),
                          _oWaitingBlocks.end( ),
                          cmp ) ){
         OSAL_tEventMask hPSEvRequest = SPM_U32_SYNC_POINT;

         OSAL_s32EventPost( _hSyncEvent, hPSEvRequest, OSAL_EN_EVENTMASK_OR );
      }
   }

      SPM_NULL_POINTER_CHECK( _poclSyncHandler );
      _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

   if ( _poclStartupInvest ){
      // store in startup investigation
      SPM_STARTUPINVEST_INIT_STARTUPITEM
         SPM_STARTUPINVEST_FORCE_ENTRY( "SYNCPOINT", ( (std::string)"New SyncPoint reached: '" + strSyncPointName + "'" ) )
   }

   {
      // update CCA prperty for SyncPoints

      spm_corefi_tclMsgActiveStartUpSyncPointsStatus tSyncPoints;

      tSyncPoints.tStartupSyncPointList.clear( );

      TSpmSwBlockSet::const_iterator                 it;
      for ( it = _oSyncPoints.begin( ); it != _oSyncPoints.end( ); ++it ){
         tSyncPoints.tStartupSyncPointList.push_back( it->c_str( ) );
      }
      SPM_GET_IF_REFERENCE_NEW_VAR( pCcaSrv, ISpmCcaServiceServer );
      pCcaSrv->vUpdateProperty( SPM_COREFI_C_U16_ACTIVESTARTUPSYNCPOINTS, &tSyncPoints );
   }

} // vOnSyncPointReached

void spm_tclStartupSystem::vOnCheckStartup( ){
      if ( _startWaitTime != 0 ){
         SPM_NULL_POINTER_CHECK( _poclSyncHandler );
         _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );
      }

}

tBool spm_tclStartupSystem::bStartProcess( const std::string& strProcessname ){
/*!
  * \fn
  *  \brief
  *   Method to spawn a given process.

  *  \param[in] strName: Process Name.
  ******
  */

   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );
   SPM_NULL_POINTER_CHECK_VAL( _poclRegistry );

   TMapProcessConfiguration   & oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
   TMapProcStartOrder::iterator it;
   tBool                        bRet              = FALSE;

   ETG_TRACE_USR1( ( "bStartProcess, [%d] [re]start process '%s'", OSAL_ClockGetElapsedTime( ), strProcessname.c_str( ) ) );

   for ( it = _oProcStartOrder.begin( ); it != _oProcStartOrder.end( ); ++it ){
      ETG_TRACE_USR4( ( "bStartProcess, check name '%s'", it->second._process.c_str( ) ) );
      if ( it->second._process == strProcessname ){

         // and now check if process configuration is available
         TMapProcessConfiguration::iterator posProc = oMapProcessConfig.find( it->second._process );
         if ( posProc != oMapProcessConfig.end( ) ){
            TProcConfiguration& _oConf = posProc->second;
            ETG_TRACE_USR4( ( "bStartProcess, now [%d] start process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );
            if ( !_oConf.bProcEnabled ){
               ETG_TRACE_USR4( ( "Process \"%s\" is disabled", _oConf.strProcessName.c_str( ) ) );
               return( TRUE );
            }

            // check if registry for this process needs to be read
            if ( !_oConf.strRegPath.empty( ) ){

               ETG_TRACE_USR4( ( "bStartProcess, import registry now [%d] for process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );

               if ( !_poclRegistry->bImportDataByFile( _oConf.strRegPath, FALSE ) ){
                  ETG_TRACE_FATAL( ( "Could not read registry of process \"%s\"", _oConf.strRegPath.c_str( ) ) );
                  return( FALSE );
               }
               ETG_TRACE_USR4( ( "bStartProcess, registry is loaded [%d], process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );

               // check if due to a valid variant the registry has to be modified
               _poclStartupCommon->vUpdateRegistryDueToValidVariant( );
            }

            // find SwBlocks to start
            TBlockList::iterator pos;
            for ( pos = _oConf.listSwBlocks.begin( ); pos != _oConf.listSwBlocks.end( ); ++pos ){
               // \todo: do not use the magic 0xffffffff
               if ( pos->u32BlockConf != 0xffffffff ){
                  tBool bBlockMode = FALSE;
                  // start block with process name
                  if ( pos->u32BlockConf ){
                     bBlockMode = TRUE;
                     // changed parameter list
                  }
                  if ( _poclStartupCommon->bAddSwBlock( pos->strBlockName.c_str( ),
                                                        bBlockMode,
                                                        bIsLocal( ),
                                                        _oConf.strProcessLocation,
                                                        _oConf.strProcessName,
                                                        _oConf.strProcStartType ) ){
                  }
                  pos->bActive = TRUE;
               }
            }

            // Check if we have to start a process
            ETG_TRACE_USR4( ( "bStartProcess, and now [%d] start the process '%s':", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );
            ETG_TRACE_USR4( ( "Priority:  %d", _oConf.u32ProcPrio ) );
            ETG_TRACE_USR4( ( "Path:      %s", _oConf.strProcessLocation.c_str( ) ) );

            if ( bProcessSpawn( _oConf ) ){
               _poclStartupCommon->vSetProcId( _oConf.strProcessName, _oConf.s32ProcId );
               bRet = TRUE;
            }
         }
      }
   }
   return( bRet );

} // bStartProcess

tBool spm_tclStartupSystem::bStopProcess( const std::string& strProcessname ){
/*!
  * \fn
  *  \brief
  *   Method to stop a given process.
  *
  *  \param[in] strProcessname: Name of the process to be stopped.
  ******
  */

   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );

   TMapProcessConfiguration   & oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
   TMapProcStartOrder::iterator it;

   // iterate over all entries. These are sorted by the less comparator
   // so we start with the lowest number and go then to the next in line
   // automatically. We do not have to find the next in line separately by
   // the find functionality.
   for ( it = _oProcStartOrder.begin( ); it != _oProcStartOrder.end( ); ++it ){
      if ( it->second._process == strProcessname ){

         // and now check if process configuration is available
         TMapProcessConfiguration::iterator posProc = oMapProcessConfig.find( it->second._process );
         if ( posProc != oMapProcessConfig.end( ) ){
            if ( posProc->second.bProcSpawned ){
               // find SwBlocks to start
               TBlockList::iterator pos;
               for ( pos = posProc->second.listSwBlocks.begin( ); pos != posProc->second.listSwBlocks.end( ); ++pos ){
                  // we are already in process of terminating this process -> DO NOT CALL WITH "TRUE"
                  _poclStartupCommon->vRemoveSwBlock( pos->strBlockName.c_str( ), FALSE );
               }
               vRemoveProcess( posProc->second );
               posProc->second.bProcSpawned = FALSE;

               // one more process is killed -> trigger message
               _poclWorkerServer->bPostMessage( "ISpmStartupCommon", SPM_U32_WORKER_STA_PROCESS_KILLED );
               ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bStopProcess(): Processed killed ready '%s'", posProc->first.c_str( ) ) );

               return( TRUE );
            }
         }
      }
   }
   return( FALSE );
} // bStopProcess

tBool spm_tclStartupSystem::bKillProcess( TProcConfiguration& oProcConf ){
   /*!
     * \fn
     *  \brief
     *    Kills the given process and removes from supervision list.
     *
     *  \param[in] oProcConf: process configuration.
     *  \return  TRUE if process is successfully killed otherwise FALSE.
     *
     ******
     */
   OSAL_tEventHandle hShutdown;
   OSAL_tEventMask   hEvRequest                   = 0x00000001;
   tU32              u32handle                    = 0;
   tBool             fHandleFound                 = FALSE;
   tBool             bProcessTerminationConfirmed = FALSE;
   tBool             bIsSystemdProcess            = FALSE;

   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bKillProcess(): Now remove process '%s'", oProcConf.strProcessName.c_str( ) ) );
   // tBool bWaitForProcessTermination = TRUE;
   tBool             bSuccess                     = FALSE;

   tBool             fProcessStopSuccessfull      = FALSE;


   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclSupervision, ISpmProcessSupervision );

   if ( poclSupervision->bGetProcessHandleByName( oProcConf.strProcessName, &u32handle ) == TRUE ){
      fHandleFound = TRUE;
   }

   if ( poclSupervision->bRemoveProcessFromSupervisionList( oProcConf.strProcessName ) ){
      ETG_TRACE_USR4( ( "Process %s removed from list", oProcConf.strProcessName.c_str( ) ) );
   } else {
      ETG_TRACE_USR4( ( "Process %s NOT removed from list", oProcConf.strProcessName.c_str( ) ) );
   }


   // Kill via EVENT
   if ( OSAL_s32EventOpen( oProcConf.strProcShutdownEvent.c_str( ), &hShutdown ) == OSAL_OK ){
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bKillProcess(): send shutdown event to terminate process '%s'", oProcConf.strProcShutdownEvent.c_str( ) ) );
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bKillProcess(): Shutdown Event Name ='%s'", oProcConf.strProcShutdownEvent.c_str( ) ) );
      if ( OSAL_s32EventPost( hShutdown, hEvRequest, OSAL_EN_EVENTMASK_OR ) != OSAL_OK ){
      } else {
         fProcessStopSuccessfull = TRUE;
      }
      if ( OSAL_s32EventClose( hShutdown ) != OSAL_OK ){
          ETG_TRACE_ERRMEM( ( "spm_tclStartupSystem::bKillProcess(): Shutdown Event Name ='%s' still exists -- Process seems to be still active", oProcConf.strProcShutdownEvent.c_str( ) ) );
      }
      OSAL_s32ThreadWait( 200 );
      /* Event should now be deleted: */
      if ( OSAL_s32EventOpen( oProcConf.strProcShutdownEvent.c_str( ), &hShutdown ) == OSAL_OK ){
          ETG_TRACE_ERRMEM( ( "spm_tclStartupSystem::bKillProcess(): Shutdown Event Name ='%s' still exists -- Process seems to be still active", oProcConf.strProcShutdownEvent.c_str( ) ) );
          OSAL_s32EventClose( hShutdown );
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bKillProcess(): Can't open shutdown event to terminate process! Eventname '%s'. Trying now to kill process via killall command", oProcConf.strProcShutdownEvent.c_str( ) ) );
   }

   if ( fProcessStopSuccessfull == FALSE ){
      // Extract process NAME from process LOCATION by parsing from the end of the string until the first occurrence of the slash '/' char.
      // e.g. extract 'lcmtestappcomponent_out.out' from '/opt/bosch/base/bin/lcmtestappcomponent_out.out'
      std::string strProcessName;
      std::size_t pos = oProcConf.strProcessLocation.rfind( '/' );
      if ( pos != std::string::npos ){
         strProcessName = oProcConf.strProcessLocation.substr( pos + 1 );
      }
      if ( !strProcessName.empty( ) ){
         ETG_TRACE_USR4( ( "Calling SPM_ROOTDAEMON_CALLER_rPerformRootOp(KILL_PROCESS, '%s')", strProcessName.c_str( ) ) );
         SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", KILL_PROCESS, strProcessName.c_str( ) );
      } else {
         ETG_TRACE_FATAL( ( "Process name could not be extracted from process location" ) );
      }
   }
   if ( fHandleFound == TRUE ){
      // if (bWaitForProcessTermination)
      {
         int w;
         int iWaitTime    = 0;
         int iMaxWaitTime = 3000;
         int iSleepTime   = 100;
         int status       = 0;
         bProcessTerminationConfirmed = FALSE;
         while ( !bProcessTerminationConfirmed && ( iWaitTime < iMaxWaitTime ) ){
            iWaitTime += iSleepTime;

            w          = waitpid( u32handle, &status, WNOHANG );
            int waitPidErr = errno;
            if ( w == - 1 ){
               if ( ( waitPidErr != EINTR ) ){

                  ETG_TRACE_USR4( ( "Process with handle %d was killed!", u32handle ) );
               } else {
                  ETG_TRACE_USR4( ( "Process with handle %d was maybe killed, waitpid returns with error! (errno=%d=%s)", u32handle, waitPidErr, strerror( waitPidErr ) ) );
               }
               bProcessTerminationConfirmed = TRUE;
               bSuccess                     = TRUE;
            }

            OSAL_s32ThreadWait( iSleepTime );
         } // while
      }

   } else {
         bIsSystemdProcess = bIsSystemdProcess; // LINT
   }
   if ( bProcessTerminationConfirmed ){
      ETG_TRACE_USR4( ( "Kill of process'%s' confirmed!", oProcConf.strProcessName.c_str( ) ) );
   } else {
      ETG_TRACE_USR4( ( "Kill of process'%s' NOT confirmed!", oProcConf.strProcessName.c_str( ) ) );
   }
   ETG_TRACE_USR4( ( "bKillProcess() == %d", bSuccess ) );
   return( bSuccess );
} // bKillProcess

tBool spm_tclStartupSystem::bForceProcessStop( const std::string& strProcessname ){
   /*!
     * \fn
     *  \brief
     *    Function to stop a process. The process name has to be the same as used in
      the base.reg key_ "PROC_NAME".
      1) This functions checks if the given process was started from this startup-instance (via isPartOff()),
       if not, this function returns with FALSE.
      2) All applications of all swblocks of the process "PROC_NAME" are then forced to AMT_C_U32_STATE_OFF state
      3) After all applications ack. the offstate (synced via vWaitForSync(SPM_U32_SYNC_FORCE), the sw-block
       entries are removed from the registry
      4) Final action: The process is removed/killed. Kill method depends on the kind of the startup thread (local/remote/etc),
       and is implemented in the derived local/remote classed.
     *
     *  \param[in] strProcessname : Process name.
     *  \return  TRUE if process is stopped successfully otherwise FALSE.
     *
     ******
     */
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );

   TMapProcessConfiguration   & oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
   TMapProcStartOrder::iterator it;
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclDatabase, ISpmApplicationDatabase );
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclLam,      ISpmLocalAppManager );

   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s)", strProcessname.c_str( ) ) );

   _poclStartupCommon->vAddClient( this );
   if ( isPartOf( strProcessname ) ){
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop() called and %s isPartOf this startup-thread", strProcessname.c_str( ) ) );
      // and now check if process configuration is available

      TMapProcessConfiguration::iterator posProc;
      tBool                              bProcConfigFound = FALSE;

      for ( posProc = oMapProcessConfig.begin( ); posProc != oMapProcessConfig.end( ); posProc++ ){
         if ( posProc->second.strProcessName == strProcessname ){
            bProcConfigFound = TRUE;
            break;
         }
      }


      if ( bProcConfigFound ){
         if ( posProc->second.bProcSpawned ){
            ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) forcing now all sw-blocks to OFF and waiting for sync", strProcessname.c_str( ) ) );

            /* Remember the sw-blocks to be forced and we are waiting for*/
            TBlockList::const_iterator itBlockList;
            for ( itBlockList = posProc->second.listSwBlocks.begin( ); itBlockList != posProc->second.listSwBlocks.end( ); ++itBlockList ){
               _oWaitingToBeForcedSwBlocks.insert( itBlockList->strBlockName );
               /* Now force all SW-BLOCKs to the OFF state*/
            }
            TBlockList::const_iterator it3;
            tBool                      bAllBlocksInTargetState = TRUE;
            const TBlockList         & swList                  = posProc->second.listSwBlocks;
            tU32                       u32State                = AMT_C_U32_STATE_OFF;

            for ( it3 = swList.begin( ); it3 != swList.end( ); ++it3 ){
               spm_tclSoftwareBlock *poSwBlock;
               poSwBlock = poclDatabase->poGetSoftwareBlock( it3->strBlockName );
               if ( poSwBlock ){

                  if ( poSwBlock->bIsBlockMode( ) ){
                     ETG_TRACE_USR4( ( "SwBlock %s has enabled BLOCK MODE. Switching BLOCKMODE off", poSwBlock->pcGetName( ).c_str( ) ) );
                     poSwBlock->vSetBlockMode( FALSE );
                  } else {
                     ETG_TRACE_USR4( ( "SwBlock %s has no enabled BLOCK MODE. ", poSwBlock->pcGetName( ).c_str( ) ) );
                  }
                  ETG_TRACE_USR4( ( "Forcing SwBlock %s to state", poSwBlock->pcGetName( ).c_str( ) ) );
                  ETG_TRACE_USR4( ( "State =%d", u32State ) );
                  if ( !poclLam->bForceSoftwareBlock( it3->strBlockName, u32State ) ){
                     ETG_TRACE_USR4( ( "SwBlock %s currently NOT in forced state!", poSwBlock->pcGetName( ).c_str( ) ) );
                     bAllBlocksInTargetState = FALSE;
                  } else {
                     ETG_TRACE_USR4( ( "SwBlock %s allready in forced state!", poSwBlock->pcGetName( ).c_str( ) ) );
                     // poj2hi \todo: poSwBlock->bSetSupplierStateToUnavailableForAllSwBlockApplications(lnkspm_tclLocalApplicationManager);
                  }
               }
            }
            /* END */
            if ( bAllBlocksInTargetState ){
               ETG_TRACE_USR4( ( "bAllBlocksInTargetState() == TRUE " ) );
               ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) all SW-Blocks allready in OFF state, no WaitForSync necessary.", strProcessname.c_str( ) ) );
            } else {
               ETG_TRACE_USR4( ( "bForceAllSwBlocks() == FALSE " ) );
               vWaitForSync( SPM_U32_SYNC_FORCE );
               ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) SPM_U32_SYNC_FORCE received, removing now sw-blocks from registry", strProcessname.c_str( ) ) );
            }

            /* now remove all sw-blocks:*/
            {
               TBlockList::const_iterator it2;
               for ( it2 = swList.begin( ); it2 != swList.end( ); ++it2 ){
                  poclDatabase->vRemoveSoftwareBlock( it2->strBlockName );
               }
            }

            ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) killing now process..", strProcessname.c_str( ) ) );
            if ( bKillProcess( posProc->second ) ){
               ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) process kill OK!", strProcessname.c_str( ) ) );
            } else {
               ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bForceProcessStop(%s) process kill NOK!", strProcessname.c_str( ) ) );
            }
            posProc->second.bProcSpawned = FALSE;
            _oWaitingToBeForcedSwBlocks.clear( );
            return( TRUE );
         } else {
            ETG_TRACE_USR4( ( "proc config found for %s, but process is currently not spawned.", strProcessname.c_str( ) ) );
         }
      } else {
         ETG_TRACE_USR4( ( "proc config not found for %s", strProcessname.c_str( ) ) );
      }
   } else {
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop() called but %s is NOT PartOf this startup-thread", strProcessname.c_str( ) ) );
   }
   _poclStartupCommon->vRemoveClient( this );
   return( FALSE );
} // bForceProcessStop

tBool spm_tclStartupSystem::bForceProcessStopSyncronous( const std::string& strProcessName,
                                                         std::string      & strForcedSwBlockNameToWaitFor ){
   /*!
     * \fn
     *  \brief
     *    Function to stop a process. The process name has to be the same as used in
      the base.reg key_ "PROC_NAME".
      1) This functions checks if the given process was started from this startup-instance (via isPartOff()),
       if not, this function returns with FALSE.
      2) All applications of all swblocks of the process "PROC_NAME" are then forced to AMT_C_U32_STATE_OFF state
      3) After all applications ack. the offstate (synced via vWaitForSync(SPM_U32_SYNC_FORCE), the sw-block
       entries are removed from the registry
      4) Final action: The process is removed/killed. Kill method depends on the kind of the startup thread (local/remote/etc),
       and is implemented in the derived local/remote classed.
     *
     *  \param[in]  strProcessName : process name.
     *  \param[out] strForcedSwBlockNameToWaitFor: process name that is stopped.
     *  \return  TRUE if successfully stops process otherwise FALSE.
     *
     ******
     */
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );

   TMapProcessConfiguration   & oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
   TMapProcStartOrder::iterator it;
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclDatabase, ISpmApplicationDatabase );
   SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclLam,      ISpmLocalAppManager );

   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStopSyncronous(%s)", strProcessName.c_str( ) ) );

   if ( isPartOf( strProcessName ) ){
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStopSyncronous() called and %s isPartOf this startup-thread", strProcessName.c_str( ) ) );
      // and now check if process configuration is available

      tBool bProcConfigFound = FALSE;

      for ( posProcToKill = oMapProcessConfig.begin( ); posProcToKill != oMapProcessConfig.end( ); posProcToKill++ ){
         if ( posProcToKill->second.strProcessName == strProcessName ){
            bProcConfigFound = TRUE;
            break;
         }
      }


      if ( bProcConfigFound ){
         if ( posProcToKill->second.bProcSpawned ){
            ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStopSyncronous(%s) forcing now all sw-blocks to OFF and waiting for sync", strProcessName.c_str( ) ) );

            /* Remember the sw-blocks to be forced and we are waiting for*/
            TBlockList::const_iterator itBlockList;
            for ( itBlockList = posProcToKill->second.listSwBlocks.begin( ); itBlockList != posProcToKill->second.listSwBlocks.end( ); ++itBlockList ){
               _oWaitingToBeForcedSwBlocks.insert( itBlockList->strBlockName );
               /* Now force all SW-BLOCKs to the OFF state*/
            }
            TBlockList::const_iterator it3;
            tBool                      bAllBlocksInTargetState = TRUE;
            const TBlockList         & swList                  = posProcToKill->second.listSwBlocks;
            tU32                       u32State                = AMT_C_U32_STATE_OFF;

            for ( it3 = swList.begin( ); it3 != swList.end( ); ++it3 ){
               spm_tclSoftwareBlock *poSwBlock;
               poSwBlock = poclDatabase->poGetSoftwareBlock( it3->strBlockName );
               if ( poSwBlock ){

                  if ( poSwBlock->bIsBlockMode( ) ){
                     ETG_TRACE_USR4( ( "SwBlock %s has enabled BLOCK MODE. Switching BLOCKMODE off", poSwBlock->pcGetName( ).c_str( ) ) );
                     poSwBlock->vSetBlockMode( FALSE );
                  } else {
                     ETG_TRACE_USR4( ( "SwBlock %s has no enabled BLOCK MODE. ", poSwBlock->pcGetName( ).c_str( ) ) );
                  }
                  ETG_TRACE_USR4( ( "Forcing SwBlock %s to state", poSwBlock->pcGetName( ).c_str( ) ) );
                  ETG_TRACE_USR4( ( "State =%d", u32State ) );
                  if ( !poclLam->bForceSoftwareBlock( it3->strBlockName, u32State ) ){
                     ETG_TRACE_USR4( ( "SwBlock %s currently NOT in forced state!", poSwBlock->pcGetName( ).c_str( ) ) );
                     bAllBlocksInTargetState = FALSE;
                  } else {
                     ETG_TRACE_USR4( ( "SwBlock %s allready in forced state!", poSwBlock->pcGetName( ).c_str( ) ) );
                     // poj2hi \todo: poSwBlock->bSetSupplierStateToUnavailableForAllSwBlockApplications(lnkspm_tclLocalApplicationManager);
                  }
               }
            }
            /* END */
            if ( bAllBlocksInTargetState ){
               ETG_TRACE_USR4( ( "bAllBlocksInTargetState() == TRUE " ) );
               ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStopSyncronous(%s) all SW-Blocks allready in OFF state, no WaitForSync necessary.", strProcessName.c_str( ) ) );
               strForcedSwBlockNameToWaitFor.clear( );
               vRemoveSwBlockAndKillProcessSyncronous( );
            } else {
               ETG_TRACE_USR4( ( "bForceAllSwBlocks() == FALSE " ) );
               strForcedSwBlockNameToWaitFor = strProcessName;
            }

            return( TRUE );
         } else {
            ETG_TRACE_FATAL( ( " posProc->second.bProcSpawned=%c", posProcToKill->second.bProcSpawned ) );
            ETG_TRACE_USR4( ( "proc config found for %s, but process is currently not spawned.", strProcessName.c_str( ) ) );
         }
      } else {
         ETG_TRACE_USR4( ( "proc config not found for %s", strProcessName.c_str( ) ) );
      }
   } else {
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop() called but %s is NOT PartOf this startup-thread", strProcessName.c_str( ) ) );
   }
   return( FALSE );
} // bIsSwBlockInTargetStateForForcedProcessStop

tBool spm_tclStartupSystem::bWaitForSwBlockRemoval( tU32               u32SyncMode,
                                                    const std::string& strSwBlockName ){
   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bWaitForSwBlockRemoval() SPM_U32_SYNC_FORCE received, checking SwBlock %s", strSwBlockName.c_str( ) ) );

   if ( ( _oWaitingBlocks.size( ) == 0 ) || ( u32SyncMode == SPM_U32_SYNC_NONE ) ){
      //start now
      return( TRUE );
   } else {
      //check for blocks
      spm_tclCompareSwBlocks cmp;
      if ( u32SyncMode == SPM_U32_SYNC_UP ){
         if ( std::includes( _oUpBlocks.begin( ), _oUpBlocks.end( ), _oWaitingBlocks.begin( ), _oWaitingBlocks.end( ), cmp ) ){ // we are already up
            return( TRUE );
         }
      } else if ( u32SyncMode == SPM_U32_SYNC_CONNECTED ){
         if ( std::includes( _oConnectedBlocks.begin( ), _oConnectedBlocks.end( ), _oWaitingBlocks.begin( ), _oWaitingBlocks.end( ), cmp ) ){    // we are already connected
            return( TRUE );
         }
      } else if ( u32SyncMode == SPM_U32_SYNC_LOADED ){
      } else if ( u32SyncMode == SPM_U32_SYNC_FORCE ){
         if ( std::includes( _oForcedSwBlocks.begin( ), _oForcedSwBlocks.end( ), _oWaitingToBeForcedSwBlocks.begin( ), _oWaitingToBeForcedSwBlocks.end( ), cmp ) ){  // blocks
            return( TRUE );
         }
      }

   }

   return( FALSE ); // swblock not up, have to wait...
} // bWaitForSwBlockRemoval

tVoid spm_tclStartupSystem::vRemoveSwBlockAndKillProcessSyncronous( ){
   /*!
     * \fn
     *  \brief
     *    Fetch list of process from posProcToKill and Removes the corresponding software blocks from database
        and kills process.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   const TBlockList& swList = posProcToKill->second.listSwBlocks;

   /* now remove all sw-blocks:*/
   {
      TBlockList::const_iterator it2;
      SPM_GET_IF_REFERENCE_NEW_VAR( poclDatabase, ISpmApplicationDatabase );

      for ( it2 = swList.begin( ); it2 != swList.end( ); ++it2 ){
         poclDatabase->vRemoveSoftwareBlock( it2->strBlockName );
      }
   }

   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) killing now process..", posProcToKill->second.strProcessName.c_str( ) ) );
   if ( bKillProcess( posProcToKill->second ) ){
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::bForceProcessStop(%s) process kill OK!", posProcToKill->second.strProcessName.c_str( ) ) );
   } else {
      ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bForceProcessStop(%s) process kill NOK!", posProcToKill->second.strProcessName.c_str( ) ) );
   }
   posProcToKill->second.bProcSpawned = FALSE;
   _oWaitingToBeForcedSwBlocks.clear( );

} // vRemoveSwBlockAndKillProcessSyncronous

tBool spm_tclStartupSystem::isPartOf( const std::string& strProcessname ){
   TMapProcStartOrder::iterator it;

   for ( it = _oProcStartOrder.begin( ); it != _oProcStartOrder.end( ); ++it ){
      // Not really object oriented, but it does the job
      if ( ( _poclGrp != NULL ) && ( it->second._s32GroupNr >= 0 ) ){
         if ( _poclGrp->find( it->second._s32GroupNr, strProcessname ) ){
            return( TRUE );
         }
      } else {
         // do the processnames match?
         if ( it->second._process.compare( strProcessname ) == 0 ){
            return( TRUE );
         }
      }
   }
   // process not part of this startup.
   return( FALSE );
} // isPartOf

tBool spm_tclStartupSystem::bPrepareStart( ){
   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );

   ETG_TRACE_USR4( ( "spm_tclStartupSystem::bPrepareStart(): %d", OSAL_ClockGetElapsedTime( ) ) );

   // if process start order is empty then return TRUE
   if ( _oProcStartOrder.size( ) == 0 ){
      // TRACE_SPM_INFO_STRING("Startup order is empty");
      return( TRUE );
   }

   _oRemainingProc2Start = _oProcStartOrder;

   return ( FALSE );
}

// start the processes in the order they are defined in the registry
tBool spm_tclStartupSystem::bGetNextProcess2Start( TProcConfiguration& roProc2StartConfig,
                                                   tU32              & ru32SyncMode,
                                                   TSpmSwBlockSet    & roWaitingBlocks ){
/*!
  * \fn
  *  \brief
  *   Method to start processes in the order defined in the registry.
  *
  *  \param[in] roProc2StartConfig: process configuration.
  *  \param[out]   ru32SyncMode: output sync mode
  *  \param[in] roWaitingBlocks: set of waiting blocks
  *  \return   TRUE on success otherwise FALSE.
  ******
  */

   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );

   tBool bProcessConfigFound = FALSE;

   ETG_TRACE_USR4( ( "bGetNextProcess2Start(), %d", OSAL_ClockGetElapsedTime( ) ) );

   // if process start order is empty then return TRUE
   if ( _oRemainingProc2Start.size( ) == 0 ){
      // TRACE_SPM_INFO_STRING("Startup order is empty");
      return( FALSE );
   }
   TMapProcessConfiguration   & oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
   TMapProcStartOrder::iterator it                = _oRemainingProc2Start.begin( );

   if ( it != _oRemainingProc2Start.end( ) ){

      // we have to iterate here over a possible group. In case we have a normal
      // process configuration then we only iterate once
      // and _poclGrp is only optional that does not make it simpler here
      #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING_GRP
         tBool    bInitialized    = FALSE;
      #endif
      tU32        u32HighPrioSync = SPM_U32_SYNC_UNKNOWN;
      std::string strProcessname;

      #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING_GRP
         if ( bInitialized ){
            // here we are already in group iteration
            if ( !_poclGrp->bHasNext( it->second._s32GroupNr ) ){
            }
            u32HighPrioSync = _poclGrp->u32GetSync( it->second._s32GroupNr );
            strProcessname  = _poclGrp->oGetNext( it->second._s32GroupNr );
         } else if ( _poclGrp && ( it->second._s32GroupNr >= 0 ) ){  // group available
            // here we have found a group: initialize and then start with first group element
            bInitialized = TRUE;
            _poclGrp->vStart( it->second._s32GroupNr );
            if ( !_poclGrp->bHasNext( it->second._s32GroupNr ) ){
            }
            u32HighPrioSync = _poclGrp->u32GetSync( it->second._s32GroupNr );
            strProcessname  = _poclGrp->oGetNext( it->second._s32GroupNr );
         } else
      #endif // ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING_GRP

      // here normal process startup
      strProcessname = it->second._process;

      // and now check if process configuration is available
      TMapProcessConfiguration::iterator posProc = oMapProcessConfig.find( strProcessname );
      if ( posProc != oMapProcessConfig.end( ) ){
         roProc2StartConfig  = posProc->second;

         ru32SyncMode        = u32GetSyncBlock( roProc2StartConfig, u32HighPrioSync, roWaitingBlocks );

         bProcessConfigFound = TRUE;
      }
      _oRemainingProc2Start.erase( it );
   }
   return( bProcessConfigFound );
} // bStartSystem

tU32 spm_tclStartupSystem::u32GetSyncBlock( TProcConfiguration&  _oConf,
                                            tU32                u32HighPrioSync,
                                            TSpmSwBlockSet    & roWaitingBlocks ){
   /*!
     * \fn
     *  \brief
     *    Returns sync mode for the input process configuration.
     *
     *  \param[in] _oConf: process configuration
     *  \param[in] u32HighPrioSync: highest sync value
     *  \param[out]	roWaitingBlocks: list of waiting blocks.
     *  \return   Returns sync mode.
     *
     ******
     */
   // find SwBlocks waiting for
   // handle specialty here: predecessor
   tU32                 u32SyncMode  = SPM_U32_SYNC_NONE;
   tU32                 u32DelaySync = 0; // no waiting time

   TBlockList::iterator pos;

   roWaitingBlocks.clear( );

   if ( _oConf.bPredecessorSeen ){
      // synchronize with predecessor process
      u32SyncMode  = _oConf.u32LastSync & 0xffff; // only consider least 16 bits
      u32DelaySync = _oConf.u32LastSync >> 16;    // extract the waiting time

      // get the current set of predecessor sw blocks and reduce them explicitly by the
      // set of excluded sw blocks. Then assign that to roWaitingBlocks.
      if ( ( _oPredecessorSwBlocks.size( ) > 0 ) && ( _oConf.setSwBlocksExclude.size( ) > 0 ) ){
         roWaitingBlocks.clear( ); // start with empty set of waiting blocks
         std::set_symmetric_difference( _oPredecessorSwBlocks.begin( ), _oPredecessorSwBlocks.end( ),
                                        _oConf.setSwBlocksExclude.begin( ), _oConf.setSwBlocksExclude.end( ),
                                        std::inserter( roWaitingBlocks, roWaitingBlocks.begin( ) ) );
      } else {
         roWaitingBlocks = _oPredecessorSwBlocks; // just copy if nothing to consider
      }
      if ( u32HighPrioSync < SPM_U32_SYNC_IGNORE ){
         // if in a group we have a high prio sync then override here the
         // current synchronization setting.
         ETG_TRACE_USR4( ( "High Prio Predecessor Synchronization: %d for %s", u32HighPrioSync, _oConf.strProcessName.c_str( ) ) );
         u32SyncMode = u32HighPrioSync;
      }

      #ifdef __SPM_TRACE_PREDECESSOR
         {
            std::string                    strBuffer;
            TSpmSwBlockSet::const_iterator it;

            strBuffer = "bStartProcess[Predecessor]: ";

            for ( it = roWaitingBlocks.begin( ); it != roWaitingBlocks.end( ); ++it ){
               strBuffer += * it + " ";
            }
            ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
         }
      #endif // ifdef __SPM_TRACE_PREDECESSOR
   } else {
      for ( pos = _oConf.listBlocksToSync.begin( ); pos != _oConf.listBlocksToSync.end( ); ++pos ){
         roWaitingBlocks.insert( pos->strBlockName.c_str( ) );
         tU32 t = pos->u32BlockConf & 0xffff;
         if ( u32SyncMode < t ){
            u32SyncMode = t;
         }
         t = pos->u32BlockConf >> 16;
         if ( u32DelaySync < t ){
            u32DelaySync = t;
         }
         if ( u32HighPrioSync < SPM_U32_SYNC_IGNORE ){
            // if in a group we have a high prio sync then override here the
            // current synchronization setting.
            ETG_TRACE_USR4( ( "High Prio Synchronization: %d for %s", u32HighPrioSync, _oConf.strProcessName.c_str( ) ) );
            u32SyncMode = u32HighPrioSync;
         }
      }
   }

   ETG_TRACE_USR1( ( "u32GetSyncBlock, now [%d] process '%s' is waiting for sync.", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );

   return( u32SyncMode );
} // u32GetSyncBlock

tVoid spm_tclStartupSystem::vTraceSwBlock( const std::string& strName,
                                           TSpmSwBlockSet     oSwBlocks ){
   /*!
     * \fn
     *  \brief
     *    Add list of software block name into trace.
     *
     *  \param[in] strName: name to be added in trace.
     *  \param[in]   oSwBlocks: set of software blocks.
     *  \return  void
     *
     ******
     */
   std::string                    strBuffer;
   TSpmSwBlockSet::const_iterator it;

   strBuffer = (std::string)"spm_tclStartupSystem::vTraceSwBlock(): " + strName;

   for ( it = oSwBlocks.begin( ); it != oSwBlocks.end( ); ++it ){

      strBuffer += * it + " ";
   }
   ETG_TRACE_USR4( ( "%s", strBuffer.c_str( ) ) );
} // vTraceSwBlock

tBool spm_tclStartupSystem::bStartMyProcess( TProcConfiguration& _oConf,
                                             tU32                u32SyncMode ){
/*!
  * \fn
  *  \brief
  *        Method to start one process considering the synchronization.
  *
  *  \param[in] _oConf: Process Configuration information.
  *  \param[in] u32HighPrioSync: synchronization mode.
  *  \return  TRUE on success otherwise FALSE.
  ******
  */
   tBool bStartProc = FALSE;
   tBool bRet       = FALSE;
   tBool bBlockMode = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poclStartupCommon );
   SPM_NULL_POINTER_CHECK_VAL( _poclRegistry );
   _startWaitTime = 0;

   ETG_TRACE_USR1( ( "spm_tclStartupSystem::bStartMyProcess(): now [%d] start process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );

   if ( !_oConf.bProcEnabled ){
      ETG_TRACE_USR1( ( "spm_tclStartupSystem::bStartMyProcess(): Process \"%s\" is disabled", _oConf.strProcessName.c_str( ) ) );
      return( TRUE );
   }

   // check if registry for this process needs to be read
   if ( ( SPM_STARTUP_REG_TYPE_LOAD_DIRECTLY & _oConf.u32RegType )                                       // no early reading
           && ( !_oConf.strRegPath.empty( ) )
   ){

       ETG_TRACE_USR4( ( "spm_tclStartupSystem::bStartMyProcess(): import registry now [%d] for process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strProcessName.c_str( ) ) );
       if (bCheckStartCondition(_oConf, &_oConf.strProcessName[0])) {
           if ( !_poclRegistry->bImportDataByFile( _oConf.strRegPath, FALSE ) ){
               ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bStartMyProcess(): Could not read registry of process \"%s\"", _oConf.strRegPath.c_str( ) ) );
               return( FALSE );
           }
           ETG_TRACE_USR1( ( "spm_tclStartupSystem::bStartMyProcess(): registry is loaded [%d], process '%s'", OSAL_ClockGetElapsedTime( ), _oConf.strRegPath.c_str( ) ) );
       } else {
           ETG_TRACE_USR1( ( "spm_tclStartupSystem::bStartMyProcess(): registry '%s' not loaded due to Start Condition.",_oConf.strRegPath.c_str( ) ) );
       }
      // check if due to a valid variant the registry has to be modified
      _poclStartupCommon->vUpdateRegistryDueToValidVariant( );
   }

   // call project depending registry handling for this process
   if ( !bAlternativeProcConfig( _oConf, &_oConf.strProcessName[0] ) ){
      ETG_TRACE_USR1( ( "spm_tclStartupSystem::bStartMyProcess(): Process \"%s\" is disabled by project configuration", _oConf.strProcessName.c_str( ) ) );

      //here add block as connected / up --> to avoid hanging startup if someone sync on this process
      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclLocalAppManager, ISpmLocalAppManager);
      poclLocalAppManager->vOnWholeSwBlockIsConnected( _oConf.strProcessName );
      poclLocalAppManager->vOnWholeSwBlockIsInRequestedState( _oConf.strProcessName );

      return( TRUE );
   }

   tBool bFileDetect = FALSE;
   if ( _oConf.u32ProcType == 0 ){
      for ( tU8 i = 0; ( i < 10 ) && !bFileDetect; i++ ){
         if ( SPM_STARTUP_VALUE_OSAL_START_TYPE == _oConf.strProcStartType ){
            bFileDetect = _poSpmOsLinux->bIsFileInFs( _oConf.strProcessLocation );
            if ( !bFileDetect ){
               ETG_TRACE_SYS_MIN( ( "Could not find process \"%s\". Wait ....", _oConf.strProcessLocation.c_str( ) ) );
               OSAL_s32ThreadWait( 200 );
            }
         } else {
            bFileDetect = TRUE;
         }
      }
   } else {
      bFileDetect = TRUE;
   }
   if ( ( !_oConf.strProcessLocation.empty( ) ) && !bFileDetect ){
      ETG_TRACE_FATAL( ( "Could not find process \"%s\"", _oConf.strProcessLocation.c_str( ) ) );
      return( FALSE );
   }

   tBool                bClearPredecessor = FALSE;
   TBlockList::iterator pos;

   if ( 0 == ( _oConf.u32RegType & SPM_STARTUP_REG_TYPE_NOREG ) ){
      // find SwBlocks to start
      for ( pos = _oConf.listSwBlocks.begin( ); pos != _oConf.listSwBlocks.end( ); ++pos ){
         // \todo: do not use the magic 0xffffffff
         if ( pos->u32BlockConf != 0xffffffff ){
            // start block with process name
            if ( pos->u32BlockConf ){
               bBlockMode = TRUE;
            }
            if ( _poclStartupCommon->bAddSwBlock( pos->strBlockName.c_str( ),
                                                  bBlockMode,
                                                  bIsLocal( ),
                                                  _oConf.strProcessLocation,
                                                  _oConf.strProcessName,
                                                  _oConf.strProcStartType ) ){
               // changed parameter list
               // system power manager
               bStartProc        = TRUE;
               bClearPredecessor = TRUE; // has a legal sw block for predecessor
            }
            pos->bActive = TRUE;
         } else {
            bStartProc = TRUE;
         }
      }
   } else {
      // start process (no CCA apps will be started)
      bStartProc = TRUE;
   }
   if ( bStartProc ){
      // Check if we have to start a process
      OSAL_tMSecond nMS = OSAL_ClockGetElapsedTime( );
      ETG_TRACE_USR1( ( "bStartMyProcess, and now [%d]ms start the process '%s':", nMS, _oConf.strProcessName.c_str( ) ) );
      ETG_TRACE_USR4( ( "Blockmode: %d", bBlockMode ) );
      ETG_TRACE_USR4( ( "Priority:  %d", _oConf.u32ProcPrio ) );
      ETG_TRACE_USR4( ( "Path:      %s", _oConf.strProcessLocation.c_str( ) ) );

      if ( bProcessSpawn( _oConf ) ){
         // this process could be started, now update the predecessor block
         // and the list of all blocks which are up now.
         _poclStartupCommon->vSetProcId( _oConf.strProcessName, _oConf.s32ProcId );

         // Set proc spawned flag for this procconfig, direct acess/set of spawned flag in above
         // bProcessSpawn() function is not working anymore, as it set it in the given _oConf object,
         // which is not a reference to the original oProcessConfigurationMap. So set is lost after this function is leaved.
         _poclStartupCommon->vSetProcessStarted( _oConf.strProcessName, TRUE );

         SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclDatabase, ISpmApplicationDatabase );
         spm_tclSoftwareBlock *poSwBlock = poclDatabase->poGetSoftwareBlock( _oConf.strProcessName );
         if ( poSwBlock != NULL ){
            poSwBlock->vSetWholeBlockLoaded( );
            SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclLam, ISpmLocalAppManager );
            poclLam->vOnWholeSwBlockIsLoaded( _oConf.strProcessName );
         } else {
            ETG_TRACE_USR1( ( "No SW block found for name %s", _oConf.strProcessName.c_str( ) ) );
            // in case of non-CCA applications this is allowed to happen
            // still leave the FATAL trace in to be able to see if a CCA-Process does show this
            // behaviour. Then it is wrong. We just cannot find out here if it is a CCA
            // process or not.
            // ETG_TRACE_ERRMEM( ( "No SW block found for name %s", _oConf.strProcessName ) );
         }

         // support predecessor handling in configuration
         if ( bClearPredecessor ){
            _oPredecessorSwBlocks.clear( );
            std::list < TBlockElement >::iterator it;
            for ( it = _oConf.listSwBlocks.begin( ); it != _oConf.listSwBlocks.end( ); ++it ){
               // shall sw block be inserted?
               if ( it->u32BlockConf != 0xffffffff ){
                  _oPredecessorSwBlocks.insert( it->strBlockName.c_str( ) );
                  _oAllSwBlocks.insert( it->strBlockName.c_str( ) );
               }
            }
         }

         bRet = TRUE;

         // add startup investigation information
         std::string strName = "Started";
         if ( u32SyncMode != SPM_U32_SYNC_UNKNOWN ){
            strName += " Grouped";
         }
         strName += " Process \"" + _oConf.strProcessName + "\"";
         {
            strName += ", ProcessSpawn Duration: [" + std::to_string( OSAL_ClockGetElapsedTime( ) - nMS ) + "ms]";
         }
         if ( u32SyncMode != SPM_U32_SYNC_UNKNOWN ){
            strName += ", High Prio = " + std::to_string( u32SyncMode );
         }
         SPM_STARTUPINVEST_INIT_STARTUPITEM
         SPM_STARTUPINVEST_FORCE_ENTRY( _context, strName )

         reg_tclRegKey oRegProc;
         std::string strRegPath = SPM_REG_PROCESS_BASE_PATH;
         strRegPath += _oConf.strProcessName;
         if ( oRegProc.bOpen( strRegPath.c_str( ) ) ){
            if ( FALSE == oRegProc.bSetU32( SPM_STARTUP_SYNC_REG_NAME, (tU32)TRUE ) ){
               ETG_TRACE_FATAL( ( "set SPM_STARTUP_SYNC_REG_NAME regkey is fail for process %s ", _oConf.strProcessName.c_str( ) ) );
            }
         }
      } else {

         // add startup investigation information
         std::stringstream sstream;
         sstream << std::setw( 8 ) << std::setfill( '0' ) << std::hex << OSAL_u32ErrorCode( );
         std::string strName = "Could not start process \"" + _oConf.strProcessName + "\", Errorcode = " + sstream.str( );

         SPM_STARTUPINVEST_INIT_STARTUPITEM
         SPM_STARTUPINVEST_FORCE_ENTRY( _context, strName )

         _poclStartupInvest->vDump();


         ETG_TRACE_ERRMEM( ( "%s", strName.c_str( ) ) );
      }
   } else {

      // add startup investigation information
      std::string strName = "Could not start process \"" + _oConf.strProcessName + "\", SwBlockConfiguration not ok";

      SPM_STARTUPINVEST_INIT_STARTUPITEM
                  SPM_STARTUPINVEST_FORCE_ENTRY( _context, strName )

      ETG_TRACE_ERRMEM( ( "%s", strName.c_str( ) ) );

      //here add block as connected / up --> to avoid hanging startup if someone sync on this process
      SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poclLocalAppManager, ISpmLocalAppManager);
      poclLocalAppManager->vOnWholeSwBlockIsConnected( _oConf.strProcessName );
      poclLocalAppManager->vOnWholeSwBlockIsInRequestedState( _oConf.strProcessName );

   }
   return( bRet );
} // bStartMyProcess

tVoid spm_tclStartupSystem::vCheckForProcessToStart( ){
   /*!
     * \fn
     *  \brief
     *    checks for next process to be started.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   ETG_TRACE_USR4( ( "spm_tclStartupSystem::vCheckForProcessToStart()" ) );
   if ( ( _oNextProc2StartConfig.strProcessName.empty( ) ) && ( _oRemainingProc2Start.size( ) != 0 ) ){
      tU32               u32SyncMode = 0;
      TProcConfiguration oProc2StartConfig;
      TSpmSwBlockSet     oWaitingBlock;
      TSpmSwBlockSet     oWaitingSyncBlock;

      if ( bGetNextProcess2Start( oProc2StartConfig, u32SyncMode, oWaitingBlock ) ){
         tBool bStartProc = FALSE;

         _startWaitTime = OSAL_ClockGetElapsedTime( );
         if ( oProc2StartConfig.u32DelayBeforeStartThisProc && ( !oProc2StartConfig.strSyncPointName.empty( ) ) ){
            // waiting time is set --> use as timeout for syncpoint
            _syncWaitTimeout          = oProc2StartConfig.u32DelayBeforeStartThisProc;
            _bKeepWaitingAfterTimeout = FALSE;
         } else {
            // no timeout set --> set default timeout and keep waiting for syncpoint if set (errmem entry /startupinvest is done)
            _syncWaitTimeout          = SPM_U32_SYNC_WAITING_TIMEOUT;
            _bKeepWaitingAfterTimeout = TRUE;
         }

         tBool bDisabled = FALSE;
         if ( !bAlternativeProcConfig( oProc2StartConfig, &oProc2StartConfig.strProcessName[0] ) ){
            ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Process \"%s\" is disabled by project configuration", oProc2StartConfig.strProcessName.c_str( ) ) );
            //here we ignore any delay or syncmode --> process will be not started now :-)
            bDisabled = TRUE;
         }

         //check start delay (wait before starting this process)
         if ( !bDisabled && ( oProc2StartConfig.u32DelayBeforeStartThisProc != 0 )){
            ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Delay for this process found: %dms.", oProc2StartConfig.u32DelayBeforeStartThisProc ) );
            oProc2StartConfig.u32AbsStartTime = oProc2StartConfig.u32DelayBeforeStartThisProc + OSAL_ClockGetElapsedTime( );
         }

         //configuration of last process defines a delay before to start the next process --> add this delay here
         if ( _u32DelayBeforeStartNextProc != 0 ){
            if ( oProc2StartConfig.u32AbsStartTime == 0 ){
               oProc2StartConfig.u32AbsStartTime = OSAL_ClockGetElapsedTime( );
            }
            oProc2StartConfig.u32AbsStartTime += _u32DelayBeforeStartNextProc;
            _u32DelayBeforeStartNextProc       = 0;
         }

         _u32DelayBeforeStartNextProc = oProc2StartConfig.u32DelayBeforeStartNextProc;

         // add defined sync points
         if ( !oProc2StartConfig.strSyncPointName.empty( ) ){
            ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Find syncpoint: '%50s'", oProc2StartConfig.strSyncPointName.c_str( ) ) );
            oWaitingSyncBlock.clear( );
            oWaitingSyncBlock.insert( oProc2StartConfig.strSyncPointName );
         }

            if ( SPM_STARTUP_VALUE_SYSTEMD_START_TYPE == oProc2StartConfig.strProcStartType ){
               //add additional sync point to wait for LcmLateReady --> systemd interface depends on DBUS and DBUS is ready with LCM Late
               oWaitingSyncBlock.insert( SPM_STR_SYNC_POINT_LCMLATE_STARTED );
            }

         ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): new process to start: '%30s', sync mode=%d, SyncBlocks: '%30s', SyncPoints: '%30s', StartTime: %d, StartNextProc: %d",
                           oProc2StartConfig.strProcessName.c_str( ),
                           ETG_ENUM( SPM_SYNC_MODE, u32SyncMode ),
                           strGetListOfBlocks( &oWaitingBlock ).c_str( ),
                           strGetListOfBlocks( &oWaitingSyncBlock ).c_str( ),
                           oProc2StartConfig.u32AbsStartTime,
                           _u32DelayBeforeStartNextProc
                           ) );

         if ( ( ( oWaitingBlock.size( ) == 0 ) && ( oWaitingSyncBlock.size( ) == 0 ) ) || bDisabled ){
            //start now
            bStartProc = TRUE;
         } else {
            //check for CCA SW blocks
            spm_tclCompareSwBlocks cmp;
            if ( u32SyncMode == SPM_U32_SYNC_UP ){
               if ( std::includes( _oUpBlocks.begin( ),
                                   _oUpBlocks.end( ),
                                   oWaitingBlock.begin( ),
                                   oWaitingBlock.end( ),
                                   cmp ) ){ // we are already up
                  bStartProc = TRUE;
               }
            } else if ( u32SyncMode == SPM_U32_SYNC_CONNECTED ){
               if ( std::includes( _oConnectedBlocks.begin( ),
                                   _oConnectedBlocks.end( ),
                                   oWaitingBlock.begin( ),
                                   oWaitingBlock.end( ),
                                   cmp ) ){    // we are already connected
                  bStartProc = TRUE;
               }
            } else if ( u32SyncMode == SPM_U32_SYNC_LOADED ){
               if ( std::includes( _oLoadedBlocks.begin( ),
                                   _oLoadedBlocks.end( ),
                                   oWaitingBlock.begin( ),
                                   oWaitingBlock.end( ),
                                   cmp ) ){    // we are loaded
                  bStartProc = TRUE;
               }
            } else if ( u32SyncMode == SPM_U32_SYNC_FORCE ){
               if ( std::includes( _oForcedSwBlocks.begin( ),
                                   _oForcedSwBlocks.end( ),
                                   _oWaitingToBeForcedSwBlocks.begin( ),
                                   _oWaitingToBeForcedSwBlocks.end( ),
                                   cmp ) ){  // blocks allready in forced state
                  bStartProc = TRUE;
               }
            }

            // and check for additional sync points
            if ( oWaitingSyncBlock.size( ) != 0 ){

               if ( std::includes( _oSyncPoints.begin( ), _oSyncPoints.end( ), oWaitingSyncBlock.begin( ), oWaitingSyncBlock.end( ), cmp ) ){    // we are already connected
                  ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): SyncPoint already reached --> start process" ) );
                  if ( u32SyncMode == SPM_U32_SYNC_NONE ){
                     bStartProc = TRUE;
                  }
               } else {
                  bStartProc = FALSE;
               }
            }
         }

         if ( bStartProc && ( oProc2StartConfig.u32AbsStartTime <= OSAL_ClockGetElapsedTime( ) ) ){
            bStartMyProcess( oProc2StartConfig, u32SyncMode );
            _oNextProc2StartConfig.strProcessName = "";
            _u32NextProcSyncMode                  = SPM_U32_SYNC_NONE;
            _bIgnoreSyncBlocks                    = FALSE;
            _oNextProcWaitingBlocks.clear( );
            _oNextProcWaitingSyncBlocks.clear( );

               SPM_NULL_POINTER_CHECK( _poclSyncHandler );
               _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

         } else {
            //set waiting process information

            _oNextProc2StartConfig      = oProc2StartConfig;
            _u32NextProcSyncMode        = u32SyncMode;
            _oNextProcWaitingBlocks     = oWaitingBlock;
            _oNextProcWaitingSyncBlocks = oWaitingSyncBlock;

            if ( oProc2StartConfig.u32AbsStartTime > OSAL_ClockGetElapsedTime( ) ){
               ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Start time not reached, wait for %dms", oProc2StartConfig.u32AbsStartTime - OSAL_ClockGetElapsedTime( ) ) );
               //start timer to trigger checking process to start
               if ( OSAL_s32TimerSetTime( _hStartProcessTimer, oProc2StartConfig.u32AbsStartTime - OSAL_ClockGetElapsedTime( ), 0 ) != OSAL_OK ){
                  ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
               }
            }

            if ( !_oNextProc2StartConfig.strSyncPointName.empty( ) ){
               ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Find syncpoint: '%50s'", _oNextProc2StartConfig.strSyncPointName.c_str( ) ) );
               _oNextProcWaitingSyncBlocks.insert( _oNextProc2StartConfig.strSyncPointName );
            }

            SPM_NULL_POINTER_CHECK( _poclSyncHandler );
            _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );
         }
      }
   } else if ( !_oNextProc2StartConfig.strProcessName.empty( ) ){
      //process is waiting for sync
      ETG_TRACE_USR4( ( "spm_tclStartupSystem::vCheckForProcessToStart(): check waiting process to start: '%50s'", _oNextProc2StartConfig.strProcessName.c_str( ) ) );

      //check for blocks
      tBool                  bStartProc = FALSE;
      spm_tclCompareSwBlocks cmp;

      if ( _bIgnoreSyncBlocks ){
         _oNextProcWaitingBlocks.clear( );
         _oNextProcWaitingSyncBlocks.clear( );
         _u32NextProcSyncMode                   = SPM_U32_SYNC_NONE;
         _oNextProc2StartConfig.u32AbsStartTime = 0;
      }

      if ( ( _oNextProcWaitingBlocks.size( ) == 0 ) && ( _oNextProcWaitingSyncBlocks.size( ) == 0 ) ){
         bStartProc = TRUE;
      } else if ( _u32NextProcSyncMode == SPM_U32_SYNC_UP ){
         if ( std::includes( _oUpBlocks.begin( ),
                             _oUpBlocks.end( ),
                             _oNextProcWaitingBlocks.begin( ),
                             _oNextProcWaitingBlocks.end( ),
                             cmp ) ){ // we are already up
            bStartProc = TRUE;
         }
      } else if ( _u32NextProcSyncMode == SPM_U32_SYNC_CONNECTED ){
         if ( std::includes( _oConnectedBlocks.begin( ),
                             _oConnectedBlocks.end( ),
                             _oNextProcWaitingBlocks.begin( ),
                             _oNextProcWaitingBlocks.end( ),
                             cmp ) ){    // we are already connected
            bStartProc = TRUE;
         }
      } else if ( _u32NextProcSyncMode == SPM_U32_SYNC_LOADED ){
         if ( std::includes( _oLoadedBlocks.begin( ),
                             _oLoadedBlocks.end( ),
                             _oNextProcWaitingBlocks.begin( ),
                             _oNextProcWaitingBlocks.end( ),
                             cmp ) ){    // we are already connected
            bStartProc = TRUE;
         }
      } else if ( _u32NextProcSyncMode == SPM_U32_SYNC_FORCE ){
         if ( std::includes( _oForcedSwBlocks.begin( ),
                             _oForcedSwBlocks.end( ),
                             _oWaitingToBeForcedSwBlocks.begin( ),
                             _oWaitingToBeForcedSwBlocks.end( ),
                             cmp ) ){  // blocks allready in forced state
            bStartProc = TRUE;
         }
      }

      if ( _oNextProcWaitingSyncBlocks.size( ) != 0 ){
         if ( std::includes( _oSyncPoints.begin( ), _oSyncPoints.end( ), _oNextProcWaitingSyncBlocks.begin( ), _oNextProcWaitingSyncBlocks.end( ), cmp ) ){    // we are already
                                                                                                                                                               // connected
            if ( _u32NextProcSyncMode == SPM_U32_SYNC_NONE ){
               bStartProc = TRUE;
            }

         } else {
            bStartProc = FALSE;
         }
      }

      if ( bStartProc && ( _oNextProc2StartConfig.u32AbsStartTime <= OSAL_ClockGetElapsedTime( ) ) ){
         bStartMyProcess( _oNextProc2StartConfig, _u32NextProcSyncMode );
         _oNextProc2StartConfig.strProcessName = "";
         _u32NextProcSyncMode                  = SPM_U32_SYNC_NONE;
         _bIgnoreSyncBlocks                    = FALSE;
         _oNextProcWaitingBlocks.clear( );
         _oNextProcWaitingSyncBlocks.clear( );


         SPM_NULL_POINTER_CHECK( _poclSyncHandler );
         _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );

      } else if ( _oNextProc2StartConfig.u32AbsStartTime > OSAL_ClockGetElapsedTime( ) ){
         //start timer to trigger checking process to start
         ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Start time not reached, wait for %dms", _oNextProc2StartConfig.u32AbsStartTime - OSAL_ClockGetElapsedTime( ) ) );

         if ( OSAL_s32TimerSetTime( _hStartProcessTimer, _oNextProc2StartConfig.u32AbsStartTime - OSAL_ClockGetElapsedTime( ), 0 ) != OSAL_OK ){
            ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         }
      } else {
         ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Process cannot be started yet!" ) );
      }

   } else if ( !_bStartUpFinished && ( _oRemainingProc2Start.size( ) == 0 )  &&
                (_oSyncPoints.find(SPM_STR_SYNC_POINT_LCMLATE_STARTED) != _oSyncPoints.end() ) ){
        // Sending StartupFinished message to complete the startup when
        // there is no remain process to be started and LCM LATE was already started(also synchronized) with LCM EARLY

      _bStartUpFinished = TRUE;

      SPM_NULL_POINTER_CHECK( _poclStartupSupervisor );
      SPM_NULL_POINTER_CHECK( _poclStartupCommon );
      SPM_NULL_POINTER_CHECK( _poclWorkerServer );

      ETG_TRACE_USR1( ( "spm_tclStartupSystem::vCheckForProcessToStart(): Startup process \"%s\" begins to work in parallel...", _regConfPath.c_str( ) ) );

      //setting the StartupTime taken to complete the startup
      _poclWorkerServer->vSetStartupTime( );
      _poclStartupCommon->vRemoveClient( this );
      _poclStartupSupervisor->vStartupSystemProcessed( );

      SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR(poDbusTmp, spm_ISpmLateServiceHandler);
      if ( poDbusTmp ){
         if ( !poDbusTmp->bStartupReady( ) ){
             ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupSystem::vCheckForProcessToStart() !!!!!! Error detected !!!!!! poDbusTmp->bStartupReady() returns FALSE" ) );
         }
      }
   }

   //check if a one sync block possibly not reachable
   if ( ( _startWaitTime != 0 ) && ( ( OSAL_ClockGetElapsedTime( ) - _startWaitTime ) > _syncWaitTimeout ) ){

      // store in startup investigation
      std::string                    strName = "Sync error: process '" + _oNextProc2StartConfig.strProcessName + "' is waiting for '";

      TSpmSwBlockSet::const_iterator it;

      for ( it = _oNextProcWaitingBlocks.begin( ); it != _oNextProcWaitingBlocks.end( ); ++it ){
         strName += * it + " ";
      }
      for ( it = _oNextProcWaitingSyncBlocks.begin( ); it != _oNextProcWaitingSyncBlocks.end( ); ++it ){
         strName += * it + " ";
      }
      strName += "'";

      if ( _poclStartupInvest ){
         SPM_STARTUPINVEST_INIT_STARTUPITEM
            SPM_STARTUPINVEST_FORCE_ENTRY( "STARTUP", strName )

         _poclStartupInvest->vDump();
      }

      //trigger emtrace entry
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strName.c_str( ), (tU16)strName.length( ), SPM_WRITE_DIRECTLY );

      if ( !_bKeepWaitingAfterTimeout ){
         // start next process with cleared waiting list
         _bIgnoreSyncBlocks = TRUE;

         //and retrigger proc start check
         SPM_NULL_POINTER_CHECK( _poclSyncHandler );
         _poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );
      }

      _startWaitTime   = 0;
      _syncWaitTimeout = SPM_U32_SYNC_WAITING_TIMEOUT;
   }

} // vCheckForProcessToStart

tVoid spm_tclStartupSystem::vStartProcessTimerCallback( tVoid *pArg ){
/*!
  * \fn
  *  \brief
  *    Timer callback.
  *
  *  \param[in] pArg: pointer to startup handler.
  *  \return void
  *  \version
  *    1.0   - Initial
  ******
  */
   spm_tclStartupSystem *poStartupHandler = (spm_tclStartupSystem*)pArg;

   SPM_NULL_POINTER_CHECK( poStartupHandler );
   SPM_NULL_POINTER_CHECK( poStartupHandler->_poclSyncHandler );

   poStartupHandler->_poclSyncHandler->vSwitchContext( SPM_CTX_SWITCH_PROC_CHECK_TRIGGER );
}

std::string spm_tclStartupSystem::strGetListOfBlocks( const TSpmSwBlockSet *poBlock ){

   TSpmSwBlockSet::const_iterator it;
   std::string                    strBlockName;

   for ( it = poBlock->begin( ); it != poBlock->end( ); ++it ){
      strBlockName += it->c_str( );
      strBlockName += " ";
   }
   return( strBlockName );
}

tU32 spm_tclStartupSystem::u32CalculateStartupRegistry(tU32 u32StartupConfig){
   /*!
     * \fn
     *  \brief
     *    Check if configuration has to be updated.
     *    Can be overloaded by project configuration
     *
     *  \param
     *  \return  tU32
     *
     ******
     */

    // no change for base component
    return u32StartupConfig;
}

spm_tclStartupSystem::TOperationResult spm_tclStartupSystem::eCheckNextOperation(std::string strCommad) {
    /*!
      * \fn
      *  \brief
      *    Parser for KDS configuration check via datapool.
      *    example based on AIVI
      *    SDS starten, wenn --> "VoicePresent" || "!DA"
      *    (<SystemConfiguration1.VoiceRecognition> == 2) || (<AIVIVariantCoding.VariantSelection> != 2)
      *
      *    bosch navi starten, wenn --> "!DA" && "!JP" && "!KOR"
      *    (<AIVIVariantCoding.VariantSelection> != 2) && (<VehicleInformation.DestinationRegion1> != 25) && (<VehicleInformation.DestinationRegion1> != 26)
      *
      *    japan navi starten, wenn --> "!DA" && "JP"
      *    (<AIVIVariantCoding.VariantSelection> != 2) && (<VehicleInformation.DestinationRegion1> == 25)
      *
      *    korean navi starten, wenn --> "!DA" && "KOR"
      *    (<AIVIVariantCoding.VariantSelection> != 2) && (<VehicleInformation.DestinationRegion1> == 26)
      *
      *  \param
      *  \return  tU32
      *
      ******
      */

    tU8 KdsResult = 0;
    spm_tclStartupSystem::TOperationResult eResult;

    eResult.bResult = FALSE;
    eResult.eNextOperation = eFailed;
    eResult.strRemaining = "";

    ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): Analyse command: '%s'",strCommad.c_str()));

    // get KDS key
    size_t startPos = strCommad.find("<");
    size_t endPos = strCommad.find(">");
    size_t splitPos = strCommad.find(".");

    if ( (startPos != std::string::npos) && (splitPos != std::string::npos)) {
        startPos += 1;
        std::string strConfigElement = strCommad.substr(startPos, splitPos-startPos);
        std::string strConfigItem    = strCommad.substr(splitPos+1, endPos-splitPos-1);

        ETG_TRACE_USR4(("spm_tclStartupSystem::eCheckNextOperation(): Check for KDS Key: '%40s.%40s'",strConfigElement.c_str(), strConfigItem.c_str()));

        if ( DP_S32_NO_ERR == DP_s32GetConfigItem(strConfigElement.c_str(), strConfigItem.c_str(), &KdsResult, 1) ){
            // now we have the KDS value --> go on parsing command
            ETG_TRACE_USR4(("spm_tclStartupSystem::eCheckNextOperation(): KDS Key: '%d'",KdsResult));
            if (strCommad.length() > endPos+1) {
                std::string strRemainOp = strCommad.c_str()+endPos+1;
                std::string strOperand;
                std::string strValue;

                ETG_TRACE_USR4(("spm_tclStartupSystem::eCheckNextOperation(): Remaining string (%d): '%s'", strRemainOp.size(), strRemainOp.c_str()));

                for(std::string::size_type i = 0; (i < strRemainOp.size()) && (strRemainOp[i] != ')'); ++i) {
                    if ((strRemainOp[i] == '!') || (strRemainOp[i] == '=') || (strRemainOp[i] == '>') || (strRemainOp[i] == '<')) {
                        strOperand.push_back(strRemainOp[i]);
                    }else if ((strRemainOp[i] >= '0') || (strRemainOp[i] >= '9')) {
                        strValue.push_back(strRemainOp[i]);
                    }
                }

                if ((strValue.size() != 0) && (strOperand.size() != 0)) {
                    int iValue = atoi(strValue.c_str());

                    EOperation eOp = eFinal;
                    // now we should have everything to execute command
                    if (0 == strOperand.compare("==")) {
                        eOp = eEqual;
                        if ((int)KdsResult == iValue) eResult.bResult = TRUE;
                    } else if (0 == strOperand.compare("!=")) {
                        eOp = eNotEqual;
                        if ((int)KdsResult != iValue) eResult.bResult = TRUE;
                    } else if (0 == strOperand.compare(">")) {
                        eOp = eBigger;
                        if ((int)KdsResult > iValue) eResult.bResult = TRUE;
                    } else if (0 == strOperand.compare(">=")) {
                        eOp = eBiggerEqual;
                        if ((int)KdsResult >= iValue) eResult.bResult = TRUE;
                    } else if (0 == strOperand.compare("<")) {
                        eOp = eLess;
                        if ((int)KdsResult < iValue) eResult.bResult = TRUE;
                    } else if (0 == strOperand.compare("<=")) {
                        eOp = eLessEqual;
                        if ((int)KdsResult <= iValue) eResult.bResult = TRUE;
                    }

                    ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): Key: '%40s.%40s(%d)', Operand: '%2s(%d)'",
                            strConfigElement.c_str(), strConfigItem.c_str(), KdsResult,
                            strOperand.c_str(), eOp));

                    ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): Value: '%3s(%d), Result: %d'",
                            strValue.c_str(), iValue,
                            eResult.bResult));

                    eResult.eNextOperation = eFinal;

                    //and now check if there is an additional operation to check
                    eResult.strRemaining = "";
                    size_t posAnd = strRemainOp.find("&&");
                    size_t posOr = strRemainOp.find("||");
                    if ( (posAnd != std::string::npos) || (posOr != std::string::npos)) {
                        if (posAnd < posOr) {
                            eResult.eNextOperation = eAnd;
                        } else {
                            eResult.eNextOperation = eOr;
                        }
                        eResult.strRemaining = strRemainOp.c_str()+strRemainOp.find("(");;
                    }

                    ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): Result: '%d', NextOperation: '%d', nextCmd: '%100s'",eResult.bResult, eResult.eNextOperation, eResult.strRemaining.c_str()));
                    return eResult;

                } else {
                    ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): Failed --> remaining string mismatch, no args!"));
                }

            } else {
                ETG_TRACE_USR1( ("spm_tclStartupSystem::eCheckNextOperation(): Failed --> remaining string mismatch!"));
                return eResult;
            }

        } else {
            ETG_TRACE_USR1( ("spm_tclStartupSystem::eCheckNextOperation(): Failed to read KDS value!"));
            return eResult;
        }
    } else {
        ETG_TRACE_USR1(("spm_tclStartupSystem::eCheckNextOperation(): KDS Key not configured!"));
        return eResult;
    }

    return eResult;
}

tBool spm_tclStartupSystem::bGetCondition(std::string strCommad, tBool bStartOnError) {
    tBool bFinalResult = FALSE;

    tBool bFailureDetected    = FALSE;
    EOperation eNextOperation = eOr;

    ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): Start checking condition: %s", strCommad.c_str()));
    TOperationResult cmd = eCheckNextOperation(strCommad);

    bFinalResult = cmd.bResult;

    if (cmd.eNextOperation == eFailed) {
        bFailureDetected = TRUE;
    }

    while ((cmd.eNextOperation != eFinal) && (cmd.eNextOperation != eFailed) ) {
        if (eNextOperation == eAnd) {
            bFinalResult = bFinalResult&&cmd.bResult;
            ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): New result after AND: %d",bFinalResult));
        } else {
            bFinalResult = bFinalResult||cmd.bResult;
            ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): New result after OR: %d",bFinalResult));
        }
        eNextOperation = cmd.eNextOperation;
        cmd = eCheckNextOperation(cmd.strRemaining);

        if (cmd.eNextOperation == eFailed) {
            bFailureDetected = TRUE;
        }
    }

    if (eNextOperation == eAnd) {
        bFinalResult = bFinalResult&&cmd.bResult;
        ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): New result after final AND: %d",bFinalResult));

    } else {
        bFinalResult = bFinalResult||cmd.bResult;
        ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): New result after final OR: %d",bFinalResult));
    }

    if (bFailureDetected && bStartOnError) {
        bFinalResult = TRUE;
        ETG_TRACE_USR1(("spm_tclStartupSystem::bGetCondition(): Start process due to 'StartOnError' --> %d",bFinalResult));
    }
    return bFinalResult;
}

tBool spm_tclStartupSystem::bGetStartCondition(std::string strCommad, tBool bStartOnError) {

    tBool bFinalResult = FALSE;

    size_t startPos = strCommad.find("[");
    size_t endPos = strCommad.find("]");

    if ((startPos != std::string::npos) && (endPos != std::string::npos)) {
        bFinalResult = bGetCondition(strCommad.substr(startPos+1,endPos-startPos), bStartOnError);

        std::string strRemainingCommand = strCommad.substr(endPos+1);
        startPos = strRemainingCommand.find("[");
        endPos = strRemainingCommand.find("]");

        while ((startPos != std::string::npos) && (endPos != std::string::npos)) {
            size_t posAnd = strRemainingCommand.find("&&");
            size_t posOr = strRemainingCommand.find("||");

            if ( (posAnd != std::string::npos) || (posOr != std::string::npos)) {
                tBool bResult = bGetCondition(strRemainingCommand.substr(startPos+1,endPos-startPos), bStartOnError);
                if (posAnd < posOr) {
                    ETG_TRACE_USR1(("spm_tclStartupSystem::bGetStartCondition(): 'AND' "));
                    bFinalResult = bFinalResult && bResult;
                } else {
                    ETG_TRACE_USR1(("spm_tclStartupSystem::bGetStartCondition(): 'OR' "));
                    bFinalResult = bFinalResult || bResult;
                }
            }

            strRemainingCommand = strRemainingCommand.substr(endPos+1);
            startPos = strRemainingCommand.find("[");
            endPos = strRemainingCommand.find("]");
        }
    } else {
        bFinalResult = bGetCondition(strCommad, bStartOnError);
    }

    ETG_TRACE_USR1(("spm_tclStartupSystem::bGetStartCondition(): Final result: %d",bFinalResult));

    return bFinalResult;
}

tBool spm_tclStartupSystem::bCheckStartCondition(TProcConfiguration& oProcConf, tChar *strProcessName) {
    tBool bFinalResult = TRUE;

    ETG_TRACE_USR1(("spm_tclStartupSystem::bCheckStartCondition(): Start checking condition for process: %s",strProcessName));

    if (!oProcConf.strCondition.empty()) {
        bFinalResult = bGetStartCondition(oProcConf.strCondition);
    }

    if (!bFinalResult) {
        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bCheckStartCondition(): disable process '%s' by configuration", strProcessName ) );
    }
    return bFinalResult;
}

tBool spm_tclStartupSystem::bAlternativeProcConfig(const TProcConfiguration& oProcConf, tChar *strProcessName, tBool bValidationOnly) {

    tBool bFinalResult = FALSE;
    tBool bStartOnError = FALSE;
    tBool bLoadSelectorReg = FALSE;

    if ( !oProcConf.strAdditionCondition.empty() ) {
        ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): Addition command detected: '%s'", oProcConf.strAdditionCondition.c_str() ) );
        std::size_t found = oProcConf.strAdditionCondition.find("START_ON_ERROR");
        if (found!=std::string::npos) {
            ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): 'StartOnError' is set") );
            bStartOnError = TRUE;
        }
        found = oProcConf.strAdditionCondition.find("LOAD_SELECTOR_REG");
        if (found!=std::string::npos) {
            ETG_TRACE_USR1( ( "spm_tclStartupSystem::bAlternativeProcConfig(): 'LoadSelectorReg' is set") );
            bLoadSelectorReg = TRUE;
        }
    }

    if (bLoadSelectorReg) {
        vLoadSelectorRegistries(strProcessName);
    }

    if (!oProcConf.strCondition.empty()) {

        ETG_TRACE_USR1(("spm_tclStartupSystem::bAlternativeProcConfig(): Start checking condition for process: %s",strProcessName));

        bFinalResult = bGetStartCondition(oProcConf.strCondition, bStartOnError);

    } else {
        bFinalResult = bAlternativeProcConfig(strProcessName, bValidationOnly);
    }

    if (!bFinalResult) {
        ETG_TRACE_FATAL( ( "spm_tclStartupSystem::bAlternativeProcConfig(): disable process '%s' by configuration", strProcessName ) );
    }
    return bFinalResult;
}

tVoid spm_tclStartupSystem::vLoadSelectorRegistries( tChar *strProcessName) {
    (tVoid)strProcessName;
    ETG_TRACE_USR1(("spm_tclStartupSystem::vLoadSelectorRegistries(): Project method should be overloaded if needed in class 'spm_tclStartupSystemConfig'"));
}


