/*!
  * \file spm_StartupLocal.cpp
  *  \brief
  *    configuration of system startup and process spawn.
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * 1.0 | 27.01.11  | TMS Fischer       |  initial version\n
  * 1.1 | 27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_StartupLocal.h"
#include "spm_ProcessDatabase.h"
#include "spm_IStartupCommon.h" //to get SPM_STARTUP_VALUE_OSAL_START_TYPE
#include "spm_ILateServiceHandler.h"
// interfaces class definitions
#include "spm_factory.h"

#include "spm_IOsalProxy.h"
#include "timeConvert.h"

#include "spm_IWorkerClient.h"
#include "spm_ProcessSupervision.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IOsLinux.h"
// FI-zone starts
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

// spm helper

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_StartupLocal.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#if OSAL_CONF == OSAL_LINUX
#include <sys/stat.h>
#include <sys/dir.h>
#include <sched.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#endif

#include "spm_rootdaemon.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID   SPM_FILE_STARTUPLOCAL


spm_tclStartupLocal::spm_tclStartupLocal( const ISpmFactory& factory,
                                          const std::string& strConfigPath,
                                          const std::string& strInstance )
   : spm_tclStartupSystemConfig( factory, strConfigPath, strInstance, TRUE )
   , _poProcessDatabase( NULL )
   , _poWorkerServer( NULL )
   , _poclProcessSupervision( NULL )
   , _poclCcaSrv( NULL )
   , _fStartUpFinishedConfigSet( FALSE ){
   /*!
     * \fn
     *  \brief
     *    Constructor
     *
     *  \param[in] factory: spm factory object.
     *  \param[in] strConfigPath: config path
     *  \param[in] strInstance: instance number
     ******
     */
}

spm_tclStartupLocal::~spm_tclStartupLocal( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param
  ******
  */
   _poProcessDatabase      = NULL;
   _poclProcessSupervision = NULL;
   _poWorkerServer         = NULL;
   _poclCcaSrv             = NULL;
}

tVoid spm_tclStartupLocal::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method to initialize the dependency of the spm_tclStartupLocal.
  *
  *  \param
  ******
  */
// get all needed references now -> all SPM objects are now available
   spm_tclStartupSystem::vGetReferences( );
   SPM_GET_IF_REFERENCE_USE_VAR( _poProcessDatabase,      ISpmProcessDatabase );
   SPM_GET_IF_REFERENCE_USE_VAR( _poWorkerServer,         ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclProcessSupervision, ISpmProcessSupervision );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaSrv,             ISpmCcaServiceServer );
} // vGetReferences

tVoid spm_tclStartupLocal::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   SPM factory invokes this method after invoking the method - vGetReferences.

  *  \param
  ******
  */
   spm_tclStartupSystem::vStartCommunication( );

   SPM_NULL_POINTER_CHECK( _poWorkerServer );
   _poWorkerServer->vAddClient( this );
}

tBool spm_tclStartupLocal::bProcessSpawn( TProcConfiguration& _oConf ){
/*!
  * \fn
  *  \brief
  *   Method to spawn a given process.

  *  \param[in] _oConf: Process Configuration Information.
  *  \note
  *     If start type of a process in process configuration is "OSAL", then spawn the same using OSAL Process Spawn.
  *     If start type of a process in process configuration is "SYSTEMD", them spawn the same using SYSTEMD interface.\n
  *
  *     This method is implemented for Operating Systems other than WINNT like LINUX.
  *  \todo
  *    Interface to systemd to spawn a process if start type if "SYSTEMD".
  *  \version
  ******
  */

   tBool bRet = FALSE;

   SPM_NULL_POINTER_CHECK_VAL( _poProcessDatabase );
   SPM_NULL_POINTER_CHECK_VAL( _poclProcessSupervision );

   ETG_TRACE_USR4( ( "spm_tclStartupLocal::bProcessSpawn : Process Start Type (Arg=%d) %s ", _oConf.u32ProcTypeArg, _oConf.strProcStartType.c_str( ) ) );

   {
      if ( _poclStartupCommon ){
         TMapProcessConfiguration& oMapProcessConfig = _poclStartupCommon->oGetProcessConfig( );
         for ( TMapProcessConfiguration::iterator posProc = oMapProcessConfig.begin( ); posProc != oMapProcessConfig.end( ); ++posProc ){
            if ( posProc->second.strProcessName == _oConf.strProcessName ){

               OSAL_trTimeDate rCurrentTime = { 0,0,0,0,0,0,0,0,0 };

               SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poOsalProxy, ISpmOsalProxy );
               (tVoid)poOsalProxy->bGetUtcTime( &rCurrentTime );

               rCurrentTime.s32Year           += 1900;

               tclTimeConvert oTimeConv;
               posProc->second.u32StartTimeAbs = oTimeConv.u32GetSeconds( &rCurrentTime );
               posProc->second.u32StartTimeRel = OSAL_ClockGetElapsedTime( );
               posProc->second.bProcSpawned    = TRUE;
            }
         }
      }
   }

   if ( SPM_STARTUP_VALUE_OSAL_START_TYPE == _oConf.strProcStartType ){
      OSAL_trProcessAttribute rAttr;

      ETG_TRACE_USR4( ( "spm_tclStartupLocal::bProcessSpawn(): '%s'", _oConf.strProcessName.c_str( ) ) );

      if ( _oConf.strProcessLocation.empty( ) ){
         // process already running, does not need to be started again
         _oConf.bProcSpawned = TRUE;
         return( TRUE );
      }
      ETG_TRACE_USR4( ( "spm_tclStartupLocal::bProcessSpawn: OSAL START " ) );
      #ifdef SPM_FEATURE_ENABLE_CONFIG_WO_ABS_PATHES
         // check if we need to add default path to process folder
         std::string strPathBuffer;
         if ( std::string::npos != _oConf.strProcessLocation.find( SPM_PROCESS_BASE_PATH ) ){
            strPathBuffer = _oConf.strProcessLocation;
         } else if ( std::string::npos == _oConf.strProcessLocation.find( SPM_PROCESS_LOCAL_PATH ) ){
            ETG_TRACE_USR1( ( "Conversion of process location path needed: '%s'", _oConf.strProcessLocation.c_str( ) ) );
            strPathBuffer = (std::string)SPM_PROCESS_LOCAL_PATH + _oConf.strProcessLocation;
         } else {
            strPathBuffer = _oConf.strProcessLocation;
         }
         _oConf.strProcessLocation = strPathBuffer;
            ETG_TRACE_USR1( ( "Process location is '%s'", _oConf.strProcessLocation.c_str( ) ) );
      #endif


      tBool        bGdbFound = FALSE;
      // try to start gdb script debug
      std::string  strGdbScript;
      strGdbScript = _oConf.strProcessLocation + ".gdb";

      ETG_TRACE_USR1( ( "spm_tclStartupLocal::bProcessSpawn(): lookup gdb trigger file '%s'", strGdbScript.c_str( ) ) );

      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(poOsLinux, ISpmOsLinux);
      if ( poOsLinux->bIsFileInFs( strGdbScript ) ){
         ETG_TRACE_USR1( ( "spm_tclStartupLocal::vProcessSpawn(): try to start '%80s' via gdb", _oConf.strProcessLocation.c_str( ) ) );
         SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", START_PROCESS_VIA_GDB, _oConf.strProcessLocation.c_str( ) );
         bGdbFound = TRUE;
      }
      if ( !bGdbFound ){
         /* start the process */
         rAttr.szName        = &_oConf.strProcessLocation[0];
         rAttr.szAppName     = &_oConf.strProcessLocation[0];
         rAttr.u32Priority   = _oConf.u32ProcPrio;
         rAttr.szCommandLine = 0;  // command line parameter

         ETG_TRACE_USR4( ( "spm_tclStartupLocal::bProcessSpawn(): Spawn process now %d", OSAL_ClockGetElapsedTime( ) ) );

         onPreProcessStart( &rAttr );
         /* Check if a special script to excecute for this process before spawn is performend: */

         _oConf.s32ProcId = (tS32)OSAL_ProcessSpawn( &rAttr );
         if ( OSAL_ERROR != _oConf.s32ProcId ){
            TProcessConfiguration processConfig;
            processConfig.handle   = _oConf.s32ProcId;
            // \todo PROCMON - How should instance,affinity,nicelevel and LINUX_BUILD be handled?
            // processConfig.instance;
            processConfig.instance = _u32Instance;
            #ifdef LINUX_BUILD
               /* Get current cpu affinity (needed for restauration if affinity is changed below)*/
               sched_getaffinity( _oConf.s32ProcId, sizeof( cpu_set_t ), &processConfig.cpuMask );
            #endif

            processConfig.strProcessName  = _oConf.strProcessName;
            processConfig.ProcPrio        = _oConf.u32ProcPrio;
            processConfig.Affinity        = _oConf.u32Affinity;
            processConfig.AffinityEnd     = _oConf.u32AffinityEnd;
            processConfig.NiceLevel       = _oConf.u32NiceLevel;
            processConfig.NiceLevelEnd    = _oConf.u32NiceLevelEnd;
            processConfig.CGroupMemLevel1 = _oConf.u32CGroupMemLevel1;
            processConfig.CGroupMemLevel2 = _oConf.u32CGroupMemLevel2;
            processConfig.startTimeStamp  = OSAL_ClockGetElapsedTime( );
            processConfig.isMissing       = FALSE;

            // m_processDB->_oStartedProcesses[processConfig.handle] = processConfig;
            _poProcessDatabase->bUpdateMapStartedProc( processConfig );

            /* Is nicelevel != default (0), set new nice level*/
            if ( processConfig.NiceLevel != 0 ){
               ETG_TRACE_USR4( ( "slv_nice: Changing nicelevel for process %s", processConfig.strProcessName.c_str( ) ) );
               ETG_TRACE_USR4( ( "slv_nice: New Nicelevel=%d", processConfig.NiceLevel ) );
               bSetNiceLevel( processConfig.NiceLevel, processConfig.handle );

               /* If there was allready some task created before bSetNiceLevel() was executed,
                  we set here also the allready forked tasks to the new nice level:*/
               bRenice( processConfig.NiceLevel, processConfig.handle );
            }

            if ( processConfig.Affinity != 0 ){
                  ETG_TRACE_USR4( ( "slv_cpu: Changing CPU Affinity for process %s", processConfig.strProcessName.c_str( ) ) );
                  ETG_TRACE_USR4( ( "slv_cpu: New cpu affinity=%d", processConfig.Affinity ) );
               #ifdef LINUX_BUILD
                  bSetCpuAffinity( processConfig.Affinity, processConfig.handle, processConfig.cpuMask );
               #endif
            }

            // process now running
            _oConf.bProcSpawned = TRUE;
            bRet                = TRUE;
            onPostProcessStart( &rAttr );
         } else {
            tU32 u32ErrorCode = OSAL_u32ErrorCode( );
            ETG_TRACE_FATAL( ( "spm_tclStartupLocal::bProcessSpawn: %08x (%s)", u32ErrorCode, OSAL_coszErrorText( u32ErrorCode ) ) );
         }
         ETG_TRACE_USR4( ( "spm_tclStartupLocal::bProcessSpawn(): Process running now %d", OSAL_ClockGetElapsedTime( ) ) );
      }
   } else if ( SPM_STARTUP_VALUE_SYSTEMD_START_TYPE == _oConf.strProcStartType ){
      // \todo SYSTEMD --SYSTEMD START PROCESSES
      SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL(poDbusTmp, spm_ISpmLateServiceHandler);
      ETG_TRACE_FATAL( ( "spm_tclStartupLocal::bProcessSpawn(): address %p", poDbusTmp ) );

      if ( poDbusTmp ){
         poDbusTmp->s32StartProcess( _oConf.strProcessLocation.c_str( ), ( tU32 ) spm_fi_tcl_SPM_e32_START_PROC_TYPE::FI_EN_SPM_U32_START_PROC_TYPE_SYSTEMD );
         bRet = TRUE;
      }

   } else if ( "OSALLATE" == _oConf.strProcStartType ){
      SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL(poDbusTmp, spm_ISpmLateServiceHandler);
      ETG_TRACE_FATAL( ( "spm_tclStartupLocal::bProcessSpawn(): address %p", poDbusTmp ) );

      if ( poDbusTmp ){
         poDbusTmp->s32StartProcess( _oConf.strProcessLocation.c_str( ), ( tU32 ) spm_fi_tcl_SPM_e32_START_PROC_TYPE::FI_EN_SPM_U32_START_PROC_TYPE_OSAL_SPAWN );
         bRet = TRUE;
      }

   } else if ( "SYSTEMCTL" == _oConf.strProcStartType ){
      SPM_ROOTDAEMON_CALLER_rPerformRootOp( "lcm", SYSTEMCTL_START_UNIT, _oConf.strProcessLocation.c_str( ) );
      bRet = TRUE;
   } else if ( SPM_STARTUP_VALUE_SYSTEM_RECOVERY_START_TYPE == _oConf.strProcStartType ){
      bRet = _poclWorkerServer->bPostMessageToWorker( SPM_U32_SYSTEM_RECOVERY, 0 );
   } else if ( SPM_STARTUP_VALUE_SUPERVISION_START_TYPE == _oConf.strProcStartType ){
      ETG_TRACE_USR1( ( "Process to supervise=%s ", _oConf.strProcessLocation.c_str( ) ) );
      _poclProcessSupervision->vRememberAdditionalProzessToSupervise( _oConf.strProcessLocation, _oConf.u32ProcTypeArg );
      bRet = TRUE;
   }

   return( bRet );
} // bProcessSpawn

tVoid spm_tclStartupLocal::vRemoveProcess( TProcConfiguration& oProcConf ){
/*!
  * \fn
  *  \brief
  *   Method to remove a given process.
  *
  *  \param[in] oProcConf: Process Configuration Information.
  ******
  */
   OSAL_tEventHandle hShutdown;
   OSAL_tEventMask   hEvRequest = 0x00000001;

         ETG_TRACE_USR4( ( "vRemoveProcess(): Now remove process '%s'", oProcConf.strProcessName.c_str( ) ) );

   if ( OSAL_s32EventOpen( oProcConf.strProcShutdownEvent.c_str( ), &hShutdown ) == OSAL_OK ){
         ETG_TRACE_USR4( ( "vRemoveProcess(): send shutdown event to terminate process '%s'",
                        oProcConf.strProcShutdownEvent.c_str( ) ) );
      if ( OSAL_s32EventPost( hShutdown, hEvRequest, OSAL_EN_EVENTMASK_OR ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!, Could not post ShutdownEvent '%s'",
                             oProcConf.strProcShutdownEvent.c_str( ) ) );
      }
      if ( OSAL_s32EventClose( hShutdown ) != OSAL_OK ){
         ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!, Could not close ShutdownEvent '%s'",
                             oProcConf.strProcShutdownEvent.c_str( ) ) );
      }
         OSAL_s32ThreadWait( 200 );
      /* Event should now be deleted: */
      if ( OSAL_s32EventOpen( oProcConf.strProcShutdownEvent.c_str( ), &hShutdown ) == OSAL_OK ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         if ( OSAL_s32EventClose( hShutdown ) != OSAL_OK ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         }
      }
   } else {
      ETG_TRACE_ERRMEM( ( "SPM: !!!!!! Error detected !!!!!!, Could not open Shutdown event '%s'",
                          oProcConf.strProcShutdownEvent.c_str( ) ) );
   }
} // vRemoveProcess

tVoid spm_tclStartupLocal::vSetStartUpFinishedProcessConfiguration( tU32 instance ){
   TMapStartedProcesses::iterator it;

   SPM_NULL_POINTER_CHECK( _poProcessDatabase );

   TMapStartedProcesses         & oStartedProcesses = _poProcessDatabase->mapGetStartedProcesses( );

   for ( it = oStartedProcesses.begin( ); it != oStartedProcesses.end( ); ++it ){
      TProcessConfiguration processConfig = ( * it ).second;
      if ( processConfig.instance == instance ){
         if ( processConfig.NiceLevel != processConfig.NiceLevelEnd ){
            ETG_TRACE_USR4( ( "spm_tclStartupLocal::vSetStartUpFinishedProcessConfiguration(%d)", instance ) );

            /* renice to NiceLevelEnd:
               renice processConfig.NiceLevelEnd -p processConfig.handle
              */
            ETG_TRACE_USR4( ( "slv: Renice for process %s after startup finished", processConfig.strProcessName.c_str( ) ) );
            ETG_TRACE_USR4( ( "slv: Renice to %d", processConfig.NiceLevelEnd ) );
            bRenice( processConfig.NiceLevelEnd, processConfig.handle );
         } else {
            // ETG_TRACE_USR4(("slv: No renice for process %s (hdl=%d) after startup finished", processConfig.strProcessName.c_str(), processConfig.handle));
         }

         if ( processConfig.Affinity != processConfig.AffinityEnd ){
               ETG_TRACE_USR4( ( "slv: Changing CPU affinity for process %s after startup finished", processConfig.strProcessName.c_str( ) ) );
               ETG_TRACE_USR4( ( "slv: New affinity: %d", processConfig.AffinityEnd ) );
            #ifdef LINUX_BUILD
               bSetCpuAffinity( processConfig.AffinityEnd, processConfig.handle, processConfig.cpuMask );
            #endif
         }
      }
   }
   _fStartUpFinishedConfigSet = TRUE;
} // vSetStartUpFinishedProcessConfiguration

tBool spm_tclStartupLocal::bRenice( tS32 s32NewNiceLevel,
                                    tS32 s32handle ){
   /*!
     * \fn
     *  \brief
     *    Changes the nice of process to new given nice value.
     *
     *  \param[in] s32NewNiceLevel : new nice value.
     *  \param[in]  s32handle : handle of the process (process id).
     *  \return  FALSE if process was not found otherwise TRUE.
     *
     ******
     */
   std::string strScriptCmdLine = "NiceLevel=" + std::to_string(s32NewNiceLevel) + ",hdl=" + std::to_string(s32handle);

   //OSAL_s32PrintFormat(strScriptCmdLine, "NiceLevel=%d,hdl=%d", s32NewNiceLevel, s32handle);
               ETG_TRACE_USR4( ( "bRenice(%s)", strScriptCmdLine.c_str( ) ) );

   strScriptCmdLine = "/proc/" + std::to_string(s32handle) + "/task";
   //OSAL_s32PrintFormat(strScriptCmdLine, "/proc/%d/task", s32handle);
               ETG_TRACE_USR4( ( "ReniceCmdLine=%s", strScriptCmdLine.c_str( ) ) );

   DIR           *dp;
   struct dirent *dirp;
   if ( ( dp = opendir( strScriptCmdLine.c_str( ) ) ) == NULL ){
               ETG_TRACE_USR4( ( "Error opening dir=%s", strScriptCmdLine.c_str( ) ) );
      return( FALSE );
   }

   while ( ( dirp = readdir( dp ) ) != NULL ){
      if ( dirp->d_type == DT_DIR ){
         if ( strcmp( dirp->d_name, "." ) && strcmp( dirp->d_name, ".." ) ){
            int oldp = getpriority( PRIO_PROCESS, (id_t)atol( dirp->d_name ) );
            if ( setpriority( PRIO_PROCESS, (id_t)atol( dirp->d_name ), s32NewNiceLevel ) == 0 ){
               //OSAL_s32PrintFormat(&strScriptCmdLine[0], "Changed /proc/%d/%d from prio %d to %d", s32handle, atol(dirp->d_name), oldp, s32NewNiceLevel);
               strScriptCmdLine = "Changed /proc/" + std::to_string(s32handle) + "/" + std::to_string(atol( dirp->d_name )) + " from prio " + std::to_string(oldp) + " to " + std::to_string(s32NewNiceLevel);
               ETG_TRACE_USR4( ( "ReniceCmdLine=%s", strScriptCmdLine.c_str( ) ) );
            }
         }
      }
   }
   closedir( dp );

   return( TRUE );
} // bRenice

tBool spm_tclStartupLocal::bSetCpuAffinity( tS32      s32NewCpuAffinity,
                                            tS32      s32handle,
                                            cpu_set_t cpuMask ){
   /*!
     * \fn
     *  \brief
     *    Set the cpu affinity of process
        By setting cpu affinity process is eligible to run on only those cpus.
     *
     *  \param[in] s32NewCpuAffinity : cpu affinity.
     *  \param[in] s32handle:          process handle or id.
     *  \param[in] cpuMask:            set of cpus to set affinity.
     *  \return  always return TRUE.
     *
     ******
     */
// #define BINDCPU0 "bindcpu0"
// #define BINDCPU1 "bindcpu1"
   #define CPU0 0
   #define CPU1 1

   int         err;
   cpu_set_t   newCpuSet;
   std::string strScriptCmdLine;

   strScriptCmdLine = "Affinity=" + std::to_string( s32NewCpuAffinity ) + ",hdl=" + std::to_string( s32handle ) + " (CPU_COUNT=" + std::to_string( CPU_COUNT( &cpuMask ) ) + ")";
   ETG_TRACE_USR4( ( "bSetCpuAffinity(%s)", strScriptCmdLine.c_str( ) ) );

   CPU_ZERO( &newCpuSet );           /* clear all CPUs*/
   switch ( s32NewCpuAffinity ){
      case 1:
         CPU_SET( CPU0, &newCpuSet );   /*allow core #0*/
         break;

      case 2:
         CPU_SET( CPU1, &newCpuSet );   /*allow core #1*/
         break;

      case 0:
      default:
         newCpuSet = cpuMask;
         break;
   }

   err = sched_setaffinity( s32handle, sizeof( cpu_set_t ), &newCpuSet );
   if ( err ){
      ETG_TRACE_FATAL( ( "sched_setaffinity() for %s returns with ERROR!", strScriptCmdLine.c_str( ) ) );
      /* \todo*/
   }
   return( TRUE );
} // bSetCpuAffinity

tBool spm_tclStartupLocal::bSetNiceLevel( tS32 s32NewNiceLevel,
                                          tS32 s32handle ){
   /*!
     * \fn
     *  \brief
     *    Set the nice value of the process.
     *
     *  \param[in] s32NewNiceLevel: new nice value.
     *  \param[in] s32handle:       process id.
     *  \return  always return TRUE.
     *
     ******
     */

   int         err;

   std::string strScriptCmdLine;

   strScriptCmdLine = "NiceLevel=" + std::to_string(s32NewNiceLevel) + ",hdl=" + std::to_string(s32handle);
   ETG_TRACE_USR4( ( "bSetNiceLevel(%s)", strScriptCmdLine.c_str( ) ) );

   err              = setpriority( PRIO_PROCESS, s32handle, s32NewNiceLevel );
   if ( err ){
      ETG_TRACE_FATAL( ( "bSetNiceLevel() for %s returns with ERROR!", strScriptCmdLine.c_str( ) ) );

   }
   return( TRUE );
} // bSetNiceLevel

tVoid spm_tclStartupLocal::vHandleMessage( tU32 u32Message,
                                           tU32 u32Parm ){
      ETG_TRACE_FATAL( ( "spm_tclStartupLocal::vHandleMessage(%d)", u32Message ) );
   switch ( u32Message ){
      default:
         break;
   } //lint !e764: switch  statement does not have a case
   u32Parm = u32Parm;
}

tBool spm_tclStartupLocal::bHandleSynchrounousCall( tU32   u32Message,
                                                    tVoid *args ){
   (tVoid)args;
            ETG_TRACE_USR4( ( "spm_tclStartupLocal::bHandleSynchrounousCall(%d)", u32Message ) );
   switch ( u32Message ){
      case SPM_U32_WORKER_BROADCAST_STARTUP_DELAY:
            ETG_TRACE_USR4( ( "spm_tclStartupLocal::bHandleSynchrounousCall(SPM_U32_WORKER_BROADCAST_STARTUP_DELAY)" ) );
         if ( !_fStartUpFinishedConfigSet ){
            ETG_TRACE_USR4( ( "StartUp finished config not set yet: Calling vSetStartUpFinishedProcessConfiguration(%d)", _u32Instance ) );
            vSetStartUpFinishedProcessConfiguration( _u32Instance );
            #ifndef LCM_UNIT_TESTS
               // \todo poj2hi

               /* Not sure if startup finished should also set in time out case, but for the LCM the startup
                  has finished yet...*/
               spm_corefi_tclMsgStartupFinishedStatus startUpFinishedStatus;
               startUpFinishedStatus.StartupFinishedState = 1;
               _poclCcaSrv->vUpdateProperty( SPM_COREFI_C_U16_STARTUPFINISHED, &startUpFinishedStatus );
            #endif

         }
         break;

      case SPM_U32_WORKER_BROADCAST_STARTUP_ENDED:
            ETG_TRACE_USR4( ( "spm_tclStartupLocal::bHandleSynchrounousCall(SPM_U32_WORKER_BROADCAST_STARTUP_ENDED)" ) );
         if ( !_fStartUpFinishedConfigSet ){
            ETG_TRACE_USR4( ( "StartUp finished: vSetStartUpFinishedProcessConfiguration(%d)", _u32Instance ) );
            vSetStartUpFinishedProcessConfiguration( _u32Instance );
            #ifndef LCM_UNIT_TESTS
               // \todo poj2hi
               spm_corefi_tclMsgStartupFinishedStatus startUpFinishedStatus;
               startUpFinishedStatus.StartupFinishedState = 1;
               _poclCcaSrv->vUpdateProperty( SPM_COREFI_C_U16_STARTUPFINISHED, &startUpFinishedStatus );
            #endif
         }
         break;

      default:
         break;
   } // switch
   return( FALSE );
}    // bHandleSynchrounousCall

