/*!
  * \file spm_StartupInvestigation.cpp
  *  \brief
  *    Gathers all information during startup
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 11.03.11  | TMS Fischer       | initial version
  * 27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */
#include <sstream>
#include <iomanip>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_StartupInvestigation.h"

// interfaces class definitions

// spm helper
#include "spm_CriticalSection.h"
#include "spm_IFactory.h"
#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"
#include "spm_ISystemPowerManager.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_StartupInvestigation.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID   SPM_FILE_STARTUPINVESTIGATION

spm_tclStartupInvestigation::spm_tclStartupInvestigation( const std::string& strSemname,
                                                          const ISpmFactory& factory ) : ISpmStartupInvestigationServer( factory )
   , _hSem(OSAL_C_INVALID_HANDLE)
   , _strSemname( strSemname )
   , _u32TmrOffset( 0 )
   , _bStartupPerformed( FALSE ){
   /*!
     * \fn
     *  \brief
     *    Constructor
     *
     *  \param[in] strSemname: semaphore name.
     *  \param[in] factory: spm factory object.
     ******
     */
   ETG_TRACE_USR4( ( "Entered constructor spm_tclStartupInvestigation" ) );

   if( OSAL_s32SemaphoreCreate( strSemname.c_str( ),
                                &_hSem,
                                1 )  != OSAL_OK ) {
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupInvestigation !!!!!! Error detected !!!!!! cannot create Semaphore: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
  }

   _u32TmrOffset          = 0;
   _bStartupPerformed     = FALSE;
   _bShutdownInvestActive = FALSE;
}

spm_tclStartupInvestigation::~spm_tclStartupInvestigation( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param
  ******
  */
   OSAL_s32SemaphoreClose( _hSem );
   OSAL_s32SemaphoreDelete( _strSemname.c_str( ) );
}

tVoid spm_tclStartupInvestigation::vAddStartupItem( const spm_tclStartupItem& oStartupItem ){
   if ( _bStartupPerformed ){
      return;
      // protect against parallel access, release is done by destructor
   }
   spm_tclHandleSemaphore crit( _hSem );

   // tChar buf[SPM_INTERNAL_BUFFER_LENGTH];
   // OSALUTIL_s32SaveNPrintFormat(buf, sizeof(buf), "vAddStartupItem: %s, %s", oStartupItem._context.c_str(), oStartupItem._name.c_str());
   // ETG_TRACE_USR4(("%s",buf));

   _items.insert( std::pair < tU32, spm_tclStartupItem >( OSAL_ClockGetElapsedTime( ),
                                                          oStartupItem ) );
}

// private class to be used with the stl find_if algorithm to find a specific item
// in the array.
class find_op
{
public:
find_op( tU32 u32AppId ){ _u32AppId = u32AppId; }

bool operator()( const TStartupInvestigationItem& op ) const
{ return( op._u32AppId == _u32AppId ); }

private:
tU32 _u32AppId;
};


tVoid spm_tclStartupInvestigation::vUpdateApplicationItem( tU32             u32AppId,
                                                           tenSpmStartupReq enReqState,
                                                           tU32             u32RequestedState ){
   if ( _bStartupPerformed ){
      /* Check for SHUTDOWN investigation entrys to made. */
      if (
         _bShutdownInvestActive
         && ( u32RequestedState == AMT_C_U32_STATE_OFF )
         && ( ( enReqState == SPM_STARTUP_APPLICATION_STATEREQ )
              || ( enReqState == SPM_STARTUP_APPLICATION_STATECONF )
              )
         ){
         spm_tclHandleSemaphore crit( _hSem );

         // tChar buf[SPM_INTERNAL_BUFFER_LENGTH];
         // OSALUTIL_s32SaveNPrintFormat(buf, sizeof(buf),"vUpdateApplicationItem OFF-State: AppId=%d, enReqState=%d, %d, %d", u16AppId, enReqState );
         // TRACE_SPM_INFO_STRING(buf);

         std::vector < TStartupInvestigationItem >::iterator it;
         find_op                                             op( u32AppId );
         it = std::find_if( _apps.begin( ), _apps.end( ), op );

         if ( it != _apps.end( ) ){
            /* Only this two states are importent for shutdown investigation.  */
            if ( enReqState == SPM_STARTUP_APPLICATION_STATEREQ ){
               it->_u32RequestedState                                 = u32RequestedState;
               it->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time = OSAL_ClockGetElapsedTime( );
            } else if ( enReqState == SPM_STARTUP_APPLICATION_STATECONF ){
               it->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time = it->_u32RequestedState == u32RequestedState ? OSAL_ClockGetElapsedTime( ) : 0xffffffff;
            }
         } else {
            ETG_TRACE_ERRMEM( ( "vUpdateApplicationItem() Shutdown error. Unknown application (AppId=%d) was detected!", (tUInt)u32AppId ) );
         }
      }
   } else {
      // protect against parallel access, release is done by destructor
      spm_tclHandleSemaphore                              crit( _hSem );

      // tChar buf[SPM_INTERNAL_BUFFER_LENGTH];
      // OSALUTIL_s32SaveNPrintFormat(buf, sizeof(buf),"vUpdateApplicationItem: %d, %d, %d, %d", u16AppType, u32AppId, enReqState, u32TimeIdx);
      // ETG_TRACE_USR4(("%s",buf));
      std::vector < TStartupInvestigationItem >::iterator it;
      find_op                                             op( u32AppId );

      it = std::find_if( _apps.begin( ), _apps.end( ), op );
      if ( it != _apps.end( ) ){
         // item already available so update changed values
         if ( ( enReqState == SPM_STARTUP_APPLICATION_STATEREQ )
              || ( enReqState == SPM_STARTUP_APPLICATION_STATECONF ) ){
            if ( it->_times[enReqState]._u32Time == 0 ){  // not already set?
               // first contact
               if ( enReqState == SPM_STARTUP_APPLICATION_STATEREQ ){
                  it->_u32RequestedState          = u32RequestedState;
                  it->_times[enReqState]._u32Time = OSAL_ClockGetElapsedTime( );
               } else {
                  // state configuration
                  // if wrong answer from component then set time to (unsigned) -1
                  it->_times[enReqState]._u32Time = it->_u32RequestedState == u32RequestedState ? OSAL_ClockGetElapsedTime( ) : 0xffffffff;
               }
            }
         } else {
            it->_times[enReqState]._u32Time = OSAL_ClockGetElapsedTime( );
         }
      } else {
         // add a new item to the investigation array
         TStartupInvestigationItem item;

         item._u32AppId                   = u32AppId;
         item._times[enReqState]._u32Time = OSAL_ClockGetElapsedTime( );
         item._u32RequestedState          = u32RequestedState == SPM_REJECTED_REQUESTED_STATE ? SPM_REJECTED_REQUESTED_STATE : 0; // default

         _apps.push_back( item );
      }
   }

} // vUpdateApplicationItem

tVoid spm_tclStartupInvestigation::vTraceItems( ){

   std::multimap < tU32, spm_tclStartupItem >::const_iterator it;
   tU32                                                       u32LastTime = 0;

      ETG_TRACE_FATAL( ( "Startup investigation (timer difference: %dms)", _u32TmrOffset ) );
   for ( it = _items.begin( ); it != _items.end( ); ++it ){
      std::string strOutput = "            " + it->second._context;
      strOutput = strOutput.substr( strOutput.length( ) - 12, 12 );
      ETG_TRACE_FATAL( ( "Startup investigation: time %6d (osal: %6d) [diff: %4d], (%12s) %s",
                         it->first + _u32TmrOffset,
                         it->first,
                         it->first + _u32TmrOffset - u32LastTime,
                         strOutput.c_str( ),
                         it->second._name.c_str( )
                         ) );
      u32LastTime = it->first + _u32TmrOffset;
   }

   TVectorInvestigation::iterator it2;

   for ( it2 = _apps.begin( ); it2 != _apps.end( ); ++it2 ){

      SPM_ETG_CREATE_FORMAT_STRING
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 1 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 2 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 3 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 4 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 5 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 6 )
      SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 7 )

      etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM, TR_LEVEL_FATAL ), 0, 0, u32Format,
                        I_SPM_03 /* param1 */,
                        it2->_u32AppId /* param2 */,
                        it2->_u32RequestedState /* param3 */,
                        ( ( it2->_times[SPM_STARTUP_APPLICATION_STARTREQ]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STARTREQ]._u32Time + _u32TmrOffset ) : 0 )       /*
                                                                                                                                                                                     param4
                                                                                                                                                                                    */
                        ,
                        ( ( it2->_times[SPM_STARTUP_APPLICATION_STARTCONF]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STARTCONF]._u32Time + _u32TmrOffset ) : 0 )     /*
                                                                                                                                                                                     param5
                                                                                                                                                                                    */
                        ,
                        ( ( it2->_times[SPM_STARTUP_APPLICATION_INITIALIZED]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_INITIALIZED]._u32Time + _u32TmrOffset ) : 0 ) /*
                                                                                                                                                                                     param6
                                                                                                                                                                                    */
                        ,
                        ( ( it2->_times[SPM_STARTUP_APPLICATION_STATEREQ]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STATEREQ]._u32Time + _u32TmrOffset ) : 0 )       /*
                                                                                                                                                                                     param7
                                                                                                                                                                                    */
                        ,
                        ( ( it2->_times[SPM_STARTUP_APPLICATION_STATECONF]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STATECONF]._u32Time + _u32TmrOffset ) : 0 ) )   /*
                                                                                                                                                                                     param8
                                                                                                                                                                                    */
      ;
   }

} // vTraceItems

tVoid spm_tclStartupInvestigation::vTraceShutdownItems( ){

   if ( _bShutdownInvestActive ){
      ETG_TRACE_FATAL( ( "Shutdown investigation (timer difference: %dms)", _u32TmrOffset ) );

      TVectorInvestigation::iterator it2;

      for ( it2 = _apps.begin( ); it2 != _apps.end( ); ++it2 ){
         if ( it2->_u32RequestedState != SPM_REJECTED_REQUESTED_STATE ){

            SPM_ETG_CREATE_FORMAT_STRING
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T16, 1 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 2 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T16, 3 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T16, 4 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 5 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 6 )
            SPM_ETG_ADD_NEW_FORMAT_TO_FORMAT_STRING( SPM_ETG_FORMAT_VALUE_EN_T32, 7 )
            etg_vTraceBinary( SPM_CALC_ETG_TRACE_CLASS( SPM_TRACE_CLASS_SPM, TR_LEVEL_FATAL ), 0, 0, u32Format,
                              I_SPM_04 /* param1 */,
                              0 /* param2 */,                                                                                                                                                                                             //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          // apptype,
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          // remove
                              it2->_u32AppId /* param3 */,
                              0 /* param4 */,                                                                                                                                                                                             //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          // apptype,
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          //
                                                                                                                                                                                                                                          // remove
                              it2->_u32RequestedState /* param5 */,
                              ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time > 0 ) ? ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time + _u32TmrOffset ) : 0 )                                                       /*
                                                                                                                                                                                                                                             param6
                                                                                                                                                                                                                                            */
                              ,
                              ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time > 0 ) ? ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time + _u32TmrOffset ) : 0 )                                                     /*
                                                                                                                                                                                                                                             param7
                                                                                                                                                                                                                                            */
                              ,
                              ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time > 0 ) ? ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time ) - ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time ) ) : 0 ) ) /*
                                                                                                                                                                                                                                             param8
                                                                                                                                                                                                                                            */
            ;


         } // if (it2->_u32RequestedState != SPM_REJECTED_REQUESTED_STATE )
      }    // for
      ETG_TRACE_FATAL( ( "-------- SHUTDOWN INVESTIGATION END ------------" ) );
      OSAL_s32ThreadWait( 250 );
   } // if (_bShutdownInvestActive)

}       // vTraceShutdownItems

tVoid spm_tclStartupInvestigation::vDump( ){
   /*!
     * \fn
     *  \brief
     *    Dumps application startup-Timing into error memory.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   spm_tclHandleSemaphore                                     crit( _hSem );

   std::string                                                strBuffer;

   std::multimap < tU32, spm_tclStartupItem >::const_iterator it;

      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );

   strBuffer = "Startup investigation:";
   poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

   for ( it = _items.begin( ); it != _items.end( ); ++it ){
      std::string strOutput = "            " + it->second._context;
      strOutput = strOutput.substr( strOutput.length( ) - 12, 12 );
      std::stringstream sstreamStartupInv;
      sstreamStartupInv << std::setw( 6 ) << std::dec << std::to_string( it->first + _u32TmrOffset ) << ", (" << std::setw(12) << strOutput << ") " << it->second._name;
      strBuffer =  sstreamStartupInv.str( );
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
   }

   TVectorInvestigation::iterator it2;

   for ( it2 = _apps.begin( ); it2 != _apps.end( ); ++it2 ){
      tU32 u32StartReq = ( ( it2->_times[SPM_STARTUP_APPLICATION_STARTREQ]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STARTREQ]._u32Time + _u32TmrOffset ) : 0 );
      tU32 u32StartConf = ( ( it2->_times[SPM_STARTUP_APPLICATION_STARTCONF]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STARTCONF]._u32Time + _u32TmrOffset ) : 0 );
      tU32 u32Init = ( ( it2->_times[SPM_STARTUP_APPLICATION_INITIALIZED]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_INITIALIZED]._u32Time + _u32TmrOffset ) : 0 );
      tU32 u32StarterReq = ( ( it2->_times[SPM_STARTUP_APPLICATION_STATEREQ]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STATEREQ]._u32Time + _u32TmrOffset ) : 0 );
      tU32 u32StateConf = ( ( it2->_times[SPM_STARTUP_APPLICATION_STATECONF]._u32Time > 0 ) ? ( it2->_times[SPM_STARTUP_APPLICATION_STATECONF]._u32Time + _u32TmrOffset ) : 0 );
      std::stringstream sstreamStartupInvestigation;
      sstreamStartupInvestigation << "AppId=" << std::setw( 8 ) << std::hex << std::to_string( it2->_u32AppId ) << " ";
      sstreamStartupInvestigation << "ReqState=" << std::setw( 8 ) << std::hex << std::to_string( it2->_u32RequestedState ) << " ";
      sstreamStartupInvestigation << "SR:" << std::setw( 8 ) << std::dec << std::to_string( u32StartReq ) << " ";
      sstreamStartupInvestigation << "SC:" << std::setw( 8 ) << std::dec << std::to_string( u32StartConf ) << " ";
      sstreamStartupInvestigation << "INIT:" << std::setw( 8 ) << std::dec << std::to_string( u32Init ) << " ";
      sstreamStartupInvestigation << "STR:" << std::setw( 8 ) << std::dec << std::to_string( u32StarterReq ) << " ";
      sstreamStartupInvestigation << "STC:" << std::setw( 8 ) << std::dec << std::to_string( u32StateConf );
      strBuffer = sstreamStartupInvestigation.str( );

      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

   }
} // vDump

tVoid spm_tclStartupInvestigation::vDumpShutdownInvestigation( ){
   /*!
     * \fn
     *  \brief
     *    Dumps application information into error memory if shutdown investigation is active.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   if ( _bShutdownInvestActive ){
      spm_tclHandleSemaphore crit( _hSem );

      std::string            strBuffer;
      SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );

      strBuffer = "Shutdown investigation:";
      poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );

      TVectorInvestigation::iterator it2;

      for ( it2 = _apps.begin( ); it2 != _apps.end( ); ++it2 ){
         std::stringstream sstreamDumpShutdownInvestigation;
         tU32 u32AppStateReq = ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time > 0 ) ? ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATEREQ]._u32Time + _u32TmrOffset ) : 0 );
         tU32 u32AppStateConf = ( ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time > 0 ) ? ( it2->_times[SPM_SHUTDOWN_APPLICATION_STATECONF]._u32Time + _u32TmrOffset ) : 0 );
         sstreamDumpShutdownInvestigation << "AppId=" << std::setw( 8 ) << std::to_string( it2->_u32AppId ) << " ";
         sstreamDumpShutdownInvestigation << "ReqState=" << std::setw( 2 ) << std::to_string( it2->_u32RequestedState ) << " ";
         sstreamDumpShutdownInvestigation << "STR=" << std::setw( 8 ) << std::to_string( u32AppStateReq ) << " ";
         sstreamDumpShutdownInvestigation << "STC=" << std::setw( 8 ) << std::to_string( u32AppStateConf );
         strBuffer = sstreamDumpShutdownInvestigation.str( );
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str( ), (tU16)strBuffer.length( ) );
      }
   }

} // vDumpShutdownInvestigation

