/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_StartupCommon.h
  * @brief        This is header file contains functions for StartupCommon (start-up configuration).
  *               also for add start-up system and update registry to valid variant.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <set>
#include <list>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_GlobDefs.h"
#include "spm_IStartupCommon.h"

class ISpmApplicationDatabase;
class ISpmCcaSupplierServer;
class ISpmStartupSystem;
class ISpmStartupSystemVariant;
class ISpmRegistry;

typedef std::list < ISpmStartupSystem* > TListStartup;
// //////////////////////////////////////////////////////////////////////////////
// class definition spm_tclStartupCommon

/*!
  *  \class spm_tclStartupCommon
  *  \brief Common part of the startup process(es).Reads out the process configurations of every process.
  *
  */
class spm_tclStartupCommon : public ISpmStartupCommon
{
public:
spm_tclStartupCommon( tU16               u16SpmId,
                      const ISpmFactory& factory );
virtual ~spm_tclStartupCommon( );

tVoid vSetStartupSystemVariant( ISpmStartupSystemVariant *variant ){ _variant = variant; }

virtual tVoid vAddStartupSystem( ISpmStartupSystem *variant ){ _oListStartup.push_back( variant ); }

virtual tBool bStartProcess( const std::string& strProcessName );

virtual tBool bStartProcess( TProcConfiguration& /*_oConf*/,
                             tU32                /*u32HighPrioSync*/ ){ return( FALSE ); } //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used
virtual tBool bStopProcess( const std::string& strProcessName );

virtual tBool bForceProcessStop( const std::string& strProcessName );

virtual tU32 u32GetSupervisionMode( tU32 ProcId );

virtual std::string strGetProcName( tU32 u32ProcId );

virtual tU32 u32GetProcId( const std::string& strName );

virtual void vSetProcId( const std::string& strName,
                         const tU32         PID );

virtual tBool bAddSwBlock( const std::string& strSwBlockName,
                           const tBool        bBlockMode = FALSE,
                           const tBool        bLocal = TRUE,
                           const std::string& strExecPath = "",
                           const std::string& strProcessName = "",
                           const std::string& strStartType = SPM_STARTUP_VALUE_OSAL_START_TYPE );

virtual tBool bAddAdditionalSwBlock( const std::string& strBusShortName,
                                     const std::string& strServiceName,
                                     const tU32         u32AppId,
                                     const std::string& strStartType = SPM_STARTUP_VALUE_OSAL_START_TYPE );

virtual tVoid vRemoveSwBlock( const std::string& strSwBlockName,
                              tBool              bStopProcWithLastBlock = TRUE );

virtual tBool bRemoveSwBlockFromProcessList( const std::string& strSwBlockName,
                                             std::string      & strProcessName );

virtual tVoid vRemoveStoppedService( const std::string& strServiceName );

virtual tVoid vUpdateStartedService( const std::string& strServiceName );

// virtual tVoid vRemoveProcess(const std::string& strName);

// implementation of interface ISpmSwBlockClient
virtual void vOnSwBlockLoaded( const TSpmSwBlockSet& SwBlocksLoaded );

virtual void vOnSwBlockConnected( const TSpmSwBlockSet& SwBlocksConnected );

virtual void vOnSwBlockUp( const TSpmSwBlockSet& SwBlocksUp );

virtual void vOnSyncPointReached( const std::string& strSyncPointName );

virtual void vOnSwBlockForced( const TSpmSwBlockSet& SwBlocksForced );

virtual tVoid vAddClient( ISpmSwBlockClient *clt );

virtual tVoid vRemoveClient( ISpmSwBlockClient *clt );

virtual std::string strGetProcLocationByBlockName( const std::string& strBlockName );

virtual std::string strGetProcNameByBlockName( const std::string& strBlockName );

virtual tVoid vReadBaseRegistry( );

virtual tVoid vReadStartupConfiguration( );

virtual tVoid vReadEarlyRegistries( );

virtual tVoid vReadRegistries( );

virtual tVoid vEvaluateProcessStartup( );

virtual tVoid vSetSwBlockServer( ISpmApplicationDatabase */*pDatabase*/,
                                 ISpmCcaSupplierServer   */*pSupplier*/,
                                 ISpmSwBlockServer       */*pServer*/ ){ } //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual tBool bIsProcessStarted( const std::string& strProcessname ) const;

virtual tVoid vSetProcessStarted( const std::string& strProcessname,
                                  tBool              bIsStarted );

virtual tVoid vReadRegistries( const TMapRegistries& registry );

virtual tVoid vEvaluateRegistry( const std::string& strConf,
                                 TMapRegistries   & registry ) const;

virtual tVoid vUpdateRegistryDueToValidVariant( ) const;

virtual TMapProcessConfiguration& oGetProcessConfig( ){ return( _oMapProcessConfig ); }

virtual tVoid vTraceProcessConfiguration( );

TMapProcessConfiguration _oMapProcessConfig;

protected:
virtual tVoid vEvaluateProcessConfiguration( );

virtual std::string sDetermineStartupRegistry( std::string strRegistryName );

tVoid vReadEarlyProcessRegistries( );

private:
tU16                            _u16SpmId;
OSAL_tSemHandle                 _hSemHandle;
std::string                     _strSemName;

tVoid vDeactivateProcesses( const std::set < std::string >& procs );


TListStartup                    _oListStartup;
std::list < std::string >       _oListPid0Services;

std::set < ISpmSwBlockClient* > _clients;

ISpmSwBlockServer              *_poclSwBlockServer;
ISpmCcaSupplierServer          *_poclSupplierServer;
ISpmApplicationDatabase        *_poclDatabase;
ISpmStartupSystemVariant       *_variant;
ISpmWorkerServer               *_poclWorkerServer;
ISpmRegistry                   *_poclRegistry;

public:
// ISpmBase interface functions
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 );

virtual tBool bHandleSynchrounousCall( tU32   /*u32Message*/,
                                       tVoid */*args*/ ){ return( FALSE ); }  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

protected:
virtual const tChar*getName( ) const { return( "spm_tclStartupCommon" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

};

