/*!
  * \file spm_StartupCommon.cpp
  *  \brief
  *   Common part of the startup process(es). Reads out the process configurations of every process.
  *
  *  \note
  *  PROJECT: NextGen
  *  SW-COMPONENT: FC SPM
  *  COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim
  *  \see
  *  \version
  * 1.0 |  27.01.11  | TMS Fischer       | initial version\n
  * 1.1 |  27.11.12  | CM-AI/CB32 kollai | Adaptation for GENERIC PLATFORM
  ******
  */

#include <sstream>
#include <iomanip>
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_StartupCommon.h"

// interfaces class definitions
#include "spm_IApplicationDatabase.h"
#include "spm_ICcaSupplierServer.h"
#include "spm_ICcaServiceServer.h"
#include "spm_IStartupSystemVariant.h"
#include "spm_IStartupSupervisor.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_IOsalProxy.h"
#include "spm_IFactory.h"
#include "spm_IRegistry.h"
#include "spm_IStartupSystem.h"
#include "spm_ILateServiceHandler.h"

// spm helper
#include "spm_CriticalSection.h"
#include "spm_StringTokenizer.h"
#include "spm_SoftwareBlock.h"
#include "spm_Registry_pathes.h"
#include "timeConvert.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_StartupCommon.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// -----------------------------------------------------------------------------
// defines
// -----------------------------------------------------------------------------
// #define SPM_TRACE_FILE_ID   SPM_FILE_STARTUPCOMMON
#ifndef SPM_DOWNLOAD_PROCS_2_KILL
   #define SPM_DOWNLOAD_PROCS_2_KILL
#endif
#ifndef SPM_PRJ_SYSTEM_STATE_DOWNLOAD
   #define SPM_PRJ_SYSTEM_STATE_DOWNLOAD    SPM_SYSTEM_DOWNLOAD
#endif

spm_tclStartupCommon::spm_tclStartupCommon( tU16               u16SpmId,
                                            const ISpmFactory& factory ) : ISpmStartupCommon( factory )
   , _u16SpmId( u16SpmId )
   , _hSemHandle(OSAL_C_INVALID_HANDLE)
   , _poclSwBlockServer( NULL )
   , _poclSupplierServer( NULL )
   , _poclDatabase( NULL )
   , _variant( NULL )
   , // may be optional if not supported
   _poclWorkerServer( NULL )
   , _poclRegistry( NULL ){
   /*!
     * \fn
     *  \brief
     *    Constructor
     *
     *  \param[in] u16SpmId: spm application id.
     *  \param[in] factory: spm factory object.
     ******
     */
   _strSemName = "SpmSCm" + std::to_string(u16SpmId);
   if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( _strSemName.c_str( ), &_hSemHandle, (tU32)1 ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclStartupCommon !!!!!! Error detected !!!!!! cannot create Semaphore: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclStartupCommon::~spm_tclStartupCommon( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param
  ******
  */
   OSAL_s32SemaphoreClose( _hSemHandle );
   OSAL_s32SemaphoreDelete( _strSemName.c_str( ) );

   SPM_NULL_POINTER_CHECK( _poclSwBlockServer );
   _poclSwBlockServer->vRemoveClient( this ); // unregister

   _poclSwBlockServer  = NULL;
   _poclSupplierServer = NULL;
   _poclDatabase       = NULL;
   _variant            = NULL;
   _poclWorkerServer   = NULL;
   _poclRegistry       = NULL;
}

tVoid spm_tclStartupCommon::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclSwBlockServer );
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );

   _poclSwBlockServer->vAddClient( this ); // register to sw block information
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclStartupCommon::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclDatabase, ISpmApplicationDatabase );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSwBlockServer, ISpmLocalAppManager, ISpmSwBlockServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclSupplierServer, ISpmCcaSupplierServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,   ISpmWorkerServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclRegistry,       ISpmRegistry );

   #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
      SPM_GET_IF_REFERENCE_USE_VAR( _variant,            ISpmStartupSystemVariant );
   #endif
} // vGetReferences

tVoid spm_tclStartupCommon::vHandleMessage( tU32 u32Message,
                                            tU32 u32Parm ){
   /*!
     * \fn
     *  \brief
     *    handles the message.
     *
     *  \param[in] u32Message: type of message.
     *  \param[in] u32Parm: extra parameter to handle message (process id).
     *  \return  void
     ******
     */
   if ( u32Message == SPM_U32_WORKER_STA_PROCESS_STOP ){
      ETG_TRACE_USR1( ( "spm_tclStartupCommon::vHandleMessage(): Message to stop process (pid %d) detected", u32Parm ) );
      bStopProcess( strGetProcName( u32Parm ) );
   } else if ( u32Message == SPM_U32_WORKER_STA_PROCESS_START ){
      bStartProcess( strGetProcName( u32Parm ) );
   } else if ( u32Message == SPM_U32_WORKER_STA_PROCESS_KILLED ){

      dp_tclSpmDpInternDataSystemState oNewSysState;
                  ETG_TRACE_USR1( ( "spm_tclStartupCommon::vHandleMessage(): Message process KILLED detected" ) );

      if ( SPM_PRJ_SYSTEM_STATE_DOWNLOAD == oNewSysState.tGetData( ) ){
         tBool       bAllDlProcsDead   = TRUE;
         std::string astrProcss2Kill[] = { SPM_DOWNLOAD_PROCS_2_KILL };
         tU8         u8ArrayCount      = ELEMENTE( astrProcss2Kill );

                  ETG_TRACE_USR1( ( "spm_tclStartupCommon::vHandleMessage(): We are in state DOWNLOAD --> another process is killed." ) );
         for ( tU8 i = 0; i < u8ArrayCount; i++ ){
            TMapProcessConfiguration::iterator posProc = _oMapProcessConfig.find( astrProcss2Kill[i] );
            if ( posProc != _oMapProcessConfig.end( ) ){
               if ( posProc->second.bProcSpawned != TRUE ){
                  ETG_TRACE_USR1( ( "spm_tclStartupCommon::vHandleMessage(): Process is dead '%s' now.", astrProcss2Kill[i].c_str( ) ) );
               } else {
                  bAllDlProcsDead = FALSE;
               }
            }
         }
         if ( bAllDlProcsDead ){
                  ETG_TRACE_USR1( ( "spm_tclStartupCommon::vHandleMessage(): All processes for DL are killed." ) );
                  SPM_NULL_POINTER_CHECK( _poclWorkerServer );
            _poclWorkerServer->bPostMessage( "ISpmCcaServiceServer", SPM_U32_WORKER_CSH_SYSTEM_MODE_DL );
         }
      }
   }
} // vHandleMessage

tBool spm_tclStartupCommon::bStartProcess( const std::string& strProcessName ){
   /*!
     * \fn
     *  \brief
     *    start the process.
     *
     *  \param[in] strProcessName: process name to be started.
     *  \return    always TRUE
     *
     ******
     */

   TListStartup::iterator pos;

   for ( pos = _oListStartup.begin( ); pos != _oListStartup.end( ); ++pos ){
      if ( ( * pos )->bStartProcess( strProcessName ) ){
         break;
      }
   }
   return( TRUE );
} // bStartProcess

tBool spm_tclStartupCommon::bStopProcess( const std::string& strProcessName ){
   /*!
     * \fn
     *  \brief
     *    stop the process.
     *
     *  \param[in] strProcessName: process name to be stoped.
     *  \return    always TRUE
     *
     ******
     */
         ETG_TRACE_USR4( ( "spm_tclStartupCommon::bStopProcess(): Stop process '%s'", strProcessName.c_str( ) ) );

   // get process name from location
   TListStartup::iterator pos;
   for ( pos = _oListStartup.begin( ); pos != _oListStartup.end( ); ++pos ){
      if ( ( * pos )->bStopProcess( strProcessName ) ){
         break;
      }
   }
   return( TRUE );
}

tBool spm_tclStartupCommon::bForceProcessStop( const std::string& strProcessName ){
   /*!
     * \fn
     *  \brief
     *    ENTRY Function to stop a process. The process name has to be the same as used in
      the base.reg key_ "PROC_NAME". This function sends the given procname to all
      known/registered startup threads (via bStopProcess()), until one the threads
      returns with TRUE (TRUE= The startup thread was the one how started the thread and
      knows how to kill them. So process is then also killed here. If this function returns
      with true, the complete process stop action is done! (all process apps in OFF and process is terminated).
     *
     *  \param[in] strProcessName: process name to be stoped.
     *  \return    TRUE  if process is stopped.
     *
     ******
     */
   tBool                  bProcessKilled = FALSE;

         ETG_TRACE_USR4( ( "spm_tclStartupCommon::bForceProcessStop(): Stop process '%s'", strProcessName.c_str( ) ) );

   TListStartup::iterator pos;
   for ( pos = _oListStartup.begin( ); pos != _oListStartup.end( ); ++pos ){
      bProcessKilled = ( * pos )->bForceProcessStop( strProcessName );
      if ( bProcessKilled ){
         break;
      }
   }

   return( bProcessKilled );
} // bForceProcessStop

tU32 spm_tclStartupCommon::u32GetSupervisionMode( tU32 u32ProcId ){
   /*!
     * \fn
     *  \brief
     *    This function returns supervision state for the input process id.
     *
     *  \param[in] u32ProcId:  Process id.
     *  \return  supervision state for the process.
     *
     ******
     */
   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
      if ( posProc->second.s32ProcId == (tS32)u32ProcId ){
         return( posProc->second.u32ProcSupervision );
      }
   }
   return( 0 );
}

std::string spm_tclStartupCommon::strGetProcName( tU32 u32ProcId ){
   /*!
     * \fn
     *  \brief
     *    Returns process name for the input process id.
     *
     *  \param[in] u32ProcId: Process id.
     *  \return  returns process name if found otherwise empty string.
     *
     ******
     */
   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
      if ( posProc->second.s32ProcId == (tS32)u32ProcId ){
         return( posProc->second.strProcessName );
      }
   }
   return( "" );
}

tU32 spm_tclStartupCommon::u32GetProcId( const std::string& strProcessName ){
/*!
  * \fn
  *  \brief  get the process id of a given process
  *  \param [in] strProcessName: name of the process.
  *  \return tU32: Process id.
  ******
  */
   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
      if ( posProc->second.strProcessName == strProcessName ){
         ETG_TRACE_USR4( ( "spm_tclStartupCommon::u32GetProcId: PID '%d' and ProcessName '%s'", posProc->second.s32ProcId, strProcessName.c_str( ) ) );
         return( (tU32)posProc->second.s32ProcId );
      }
   }
   return( (tU32)PROC_UNKNOWN_PROC_ID );
}

void spm_tclStartupCommon::vSetProcId( const std::string& strSwBlockName,
                                       const tU32         u32PID ){
/*!
  * \fn
  *  \brief Function to update the process id.for the given process in process configuration structure.
  *  \param [in] strSwBlockName: name of the process.
  *  \param [in] u32PID: process.id which is to be set.
  *  \return none
  *  \details This method will also update the PID for software bolck of the process.
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclDatabase );
   ETG_TRACE_USR4( ( "spm_tclStartupCommon::vSetProcId: PID '%d' and ProcessName '%s'", u32PID, strSwBlockName.c_str( ) ) );

   tBool                              bFound = FALSE;
   TMapProcessConfiguration::iterator posProc;
   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
      if ( posProc->second.strProcessName == strSwBlockName ){
         posProc->second.s32ProcId = ( tS32 )u32PID;
         bFound                    = TRUE;
      }
   }
   if ( bFound == FALSE ){
      ETG_TRACE_USR1( ( "spm_tclStartupCommon::vSetProcId: ProcessConfig not found for PID=%d, SwBlockName=%s", u32PID, strSwBlockName.c_str( ) ) );
   }

   spm_tclSoftwareBlock *poSwBlock = _poclDatabase->poGetSoftwareBlock( strSwBlockName );
   if ( poSwBlock != NULL ){
      poSwBlock->vSetPID( u32PID );
   } else {
      ETG_TRACE_USR1( ( "spm_tclStartupCommon::vSetProcId: SwBlock not found for PID=%d, SwBlockName=%s", u32PID, strSwBlockName.c_str( ) ) );
   }
} // vSetProcId

tBool spm_tclStartupCommon::bAddSwBlock( const std::string& strSwBlockName,
                                         const tBool        bBlockMode,
                                         const tBool        bLocal,
                                         const std::string& strExecPath,
                                         const std::string& strProcessName,
                                         const std::string& strStartType ){
/*!
  * \fn
  *  \brief Function to add the software block to the application database and update its details.
  *         The ExecPath, SWBlockName, StartType, ProcID are updated.
  *  \param [in] strSwBlockName: name of the software block which need to be added to AppDataBase.
  *  \param [in] bBlockMode:     enable the blockmode mode or not.
  *  \param [in] bLocal:         local to FC LCM or remote
  *  \param [in] strExecPath:    process location or executable/service file path
  *  \param [in] strProcessName: process name of the added software block.
  *  \param [in] strStartType:   start type of the process OSAL/SYSTEMD.
  *  \return tBool: True if successfuly added to application database.
  ******
  */
      ETG_TRACE_USR1( ( "spm_tclStartupCommon::bAddSwBlock strSwBlockName=%16s, bBlockMode=%d, bLocal=%d, strExecPath=%30s, strProcessName=%16s, strStartType=%16s",
                     strSwBlockName.c_str( ),
                     bBlockMode,
                     bLocal,
                     strExecPath.c_str( ),
                     strProcessName.c_str( ),
                     strStartType.c_str( ) ) );

   tBool                 bSuccess = FALSE;
   spm_tclSoftwareBlock *poSwBlock;

   SPM_NULL_POINTER_CHECK_VAL( _poclDatabase );
   SPM_NULL_POINTER_CHECK_VAL( _poclSupplierServer );
   SPM_NULL_POINTER_CHECK_VAL( _poclRegistry );

   if ( _poclRegistry->bIsSoftwareBlockInRegistry( strSwBlockName ) ){
      poSwBlock = _poclDatabase->poGetSoftwareBlock( strSwBlockName );

      // is software block already set and is this block still disconnected?
      if ( poSwBlock == NULL ){
         // first get from registry all information about app ids and svcids
         // in that software block of the new process.
         // appId-->vector svcId
         std::map < tU16, std::vector < tU16 > > mapAppIds;

         if ( _poclRegistry->bGetAllApplicationsOfSwBlock( _u16SpmId, strSwBlockName, mapAppIds )
              && ( mapAppIds.size( ) > 0 ) ){
            // sw block was found in registry, now copy all information to the
            // application database and the cca supplier server.

            // first create the new software block in the application database
            poSwBlock = _poclDatabase->poAddSoftwareBlock( strSwBlockName );
            poSwBlock->vSetExecPath( strExecPath );
            poSwBlock->vSetSwBlockName( strSwBlockName );
            poSwBlock->vSetStartType( strStartType );
            tU32 u32ProcId = u32GetProcId( strProcessName.c_str( ) );
            if ( u32ProcId != (tU32)PROC_UNKNOWN_PROC_ID ){
               poSwBlock->vSetPID( u32ProcId );
            } else {
               SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL( _poclDbusTmp, spm_ISpmLateServiceHandler );
               if ( _poclDbusTmp->bIsLcmLateStarted( ) ) {
                  // spm late started now. trigger the GetProcessId method for the list of services
                  for (std::list< std::string >::iterator it = _oListPid0Services.begin(); it != _oListPid0Services.end(); ++it) {
                     _poclDbusTmp->vGetProcessId( *it );
                  }
                  // finish the work clear all now
                  _oListPid0Services.clear( );
                  if ( SPM_STARTUP_VALUE_SYSTEMD_START_TYPE == strStartType ) {
                     _poclDbusTmp->vGetProcessId( strExecPath );
                  }
               } else {
                  // Save the services for waiting spm late available
                  if ( SPM_STARTUP_VALUE_SYSTEMD_START_TYPE == strStartType )
                  {
                     _oListPid0Services.push_back( strExecPath );
                  }
               }
            }

            bSuccess = TRUE;

            std::map < tU16, std::vector < tU16 > >::iterator it;

            for ( it = mapAppIds.begin( ); it != mapAppIds.end( ); ++it ){
               // add the application id to the software block. It is a placeholder
               // for all applications of the process which is about to be started.
               poSwBlock->vAddConnectedApp( it->first, bLocal );
               std::vector < tU16 >::iterator it2;

               for ( it2 = it->second.begin( ); it2 != it->second.end( ); ++it2 ){
                  // Add the services to the supplier server which were just
                  // found in the registry.
                  _poclSupplierServer->vAddNewService( * it2, it->first );
               }
            }

            // if block mode is wanted then enable the blockmode mode
            if ( bBlockMode ){
               poSwBlock->vSetBlockMode( );
               // at last enable the software block for further processing
            }
            poSwBlock->vSetWholeBlockEnabled( );

         } else {
            ETG_TRACE_ERR( ( "WARNING: SWBLOCK %s not added (perhaps no apps in registry found)!", strSwBlockName.c_str( ) ) );
         }
      } else if (!poSwBlock->bIsConnected( )) {
          ETG_TRACE_ERRMEM( ( "WARNING: SWBLOCK %s is already in system, but not connected !!!", strSwBlockName.c_str( ) ) );
      } else {
            ETG_TRACE_ERR( ( "WARNING: SWBLOCK %s is already connected!", strSwBlockName.c_str( ) ) );
      }
   } else {
            ETG_TRACE_ERR( ( "WARNING: Registry PATH for SWWBLOCK not found, SWBLOCK '%s' won't be started!", strSwBlockName.c_str( ) ) );
   }
   return( bSuccess );
} // bAddSwBlock

tBool spm_tclStartupCommon::bAddAdditionalSwBlock( const std::string& strBusShortName,
                                                   const std::string& strServiceName,
                                                   const tU32         u32AppId,
                                                   const std::string& strStartType ){
/*!
  * \fn
  *  \brief Function to add a software block to the application database and update its details that was not loaded via registry.
  *  \param [in] strBusShortName: name of the DBus bus - converted to a shorter name
  *  \param [in] strServiceName:  name of the service where this block was started from
  *  \param [in] u32AppId:     application ID (mapped from strBusShortName
  *  \param [in] strStartType: start type of the process OSAL/SYSTEMD (almost certainly always SYSTEMD).
  *  \return tBool: True if successfuly added to application database.
  ******
  */
   ETG_TRACE_USR1( ( "spm_tclStartupCommon::bAddAdditionalSwBlock strBusShortName=%16s, strServiceName=%16s, u32AppId=%d, strStartType=%16s",
                     strBusShortName.c_str( ),
                     strServiceName.c_str( ),
                     u32AppId,
                     strStartType.c_str( ) ) );

   tBool                 bSuccess = FALSE;
   spm_tclSoftwareBlock *poSwBlock;

   SPM_NULL_POINTER_CHECK_VAL( _poclDatabase );

   poSwBlock = _poclDatabase->poGetSoftwareBlock( strBusShortName );

   // is software block already set and is this block still disconnected?
   if ( ( poSwBlock == NULL ) || !poSwBlock->bIsConnected( ) ){
      // first create the new software block in the application database
      poSwBlock = _poclDatabase->poAddSoftwareBlock( strBusShortName );
      poSwBlock->vSetExecPath( strServiceName );
      poSwBlock->vSetSwBlockName( strBusShortName );
      poSwBlock->vSetStartType( strStartType );
      poSwBlock->vSetPID( u32GetProcId( strBusShortName ) );

      bSuccess = TRUE;

      poSwBlock->vAddConnectedApp( u32AppId, TRUE );
      // at last enable the software block for further processing
      poSwBlock->vSetWholeBlockEnabled( );
   } else {
      ETG_TRACE_FATAL( ( "WARNING: SWBLOCK %s is already connected!", strBusShortName.c_str( ) ) );
   }
   return( bSuccess );
} // bAddAdditionalSwBlock

tBool spm_tclStartupCommon::bRemoveSwBlockFromProcessList( const std::string& strSwBlockName,
                                                           std::string      & strProcessName ){
/*!
  * \fn
  *  \brief Function sets the SW-Block to inactive.
  *  \param [in]  strSwBlockName:  Name of the SW-Block
  *  \param [out] strProcessName:  Name of the process that contained this SW-Block
  *  \return tBool: True if successfuly removed from application database.
  ******
  */
   tBool                              bActiveSwBlock = FALSE;
   tBool                              bSwBlockFound  = FALSE;

   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); !bSwBlockFound && ( posProc != _oMapProcessConfig.end( ) ); ++posProc ){

      TBlockList::iterator pos;
      bActiveSwBlock = FALSE;

      for ( pos = posProc->second.listSwBlocks.begin( ); pos != posProc->second.listSwBlocks.end( ); ++pos ){
         if ( pos->strBlockName == strSwBlockName ){
            // OK, found SwBlock, now set to inactive
            pos->bActive   = FALSE;
            strProcessName = posProc->first;
            bSwBlockFound  = TRUE;
         }
         if ( pos->bActive ){
            // at least one block is active
            bActiveSwBlock = TRUE;
         }
      }
   }

   return( bActiveSwBlock );
} // bRemoveSwBlockFromProcessList

tVoid spm_tclStartupCommon::vRemoveSwBlock( const std::string& strSwBlockName,
                                            tBool              bStopProcWithLastBlock ){
   /*!
     * \fn
     *  \brief
     *    Removes software block.
     *
     *  \param[in] strSwBlockName :          Name of software block.
     *  \param[in] bStopProcWithLastBlock:   whether process within software block need to be removed or not.
     *  \return  void
     ******
     */
   std::string strBlockKey;
   std::string strProcessName;

         SPM_NULL_POINTER_CHECK( _poclDatabase );
         SPM_NULL_POINTER_CHECK( _poclWorkerServer );
         SPM_NULL_POINTER_CHECK( _poclRegistry );

         ETG_TRACE_USR4( ( "vRemoveSwBlock(): '%s'", strSwBlockName.c_str( ) ) );

   if ( _poclRegistry->bIsSoftwareBlockInRegistry( strSwBlockName ) ){
      _poclDatabase->vRemoveSoftwareBlock( strSwBlockName );

         ETG_TRACE_USR4( ( "vRemoveSwBlock(): vRemoveSoftwareBlock called '%s'", strSwBlockName.c_str( ) ) );

      // and here check if all SW blocks of the process are removed --> remove process
      if ( !bRemoveSwBlockFromProcessList( strSwBlockName, strProcessName ) ){

         ETG_TRACE_USR4( ( "vRemoveSwBlock(): bRemoveSwBlockFromProcessList called '%s'", strSwBlockName.c_str( ) ) );

         // no block active in process -> remove process
         strBlockKey = (std::string)SPM_REG_PROCESS_BASE_PATH + strSwBlockName;

         if ( FALSE == _poclRegistry->bRemoveSoftwareBlock( strBlockKey ) ){
            ETG_TRACE_COMP( ( "SPM: !!!!!! Warning detected !!!!!!. Could not remove SW-Block %s", strBlockKey.c_str( ) ) );
            // and now send shutdown event to main thread of process
         }
         if ( bStopProcWithLastBlock ){
            tU32 u32ProcId = u32GetProcId( strProcessName );
            if ( ( (tU32)PROC_UNKNOWN_PROC_ID == u32ProcId ) ){
               ETG_TRACE_ERRMEM( ( "spm_tclStartupCommon::vRemoveSwBlock(): Error: Process is not stopped. PID (%d) for process '%s' is not updated in TProcConfiguration !!!", PROC_UNKNOWN_PROC_ID, strProcessName.c_str( ) ) );
            } else {
               ETG_TRACE_USR4( ( "spm_tclStartupCommon::vRemoveSwBlock(): Send message 'PROCESS_STOP' to stop (pid '%d') process '%s' ", u32ProcId, strProcessName.c_str( ) ) );
               _poclWorkerServer->bPostMessage( "ISpmStartupCommon", SPM_U32_WORKER_STA_PROCESS_STOP, u32ProcId );
            }
         }
      }
   } else {
      // tChar szStr[120];
      // OSAL_s32PrintFormat( szStr, "WARNING: No %s.reg found, no start of SWBLOCK %s,", pcSwBlockName, pcSwBlockName);
      // TRACE_SPM_INFO_STRING(szStr);
   }
   return;
} // vRemoveSwBlock

tVoid spm_tclStartupCommon::vRemoveStoppedService( const std::string& strServiceName ){
    /*!
      * \fn
      *  \brief
      *    Removes software blocks and update the TMapProcessConfiguration of stopped service.
      *
      *  \param[in] strServiceName :          Name of stopped service.
      *  \return  void
      *
      *  \details Get the TMapProcessConfiguration of stopped service.
      *           Remove its software blocks by calling vRemoveSwBlock.
      *           Update the bProcSpawned of TMapProcessConfiguration to not started(FALSE).
      ******
      */
    ETG_TRACE_USR4( ( "spm_tclStartupCommon::vRemoveStoppedService(): service: '%s'", strServiceName.c_str( ) ) );
    TMapProcessConfiguration::iterator posProc;
    for( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
        if( posProc->second.strProcessLocation == strServiceName
            && posProc->second.bProcSpawned == TRUE ){
            TBlockList::iterator posSwBlock;
            for( posSwBlock = posProc->second.listSwBlocks.begin( );
                 posSwBlock != posProc->second.listSwBlocks.end( );
                 ++posSwBlock ){
                    ETG_TRACE_USR4( ( "spm_tclStartupCommon::vRemoveStoppedService(): vRemoveSwBlock('%s')", posSwBlock->strBlockName.c_str( ) ) );

                    // the service is already stopped, so call vRemoveSwBlock with FALSE to prevent from sending SPM_U32_WORKER_STA_PROCESS_STOP and calling spm_tclStartupCommon::bStopProcess
                    vRemoveSwBlock( posSwBlock->strBlockName, FALSE );
            }
            posProc->second.bProcSpawned = FALSE;
        }
    }
}

tVoid spm_tclStartupCommon::vUpdateStartedService( const std::string& strServiceName ){
    /*!
      * \fn
      *  \brief
      *    Update the TMapProcessConfiguration of started service.
      *
      *  \param[in] strServiceName :          Name of started service.
      *  \return  void
      *
      *  \details Get the TMapProcessConfiguration of started service.
      *           Update TMapProcessConfiguration:
      *               + bProcSpawned to started(TRUE)
      *               + u32StartTimeAbs
      *               + u32StartTimeRel
      ******
      */
    ETG_TRACE_USR4( ( "spm_tclStartupCommon::vUpdateStartedService(): service: '%s'", strServiceName.c_str( ) ) );
    TMapProcessConfiguration::iterator posProc;
    for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){
        if ( posProc->second.strProcessLocation == strServiceName
             && posProc->second.bProcSpawned == FALSE ){
            OSAL_trTimeDate rCurrentTime = { 0,0,0,0,0,0,0,0,0 };

            SPM_GET_IF_REFERENCE_NEW_VAR( poOsalProxy, ISpmOsalProxy );
            (tVoid)poOsalProxy->bGetUtcTime( &rCurrentTime );
            rCurrentTime.s32Year           += 1900;

            tclTimeConvert oTimeConv;
            posProc->second.u32StartTimeAbs = oTimeConv.u32GetSeconds( &rCurrentTime );
            posProc->second.u32StartTimeRel = OSAL_ClockGetElapsedTime( );

            posProc->second.bProcSpawned    = TRUE;
        }
    }
}

void spm_tclStartupCommon::vOnSwBlockLoaded( const TSpmSwBlockSet& SwBlocksLoaded ){
   /*!
     * \fn
     *  \brief distributes the info that the SwBlock is loaded to all registered clients
     *
     *  \param [in] SwBlocksLoaded: Set of SwBlockNames that are at least in state Loaded
     *  \return void
     *
     *  \details goes through the list of all registered SwBlockClients and calls vOnSwBlockLoaded there
     */
   spm_tclHandleSemaphore                    sem( _hSemHandle ); // critical section

   std::set < ISpmSwBlockClient* >::iterator it;

   for ( it = _clients.begin( ); it != _clients.end( ); ++it ){
      ( * it )->vOnSwBlockLoaded( SwBlocksLoaded );
   }
}

void spm_tclStartupCommon::vOnSwBlockConnected( const TSpmSwBlockSet& SwBlocksConnected ){
   /*!
     * \fn
     *  \brief distributes the info that the SwBlock is connected to all registered clients
     *
     *  \param [in] SwBlocksConnected Set of SwBlockNames that are at least in state Connected
     *  \return void
     *
     *  \details goes through the list of all registered SwBlockClients and calls vOnSwBlockConnected there
     */
   spm_tclHandleSemaphore                    sem( _hSemHandle ); // critical section

   std::set < ISpmSwBlockClient* >::iterator it;

   for ( it = _clients.begin( ); it != _clients.end( ); ++it ){
      ( * it )->vOnSwBlockConnected( SwBlocksConnected );
   }
}

void spm_tclStartupCommon::vOnSwBlockUp( const TSpmSwBlockSet& SwBlocksUp ){
   /*!
     * \fn
     *  \brief distributes the info that the SwBlock is up to all registered clients
     *
     *  \param [in] SwBlocksUp Set of SwBlockNames that are in state Up
     *  \return void
     *
     *  \details goes through the list of all registered SwBlockClients and calls vOnSwBlockUp there
     */
   spm_tclHandleSemaphore                    sem( _hSemHandle ); // critical section

   std::set < ISpmSwBlockClient* >::iterator it;

   for ( it = _clients.begin( ); it != _clients.end( ); ++it ){
      ( * it )->vOnSwBlockUp( SwBlocksUp );
   }
}

tVoid spm_tclStartupCommon::vOnSyncPointReached( const std::string& strSyncPointName ){
   /*!
     * \fn
     *  \brief
     *       Calls vOnSyncPointReached for all registered startups.
     *  \param [in] strSyncPointName: Name of sync point.
     *  \return void
     */
   TListStartup::iterator pos;

               ETG_TRACE_USR4( ( " spm_tclStartupCommon::vOnSyncPointReached()" ) );

   for ( pos = _oListStartup.begin( ); pos != _oListStartup.end( ); ++pos ){
               ETG_TRACE_USR4( ( "vOnSyncPointReached()..." ) );
      ( * pos )->vOnSyncPointReached( strSyncPointName );
   }
} // bStartProcess

void spm_tclStartupCommon::vOnSwBlockForced( const TSpmSwBlockSet& SwBlocksForced ){
   /*!
     * \fn
     *  \brief distributes the info that the SwBlock is up and to all registered clients
     *
     *  \param [in] SwBlocksUp Set of SwBlockNames that are in state Up
     *  \return void
     *
     *  \details goes through the list of all registered SwBlockClients and calls vOnSwBlockUp there.
     *  This is only done if vSetForceBlockEnabled was called for this SwBlock
     */
               ETG_TRACE_USR4( ( " spm_tclStartupCommon::vOnSwBlockForced()" ) );

   spm_tclHandleSemaphore                    sem( _hSemHandle ); // critical section

   std::set < ISpmSwBlockClient* >::iterator it;

   for ( it = _clients.begin( ); it != _clients.end( ); ++it ){
               ETG_TRACE_USR4( ( "vOnSwBlockForced()..." ) );
      ( * it )->vOnSwBlockForced( SwBlocksForced );
   }
}

tVoid spm_tclStartupCommon::vAddClient( ISpmSwBlockClient *clt ){
   /*!
     * \fn
     *  \brief
     *    Add the client.
     *
     *  \param[in] ISpmSwBlockClient: Pointer to the client.
     *  \return void
     *
     ******
     */
   spm_tclHandleSemaphore sem( _hSemHandle ); // critical section

   if ( clt != NULL ){
      _clients.insert( clt );
   }
}

tVoid spm_tclStartupCommon::vRemoveClient( ISpmSwBlockClient *clt ){
   /*!
     * \fn
     *  \brief
     *    Removes the client.
     *
     *  \param[in] ISpmSwBlockClient: Pointer to the client.
     *  \return void
     *
     ******
     */
   spm_tclHandleSemaphore                    sem( _hSemHandle ); // critical section

   std::set < ISpmSwBlockClient* >::iterator it;

   it = _clients.find( clt );

   if ( it != _clients.end( ) ){
      _clients.erase( it );
   }
}

tVoid spm_tclStartupCommon::vDeactivateProcesses( const std::set < std::string >& procs ){
   /*!
     * \fn
     *  \brief
     *    Deactivates set of processes.
     *
     *  \param[in] procs: Set of processes.
     *  \return  void
     *
     ******
     */
   std::set < std::string >::const_iterator it;

               ETG_TRACE_USR4( ( "spm_tclStartupCommon::vDeactivateProcesses procs size=%u _oMapProcessConfig.size=%u", (tUInt)procs.size( ), (tUInt)_oMapProcessConfig.size( ) ) );
   for ( it = procs.begin( ); it != procs.end( ); ++it ){
      TMapProcessConfiguration::iterator it2 = _oMapProcessConfig.find( * it );
      if ( it2 != _oMapProcessConfig.end( ) ){
         it2->second.bProcEnabled = FALSE;
         {
               ETG_TRACE_USR4( ( "Deactivated Process %s due to variant enabled", ( * it ).c_str( ) ) );
         }
      }
   }
}

tBool spm_tclStartupCommon::bIsProcessStarted( const std::string& strProcessname ) const {
   /*!
     * \fn
     *  \brief
     *    check whether process is started or not
     *
     *  \param[in] strProcessname: Process name.
     *  \return    TRUE if process started. FALSE if process not found or not started.
     *
     ******
     */
   // do not use here operator[] here for faster access to item of choice.
   // it is not a const function because if entry not found it will insert
   // a new item in the configuration. And that is not intended here.
   TMapProcessConfiguration::const_iterator it;

   it = _oMapProcessConfig.find( strProcessname );
   if ( it != _oMapProcessConfig.end( ) ){
      return( it->second.bProcSpawned );
   }
   return( FALSE ); // if not found at all
}

tVoid spm_tclStartupCommon::vSetProcessStarted( const std::string& strProcessname,
                                                tBool              bIsStarted ){
   /*!
     * \fn
     *  \brief
     *    Sets the process started state .
     *
     *  \param[in] strProcessname: Process name.
     *  \param[in] bIsStarted: boolean value for process started state.
     *  \return  void
     *
     ******
     */
   // do not use here operator[] here for faster access to item of choice.
   // it is not a const function because if entry not found it will insert
   // a new item in the configuration. And that is not intended here.
   TMapProcessConfiguration::iterator it;

   it = _oMapProcessConfig.find( strProcessname );
   if ( it != _oMapProcessConfig.end( ) ){
      it->second.bProcSpawned = bIsStarted;
   }
}

tVoid spm_tclStartupCommon::vEvaluateRegistry( const std::string& strConf,
                                               TMapRegistries   & registry ) const {
/*!
  * \fn
  *  \brief
  *    Evaluates the Registry.
  *
  *  \param[in]  strConf:   Name of registry.
  *  \param[out] registry:  output registry to be filled.
  *  \return  void
  *
  ******
  */
   tU32                  as32Data[255];

   std::string           strBaseName;
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_tIODescriptor    fd;
   OSAL_trIOCtrlRegistry rReg;

   strBaseName = SPM_REG_PROCESS_BASE_SPM_REGISTRY_BASE_PATH + strConf;

   fd          = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );
   if ( fd != OSAL_ERROR ){
      rDirV.fd        = fd;
      rDirV.s32Cookie = 0;

      while ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
         rReg.pcos8Name = rDirV.dirent.s8Name;
         rReg.ps8Value  = (tU8*)as32Data;
         rReg.u32Size   = sizeof( as32Data );

         if ( OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){

            if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
               // put one registry path into the configuration
               registry[(tChar*)rDirV.dirent.s8Name] = (tChar*)as32Data;

            } else {
               // Could not find correct entry
               ETG_TRACE_ERR( ( "Could not find correct entry" ) );
            }
         }
      }
      OSAL_s32IOClose( fd );
   }
} // vEvaluateRegistry

tVoid spm_tclStartupCommon::vEvaluateProcessConfiguration( ){
   /*!
     * \fn
     *  \brief
     *    Evaluates Process Configuration using registry.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   tS32                  as32Data[255];

   OSAL_trIOCtrlDir      rDir  = { 0,0,0 };
   OSAL_trIOCtrlDir      rDirV = { 0,0,0 };
   OSAL_trIOCtrlRegistry rReg;
   OSAL_tIODescriptor    fdReg, fdVal;

   std::string           strBaseName;

   strBaseName = (std::string)SPM_REG_PROCESS_CONFIG_BASE_PATH;

   fdReg       = OSAL_IOOpen( strBaseName.c_str( ), OSAL_EN_READONLY );
   if ( fdReg != OSAL_ERROR ){
      /* First, recursively show all the subkeys */
      rDir.fd        = fdReg;
      rDir.s32Cookie = 0;

      // check configuration for each process
      while ( OSAL_s32IOControl( fdReg, OSAL_C_S32_IOCTRL_FIOREADDIR, ( intptr_t )&rDir ) != OSAL_ERROR ){
         std::string        strProcDirName;
         std::string        strProcDirSwBlockName;
         std::string        strProcDirSyncBlockName;
         std::string        strProcDirLastModeName;

         TProcConfiguration tNewProcConfig;

         // new process directory found
         strProcDirName         = strBaseName + (tString)rDir.dirent.s8Name;

         //check if this is a last mode depending entry
         strProcDirLastModeName = strProcDirName + SPM_STARTUP_FOLD_CONF_LASTMODE;

            ETG_TRACE_USR1( ( "PROC-REGISTRY-CONF: %s", strProcDirName.c_str( ) ) );
            ETG_TRACE_USR1( ( "PROC-REGISTRY-CONF: %s", strProcDirLastModeName.c_str( ) ) );

         fdVal                  = OSAL_IOOpen( strProcDirLastModeName.c_str( ), OSAL_EN_READONLY );
         if ( fdVal != OSAL_ERROR ){
            ETG_TRACE_USR1( ( "PROC-REGISTRY-CONF: %s", strProcDirLastModeName.c_str( ) ) );
            OSAL_s32IOClose( fdVal );

            dp_tclSpmDpInternDataStartupConfig oStartupConfig;
            tU32                               u32CfgNb = oStartupConfig.tGetData( );
            std::stringstream sstream;
            sstream << "/" << std::setw( 4 ) << std::setfill( '0' ) << std::dec << u32CfgNb;
            strProcDirLastModeName += sstream.str( );

            fdVal                   = OSAL_IOOpen( strProcDirLastModeName.c_str( ), OSAL_EN_READONLY );
            if ( fdVal != OSAL_ERROR ){
               ETG_TRACE_USR1( ( "PROC-REGISTRY-CONF: %s", strProcDirLastModeName.c_str( ) ) );
               OSAL_s32IOClose( fdVal );
               // change to last mode folder
               strProcDirName = strProcDirLastModeName;
            }

         }

         /* Store registry base path of this specific process config into registry.
            This is used for linux startup sync workarround
           */
         tNewProcConfig.strProcConfigBasePath = strProcDirName;

         strProcDirSwBlockName                = strProcDirName + SPM_STARTUP_FOLD_CONF_SWBLOCKS;
         strProcDirSyncBlockName              = strProcDirName + SPM_STARTUP_FOLD_CONF_SYNCBLOCKS;

         // open directory for process
         fdVal                                = OSAL_IOOpen( strProcDirSwBlockName.c_str( ), OSAL_EN_READONLY );
         if ( fdVal != OSAL_ERROR ){
            rDirV.fd        = fdVal;
            rDirV.s32Cookie = 0;

            while ( OSAL_s32IOControl( fdVal, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
               // rReg.pcos8Name = rDirProc.dirent.s8Name;
               // rReg.ps8Value  = (tU8*)as32Data;
               // rReg.u32Size   = sizeof(as32Data);
               rReg.pcos8Name = rDirV.dirent.s8Name;
               rReg.ps8Value  = (tU8*)as32Data;
               rReg.u32Size   = sizeof( as32Data );

               if ( OSAL_s32IOControl( fdVal, OSAL_C_S32_IOCTRL_REGQUERYVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){

                  if ( rReg.s32Type == OSAL_C_S32_VALUE_S32 ){
                     // swBlock is enabled -> add to list
                     TBlockElement myBlock;
                     myBlock.strBlockName = (tChar*)rDirV.dirent.s8Name;
                     myBlock.u32BlockConf = as32Data[0];
                     myBlock.bActive      = TRUE;

                     tNewProcConfig.listSwBlocks.push_back( myBlock );

                  }
                  // else if (rReg.s32Type == OSAL_C_S32_VALUE_STRING)
                  // {
                  // TRACE_SPM_INFO_STRING(as32Data);
                  // }
                  // else
                  // {
                  // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
                  // }
               }
            }
               OSAL_s32IOClose( fdVal );
         }

         fdVal = OSAL_IOOpen( strProcDirSyncBlockName.c_str( ), OSAL_EN_READONLY );
         if ( fdVal != OSAL_ERROR ){
            rDirV.fd        = fdVal;
            rDirV.s32Cookie = 0;

            while ( OSAL_s32IOControl( fdVal, OSAL_C_S32_IOCTRL_REGENUMVALUE, ( intptr_t )&rDirV ) != OSAL_ERROR ){
               rReg.pcos8Name = rDirV.dirent.s8Name;
               rReg.ps8Value  = (tU8*)as32Data;
               rReg.u32Size   = sizeof( as32Data );

               if ( OSAL_s32IOControl( fdVal, OSAL_C_S32_IOCTRL_REGGETVALUE, ( intptr_t )&rReg ) != OSAL_ERROR ){

                  if ( rReg.s32Type == OSAL_C_S32_VALUE_S32 ){
                     if ( * (tU32*)as32Data != SPM_U32_SYNC_NONE ){
                        if ( OSAL_s32StringNCompare( (tString)rDirV.dirent.s8Name, SPM_STARTUP_KEY_CONF_PREDECESSOR, sizeof( rDirV.dirent.s8Name ) ) == 0 ){
                           // used to set a predecessor dependency
                           tNewProcConfig.bPredecessorSeen = TRUE; // entry seen, disable others
                           tNewProcConfig.u32LastSync      = * (tU32*)as32Data;
                        } else {
                           // swBlock is used to synchronize with former process
                           TBlockElement myBlock;
                           myBlock.strBlockName = (tChar*)rDirV.dirent.s8Name;
                           myBlock.u32BlockConf = * (tU32*)as32Data; // use the complete value
                           myBlock.bActive      = FALSE;

                           tNewProcConfig.listBlocksToSync.push_back( myBlock );
                        }
                     }
                  } else if ( rReg.s32Type == OSAL_C_S32_VALUE_STRING ){
                     if ( OSAL_s32StringNCompare( (tString)rDirV.dirent.s8Name, SPM_STARTUP_KEY_CONF_EXCLUDE, sizeof( rDirV.dirent.s8Name ) ) == 0 ){
                        spm_tclStringTokenizer st( (tChar*)as32Data, " ,", FALSE );
                        while ( st.bHasMoreTokens( ) ){
                           tNewProcConfig.setSwBlocksExclude.insert( st.oNextToken( ) );
                        }
                     }
                  }
                  // else
                  // {
                  // ETG_TRACE_ERR(("SPM: !!!!!! Error detected !!!!!!"));
                  // }
               }
            }
               OSAL_s32IOClose( fdVal );
         }

         reg_tclRegKey oProcReg;

         if ( oProcReg.bOpen( strProcDirName.c_str( ) ) ){
            // temporary buffer used to read from registry
            std::string strTmpbuf( 256, 0 );

            strTmpbuf = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESSNAME, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }

            tNewProcConfig.strProcessName = strTmpbuf.c_str( );

            // initializing strTmpbuf to empty is required to erase previous data. if bQueryString fails then strTmpbuf must have empty string
            strTmpbuf                     = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESSLOCATION, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }

            tNewProcConfig.strProcessLocation = strTmpbuf.c_str( );

            strTmpbuf                         = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESSREGISTRY, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }

            tNewProcConfig.strRegPath = strTmpbuf.c_str( );

            strTmpbuf                 = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESS_SHUTDOWN_EVENT, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               tNewProcConfig.strProcShutdownEvent = tNewProcConfig.strProcessName + "_" + SPM_STARTUP_FORMAT_PROCESS_SHUTDOWN_EVENT;
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.strProcShutdownEvent = strTmpbuf.c_str( );
            }
            tU32 u32ProcPrio;
            tNewProcConfig.u32ProcPrio = 0; // 0 = Use linux default process priority
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_PRIORITY, &u32ProcPrio ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32ProcPrio = u32ProcPrio;
            }
            tU32 u32ProcEnabled;
            tNewProcConfig.bProcEnabled = FALSE;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_ENABLED, &u32ProcEnabled ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else if ( u32ProcEnabled != 0 ){
               tNewProcConfig.bProcEnabled = TRUE;
            }
            tU32 u32ProcAffinity;
            tNewProcConfig.u32Affinity = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_AFFINITY, &u32ProcAffinity ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32Affinity = u32ProcAffinity;
            }
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_AFFINITY_END, &u32ProcAffinity ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32AffinityEnd = u32ProcAffinity;
            }
            tU32 u32ProcType;
            tNewProcConfig.u32ProcType = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_TYPE, &u32ProcType ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32ProcType = u32ProcType;
            }
            tU32 u32ProcSupervision;
            tNewProcConfig.u32ProcSupervision = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_SUPERVISION, &u32ProcSupervision ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32ProcSupervision = u32ProcSupervision;
            }
            tU32 u32ProcTypeArg;
            tNewProcConfig.u32ProcTypeArg = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_TYPE_ARG, &u32ProcTypeArg ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32ProcTypeArg = u32ProcTypeArg;
            }
            tU32 u32RegType;
            tNewProcConfig.u32RegType = SPM_STARTUP_REG_TYPE_LOAD_DIRECTLY; // load directly before process is started
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_REG_TYPE, &u32RegType ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32RegType = u32RegType;
            }

            strTmpbuf = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_CGROUP_CPU_PATH_SCRIPT, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }
            tNewProcConfig.strCGroupCpuScriptPath = strTmpbuf.c_str( );

            strTmpbuf                             = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_CGROUP_MEM_PATH_SCRIPT, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }
            tNewProcConfig.strCGroupMemScriptPath = strTmpbuf.c_str( );

            tU32 u32MemLevel;
            tNewProcConfig.u32CGroupMemLevel1     = 0; // load directly before process is started
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_CGROUP_MEM_LEVEL1, &u32MemLevel ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32CGroupMemLevel1 = u32MemLevel;
            }
            tNewProcConfig.u32CGroupMemLevel2 = 0; // load directly before process is started
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_CGROUP_MEM_LEVEL2, &u32MemLevel ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32CGroupMemLevel2 = u32MemLevel;
            }
            tU32 u32NiceLevel;
            tNewProcConfig.u32NiceLevel = 0; // nice level not set
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_NICE_LEVEL, &u32NiceLevel ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32NiceLevel = u32NiceLevel;
            }
            tNewProcConfig.u32NiceLevelEnd = 0; // nice level not set
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_NICE_LEVEL_END, &u32NiceLevel ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32NiceLevelEnd = u32NiceLevel;
               // Fetch Process Start Type from Configuration in RAM Registry
            }

            strTmpbuf = "";
            // Fetch PROC_START_TYPE from Registry
            if ( oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESS_START_TYPE, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_USR4(("spm_tclStartupCommon::vEvaluateProcessConfiguration(): Setting ProcStartType to :%s",tNewProcConfig.strProcStartType.c_str()));
               tNewProcConfig.strProcStartType = strTmpbuf.c_str( );
            } else { // if error in fetching the PROC_START_TYPE from Registry.

               // ETG_TRACE_USR4(("spm_tclStartupCommon::vEvaluateProcessConfiguration(): Setting ProcStartType to Default Type - OSAL"));
               tNewProcConfig.strProcStartType = SPM_STARTUP_VALUE_OSAL_START_TYPE;
            }
            tU32 u32SyncDelay;
            tNewProcConfig.u32DelayBeforeStartNextProc = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_DELAY_BEFORE_NEXT_PROC, &u32SyncDelay ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32DelayBeforeStartNextProc = u32SyncDelay;
            }
            tU32 u32StartDelay;
            tNewProcConfig.u32DelayBeforeStartThisProc = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_DELAY, &u32StartDelay ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32DelayBeforeStartThisProc = u32StartDelay;
            }
            tU32 u32AbsStartTime;
            tNewProcConfig.u32AbsStartTime = 0;
            if ( !oProcReg.bQueryU32( SPM_STARTUP_KEY_CONF_PROCESS_TIME_ABS, &u32AbsStartTime ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            } else {
               tNewProcConfig.u32AbsStartTime = u32AbsStartTime;
            }

            strTmpbuf = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESS_SYNC_POINT, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }
            tNewProcConfig.strSyncPointName = strTmpbuf.c_str( );

            strTmpbuf = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESS_CONDITION, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }
            tNewProcConfig.strCondition = strTmpbuf.c_str( );

            strTmpbuf = "";
            if ( !oProcReg.bQueryString( SPM_STARTUP_KEY_CONF_PROCESS_ADDITIONAL_CONDITION, &strTmpbuf[0], (tU32)strTmpbuf.capacity( ) ) ){
               // ETG_TRACE_COMP(("SPM: !!!!!! Warning detected !!!!!!"));
            }
            tNewProcConfig.strAdditionCondition = strTmpbuf.c_str( );
         }

         // currently not spawned ;-)
         tNewProcConfig.bProcSpawned = FALSE;

         if ( tNewProcConfig.bProcEnabled ){
            // if this is an internal process then do not allow synchronization to others
            if ( tNewProcConfig.strProcessLocation.empty( ) ){
               // remove all sync blocks if available.
               tNewProcConfig.listBlocksToSync.clear( );
               // and create new entry in map
            }
            _oMapProcessConfig[(tChar*)rDir.dirent.s8Name] = tNewProcConfig;
         }
      }
               OSAL_s32IOClose( fdReg );
   }
} // vEvaluateProcessConfiguration

tVoid spm_tclStartupCommon::vReadRegistries( const TMapRegistries& registry ){
   /*!
     * \fn
     *  \brief
     *    Read registries from input registry files.
     *
     *  \param[in] registry: Registry map.
     *  \return  void
     *  \todo: use a registry path as an alternative
     ******
     */
   TMapRegistries::const_iterator it;

   SPM_NULL_POINTER_CHECK( _poclRegistry );
   for ( it = registry.begin( ); it != registry.end( ); ++it ){

      // \todo: use a registry path as an alternative
      if ( !_poclRegistry->bImportData( it->second.c_str( ), TRUE, it->first.c_str( ) ) ){
         ETG_TRACE_USR4( ( "Could not read registry file: %s", it->second.c_str( ) ) );
      } else {
         ETG_TRACE_USR4( ( "Registry file %s successfully read", it->second.c_str( ) ) );
      }
   }
}

tVoid spm_tclStartupCommon::vReadEarlyProcessRegistries( ){
   /*!
     * \fn
     *  \brief
     *    Read registries for early processes.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   TMapProcessConfiguration::iterator it;

   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(_poclStartupInvest, ISpmStartupInvestigationServer);
   SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(poStartUp, ISpmStartupSystem);

   SPM_NULL_POINTER_CHECK( _poclRegistry );
   for ( it = _oMapProcessConfig.begin( ); it != _oMapProcessConfig.end( ); ++it ){

      if ( !it->second.bProcEnabled ){                              // is process enabled?
         continue;                                                  // no, then continue with next process, registry is not loaded here!!
      }
      if ( ( SPM_STARTUP_REG_TYPE_PRELOAD & it->second.u32RegType ) // early reading activated?
           && ( !it->second.strRegPath.empty( ) ) ){
         // now read all registries of the processes which are marked with
         // reg_type = 1, the path is not empty and not "NoReg"
          if ( poStartUp->bAlternativeProcConfig( it->second, &it->second.strProcessName[0] ) ){
              if ( !_poclRegistry->bImportDataByFile( it->second.strRegPath, FALSE, it->second.strProcessName ) ){
                 ETG_TRACE_ERRMEM( ( "Could not read early registry file: %s", it->second.strRegPath.c_str( ) ) );
              } else {
                 ETG_TRACE_USR1( ( "Registry file %s successfully early read", it->second.strRegPath.c_str( ) ) );
                 // store in startup investigation

                 SPM_STARTUPINVEST_INIT_STARTUPITEM
                 SPM_STARTUPINVEST_FORCE_ENTRY("STARTUP", ( (std::string)"registry: '" + it->second.strRegPath.c_str( ) + "' preloaded." ) )

              }

          }
      }
   }
} // vReadEarlyProcessRegistries

tVoid spm_tclStartupCommon::vReadBaseRegistry( ){
   /*!
     * \fn
     *  \brief
     *    Read base registry base.reg.
     *
     *  \param
     *  \return  void
     *
     *  \details read registry with name base.reg as block BASE
     ******
     */
   SPM_GET_IF_REFERENCE_USE_VAR( _poclRegistry, ISpmRegistry );
   if ( !_poclRegistry->bImportData( SPM_BASE_REG_NAME, 0, "BASE" ) ){
      // Get the base.reg from persistent memory
      _poclRegistry->bImportData( "" );
   } else {
      ETG_TRACE_USR4( ( "Registry file base.reg successfully read" ) );
   }
}

/**
  *  \brief only check if a registry actually exists
  *
  *  \param [in] strRegistryName Name of the registry to be checked
  *  \return name of the registry to be loaded
  *
  *  \details checks if the file exists, if not an entry in the startup item list is made and the default registry is returned
  */
std::string spm_tclStartupCommon::sDetermineStartupRegistry( std::string strRegistryName ){
   OSAL_tIODescriptor fd = OSAL_IOOpen( strRegistryName.c_str( ), OSAL_EN_READONLY );

   if ( fd == OSAL_ERROR ){
      ETG_TRACE_FATAL( ( "sDetermineStartupRegistry(): File not found: '%s'. Try to extract default.", strRegistryName.c_str( ) ) );
      {
         // store in startup investigation
         SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL(_poclStartupInvest, ISpmStartupInvestigationServer);
         SPM_STARTUPINVEST_INIT_STARTUPITEM
         SPM_STARTUPINVEST_FORCE_ENTRY( "STARTUP", ( (std::string)"LastMode registry '" + strRegistryName + "' not found! Load default instead." ) )
      }
      std::string astrcCfgRegs[SPM_STARTUP_CONFIGURATION_MAX_NB] = SPM_STARTUP_CONFIGURATION;
      return( astrcCfgRegs[SPM_STARTUP_CONFIGURATION_DEFAULT] );

   } else {
      OSAL_s32IOClose( fd );
      // extract now specific startup configuration
   }

   return( strRegistryName );
} // sDetermineStartupRegistry

tVoid spm_tclStartupCommon::vReadStartupConfiguration( ){
   /*!
     * \fn
     *  \brief
     *    Read the currently valid start configuration registry.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   SPM_NULL_POINTER_CHECK( _poclRegistry );
   dp_tclSpmDpInternDataStartupConfig oStartupConfig;

   tU32                               u32CfgNb                                       = oStartupConfig.tGetData( );

   std::string                        astrcCfgRegs[SPM_STARTUP_CONFIGURATION_MAX_NB] = SPM_STARTUP_CONFIGURATION;

   SPM_GET_IF_REFERENCE_NEW_VAR( poOsalProxy, ISpmOsalProxy );
   tU8                                u8ApplicationMode                              = poOsalProxy->u8GetApplicationMode( );
   tU8                                u8OperationalState                             = poOsalProxy->u8GetOperationalState( );

   std::string                        strStartConfigRegistryName;

   #ifdef SPM_STARTUP_CONFIGURATION_SWU
      if ( u8ApplicationMode == DEV_WUP_C_U8_APPLICATION_MODE_DOWNLOAD ){
         // load the registry for applicationmode download
         ETG_TRACE_ERRMEM( ( "spm_tclStartupCommon::vReadStartupConfiguration: SPM started in SWU mode!" ) );
         strStartConfigRegistryName = sDetermineStartupRegistry( SPM_STARTUP_CONFIGURATION_SWU );
      } else
   #endif
   {
   #ifdef SPM_STARTUP_CONFIGURATION_MIN_TARGET
          // check if minimal target has to be started
          SPM_GET_IF_REFERENCE_NEW_VAR( poclStartupSupervisor, ISpmStartupSupervisor );

          if ( poclStartupSupervisor->bStartMinConfig( ) ){
              ETG_TRACE_ERRMEM( ( "spm_tclStartupCommon::vReadStartupConfiguration(): Start minimal configuration." ) );
              u32CfgNb = SPM_STARTUP_CONFIGURATION_MIN_TARGET;
              strStartConfigRegistryName = sDetermineStartupRegistry( astrcCfgRegs[u32CfgNb] );
          } else
          {
   #endif



   #ifdef SPM_STARTUP_CONFIGURATION_OPERATION
              switch (u8OperationalState) {
              case DEV_WUP_C_U8_OPERATIONAL_STATE_DIAGNOSIS_0 :
              {
                  // load the registry for diagnosis mode 0 intended to start only diagnosis apps
                  ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration: SPM started in Diagnosis mode 0!" ) );
                  strStartConfigRegistryName = sDetermineStartupRegistry( SPM_STARTUP_CONFIGURATION_DIAGNOSIS_0 );
                  break;
              }
              case DEV_WUP_C_U8_OPERATIONAL_STATE_DIAGNOSIS_1 :
              {
                  //load the registry for diagnosis mode 1 intended to start diagnosis and stable apps
                  ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration: SPM started in Diagnosis mode 1!" ) );
                  strStartConfigRegistryName = sDetermineStartupRegistry( SPM_STARTUP_CONFIGURATION_DIAGNOSIS_1 );
                  break;
              }
#ifdef SPM_ENABLE_DM_VERITY_CHECK
              case DEV_WUP_C_U8_OPERATIONAL_STATE_DMVERITY_CHECK :
              {
                  //load the registry for DM Verity check intended to start only script to check filesystem
                  ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration: SPM started in DM Verity!" ) );
                  strStartConfigRegistryName = sDetermineStartupRegistry( SPM_STARTUP_CONFIGURATION_DMVERITY );
                  break;
              }
#endif
              case DEV_WUP_C_U8_OPERATIONAL_STATE_NORMAL      :
              {
                  ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration: SPM started in normal mode " ) );
                  if ( SPM_STARTUP_CONFIGURATION_MAX_NB <= u32CfgNb ){
                      u32CfgNb = SPM_STARTUP_CONFIGURATION_DEFAULT; // set default configuration
                  }
                  strStartConfigRegistryName = sDetermineStartupRegistry( astrcCfgRegs[u32CfgNb] );
                  break;
              }
              case DEV_WUP_C_U8_OPERATIONAL_STATE_UNKNOWN     :
              {
                  ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration: UNKNOWN Operation State.. now start in normal mode!" ) );
                    if ( SPM_STARTUP_CONFIGURATION_MAX_NB <= u32CfgNb ){
                      u32CfgNb = SPM_STARTUP_CONFIGURATION_DEFAULT; // set default configuration
                  }
                  strStartConfigRegistryName = sDetermineStartupRegistry( astrcCfgRegs[u32CfgNb] );
                  break;
              }
              default :
              {  
                  // In other case load default configuration
                  if ( SPM_STARTUP_CONFIGURATION_MAX_NB <= u32CfgNb ){
                      u32CfgNb = SPM_STARTUP_CONFIGURATION_DEFAULT; // set default configuration
                  }
                  strStartConfigRegistryName = sDetermineStartupRegistry( astrcCfgRegs[u32CfgNb] );
                  break;
              }
              }
   #endif
   #ifdef SPM_STARTUP_CONFIGURATION_MIN_TARGET
          }
   #endif
   #ifndef SPM_STARTUP_CONFIGURATION_OPERATION
          // check if entry is available and file exists
          if ( SPM_STARTUP_CONFIGURATION_MAX_NB <= u32CfgNb ){
             u32CfgNb = SPM_STARTUP_CONFIGURATION_DEFAULT; // set default configuration
          }
          strStartConfigRegistryName = sDetermineStartupRegistry( astrcCfgRegs[u32CfgNb] );
   #endif
   }

   #ifdef SPM_STARTUP_CONFIGURATION_PROJECT_REG
      if ( spm_tclFactory::_poFactoryRef->bIsSpecialStartupRegActive( ) ){
         OSAL_tIODescriptor fd = OSAL_IOOpen( SPM_STARTUP_CONFIGURATION_PROJECT_REG, OSAL_EN_READONLY );

         if ( fd == OSAL_ERROR ){
            ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration(): Project specific registry not found: '%s'. Try to extract default.", SPM_STARTUP_CONFIGURATION_PROJECT_REG ) );
            {
               // store in startup investigation
               ISpmStartupInvestigationServer *_poclStartupInvest = dynamic_cast < ISpmStartupInvestigationServer* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler( "ISpmStartupInvestigationServer" ) );

               if ( _poclStartupInvest ){
                  SPM_STARTUPINVEST_INIT_STARTUPITEM
                  SPM_STARTUPINVEST_FORCE_ENTRY( "STARTUP", ( (std::string)"Special Project registry '" + SPM_STARTUP_CONFIGURATION_PROJECT_REG + "' not found! Load default instead." ) )
               }
            }
         } else {
            OSAL_s32IOClose( fd );
            // extract now specific startup configuration
            strStartConfigRegistryName = SPM_STARTUP_CONFIGURATION_PROJECT_REG;
            ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration(): Start special project registry: '%s'.", strStartConfigRegistryName.c_str( ) ) );
         }

      }
   #endif // ifdef SPM_STARTUP_CONFIGURATION_PROJECT_REG

#  ifdef SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG
   if ( spm_tclFactory::_poFactoryRef->bIsTargetUnconfigured() ){
       OSAL_tIODescriptor fd = OSAL_IOOpen( SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG, OSAL_EN_READONLY );

       if ( fd == OSAL_ERROR ){
          ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration(): Project specific registry for unconfigured target not found: '%s'. Try to extract default.", SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG ) );
          {
             // store in startup investigation
             ISpmStartupInvestigationServer *_poclStartupInvest = dynamic_cast < ISpmStartupInvestigationServer* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler( "ISpmStartupInvestigationServer" ) );

             if ( _poclStartupInvest ){
                SPM_STARTUPINVEST_INIT_STARTUPITEM
                   SPM_STARTUPINVEST_FORCE_ENTRY( "STARTUP", ( (std::string)"Special Project registry (unconfig)'" + SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG + "' not found! Load default instead." ) )
             }
          }
       } else {
          OSAL_s32IOClose( fd );
          // extract now specific startup configuration
          strStartConfigRegistryName = SPM_STARTUP_CONFIGURATION_PROJECT_UNCONFIGURED_REG;
          ETG_TRACE_FATAL( ( "spm_tclStartupCommon::vReadStartupConfiguration(): Start special project registry: '%s'.",  strStartConfigRegistryName.c_str( )) );
       }

   }
#  endif

   ETG_TRACE_FATAL( ( "vReadStartupConfiguration(): extract now specific startup configuration: '%s'", strStartConfigRegistryName.c_str( ) ) );
   if ( _poclRegistry->bImportData( strStartConfigRegistryName, TRUE ) ){
            ETG_TRACE_USR4( ( "Startup configuration successfully extraced to registry." ) );
   }

   {
      // store in startup investigation
      SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY(_poclStartupInvest, ISpmStartupInvestigationServer);
      SPM_STARTUPINVEST_INIT_STARTUPITEM
      SPM_STARTUPINVEST_FORCE_ENTRY( "STARTUP", ( (std::string)"Startup registry: '" + strStartConfigRegistryName + "' loaded." ) )
   }

} // vReadStartupConfiguration

tVoid spm_tclStartupCommon::vReadRegistries( ){
   /*!
     * \fn
     *  \brief
     *    Read registries.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   TMapRegistries r;

   // read in the paths of additional local registries from /dev/registries
   vEvaluateRegistry( SPM_STR_LOCAL_REGISTRY, r );
   // now read them into the sequentially
   vReadRegistries( r );
}

tVoid spm_tclStartupCommon::vReadEarlyRegistries( ){
   /*!
     * \fn
     *  \brief
     *    Read early registries.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   TMapRegistries r;

   // read in the paths of additional local registries from /dev/registries
   vEvaluateRegistry( SPM_STR_EARLY_REGISTRY, r );
   // now read them into the sequentially
   vReadRegistries( r );
}

tVoid spm_tclStartupCommon::vEvaluateProcessStartup( ){
   /*!
     * \fn
     *  \brief
     *    Evaluate Process Startup.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   ETG_TRACE_USR4( ( "spm_tclStartupCommon::vEvaluateProcessStartup()" ) );
   vEvaluateProcessConfiguration( );

   #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
      if ( _variant != NULL ){
         ETG_TRACE_USR4( ( "spm_tclStartupCommon::vEvaluateProcessStartup() VARIANT HANDLING" ) );
         // read in the variant configuration from registry
         _variant->vEvaluateProcessVariant( );
         _variant->vEvaluateProcessVariantV2( );
         // evaluate the variant expressions
         _variant->vCalculateValidConfiguration( );
         // get the list of processes to be deactivated from variant object
         vDeactivateProcesses( _variant->oGetDisabledProcesses( ) );
      }
   #endif

   vReadEarlyProcessRegistries( );
} // vEvaluateProcessStartup

tVoid spm_tclStartupCommon::vUpdateRegistryDueToValidVariant( ) const {
   /*!
     * \fn
     *  \brief
     *    Update the registry.
     *
     *  \param
     *  \return  void
     *
     ******
     */

   #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
      if ( _variant != NULL ){
         _variant->vUpdateRegistry( );
      }
   #endif
}

tVoid spm_tclStartupCommon::vTraceProcessConfiguration( ){
   /*!
     * \fn
     *  \brief
     *    Process Configuration added to trace.
     *
     *  \param
     *  \return  void
     *
     ******
     */
   TMapProcessConfiguration::iterator it;

   for ( it = _oMapProcessConfig.begin( ); it != _oMapProcessConfig.end( ); ++it ){
      std::string          strSwBlocks = "";
      TBlockList::iterator pos;
      for ( pos = it->second.listSwBlocks.begin( ); pos != it->second.listSwBlocks.end( ); ++pos ){
         strSwBlocks += pos->strBlockName;
         strSwBlocks += " ";
      }

      std::string strSyncBlocks = "";
      for ( pos = it->second.listBlocksToSync.begin( ); pos != it->second.listBlocksToSync.end( ); ++pos ){
         strSyncBlocks += pos->strBlockName;
         strSyncBlocks += " ";
      }

      ETG_TRACE_FATAL( ( "**********************************************************************************" ) );
      ETG_TRACE_FATAL( ( "Name              : %s", it->second.strProcessName.c_str( ) ) );
      ETG_TRACE_FATAL( ( "**********************************************************************************" ) );
      ETG_TRACE_FATAL( ( "Type              : prio: %d, type: %d, reg: %d, sync: '%d'", it->second.u32ProcPrio, it->second.u32ProcType, ETG_ENUM( SPM_STARTUP_REG_TYPE, it->second.u32RegType ), ETG_ENUM( SPM_SYNC_MODE, it->second.u32LastSync ) ) );
      ETG_TRACE_FATAL( ( "Status            : enabled: %d, spawned: %d, PredecessorSeen: %d", it->second.bProcEnabled, it->second.bProcSpawned, it->second.bPredecessorSeen ) );
      ETG_TRACE_FATAL( ( "StartType         : '%s'", it->second.strProcStartType.c_str( ) ) );
      ETG_TRACE_FATAL( ( "RegPath           : '%s'", it->second.strRegPath.c_str( ) ) );
      ETG_TRACE_FATAL( ( "Location          : '%s'", it->second.strProcessLocation.c_str( ) ) );
      ETG_TRACE_FATAL( ( "ShutdownEvent     : '%s'", it->second.strProcShutdownEvent.c_str( ) ) );
      ETG_TRACE_FATAL( ( "ConfigPath        : '%s'", it->second.strProcConfigBasePath.c_str( ) ) );
      ETG_TRACE_FATAL( ( "CGroup            : MemLevel1: %d, MemLevel2: %d, MemScript : '%s'", it->second.u32CGroupMemLevel1, it->second.u32CGroupMemLevel2, it->second.strCGroupMemScriptPath.c_str( ) ) );
      ETG_TRACE_FATAL( ( "CGroupCpuScript   : '%s'", it->second.strCGroupCpuScriptPath.c_str( ) ) );
      ETG_TRACE_FATAL( ( "NiceLevel         : start: %d, end: %d", it->second.u32NiceLevel, it->second.u32NiceLevelEnd ) );
      ETG_TRACE_FATAL( ( "Affinity          : start: %d, end: %d", it->second.u32Affinity, it->second.u32AffinityEnd ) );
      ETG_TRACE_FATAL( ( "CGroupCpuScript   : '%s'", it->second.strCGroupCpuScriptPath.c_str( ) ) );
      ETG_TRACE_FATAL( ( "ProcessID         : '%d'", it->second.s32ProcId ) );
      ETG_TRACE_FATAL( ( "SwBlocks          : '%s'", strSwBlocks.c_str( ) ) );
      ETG_TRACE_FATAL( ( "SyncBlocks        : '%s'", strSyncBlocks.c_str( ) ) );
      ETG_TRACE_FATAL( ( "**********************************************************************************" ) );
   }
} // vTraceProcessConfiguration

std::string spm_tclStartupCommon::strGetProcNameByBlockName( const std::string& strBlockName ){
/*!
  * \fn
  *  \brief get the process name of a given software block
  *  \param [in] strBlockName: name of the software block.
  *  \return string: Process name.
  ******
  */
   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){

      TBlockList::iterator pos;
      for ( pos = posProc->second.listSwBlocks.begin( ); pos != posProc->second.listSwBlocks.end( ); ++pos ){
         if ( ( pos->strBlockName == strBlockName ) ){
            // OK, found SwBlock
            ETG_TRACE_USR4( ( "spm_tclStartupCommon::strGetProcNameByBlockName(): %40s (%40s)", strBlockName.c_str( ), posProc->second.strProcessName.c_str( ) ) );
            return( posProc->second.strProcessName );
         }
      }
   }
   return( "" );
} // strGetProcNameByBlockName

std::string spm_tclStartupCommon::strGetProcLocationByBlockName( const std::string& strBlockName ){
/*!
  * \fn
  *  \brief get the process location of a given software block
  *  \param [in] strBlockName: name of the software block.
  *  \return string: Process location.
  ******
  */
   TMapProcessConfiguration::iterator posProc;

   for ( posProc = _oMapProcessConfig.begin( ); posProc != _oMapProcessConfig.end( ); ++posProc ){

      TBlockList::iterator pos;
      for ( pos = posProc->second.listSwBlocks.begin( ); pos != posProc->second.listSwBlocks.end( ); ++pos ){
         if ( strBlockName == pos->strBlockName ){
            // OK, found SwBlock
            ETG_TRACE_USR4( ( "spm_tclStartupCommon::strGetProcLocationByBlockName(): %40s (%40s) and Process Location '%80s'", strBlockName.c_str( ), posProc->second.strProcessName.c_str( ), posProc->second.strProcessLocation.c_str( ) ) );
            return( posProc->second.strProcessLocation );
         }
      }
   }
   return( "" );
} // strGetProcLocationByBlockName

