/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_IStartupSystemVariant.h
  * @brief        This is header file contains interfaces to add variants , evaluate process variant and calculate valid configuration.
  * @copyright    (C) 2013 - 2016 Robert Bosch Engineering and Business Solutions Limited.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <vector>
#include <string>
#include <set>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "spm_ISpmBase.h"

#define SPM_STARTUP_VARIANT_VALUE_NUMERICAL 0
#define SPM_STARTUP_VARIANT_VALUE_STRING    1

#define SPM_STR_VARIANT_SERVICES_PATH     "SERVICES"
#define SPM_STR_VARIANT_DELETE_COMPONENT  "APPID"
#define SPM_STR_VARIANT_DELETE_SERVICE    "SERVICEID"


// used for variables which have fixed values coming from defined cells
// of the spm environment and for values which are defined in the registry
// we have 3 value classes for numerical values.
// 1.) container
// 2.) single value: is like variable
// 3.) value range: compare with the range limits
// for strings we have
// 1.) the single value
// 2.) several values

/*!
  *  \class TSpmVariantVariable
  *  \brief This abstract class provides interfaces to initialize numerical values and type.
  */
class TSpmVariantVariable
{
public:
TSpmVariantVariable( ){
   _numVal = 0;
   _type   = SPM_STARTUP_VARIANT_VALUE_STRING;
}  // necessary for STL

explicit TSpmVariantVariable( const std::string& s ){
   _type   = SPM_STARTUP_VARIANT_VALUE_STRING;
   _numVal = 0;
   _strVal = s;
}

explicit TSpmVariantVariable( tU32 n ){
   _type   = SPM_STARTUP_VARIANT_VALUE_NUMERICAL;
   _numVal = n;
}

tUInt       _type;
std::string _strVal;
tU32        _numVal;
};

/*!
  *  \class ISpmVariantCompare
  *  \brief This abstract class provides interfaces to compare variants.
  */
class ISpmVariantCompare
{
public:
virtual bool operator==( const TSpmVariantVariable& op ) const = 0;

virtual ~ISpmVariantCompare( ){}
};

/*!
  *  \class TSpmVariantSingleValue
  *  \brief This abstract class provides interfaces to compare variants and initialize numerical value and its type.
  */
class TSpmVariantSingleValue : public ISpmVariantCompare
{
public:
TSpmVariantSingleValue( tUInt              nType,
                        const std::string& strVal,
                        tU32               u32val ){
   _type   = nType;
   _strVal = strVal;
   _numVal = u32val;
}

// do not allow implicit conversion
explicit TSpmVariantSingleValue( const std::string& strVal ){
   _type   = SPM_STARTUP_VARIANT_VALUE_STRING;
   _strVal = strVal;
   _numVal = 0;
}

explicit TSpmVariantSingleValue( tU32 u32val ){
   _type   = SPM_STARTUP_VARIANT_VALUE_NUMERICAL;
   _numVal = u32val;
}

virtual bool operator==( const TSpmVariantVariable& op ) const {
   if ( _type == op._type ){
      switch ( _type ){
         case SPM_STARTUP_VARIANT_VALUE_NUMERICAL:
            return( _numVal == op._numVal );

         case SPM_STARTUP_VARIANT_VALUE_STRING:
            return( _strVal == op._strVal );

         default:
            return( false );
      }
   }
   return( false );
}

//private:
tUInt       _type;
std::string _strVal;
tU32        _numVal;
};


// type not needed here, only numerical allowed

/*!
  *  \class TSpmVariantRange
  *  \brief This abstract class provides interfaces to compare numerical value of variant from range of numerical values .
  */
class TSpmVariantRange : public ISpmVariantCompare
{
public:
// do not allow implicit conversion
explicit TSpmVariantRange( tU32 l,
                           tU32 u ){
   _numValLower = l;
   _numValUpper = u;
}

virtual bool operator==( const TSpmVariantVariable& op ) const {
   if ( op._type == SPM_STARTUP_VARIANT_VALUE_NUMERICAL ){
      return( op._numVal >= _numValLower && op._numVal <= _numValUpper );
   }
   return( false );
}

private:
tU32 _numValLower;
tU32 _numValUpper;
};


struct TSpmVariantValue : public ISpmVariantCompare
{
   std::vector < ISpmVariantCompare* > _items;

   virtual bool operator==( const TSpmVariantVariable& op ) const {
      std::vector < ISpmVariantCompare* >::const_iterator it;

      for ( it = _items.begin( ); it != _items.end( ); ++it ){
         if ( ( *it )->operator==( op ) ){
            return( true );
         }
      }
      return( false );
   }

   virtual ~TSpmVariantValue( ){
      // deallocate all ISpmVariantCompare Items
      std::vector < ISpmVariantCompare* >::iterator it;

      for ( it = _items.begin( ); it != _items.end( ); ++it ){
         delete ( * it );
      }
      _items.clear( );
   }

};


struct TSpmVariantSwitch
{
   std::map < tU32, TSpmVariantValue* > _values;
   std::map < tU32, std::string > _expressions;
   std::string _exclude;

   virtual ~TSpmVariantSwitch( ){
      std::map < tU32, TSpmVariantValue* >::iterator it;

      for ( it = _values.begin( ); it != _values.end( ); ++it ){
         delete it->second;
      }
      _values.clear( );
   }

};

struct TSpmVariant
{
   std::map < tU32, TSpmVariantVariable > _variables;
   std::map < std::string, TSpmVariantSwitch > _switch;
};

// //////////////////////////////////////////////////////////////////////////////
// class for the variant evaluation during startup

/*!
  *  \class ISpmStartupSystemVariant
  *  \brief This abstract class provides interfaces to evaluate variant during startup.
          This class also delivers interfaces to add variable , evaluate process variant and calculate valid configuration.
  */
class ISpmStartupSystemVariant : public ISpmBase
{
public:
ISpmStartupSystemVariant( const ISpmFactory& factory ) : ISpmBase( factory ){ }
virtual ~ISpmStartupSystemVariant( ){ }

// all values are fix at startup
virtual tVoid vAddVariable( const std::string& strName,
                            tU32               u32Value )                    = 0;

virtual tVoid vAddVariable( const std::string& strName,
                            const std::string& strValue )                    = 0;

virtual tVoid vEvaluateProcessVariant( )                                     = 0;

virtual tVoid vEvaluateProcessVariantV2( )                                   = 0;

virtual tVoid vCalculateValidConfiguration( )                                = 0;

virtual tVoid vUpdateRegistry( )                                             = 0;

virtual const std::set < std::string >& oGetDisabledProcesses( ) const       = 0;

virtual tVoid vTrace( )                                                      = 0;

virtual const tChar*getInterfaceName( ) const { return( "ISpmStartupSystemVariant" ); }

protected:
virtual TSpmVariantValue*oEvaluateValue( const std::string& strVal )         = 0;

virtual TSpmVariantValue*oEvaluateString( const std::string& str )           = 0;

virtual TSpmVariantValue*oEvaluateNumber( const std::string& strNum )        = 0;

virtual tBool bEvaluateVariantExpression( const std::map < tU32, TSpmVariantValue* >& values,
                                          const std::string& strExpression ) = 0;

virtual tVoid vDeactivateList( const std::string& strExcl )                  = 0;

virtual tVoid vDeactivate( const std::string& strExcl )                      = 0;

};

