/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_IStartupCommon.h
  * @brief        This is header file contains interface functions for StartupCommon (start-up configuration).
  *               also interfaces to add start-up system and update registry to valid variant.
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>
#include <set>
#include <list>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISwBlock.h"
#include "spm_GlobDefs.h"
#include "spm_baseClass.h"
#include "spm_IWorkerClient.h"
#include "spm_ProcConfig.h"

class ISpmApplicationDatabase;
class ISpmCcaSupplierServer;
class ISpmStartupSystem;
class ISpmSwBlockServer;

#define SPM_STARTUP_REG_TYPE_LOAD_DIRECTLY          0x00000001
#define SPM_STARTUP_REG_TYPE_PRELOAD                0x00000002
#define SPM_STARTUP_REG_TYPE_NOREG                  0x00000004
#define SPM_STARTUP_REG_TYPE_NOREG_EXT_BLOCK        0x00000008
#define PROC_START_TYPE_MAX                         20

#define SPM_STR_NO_REG_TO_LOAD         "NoReg"
#define SPM_STR_LOCAL_REGISTRY         "LOCAL"
#define SPM_STR_EARLY_REGISTRY         "EARLY"
#define SPM_BASE_REG_NAME              "base.reg"

#define SPM_STARTUP_FOLD_CONF_SWBLOCKS              "/SWBLOCKS"
#define SPM_STARTUP_FOLD_CONF_SYNCBLOCKS            "/SYNCBLOCKS"
#define SPM_STARTUP_FOLD_CONF_LASTMODE              "/LASTMODE"

#define SPM_STARTUP_FORMAT_PROCESS_SHUTDOWN_EVENT   "SHUTDOWN"

#define SPM_STARTUP_KEY_CONF_PROCESSNAME            "PROC_NAME"
#define SPM_STARTUP_KEY_CONF_PROCESS_START_TYPE     "PROC_START_TYPE"
#define SPM_STARTUP_KEY_CONF_PROCESSLOCATION        "PROC_LOCATION"
#define SPM_STARTUP_KEY_CONF_PROCESSREGISTRY        "PROC_REGISTRY"
#define SPM_STARTUP_KEY_CONF_PROCESS_SHUTDOWN_EVENT "PROC_SHUTD_EV"
#define SPM_STARTUP_KEY_CONF_PROCESS_PRIORITY       "PROC_PRIO"
#define SPM_STARTUP_KEY_CONF_PROCESS_ENABLED        "PROC_ENABLED"
#define SPM_STARTUP_KEY_CONF_PROCESS_AFFINITY       "PROC_AFFINITY"
#define SPM_STARTUP_KEY_CONF_PROCESS_AFFINITY_END   "PROC_AFFINITY_END"
#define SPM_STARTUP_KEY_CONF_PROCESS_TYPE           "PROC_TYPE"
#define SPM_STARTUP_KEY_CONF_PROCESS_SUPERVISION    "PROC_SUPERVISION"
#define SPM_STARTUP_KEY_CONF_PROCESS_TYPE_ARG       "PROC_TYPE_ARG"
#define SPM_STARTUP_KEY_CONF_REG_TYPE               "REG_TYPE"
#define SPM_STARTUP_KEY_CONF_CGROUP_CPU_PATH_SCRIPT "PROC_CGROUP_CPU_PATH_SCRIPT"
#define SPM_STARTUP_KEY_CONF_CGROUP_MEM_PATH_SCRIPT "PROC_CGROUP_MEM_PATH_SCRIPT"
#define SPM_STARTUP_KEY_CONF_CGROUP_MEM_LEVEL1      "PROC_CGROUP_MEM_LEVEL1"
#define SPM_STARTUP_KEY_CONF_CGROUP_MEM_LEVEL2      "PROC_CGROUP_MEM_LEVEL2"
#define SPM_STARTUP_KEY_CONF_PROCESS_NICE_LEVEL     "PROC_NICE_LEVEL"
#define SPM_STARTUP_KEY_CONF_PROCESS_NICE_LEVEL_END "PROC_NICE_LEVEL_END"
#define SPM_STARTUP_KEY_CONF_PROCESS_DELAY          "PROC_START_DELAY"
#define SPM_STARTUP_KEY_CONF_PROCESS_DELAY_BEFORE_NEXT_PROC  "PROC_START_SYNC_DELAY"
#define SPM_STARTUP_KEY_CONF_PROCESS_TIME_ABS       "PROC_START_TIME"
#define SPM_STARTUP_KEY_CONF_PROCESS_SYNC_POINT     "PROC_SYNC_POINT"
#define SPM_STARTUP_KEY_CONF_PROCESS_CONDITION      "PROC_CONDITION"
#define SPM_STARTUP_KEY_CONF_PROCESS_ADDITIONAL_CONDITION "PROC_ADDITIONAL_CONDITION"

#define SPM_STARTUP_KEY_CONF_PREDECESSOR            "PREDECESSOR"
#define SPM_STARTUP_KEY_CONF_EXCLUDE                "EXCLUDE"

#define SPM_STARTUP_KEY_VAR_PATH                    "PATH"
#define SPM_STARTUP_KEY_VAR_TYPE                    "TYPE"
#define SPM_STARTUP_KEY_VAR_KEY                     "KEY"
#define SPM_STARTUP_KEY_VAR_VALUE                   "VALUE"
#define SPM_STARTUP_KEY_VAR_TYPE_DWORD              "DWORD"
#define SPM_STARTUP_KEY_VAR_TYPE_STRING             "STRING"
#define SPM_STARTUP_KEY_VAR_PROCESSES               "PROCESSES"

#define SPM_STARTUP_KEY_HKEY                        "HKEY_"
#define SPM_STARTUP_KEY_TO_REG                      "/dev/registry/"

#define SPM_STARTUP_FORMAT_VARIABLE                 "VAR%02u"
#define SPM_STARTUP_FORMAT_VALUE                    "VAL%02u"
#define SPM_STARTUP_FORMAT_EXPRESSION               "EXPR%02u"

// //////////////////////////////////////////////////////////////////////////////
// structures and classes

// //////////////////////////////////////////////////////////////////////////////
// interface for a client class to work with spm_tclActive (inside a thread)

/*!
  *  \class ISpmStartupCommon
  *  \brief This abstract class provides interfaces regarding registry read or update and process stop or start.
   This class also delivers pure virtual functions to read and evaluate registries and start-up configuration .
   It also supplies interfaces to add startup system and update registry to valid variant.
  */
class ISpmStartupCommon : public spm_tclActive
   , public ISpmSwBlockClient
   , public ISpmWorkerClient
{
public:
ISpmStartupCommon( const ISpmFactory& factory ) : ISpmWorkerClient( factory ){ }
virtual ~ISpmStartupCommon( ){ }

virtual tBool bStopProcess( const std::string& strProcessName )                  = 0;

virtual tBool bStartProcess( const std::string& strProcessName )                 = 0;

virtual tBool bStartProcess( TProcConfiguration& _oConf,
                             tU32                u32HighPrioSync )               = 0;

virtual tVoid vRemoveSwBlock( const std::string& strSwBlockName,
                              tBool              bStopProcWithLastBlock = TRUE ) = 0;

virtual tVoid vRemoveStoppedService( const std::string& strServiceName )         = 0;

virtual tVoid vUpdateStartedService( const std::string& strServiceName )         = 0;

virtual void vOnSyncPointReached( const std::string& strSyncPointName )          = 0;

virtual tBool bForceProcessStop( const std::string& strProcessName )             = 0;

virtual tVoid vTraceProcessConfiguration( )                                      = 0;

virtual tVoid vReadBaseRegistry( )                                               = 0;

virtual tVoid vReadStartupConfiguration( )                                       = 0;

virtual tVoid vReadRegistries( )                                                 = 0;

virtual tVoid vReadRegistries( const TMapRegistries& r )                         = 0;

virtual tVoid vEvaluateProcessStartup( )                                         = 0;

virtual tVoid vAddStartupSystem( ISpmStartupSystem *var )                        = 0;

virtual tVoid vUpdateRegistryDueToValidVariant( ) const                          = 0;

virtual tBool bAddSwBlock( const std::string& strSwBlockName,
                           tBool              bBlockMode = FALSE,
                           tBool              bLocal = TRUE,
                           const std::string& strExecPath = "",
                           const std::string& strProcessName = "",
                           const std::string& strStartType = SPM_STARTUP_VALUE_OSAL_START_TYPE )           = 0;

virtual tBool bAddAdditionalSwBlock( const std::string& strBusShortName,
                                     const std::string& strServiceName,
                                     tU32               u32AppId,
                                     const std::string& strStartType = SPM_STARTUP_VALUE_OSAL_START_TYPE ) = 0;

virtual tVoid vAddClient( ISpmSwBlockClient *clt )                                                         = 0;

virtual tVoid vRemoveClient( ISpmSwBlockClient *clt )                                                      = 0;

virtual std::string strGetProcLocationByBlockName( const std::string& strBlockName )                       = 0;

virtual std::string strGetProcNameByBlockName( const std::string& strBlockName )                           = 0;

virtual tU32 u32GetSupervisionMode( tU32 ProcId )                                                          = 0;

virtual TMapProcessConfiguration& oGetProcessConfig( )                                                     = 0;

virtual std::string strGetProcName( tU32 u32ProcId )                                                       = 0;

virtual tU32 u32GetProcId( const std::string& strName )                                                    = 0;

virtual void vSetProcId( const std::string& strName,
                         const tU32         PID )                                                          = 0;

virtual tVoid vSetProcessStarted( const std::string& strProcessname,
                                  tBool              bIsStarted )                                          = 0;

virtual tVoid vEvaluateRegistry( const std::string& strConf,
                                 TMapRegistries   & registry ) const                                       = 0;

virtual tVoid vReadEarlyRegistries( )                                                                      = 0;

virtual const tChar*getInterfaceName( ) const { return( "ISpmStartupCommon" ); }
};

