/*! ****************************************************************************
 * @file:         spm_GlobDefs.h
 * @PROJECT:      NextGen
 * @SW-COMPONENT: FC SPM
 * ----------------------------------------------------------------------------
 *
 * @brief:  IDs which are used in the FC SPM
 *
 * ----------------------------------------------------------------------------
 * @COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
 * HISTORY:
 * Date      | Author            | Modification
 * 07.09.11  | TMS Fischer       | initial version
 *****************************************************************************/
#pragma once

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "spm_FITypes.h"

#include <string>

#define SPM_WRITE_DIRECTLY  TRUE

/************************************************
  ************************************************
  *
  *  NULLPOINTER CHECKS
  *
  ************************************************
  *************************************************/

/*! \def SPM_NULL_POINTER_CHECK
  *     check if pointer is valid. FATAL_ASSERT if not.
  *     Do a dummy return at the end - will never be done after FATAL_ASSERT anyway
  */
#define SPM_NULL_POINTER_CHECK( poClass ) \
   if ( poClass == NULL ){ \
      FATAL_M_ASSERT_ALWAYS( ); \
      return; \
   }

/*! \def SPM_NULL_POINTER_CHECK_VAL
  *     check if pointer is valid. FATAL_ASSERT if not.
  *     Do a dummy return at the end - will never be done after FATAL_ASSERT anyway
  */
#define SPM_NULL_POINTER_CHECK_VAL( poClass ) \
   if ( poClass == NULL ){ \
      FATAL_M_ASSERT_ALWAYS( ); \
      return 0; \
   }

/************************************************
  ************************************************
  *
  *  get interface references
  *
  ************************************************
  *************************************************/

/*! \def SPM_GET_IF_REFERENCE_USE_VAR
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_IF_REFERENCE_USE_VAR( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getSpmObjHandler(#ifName ) ); \
   SPM_NULL_POINTER_CHECK( poVarName );

/*! \def SPM_GET_IF_REFERENCE_USE_VAR_VAL
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_IF_REFERENCE_USE_VAR_VAL( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getSpmObjHandler(#ifName ) ); \
   SPM_NULL_POINTER_CHECK_VAL( poVarName );

/*! \def SPM_GET_IF_REFERENCE_NEW_VAR
  *     get the reference to an interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_IF_REFERENCE_NEW_VAR( poVarName, ifName ) \
   ifName * SPM_GET_IF_REFERENCE_USE_VAR( poVarName, ifName );

/*! \def SPM_GET_IF_REFERENCE_NEW_VAR_VAL
  *     get the reference to an interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_IF_REFERENCE_NEW_VAR_VAL( poVarName, ifName ) \
   ifName * SPM_GET_IF_REFERENCE_USE_VAR_VAL( poVarName, ifName );

/************************************************
  ************************************************
  *
  *  get interface references via static factory
  *
  ************************************************
  *************************************************/

/*! \def SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler(#ifName ) ); \
   SPM_NULL_POINTER_CHECK( poVarName );

/*! \def SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY_VAL
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY_VAL( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler(#ifName ) ); \
   SPM_NULL_POINTER_CHECK_VAL( poVarName );

/*! \def SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY
  *     get the reference to an interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY( poVarName, ifName ) \
   ifName * SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY( poVarName, ifName );

/*! \def SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL
  *     get the reference to an interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_IF_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poVarName, ifName ) \
   ifName * SPM_GET_IF_REFERENCE_USE_VAR_STATIC_FACTORY_VAL( poVarName, ifName );

/************************************************
  ************************************************
  *
  *  get class references
  *
  *************************************************
  *************************************************/

/*! \def SPM_GET_CLASS_REFERENCE_USE_VAR
  *     get the reference to a specific class and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLASS_REFERENCE_USE_VAR( poVarName, className, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getSpmObjHandler(# className ) ); \
   SPM_NULL_POINTER_CHECK( poVarName );

/*! \def SPM_GET_CLASS_REFERENCE_USE_VAR_VAL
  *     get the reference to a specific class and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLASS_REFERENCE_USE_VAR_VAL( poVarName, className, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getSpmObjHandler(# className ) ); \
   SPM_NULL_POINTER_CHECK_VAL( poVarName );

/*! \def SPM_GET_CLASS_REFERENCE_NEW_VAR
  *     get the reference to a specific class and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_CLASS_REFERENCE_NEW_VAR( poVarName, className, ifName ) \
   ifName * SPM_GET_CLASS_REFERENCE_USE_VAR( poVarName, className, ifName );

/*! \def SPM_GET_CLASS_REFERENCE_NEW_VAR_VAL
  *     get the reference to a specific class and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLASS_REFERENCE_NEW_VAR_VAL( poVarName, className, ifName ) \
   ifName * SPM_GET_CLASS_REFERENCE_USE_VAR_VAL( poVarName, className, ifName );

/*! \def SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY( poVarName, className, ifName ) \
   ifName * poVarName = dynamic_cast < ifName* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler(#className ) ); \
   SPM_NULL_POINTER_CHECK( poVarName );

/*! \def SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL
  *     get the reference to an interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLASS_REFERENCE_NEW_VAR_STATIC_FACTORY_VAL( poVarName, className, ifName ) \
   ifName * poVarName = dynamic_cast < ifName* >( spm_tclFactory::_poFactoryRef->getSpmObjHandler(#className ) ); \
   SPM_NULL_POINTER_CHECK_VAL( poVarName );

/************************************************
  ************************************************
  *
  *  get client handler interface references
  *
  ************************************************
  *************************************************/

/*! \def SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR
  *     get the reference to a client handler interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getClientHandler( #ifName ) ); \
   SPM_NULL_POINTER_CHECK( poVarName );

/*! \def SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR_VAL
  *     get the reference to a client handler interface and assign it to the given existing variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR_VAL( poVarName, ifName ) \
   poVarName = dynamic_cast < ifName* >( _crfFactory.getClientHandler( #ifName ) ); \
   SPM_NULL_POINTER_CHECK_VAL( poVarName );

/*! \def SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR
  *     get the reference to a client handler interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK
  */
#define SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR( poVarName, ifName ) \
   ifName * SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR( poVarName, ifName )

/*! \def SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL
  *     get the reference to a client handler interface and assign it to the newly created variable
  *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
  */
#define SPM_GET_CLIENT_HANDLER_IF_REFERENCE_NEW_VAR_VAL( poVarName, ifName ) \
   ifName * SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR_VAL( poVarName, ifName )

/************************************************
  ************************************************
  *
  *  create new object in spm
  *
  ************************************************
  *************************************************/

/*! \def SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF
 *     Create new object of spm
 *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
 *     Add object handler to factory and trace handling
 *     Measure creation time of object
 */
#define SPM_CREATE_NEW_SPM_CLASS_WITH_TRACEIF( poVarName, functionName, className ) \
   SPM_STARTUPINVEST_SET_STARTTIME; \
   poVarName = functionName( ); \
   SPM_NULL_POINTER_CHECK( poVarName ); \
   vAddFactoryClient( poVarName ); \
   _poclTrace->vAddTraceClient( poVarName ); \
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: " #className " is created." );
  
/*! \def SPM_CREATE_NEW_SPM_CLASS_CLIENT_HANDLER
 *     Create new object of spm
 *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
 *     Add object handler to factory client handler
 *     Measure creation time of object
 */
#define SPM_CREATE_NEW_SPM_CLASS_CLIENT_HANDLER( poVarName, functionName, className ) \
   SPM_STARTUPINVEST_SET_STARTTIME; \
   poVarName = functionName( ); \
   SPM_NULL_POINTER_CHECK( poVarName ); \
   vAddFactoryClientHandler( poVarName ); \
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: " #className " is created." );
  
/*! \def SPM_CREATE_NEW_SPM_CLASS_BASE
 *     Create new object of spm
 *     Do a NullPointer check using SPM_NULL_POINTER_CHECK_VAL
 *     Add object handler to factory as ISpmBase
 *     Measure creation time of object
 */
#define SPM_CREATE_NEW_SPM_CLASS_BASE( poVarName, functionName, className ) \
   SPM_STARTUPINVEST_SET_STARTTIME; \
   poVarName = functionName( ); \
   SPM_NULL_POINTER_CHECK( poVarName ); \
   vAddFactoryClient( ( ISpmBase* ) poVarName ); \
   SPM_STARTUPINVEST_CHECK_TIME_MEASURE( "FC SPM: " #className " is created." );
  
/************************************************
  ************************************************
  *
  *  Framework defines (CCA/DBus)
  *
  *************************************************
  *************************************************/

/*! \def SPM_U32_FRAMEWORK_TYPE_CCA
  *  CCA is used as communication framework
  *  this is added to the application ID to specify how the communication to this app is done
  */

/*! \def SPM_U32_FRAMEWORK_TYPE_DBUS
  *  DBus is used as communication framework
  *  this is added to the application ID to specify how the communication to this app is done
  */

/*! \def SPM_U32_FRAMEWORK_TYPE_MASK
  *  mask the bits used to get only the communication type value
  */

/*! \def SPM_FRAMEWORK_IS_APP_LCM
  *  returns TRUE if this is a DBus Application
  */

/*! \def SPM_FRAMEWORK_IS_APP_CCA
  *  returns TRUE if this is a CCA Application
  */
#define SPM_U32_FRAMEWORK_TYPE_CCA   ( (tU32)0x00000000 )
#define SPM_U32_FRAMEWORK_TYPE_DBUS  ( (tU32)0x01000000 )
#define SPM_U32_FRAMEWORK_TYPE_MASK  ( (tU32)0xFF000000 )

#define SPM_FRAMEWORK_IS_APP_LCM( app )  ( ( SPM_U32_FRAMEWORK_TYPE_MASK & app ) == SPM_U32_FRAMEWORK_TYPE_DBUS )
#define SPM_FRAMEWORK_IS_APP_CCA( app )  ( ( SPM_U32_FRAMEWORK_TYPE_MASK & app ) == SPM_U32_FRAMEWORK_TYPE_CCA )

/*! \def SPM_FRAMEWORK_MAKE_APPID_LCM
  *  convert from "real" app-ID to LCM internal AppId also containing the Framework info.
  */
#define SPM_FRAMEWORK_MAKE_APPID_LCM( app ) ( SPM_U32_FRAMEWORK_TYPE_DBUS | app )

#define SPM_APPLICATION_CONFIGURATION_NO_SWBLOCK      "none"

/*! \def ELEMENTE
  *  return the number of elements in an array
  */
#define ELEMENTE( array )       ( (tU16)( sizeof( array ) / sizeof( array[0] ) ) )

#define SPM_SUBSTATE_MASK_ON                (tU32)0x80000000     // extern

#define SPM_SYSTEM_INVALID    0xFFFFFFFF
#define SPM_SYSTEM_NOT_RELEVANT    SPM_SYSTEM_INVALID

#define SPM_C_U32_STATE_UNCHANGED AMT_C_U32_STATE_INVALID

/************************************************
  ************************************************
  *
  *  StartupSynv defines
  *
  *************************************************
  *************************************************/
#define SPM_U32_SYNC_NONE                     0x00000000
#define SPM_U32_SYNC_CONNECTED                0x00000001
#define SPM_U32_SYNC_UP                       0x00000002
#define SPM_U32_SYNC_IGNORE                   0x00000003 // value used in group (this value shall be ignored)
#define SPM_U32_SYNC_UNKNOWN                  0x00000004 // not in group
#define SPM_U32_SYNC_LOADED                   0x00000005

#define SPM_U32_SYNC_FORCE                    0x00000006
#define SPM_U32_SYNC_POINT                    0x00000007

#define SPM_U8_PHONE_ABORT_TIMER              0x01

// Shutdown Reasons -------------------
#define SPM_U32_SHUTDOWN_UNKOWN                            0x00
#define SPM_U32_SHUTDOWN_NORMAL                            0x01
#define SPM_U32_SHUTDOWN_RESET                             0x02
#define SPM_U32_SHUTDOWN_AFTER_NO_STATE_REQUEST            0x03
#define SPM_U32_SHUTDOWN_EMERGENCY_OFF                     0x04
#define SPM_U32_SHUTDOWN_AFTER_DOWNLOAD                    0x05
#define SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_ERROR              0x06
#define SPM_U32_SHUTDOWN_APP_END_WANTED                    0x07
#define SPM_U32_SHUTDOWN_WATCHDOG_LOW_SUPERVISION          0x08
#define SPM_U32_SHUTDOWN_APP_NOT_INITIALZIED               0x09
#define SPM_U32_SHUTDOWN_ECU_RESET                         0x0a
#define SPM_U32_SHUTDOWN_AFTER_DOWNLOAD_TERMINATED         0x0b
#define SPM_U32_SHUTDOWN_NOTIFY_PROBLEM                    0x0c
#define SPM_U32_SHUTDOWN_WATCHDOG_MID_SUPERVISION          0x0d
#define SPM_U32_SHUTDOWN_WATCHDOG_HIGH_SUPERVISION         0x0e
#define SPM_U32_SHUTDOWN_DEV_AFTER_NO_STATE_REQUEST        0x0f
#define SPM_U32_SHUTDOWN_AFTER_LONG_PRESS                  0x10
#define SPM_U32_SHUTDOWN_STATE_SUPERVISION                 0x11
#define SPM_U32_SHUTDOWN_STATE_SUPERVISION_RESET           0x12
#define SPM_U32_SHUTDOWN_OVERTEMPERATURE                   0x13
#define SPM_U32_SHUTDOWN_WATCHDOG_SLV_SUPERVISION          0x14
#define SPM_U32_SHUTDOWN_WATCHDOG_HEARTBEAT_SUPERVISION    0x15
#define SPM_U32_SHUTDOWN_RESET_ECU                         0x16
#define SPM_U32_SHUTDOWN_RESET_ECU_FAST                    0x17
#define SPM_U32_SHUTDOWN_PROCESS_SUPERVISION               0x18
#define SPM_U32_SHUTDOWN_CPU_WATCHDOG                      0x19 // Extern triggered reset
#define SPM_U32_SHUTDOWN_AIL_CCA_QUEUE_FULL                0x1A
#define SPM_U32_SHUTDOWN_POWER_ON                          0x1B
#define SPM_U32_SHUTDOWN_USER                              0x1C
#define SPM_U32_SHUTDOWN_SOFTWARE                          0x1D
#define SPM_U32_SHUTDOWN_APP_ERROR                         0x1E
#define SPM_U32_SHUTDOWN_LCM_INTERN                        0x1F
#define SPM_U32_SHUTDOWN_POWER_LOSS                        0x20
#define SPM_U32_SHUTDOWN_POWER_OFF                         0x21
#define SPM_U32_SHUTDOWN_SCC                               0x22
#define SPM_U32_SHUTDOWN_ALTSW_TESTMANAGER                 0x23
#define SPM_U32_SHUTDOWN_LATE_WAKEKUP                      0x24
#define SPM_U32_SHUTDOWN_APP_RESET                         0x25
#define SPM_U32_SHUTDOWN_SCC_RESET                         0x26
#define SPM_U32_SHUTDOWN_USER_RESET                        0x27
#define SPM_U32_SHUTDOWN_MAX                               0x28 // For pram reset reason validation
// keep this line&value allways after the last valid
// shutdown reason in this list
#define SPM_U32_SHUTDOWN_INVALID                           0xff

// emTrace entrys handled by devSpm.trc
#define SPM_U8_ERRMEM_TYPE_STOP_TT               ( (tU8)0x06 )
#define SPM_U8_ERRMEM_TYPE_STOP_TT_NORMAL        ( (tU8)0x07 )
#define SPM_U8_ERRMEM_TYPE_START_TT              ( (tU8)0x08 )
#define SPM_U8_ERRMEM_TYPE_SAVE_NOW              ( (tU8)0x09 )
#define SPM_U8_ERRMEM_TYPE_STATE_OFF_FAILED      ( (tU8)0x0b )

// emTrace entrys handled by spm.trc
#define SPM_U8_ERRMEM_TYPE_RESET                 ( (tU8)0x20 )
#define SPM_U8_ERRMEM_TYPE_STRING                ( (tU8)0x21 )
#define SPM_U8_ERRMEM_TYPE_STATE                 ( (tU8)0x22 )
#define SPM_U8_ERRMEM_TYPE_DL_FINISHED           ( (tU8)0x23 )
#define SPM_U8_ERRMEM_TYPE_DL_FAILED             ( (tU8)0x24 )
#define SPM_U8_ERRMEM_TYPE_STATE_REQUEST         ( (tU8)0x25 )
#define SPM_U8_ERRMEM_TYPE_STATE_REQUEST_IMM     ( (tU8)0x26 )
#define SPM_U8_ERRMEM_TYPE_STATE_FAULT           ( (tU8)0x27 )
#define SPM_U8_ERRMEM_TYPE_VERSION_INFO          ( (tU8)0x28 )
#define SPM_U8_ERRMEM_TYPE_APP_END_WANTED        ( (tU8)0x33 )
#define SPM_U8_ERRMEM_TYPE_LINE_ERROR_ENTRY_START        ( (tU8)0x34 )
#define SPM_U8_ERRMEM_TYPE_LINE_ERROR_ENTRY_END          ( (tU8)0x35 )
#define SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR         ( (tU8)0x36 )
#define SPM_U8_ERRMEM_TYPE_LINE_DOUBLE_SEPERATOR         ( (tU8)0x37 )
#define SPM_U8_ERRMEM_TYPE_LINE_RESET_TYPE_COUNTER       ( (tU8)0x38 )
#define SPM_U8_ERRMEM_TYPE_SPM_CONFIG_ADDITIONAL_DATA    ( (tU8)0x39 )
#define SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_S      ( (tU8)0x3A )
#define SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T      ( (tU8)0x3B )
#define SPM_U8_ERRMEM_TYPE_AIL_MESSAGE_QUELL_FULL_T_ERR  ( (tU8)0x3C )
#define SPM_U8_ERRMEM_TYPE_APP_WATCHDOG_ERROR_ERR        ( (tU8)0x3D )
#define SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST              ( (tU8)0x3E )
#define SPM_U8_ERRMEM_TYPE_UNIT_FAILED_LIST_RESULT       ( (tU8)0x3F )

#define SPM_U8_ADDITIONAL_ERRMEM_DATA            0x00
#define SPM_U8_ADDITIONAL_ERRMEM_DATA_STATE      0x01
#define SPM_U8_ADDITIONAL_ERRMEM_DATA_UBAT       0x02
#define SPM_U8_ADDITIONAL_ERRMEM_DATA_CVM        0x03
#define SPM_U8_ADDITIONAL_ERRMEM_DATA_TIME       0x04
#define SPM_U8_ADDITIONAL_ERRMEM_RESET_COUNTER   0x05

#define SPM_U8_ERRMEM_TYPE_NONE                  ( (tU8)0xff )

#define SPM_S32_THREAD_PRIO_LOW                  ( (tS32)0 )
#define SPM_S32_THREAD_PRIO_MID                  ( (tS32)0 )
#define SPM_S32_THREAD_PRIO_HIGH                 ( (tS32)0 )

#define SPM_C_S32_SYSTEMINIT_COLDSTART                 ( (tS32)0 )
#define SPM_C_S32_SYSTEMINIT_WARMSTART                 ( (tS32)1 )
#define SPM_C_S32_SYSTEMINIT_RESET_ECU_API             ( (tS32)2 )
#define SPM_C_S32_SYSTEMINIT_RESET_SYSTEM              ( (tS32)3 )
#define SPM_C_S32_SYSTEMINIT_RESET_WARMSTART           ( (tS32)4 )
#define SPM_C_S32_SYSTEMINIT_SWITCH_OFF                ( (tS32)5 )
#define SPM_C_S32_SYSTEMINIT_SWITCH_OFF_IMMEDIATELY    ( (tS32)6 )
#define SPM_C_S32_SYSTEMINIT_RESTART_ALTSW_TESTMANAGER ( (tS32)7 )
#define SPM_C_S32_SYSTEMINIT_RESET_LATE_WAKEUP         ( (tS32)8 )
#define SPM_C_S32_SYSTEMINIT_RESET_APP_API             ( (tS32)9 )
#define SPM_C_S32_SYSTEMINIT_RESET_SCC_API             ( (tS32)10 )

#define SPM_STATE_CHANGE_SEM_NAME               "SPM_StateChangeSem"
#define SPM_NOTTABLE_SEM_NAME                   "SPM_NotTableSem"
#define SPM_PROXY_LIST_SEM_NAME                 "SPM_SProxyListSem"

#define SPM_SEM_SUBSTATE                        "SpmSub"

#define SPM_MSGBOX_MAX_MESSAGE_COUNT            30
#define SPM_MSGBOX_NAME                         "NOIOSCSpmSub"

#define SPM_WORKER_SEM_NAME                     "SpmWkSm"
#define SPM_MSGQ_WORKER_SERVER                  "NOIOSCSpmWkMsQ"

#define SPM_MSGBOX_GPIO_MAX_COUNT               40
#define SPM_MSGBOX_GPIO_NAME                    "NOIOSCSpmHwGpio"

#define SPM_ERRMEM_LEN                          20

#define SPM_INFO_VALID            0xa5

#define SPM_U32_CSM_BUS_FORD_IMS                  0x00000000 // CSM_C_CAN_BUS_INFO
#define SPM_U32_CSM_BUS_FORD_BMS                  0x00000001 // CSM_C_CAN_BUS_INFO
#define SPM_U32_CSM_BUS_FORD_LIN                  0x00000002 // CSM_C_CAN_BUS_INFO
#define SPM_U32_CSM_BUS_VW_INFO                   0x00000003 // CSM_C_CAN_BUS_INFO
#define SPM_U32_CSM_BUS_UNKNOWN                   0xFFFFFFFF

// SPM threads
#define SPM_U32_SPM_WDT_HW                     (tU32)0x00000001
#define SPM_U32_SPM_WDT_HI                     (tU32)0x00000002
#define SPM_U32_SPM_WDT_MI                     (tU32)0x00000004
#define SPM_U32_SPM_WDT_LO                     (tU32)0x00000008
#define SPM_U32_SPM_CCA_IN                     (tU32)0x00000010
#define SPM_U32_SPM_REQ_CC                     (tU32)0x00000020
#define SPM_U32_SPM_FGS_IN                     (tU32)0x00000040
#define SPM_U32_SPM_SUB_ST                     (tU32)0x00000080
#define SPM_U32_SPM_CSM                        (tU32)0x00000100
#define SPM_U32_SPM_IPC_PROXY                  (tU32)0x00000200
#define SPM_U32_SPM_OSL_PRX                    (tU32)0x00000400

#define SPM_U32_STARTINFO_STARTUP               0x00000001
#define SPM_U32_STARTINFO_SHUTDOWN              0x00000002

//Wakeup configs
#define SPM_U32_WAKEUPCONFIG_1                  0x00000000 // corresponds to DEFAULT in Gen3
#define SPM_U32_WAKEUPCONFIG_2                  0x00000001 // corresponds to ON_TIPPER in Gen3
#define SPM_U32_WAKEUPCONFIG_3                  0x00000002 // corresponds to CAN in Gen3
#define SPM_U32_WAKEUPCONFIG_4                  0x00000003 // corresponds to MOST in Gen3
#define SPM_U32_WAKEUPCONFIG_5                  0x00000004 // corresponds to IGN_PIN in Gen3
#define SPM_U32_WAKEUPCONFIG_6                  0x00000005 // corresponds to TEL_MUTE in Gen3
#define SPM_U32_WAKEUPCONFIG_7                  0x00000006 // corresponds to CAN & ON_TIPPER in Gen3
#define SPM_U32_WAKEUPCONFIG_8                  0x00000007

#define SPM_U32_WAKEUPCONFIG_MAX          8
// ******************************************************
// default intervalls for used timers.
// ******************************************************

#define SPM_STDBY_TIME_SEC_MAX               (tU16)( 90 * 60 )            // 90 min
#define SPM_STDBY_TIME_SEC                   (tU16)( 1 )

#define SPM_IGNITION_OFF_TIME_MIN_MAX        (tU8)120                // in min
#define SPM_IGNITION_OFF_TIME_MIN            (tU8)0                  // in min

#define SPM_DOOR_OPEN_TIME_MIN_MAX           (tU8)40                 // in min
#define SPM_DOOR_OPEN_TIME_MIN               (tU8)10                 // in min

#define SPM_UNLOCK_ACTIVITY_TIME_MIN         (tU8)10                 // in min

#define SPM_CLOCK_DISPLAY_TIME_MIN_MAX       (tU8)40                 // in min
#define SPM_CLOCK_DISPLAY_TIME_MIN           (tU8)10                 // in min

// system state default intervalls
#define SPM_SYSTEM_BACKGRD_TIMEOUT_MSEC      40000                        // 40 sec
#define SPM_SYSTEM_DOOROPEN_TIMEOUT_MSEC     40000                        // 40 sec
#define SPM_SYSTEM_DOWNLD_TIMEOUT_MSEC       (tU32)( 6 * 60 * 60 * 1000 ) // 6*60 min (10 * 60 * 1000)
#define SPM_SYSTEM_OVERTEMP_TIMEOUT_MSEC      0                           // 0 (OFF)

#define SPM_EJECT_LONG_PRESS                2000                          // 2s

#define SPM_IGNITION_TIMEOUT_MS              30000                        // 30s Ignition timeout detected -> trigger state switch

#define SPM_START_CHECK_EMTRACE_VERS         30000

#define SPM_CONSECUTIVE_RESETCOUNT_EM_TRACE     10
#define SPM_CONSECUTIVE_RESETCOUNT_DELAY         5

#define SPM_WDT_MAX_TRACKER_TIMER         5     // let the timer tracker be on
                                                // for the defined number of seconds.
#define SPM_STARTUP_OK_COUNT_DEFAULT        150

#define SPM_U8_DEFSET_GROUPSPM                  (tU8)0x15
#define SPM_U8_DEFSET_GROUP_INFOTAINMENT_AUDIO  (tU8)0x09
#define SPM_U8_DEFSET_GROUP_MAINTENANCE         (tU8)0x0d
#define SPM_U8_DEFSET_GROUP_COMPLETE            (tU8)0xff

#define SPM_U8_DEFSET_SRC_SET_USER              (tU8)0x00
#define SPM_U8_DEFSET_SRC_SET_TEF               (tU8)0x01
#define SPM_U8_DEFSET_SRC_CODING                (tU8)0x05

// ******************************************************
// global substates for status message for CCA FID 0x0f (bitcoded)
// do not remove --> mapping needed for compatibility to old CCA CFC FIs
// ******************************************************
#define SPM_U32_SUBSTATE_TYPE_TIPPER         (tU32)0x00000001
#define SPM_U32_SUBSTATE_TYPE_DIAGNOSIS      (tU32)0x00000002
#define SPM_U32_SUBSTATE_TYPE_DOWNLOAD       (tU32)0x00000004
#define SPM_U32_SUBSTATE_TYPE_TRANSPORTMODE  (tU32)0x00000008
#define SPM_U32_SUBSTATE_TYPE_IGNITION       (tU32)0x00000010
#define SPM_U32_SUBSTATE_TYPE_S_CONTACT      (tU32)0x00000020
#define SPM_U32_SUBSTATE_TYPE_DIAG_ACTIVITY  (tU32)0x00000040
#define SPM_U32_SUBSTATE_TYPE_PHONE          (tU32)0x00000080
#define SPM_U32_SUBSTATE_TYPE_DIMMING        (tU32)0x00000100
#define SPM_U32_SUBSTATE_TYPE_DOOR_OPEN      (tU32)0x00000200
#define SPM_U32_SUBSTATE_TYPE_TRIGGER_ON     (tU32)0x00000400

// ******************************************************
// default supervision configuration if not defined in registry.
// ******************************************************
#define SPM_LOW_WATCHDOG_DEFAULT_INTERVAL             2000   // in ms
#define SPM_MID_WATCHDOG_DEFAULT_INTERVAL             2000   // in ms
#define SPM_HIGH_WATCHDOG_DEFAULT_INTERVAL            2000   // in ms
#define SPM_HW_WATCHDOG_DEFAULT_INTERVAL              1000   // in ms

#define SPM_U8_CHECK_WDT_LOW_SUPERVISION_TIME_ERROR   75

#define SPM_U8_CHECK_WDT_MID_SUPERVISION_TIME_ERROR   60
#define SPM_U8_CHECK_WDT_MID_SUPERVISION_TIME_RESET   180

#define SPM_U8_CHECK_WDT_HIGH_SUPERVISION_TIME_ERROR  3
#define SPM_U8_CHECK_WDT_HIGH_SUPERVISION_TIME_RESET  9

#define SPM_U32_SUPERVISION_HEARTBEAT_INTERVAL        5000              // in ms
#define SPM_U32_SUPERVISION_HEARTBEAT_TIMEOUT         30                // in sec

#define SPM_U32_SUPERVISION_RESET_DELAY               12000             // in ms

#define SPM_ARION_WDG_TRIGGER_INTERVAL                60000             // in ms
#define SPM_ARION_WDG_TRIGGER_INTERVAL_SHUTDOWN       60000             // in ms
#define SPM_ARION_WDG_TRIGGER_INTERVAL_APPLICATION_SHUTDOWN       60000 // in ms

#define SPM_U32_SYNC_WAITING_TIMEOUT                 30000

#define SPM_U8_DEFAULT_NUMBER_OF_STARTS_WITHOUT_SUPERVISION    8

// ******************************************************
// generic sync points supported by LCM
// ******************************************************

#define SPM_STR_SYNC_POINT_LCMEARLY_STARTED          "LCMEARLY_STARTED"
#define SPM_STR_SYNC_POINT_LCM_SYSTEM_UP             "LCM_SYSTEM_UP"
#define SPM_STR_SYNC_POINT_LCM_SYSTEM_RUNNING        "LCM_SYSTEM_RUNNING"
#define SPM_STR_SYNC_POINT_LCM_STARTUP_READY         "LCM_STARTUP_READY"
#define SPM_STR_SYNC_POINT_LCMLATE_STARTED           "LCMLATE_STARTED"
#define SPM_STR_SYNC_POINT_RVC_AVAILABLE             "RVC_AVAILABLE"
#define SPM_STR_SYNC_POINT_WELCOMESCREEN_AVAILABLE   "WELCOMESCREEN_AVAILABLE"
#define SPM_STR_SYNC_POINT_WELCOMEANIMATION_RUNNING  "WELCOMEANIMATION_RUNNING"
#define SPM_STR_SYNC_POINT_BACKGROUND_AUDIO_CONNECTED "BACKGROUND_AUDIO_CONNECTED"
#define SPM_STR_SYNC_POINT_HOME_SCEEN_VISIBLE        "HOME_SCEEN_VISIBLE"
#define SPM_STR_SYNC_POINT_PHONE_CONNECTED           "PHONE_CONNECTED"
#define SPM_STR_SYNC_POINT_MAP_VISIBLE               "MAP_VISIBLE"
#define SPM_STR_SYNC_POINT_ROUTE_GUIDANCE_RUNNING    "ROUTE_GUIDANCE_RUNNING"

// ******************************************************
// default thread configuration if not defined in registry.
// ******************************************************
#define SPM_OUTPUTTHREAD_THREADNAME "SpmOut"
#define SPM_IPNFGSIF_THREADNAME     "SpmFgs"
#define SPM_POWERVDTHREAD           "SpmCopro"
#define SPM_APPREQMON_THREAD        "SpmReq"
#define SPM_LOW_WATCHDOGTHREAD      "SpmWdtLo"
#define SPM_MID_WATCHDOGTHREAD      "SpmWdtMi"
#define SPM_HIGH_WATCHDOGTHREAD     "SpmWdtHi"
#define SPM_HW_WATCHDOGTHREAD       "SpmWdtHw"
#define SPM_INPUTTHREAD_CCA         "SpmCcaIn"
#define SPM_DEL_NAVI                "SpmWork"
#define SPM_SUBSTATE_THREADNAME     "SpmSubSt"
#define SPM_OSAL_PROXY_THREADNAME   "SpmProxy"

// ******************************************************
// default start types for processes
// ******************************************************
#define SPM_STARTUP_VALUE_OSAL_START_TYPE             "OSAL"
#define SPM_STARTUP_VALUE_SYSTEMD_START_TYPE          "SYSTEMD"

// ******************************************************
// macro for the buffer length of all internal strings
// *****************************************************
//
#define SPM_INTERNAL_BUFFER_LENGTH  200


// ******************************************************
// if an application is not configured but even less
// tries to register with FC SPM then this requested
// state meaning "rejected by SPM" is set in startup-
// investigation for a better overview.
// ******************************************************

#define SPM_REJECTED_REQUESTED_STATE    (tU32)0xfe00


#define SPM_SHUTDOWN_LEVEL_DEFAULT 0
#define SPM_SHUTDOWN_LEVEL_PWR_DOWN_BEFORE_TERMINATE 1
#define SPM_SHUTDOWN_LEVEL_PWR_DOWN_BEFORE_APP_STATE_OFF 2
#define SPM_SHUTDOWN_LEVEL_PWR_DOWN_BEFORE_PF_SHUTDOWN 3


#define SPM_SLV_USE_MSG_BOX

#define SPM_SYNC_MSGQUEUE_NAME  "SpmSync"

#define SPM_OSAL_FS_PREFIX      "/dev/root"

#define SPM_REGISTRY_DYN_PATH_LX   "/var/opt/bosch/dynamic/ffs/registry"
#define SPM_REGISTRY_PERS_PATH_LX  "/var/opt/bosch/persistent/registry"
#define SPM_REGISTRY_ROOT_PATH_LX  "/opt/bosch/base/registry"

#define SPM_REGISTRY_DYN_PATH_OSAL   SPM_OSAL_FS_PREFIX SPM_REGISTRY_DYN_PATH_LX
#define SPM_REGISTRY_PERS_PATH_OSAL  SPM_OSAL_FS_PREFIX SPM_REGISTRY_PERS_PATH_LX
#define SPM_REGISTRY_ROOT_PATH_OSAL  SPM_OSAL_FS_PREFIX SPM_REGISTRY_ROOT_PATH_LX

#if OSAL_CONF == OSAL_LINUX
   #define SPM_PROCESS_LOCAL_PATH  "/opt/bosch/base/bin/"
   #define SPM_PROCESS_REMOTE_PATH "/opt/bosch/base/bin/"
   #define SPM_PROCESS_BASE_PATH   "/"

   #define SPM_BASE_PATH_RFS_ROOT   "/dev/root/"
#elif OSAL_CONF == OSAL_WINNT
   #define SPM_PROCESS_LOCAL_PATH ""
   #define SPM_PROCESS_REMOTE_PATH ""

   #define SPM_BASE_PATH_RFS_ROOT   ""
#endif // if OSAL_CONF == OSAL_LINUX

#define SPM_U32_SPM_WORKER_BROADCAST_ID                   (tU32)0x00000000
#define SPM_U32_SPM_WORKER_SPM_ID                         (tU32)0x00010000
#define SPM_U32_SPM_WORKER_LAM_ID                         (tU32)0x00020000
#define SPM_U32_SPM_WORKER_SPV_ID                         (tU32)0x00030000
#define SPM_U32_SPM_WORKER_CVM_ID                         (tU32)0x00040000
#define SPM_U32_SPM_WORKER_CSH_ID                         (tU32)0x00050000
#define SPM_U32_SPM_WORKER_TRA_ID                         (tU32)0x00060000
#define SPM_U32_SPM_WORKER_STA_ID                         (tU32)0x00070000
#define SPM_U32_SPM_WORKER_SSH_ID                         (tU32)0x00080000
#define SPM_U32_SPM_WORKER_SSM_ID                         (tU32)0x00090000
#define SPM_U32_SPM_WORKER_PRC_ID                         (tU32)0x000a0000
#define SPM_U32_SPM_WORKER_SEC_ID                         (tU32)0x000b0000
#define SPM_U32_SPM_WORKER_WRK_ID                         (tU32)0x000c0000
#define SPM_U32_SPM_WORKER_PXY_ID                         (tU32)0x000d0000
#define SPM_U32_SPM_WORKER_TMR_ID                         (tU32)0x000e0000


#define SPM_U32_WORKER_BROADCAST_STARTUP_DELAY                   (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000000 )
#define SPM_U32_WORKER_BROADCAST_STARTUP_ENDED                   (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000001 )
#define SPM_U32_WORKER_BROADCAST_VALET_MODE                      (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000002 )
#define SPM_U32_WORKER_BROADCAST_EARLY_DELAY                     (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000003 )
#define SPM_U32_WORKER_BROADCAST_STORAGE_DELAY                   (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000004 )
#define SPM_U32_WORKER_BROADCAST_SYSSTATE_CHANGED                (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000005 )

#define SPM_U32_WORKER_LAM_APP_REQUEST_MON_STOPPED               (tU32)( SPM_U32_SPM_WORKER_LAM_ID | 0x00000100 )
#define SPM_U32_WORKER_LAM_APP_END_WANTED_TO                     (tU32)( SPM_U32_SPM_WORKER_LAM_ID | 0x00000101 )
#define SPM_U32_WORKER_LAM_APP_STATE_IMMIDEATLY                  (tU32)( SPM_U32_SPM_WORKER_LAM_ID | 0x00000102 )
#define SPM_U32_WORKER_LAM_APP_CHECK_REQ                         (tU32)( SPM_U32_SPM_WORKER_LAM_ID | 0x00000103 )

#define SPM_U32_WORKER_SPV_AIL_ERROR                             (tU32)( SPM_U32_SPM_WORKER_SPV_ID | 0x00000200 )
#define SPM_U32_WORKER_SPV_PROCESS_MISSING                       (tU32)( SPM_U32_SPM_WORKER_SPV_ID | 0x00000201 )

#define SPM_U32_WORKER_SPM_SHUTDOWN                              (tU32)( SPM_U32_SPM_WORKER_SPM_ID | 0x00000300 )
#define SPM_U32_WORKER_SPM_PREPARE_DOWNLOAD_READY                (tU32)( SPM_U32_SPM_WORKER_SPM_ID | 0x00000301 )
#define SPM_U32_WORKER_SPM_CHECK_EMTRACE_VERS                    (tU32)( SPM_U32_SPM_WORKER_SPM_ID | 0x00000302 )
#define SPM_U32_WORKER_SPM_PREPARE_SHUTDOWN                      (tU32)( SPM_U32_SPM_WORKER_SPM_ID | 0x00000303 )

#define SPM_U32_WORKER_CVM_TIMEOUT                               (tU32)( SPM_U32_SPM_WORKER_CVM_ID | 0x00000400 )
#define SPM_U32_WORKER_CVM_CRITICAL_TIMEOUT                      (tU32)( SPM_U32_SPM_WORKER_CVM_ID | 0x00000401 )

#define SPM_U32_WORKER_CSH_SYSTEM_MODE_DL                        (tU32)( SPM_U32_SPM_WORKER_CSH_ID | 0x00000500 )

#define SPM_U32_WORKER_TRA_COMMAND                               (tU32)( SPM_U32_SPM_WORKER_TRA_ID | 0x00000600 )

#define SPM_U32_WORKER_STA_PROCESS_STOP                          (tU32)( SPM_U32_SPM_WORKER_STA_ID | 0x00000700 )
#define SPM_U32_WORKER_STA_PROCESS_START                         (tU32)( SPM_U32_SPM_WORKER_STA_ID | 0x00000701 )
#define SPM_U32_WORKER_STA_PROCESS_KILLED                        (tU32)( SPM_U32_SPM_WORKER_STA_ID | 0x00000702 )

#define SPM_U32_WORKER_SSH_HEARTBEAT                             (tU32)( SPM_U32_SPM_WORKER_SSH_ID | 0x00000800 )

#define SPM_U32_WORKER_SSM_STATE_TIMEOUT                         (tU32)( SPM_U32_SPM_WORKER_SSM_ID | 0x00000900 )
#define SPM_U32_WORKER_SSM_STATE_DEBOUNCE                        (tU32)( SPM_U32_SPM_WORKER_SSM_ID | 0x00000901 )
#define SPM_U32_WORKER_SSM_ACTIVATE_AUTOMATIC                    (tU32)( SPM_U32_SPM_WORKER_SSM_ID | 0x00000902 )
#define SPM_U32_WORKER_SSM_DEACTIVATE_AUTOMATIC                  (tU32)( SPM_U32_SPM_WORKER_SSM_ID | 0x00000903 )
#define SPM_U32_WORKER_SSM_DOWNLOAD_EXPIRED_RESET                (tU32)( SPM_U32_SPM_WORKER_SSM_ID | 0x00000904 )

#define SPM_U32_WORKER_WRK_REMOVE_CLIENT                         (tU32)( SPM_U32_SPM_WORKER_WRK_ID | 0x00000c00 )
#define SPM_U32_WORKER_WRK_SUPERVISION_TIMEOUT                   (tU32)( SPM_U32_SPM_WORKER_WRK_ID | 0x00000c01 )

#define SPM_U32_WORKER_PRC_SIGCHLD                               (tU32)( SPM_U32_SPM_WORKER_PRC_ID | 0x00000a00 )

#define SPM_U32_WORKER_PRX_TIME_TRACER                           (tU32)( SPM_U32_SPM_WORKER_PXY_ID | 0x00000d00 )
#define SPM_U32_WORKER_PRX_PROC_INFO                             (tU32)( SPM_U32_SPM_WORKER_PXY_ID | 0x00000d01 )
#define SPM_U32_WORKER_PRX_PROC_LIST                             (tU32)( SPM_U32_SPM_WORKER_PXY_ID | 0x00000d02 )
#define SPM_U32_WORKER_PRX_THREAD_INFO                           (tU32)( SPM_U32_SPM_WORKER_PXY_ID | 0x00000d03 )
#define SPM_U32_WORKER_PRX_THREAD_INFO_RDY                       (tU32)( SPM_U32_SPM_WORKER_PXY_ID | 0x00000d04 )

#define SPM_U32_WORKER_TMR_TIMER_EXPIRED                         (tU32)( SPM_U32_SPM_WORKER_TMR_ID | 0x00000e00 )

#define SPM_U32_SYSTEM_RECOVERY                           (tU32)( SPM_U32_SPM_WORKER_BROADCAST_ID | 0x00000016 )



/**
 * @defgroup TRACE_MAKROS
 * @{
 */
/** tU8 */
/** add a tU8 variable into a string buffer for tracing at an 0-based index */
#define SPM_M_INSERT_STRING_AT_INDEX_T8( buf, index, Byte ) \
   ( buf ).insert( ( index ) + 1, 1, Byte;

/** add a tU8 variable into a string buffer for tracing at the start of the buffer */
#define SPM_M_INSERT_T8( buf, Byte ) SPM_M_INSERT_STRING_AT_INDEX_T8( buf, 0, Byte )

/** tU16 */
/** add a tU16 variable into a string buffer for tracing at an 0-based index */
#define SPM_M_INSERT_STRING_AT_INDEX_T16( buf, index, Short ) \
   ( buf ).insert( ( index )    , 1, (tU8)( ( ( Short ) >> 8 ) & 0xFF ) ); \
   ( buf ).insert( ( index ) + 1, 1, (tU8)(   ( Short )        & 0xFF ) );

/** add a tU!& variable into a string buffer for tracing at the start of the buffer */
#define SPM_M_INSERT_T16( buf, Short ) SPM_M_INSERT_STRING_AT_INDEX_T16( buf, 0, Short )

/** add (tU16)zero to the start of the buffer */
#define SPM_M_CLEAR_T16( buf ) SPM_M_INSERT_T16(buf, (tU16)0)

/** tU32 */
/** add a tU32 variable into a string buffer for tracing at an 0-based index */
#define SPM_M_INSERT_STRING_AT_INDEX_T32( buf, index, Long ) \
   ( buf ).insert( ( index )    , 1, (tU8)( ( ( Long ) >> 24 ) & 0xFF) ); \
   ( buf ).insert( ( index ) + 1, 1, (tU8)( ( ( Long ) >> 16 ) & 0xFF) ); \
   ( buf ).insert( ( index ) + 2, 1, (tU8)( ( ( Long ) >> 8  ) & 0xFF) ); \
   ( buf ).insert( ( index ) + 3, 1, (tU8)(   ( Long )         & 0xFF) );

/** add a tU8 variable into a string buffer for tracing at the start of the buffer */
#define SPM_M_INSERT_T32( buf, Long ) SPM_M_INSERT_STRING_AT_INDEX_T32( buf, 0, Long )

/** add (tU32)zero to the start of the buffer */
#define SPM_M_CLEAR_T32( buf ) SPM_M_INSERT_T32(buf, (tU32)0)

/** add (tU32)zero into a string buffer for tracing at an 0-based index */
#define SPM_M_CLEAR_STRING_AT_INDEX_T32( buf, index ) SPM_M_INSERT_STRING_AT_INDEX_T32( buf, index, (tU32)0 )
/** @} */

typedef std::basic_string < unsigned char > tLcmUString;

