/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_ISystemPowerManager.h
  * @brief        This is header file contains interface functions to handle error memory and shutdown functionality.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_IWorkerClient.h"

#define SPM_MAX_ERRMEM_BUF              33
#define SPM_MAX_ERRMEM_APP_FAIL_BUF     ( SPM_MAX_ERRMEM_BUF + 4 )

// //////////////////////////////////////////////////////////////////////////////
// Interface to SystemPowerManager

/*!
  *  \class ISpmSystemPowerManager
  *  \brief This abstract class provides interfaces to system power management.
          This class also supplies interfaces to check error memory for version, detect start mode and
        get start mode .
  */
class ISpmSystemPowerManager : public ISpmWorkerClient, public spm_tclActive
{
public:
ISpmSystemPowerManager( const ISpmFactory& factory ) : ISpmWorkerClient( factory ){ }
virtual ~ISpmSystemPowerManager( ){ }

// Overrides from spm_tclActive
virtual tVoid main( )                                                             = 0;

protected:
// Implementation of interface ITraceClient
virtual tVoid vHandleTraceMessage( const tUChar *puchData )                       = 0;

virtual tVoid vShowErrMem( ) const                                                = 0;

virtual tVoid vTerminateSpm( )                                                    = 0;

virtual tVoid vStartMyDummyEndlessThread( tU32 u32Prio )                          = 0;

public:
virtual tVoid vStartCommunication( )                                              = 0;

virtual tVoid vKillAllApplicationsReady( )                                        = 0;

virtual tVoid vSpmTraceThreadInfo( )                                              = 0;

virtual tVoid vWritePowerOnCounterToErrMem( )                                     = 0;

virtual tVoid vWriteProjectSpecificShutdownInfoToErrMem( tU32 u32ShutdownOption ) = 0;

virtual tVoid vShutdownSystem( tU32 u32ShutdownOption )                           = 0;

// SVM
virtual tVoid vWriteErrmem( tU16       u16Entry,
                            const tU8 *pu8Buf = NULL,
                            tU16       u16Len = SPM_MAX_ERRMEM_BUF,
                            tBool      bWriteDirectly = TRUE ) = 0;

virtual tVoid vReadFile( const std::string& strBuffer,
                         tBool              bWriteErrmem )     = 0;

virtual tBool bWriteFile( const std::string& strBuffer,
                          const std::string& strContent )      = 0;

virtual tBool bRemoveFile( const std::string& strBuffer )      = 0;

virtual tVoid vCheckErrmemForVersion( )                        = 0;

virtual tVoid vHandleMessage( tU32 u32Message,
                              tU32 u32Parm = 0 )               = 0;

virtual tBool bHandleSynchrounousCall( tU32   u32Message,
                                       tVoid *args )           = 0;

virtual tVoid vSetRestartType( tU32 u32RestartType )           = 0;

virtual tU32 u32GetRestartType( )                              = 0;

virtual const tChar*getName( ) const                           = 0;

virtual const tChar*getInterfaceName( ) const { return( "ISpmSystemPowerManager" ); }
};

