/*!
  * \file spm_OsLinux.cpp
  *  \brief
  *    Linux abstraction in generic OSAL based SPM
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author             | Modification
  * 22.11.12  | CM-AI/CB32 Kollai  | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

#include "spm_Config.h"

#include "spm_IProcessSupervision.h"
#include "spm_ISystemPowerManager.h"

#include "spm_OsLinux.h"
#include "spm_IOsalProxy.h"
#include "spm_IFactory.h"
// spm helper
#include "spm_CriticalSection.h"
#include "spm_rootdaemon.h"

#include "signal.h"
#ifndef LCM_UNIT_TESTS
   #ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_EXCHND
      #include "linux/exchnd.h"
   #endif
#endif // ifndef LCM_UNIT_TESTS

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
    #include "trcGenProj/Header/spm_OsLinux.cpp.trc.h"
#endif

#include "spm_trace.h"

#include "OsalConf.h"
#include "osal_public.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>

#define SPM_IS_OSAL_PROC                   bIsFileInFs( "/tmp/_isosal" )

#define SPM_SEM_LINUX_PROXY_NAME "SpmLxPrxSem"

spm_tclOsLinux::spm_tclOsLinux( const ISpmFactory& factory ) : ISpmOsLinux( factory )
   , _poclOsalProxy( NULL )
   , _poclSystemPowerManager( NULL )
   , _hLxPrxSem(OSAL_C_INVALID_HANDLE) {

   if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( SPM_SEM_LINUX_PROXY_NAME, &_hLxPrxSem, (tU32)1 ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclOsLinux !!!!!! Error detected !!!!!! cannot create Semaphore _hLxPrxSem: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }

}

spm_tclOsLinux::~spm_tclOsLinux( ){

   if ( _hLxPrxSem != OSAL_C_INVALID_HANDLE ){
      if ( OSAL_s32SemaphoreClose( _hLxPrxSem ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! LxSem close error detected !!!!!!" ) );
      }
      if ( OSAL_s32SemaphoreDelete( SPM_SEM_LINUX_PROXY_NAME ) != OSAL_OK ){
         ETG_TRACE_ERR( ( "SPM: !!!!!! LxSem delete error detected !!!!!!" ) );
      }
   }

   _poclOsalProxy          = NULL;
   _poclSystemPowerManager = NULL;
}

tVoid spm_tclOsLinux::vGetReferences( ){
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
}

tVoid spm_tclOsLinux::vStartCommunication( ){
}

tU32 spm_tclOsLinux::u32GetCpuLoad( tVoid ){
   return( 0 );
}

tBool spm_tclOsLinux::bSystemHalt( tVoid ){
   return( TRUE );
}

tBool spm_tclOsLinux::bSystemReboot( tVoid ){
   return( TRUE );
}

tBool spm_tclOsLinux::bSupervisionWarning( tVoid ){
   return( TRUE );
}

tBool spm_tclOsLinux::bSupervisionReset( tVoid ){
   return( TRUE );
}

tBool spm_tclOsLinux::bReniceProcess( const std::string& strProcessName ){
   (tVoid)strProcessName;
   return( TRUE );
}

tBool spm_tclOsLinux::bSetReductionLevel( tU32 u32Level ){
   (tVoid)u32Level;
   return( TRUE );
}

tBool spm_tclOsLinux::bSetSupervision( tU32 u32Action ){
   (tVoid)u32Action;
   return( TRUE );
}

tBool spm_tclOsLinux::bTriggerTimeTracker( tBool bProcesses ){
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   _poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

   SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      TRIGGER_TIME_TRACKER,
      bProcesses ? "processes" : "" );

   vReadFile( "/tmp/_spm_top.log", TRUE );

   return( TRUE );
} // bTriggerTimeTracker

tBool spm_tclOsLinux::bGetReadyThreads( tVoid ){
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   _poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

   SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      GET_RUNNING_THREADS,
      "" );

   vReadFile( "/tmp/_spm_task_info.log", TRUE );

   return( TRUE );
}

tBool spm_tclOsLinux::bGetThreadInfo( const std::string& strThreadName ){
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   _poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

   SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      GET_THREAD_INFO,
      strThreadName.c_str( ) );


   vReadFile( "/tmp/_spm_task_info.log", TRUE );

   return( TRUE );
}

/*!
  * \fn
  *  \brief
  *    Gets the callstack of the specified process
  *  \param [in] strProcessName: Name of the process where info is needed
  *  \param [in] bDumpCallstacks: boolean if callstack should be dumped or not. Defaults to TRUE
  *  \param [in] strStartType: how is the process started (OSAL or systemd)
  *  \param [out] tBool: True if successful.
  *  \note
  *    dumps the callstack information according to https://inside-docupedia.bosch.com/confluence/display/gen3generic/Supervision+and+callstack+storage
  ******
  */
tBool spm_tclOsLinux::bGetProcessInfo( const std::string& strProcessName,
                                       tBool              bDumpCallstacks,
                                       const std::string& strStartType ){
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   _poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

   CmdData rCmdData = { 0,0,0,0,0 };

   (void)strcpy( rCmdData.message, "-1" );

   /* Protect OSAL PROCESS check via semaphore to avoid unwanted delete of
      isOsal file in case function is called again from another spm thread while
      callstack generation here is not finished*/
   spm_vEnterCritical( _hLxPrxSem );

   (void)remove( "/tmp/_isosal" );

   SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      GET_PROCESS_INFO_BY_NAME,
      strProcessName.c_str( ) );

   vReadFile( "/tmp/_spm_task_info.log", TRUE );
   tBool bIsOsalProc = SPM_IS_OSAL_PROC;

   spm_bReleaseCritical( _hLxPrxSem );

   if ( bDumpCallstacks ){

      std::string cmdLine;
      std::string parameters;
      std::string cmd;

      /*
         killall does not know the symbolic name for SIGRTMIN - unlike kill - so we have to use the value
         the problem is that this value might change
        */
      if ( bIsOsalProc ){
         parameters = "34 " + strProcessName;
      } else {
         parameters = "SIGUSR2 " + strProcessName;
      }

      if ( strStartType == SPM_STARTUP_VALUE_SYSTEMD_START_TYPE ){
         cmd      = "systemctl kill -s ";

         rCmdData = SPM_ROOTDAEMON_CALLER_rPerformRootOp(
            "lcm",
            SEND_SIGNAL_TO_PROCESS_VIA_SYSTEMD,
            parameters.c_str( ) );
      } else {
         cmd      = "killall -";

         rCmdData = SPM_ROOTDAEMON_CALLER_rPerformRootOp(
            "lcm",
            SEND_SIGNAL_TO_ALL_PROCESSES,
            parameters.c_str( ) );
      }

      {  // write info to emtrace
         tU8 u8Dummy = 0;

         cmdLine = cmd + parameters;

         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (const tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)cmdLine.c_str( ), (tU16)cmdLine.length( ), SPM_WRITE_DIRECTLY );
      }
   }

   return( atoi( rCmdData.message ) < 0 ? FALSE : TRUE );
} // bGetProcessInfo

/*!
  * \fn
  *  \brief
  *    Gets the callstack of the specified process
  *  \param [in] u32ProcId: process ID of the process where info is needed
  *  \param [in] bDumpCallstacks: boolean if callstack should be dumped or not. Defaults to TRUE
  *  \param [in] strStartType: how is the process started (OSAL or systemd)
  *  \param [out] tBool: True if successful.
  *  \note
  *    dumps the callstack information according to https://inside-docupedia.bosch.com/confluence/display/gen3generic/Supervision+and+callstack+storage
  ******
  */
tBool spm_tclOsLinux::bGetProcessInfo( tU32  u32ProcId,
                                       tBool bDumpCallstacks ){

   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   _poclSystemPowerManager->vWritePowerOnCounterToErrMem( );

   CmdData     rCmdData = { 0,0,0,0,0 };

   ETG_TRACE_FATAL( ( "spm_tclOsLinux::bGetProcessInfo() with u32ProcId = %u called", u32ProcId ) );

   strcpy( rCmdData.message, "-1" );


   /* Protect OSAL PROCESS check via semaphore to avoid unwanted delete of
      isOsal file in case function is called again from another spm thread while
      callstack generation here is not finished*/
   spm_vEnterCritical( _hLxPrxSem );

   (void)remove( "/tmp/_isosal" );

   SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      GET_PROCESS_INFO_BY_ID,
      std::to_string(u32ProcId).c_str( ) );

   vReadFile( "/tmp/_spm_task_info.log", TRUE );

   tBool bIsOsalProc = SPM_IS_OSAL_PROC;
   spm_bReleaseCritical( _hLxPrxSem );

   if ( bDumpCallstacks ){

      std::string cmdLine;
      std::string parameters;
      std::string cmd;

      if ( bIsOsalProc ){
         tS32          s32RTMinSignal = u32GetSigRtMinId( (tS32)u32ProcId );
         parameters = std::to_string(s32RTMinSignal)+ " " + std::to_string(u32ProcId);
         // use OSAL exception handler
         // bTriggerExceptionHandler(u32ProcId, "LCM GetProcessInfo");
      } else {
         parameters = "SIGUSR2 " + std::to_string(u32ProcId);
         // bTriggerExceptionHandler(u32ProcId, "LCM GetProcessInfo");
      }

      cmd      = "kill -";

      rCmdData = SPM_ROOTDAEMON_CALLER_rPerformRootOp(
         "lcm",
         SEND_SIGNAL_TO_PROCESS,
         parameters.c_str( ) );

      {  // write info to emtrace
         tU8 u8Dummy = 0;

         cmdLine = cmd + parameters;

         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_LINE_SINGLE_SEPERATOR ), (tU8*)&u8Dummy, sizeof( u8Dummy ), FALSE );
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)cmdLine.c_str( ), (tU16)cmdLine.length( ), SPM_WRITE_DIRECTLY );
      }
   }

   return( atoi( rCmdData.message ) < 0 ? FALSE : TRUE );
} // bGetProcessInfo

tBool spm_tclOsLinux::bEmTraceEntry( const std::string& strEntry ){
   (tVoid)strEntry;
   return( TRUE );
}

/*!
  * \fn
  *  \brief
  *        Method to trigger a callstack generation in the exception handler driver
  *
  *  \param[in] s32Pid: Process ID
  *  \param[in] strMessage: message that is printed instead of "EXCEPTION in process [...]".
                If empty ("") the default message still printed
  ******
  */
tBool spm_tclOsLinux::bTriggerExceptionHandler( tS32        s32Pid,
                                                std::string strMessage ){
   #ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_EXCHND
      int exh_fd = open( "/dev/exchnd", O_RDONLY );
      int retccode;

      if ( exh_fd <= 0 ){
         ETG_TRACE_ERRMEM( ( "Could not open /dev/exchnd: Error = %d", exh_fd ) );
         return( FALSE );
      }
      #ifndef LCM_UNIT_TESTS
         struct exchnd_on_demand param;
         param.pid        = s32Pid;
         param.use_def    = 0; /* Deprecated - Unused */
         param.modules[0] = EHM_PROCESSOR_REGISTERS;
         param.modules[1] = EHM_STACK_DUMP;
         param.modules[2] = EHM_BACKTRACE_ALL_THREADS;
         param.modules[3] = EHM_FAULT_ADDRESS;
         param.modules[4] = EHM_CPU_USAGE;
         strncpy( param.msg, strMessage.c_str( ), 255 );
         retccode         = ioctl( exh_fd, IOCTL_EXCHND_ON_DEMAND, &param );
         if ( retccode < 0 ){
            printf( "Unable to call on demand: %s\n", strerror( errno ) );
         }
      #endif // ifndef LCM_UNIT_TESTS
   #endif    // ifndef VARIANT_S_FTR_DISABLE_USE_ADIT_EXCHND
   return( TRUE );
} // bTriggerExceptionHandler

tBool spm_tclOsLinux::bStopProcess( const std::string& strLocation,
                                    const std::string& strShEvent,
                                    tU32               u32ProcId ){
   (tVoid)strLocation;
   (tVoid)strShEvent;
   (tVoid)u32ProcId;
   return( TRUE );
}

tS32 spm_tclOsLinux::s32StartProcess( const std::string& strLocation,
                                      tU32               u32Type,
                                      tU32               u32NiceLevel,
                                      tU32               u32NiceLevelEnd,
                                      tU32               u32Prio,
                                      tU32               u32Affinity,
                                      tU32               u32AffinityEnd ){
   (tVoid)strLocation;
   (tVoid)u32Type;
   (tVoid)u32NiceLevel;
   (tVoid)u32NiceLevelEnd;
   (tVoid)u32Prio;
   (tVoid)u32Affinity;
   (tVoid)u32AffinityEnd;
   return( - 1 );
}

tBool spm_tclOsLinux::bSyncFilesystem( tU32 u32Timeout ){
   (tVoid)u32Timeout;
   sync( );
   sync( );

   return( TRUE );
}

tBool spm_tclOsLinux::bShutdown( tU32 u32Timeout ){

   CmdData rCmdData = { 0,0,0,0,0 };
   (tVoid)u32Timeout;

   dp_tclSpmDpInternDataHwmCheckEnabled oHwmCheck;
   if (oHwmCheck.tGetData() != 0) {

       std::string                          strWmFile( 255, 0 );

       dp_tclSpmDpConfigWatermarkConfigFile oFile;
       if ( DP_U8_ELEM_STATUS_INITVAL == oFile.u8GetData( &strWmFile[0], 255 ) ){
           strWmFile = "/var/opt/bosch/static/spm/watermarkConf.xml ";
       }

       SPM_ROOTDAEMON_CALLER_rPerformRootOp(
               "lcm",
               GET_PROCESS_HIGH_WATER_MARKS,
               strWmFile.c_str( ) );

       vReadFile( "/tmp/_spm_hwm_info.log", TRUE );

   }

   rCmdData = SPM_ROOTDAEMON_CALLER_rPerformRootOp(
      "lcm",
      UNMOUNT_PARTITIONS,
      "/dev/mmcblk" );

   if (rCmdData.errorNo != ERR_NONE){
      std::string strBuffer = "Execution of LCM root daemon command UNMOUNT_PARTITIONS failed.";
      ETG_TRACE_FATAL( ( "%s", strBuffer.c_str() ) );
      _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strBuffer.c_str(), (tU16)strBuffer.length() );
   } else {
      vReadFile( "/tmp/_spm_unmount.log", TRUE );

      ETG_TRACE_FATAL( ( "All partitions under /dev/mmcblk* remounted as read only => %s", (OSAL_s32StringCompare(rCmdData.message, "0") ? "FAILURE" : "SUCCESS") ) );
   }

   return( TRUE );
} // bShutdown

tBool spm_tclOsLinux::bWaitForSrv( tU32 u32Timeout ){
   (tVoid)u32Timeout;
   return( TRUE );
}

tS32 spm_tclOsLinux::s32ReadFileContent( const std::string& strFile,
                                         std::string      & strBuf,
                                         tU32               u32Len ){
/*!
  * \fn
  *  \brief
  *    Reads u32Len bytes from file strFile.
  *
  *  \param[out] strBuf: File content will be stored in this string object.
  *  \param[in]  u32Len: Number of bytes to read.
  *
  *  \return  -1 if file does not exist otherwise Number of bytes actually read from file
  ******
  */
   tS32        s32Ret = - 1;

   std::string strBuffer( u32Len + 1, 0 );
   FILE       *pFile  = fopen( strFile.c_str( ), "r" );

   if ( pFile ){
      s32Ret = (tS32)fread( ( tPS8 ) & strBuffer[0], 1, u32Len, pFile );
      if ( s32Ret > 0 ){
         strBuf.assign( strBuffer.c_str( ), (tU32)s32Ret );
      }
      fclose( pFile );
   }

   return( s32Ret );
} // s32ReadFileContent

tBool spm_tclOsLinux::bIsFileInFs( const std::string& strFile ){
   tBool bRet = FALSE;

   int   fd   = open( strFile.c_str( ), O_RDONLY );

   if ( fd != - 1 ){
      bRet = TRUE;
      if ( close( fd ) == -1 ){
         ETG_TRACE_FATAL( ( "spm_tclOsLinux::bIsFileInFs(): Failed to close the file '%s'.", strFile.c_str( ) ) );
         ETG_TRACE_FATAL( ( "Error: %s", strerror(errno) ) );
      }
   }
   return( bRet );
}

tBool spm_tclOsLinux::bMkDir( const std::string& strFile ){
   tBool bRet = FALSE;

   if ( !bIsFileInFs( strFile.c_str( ) ) ){
      ETG_TRACE_FATAL( ( "spm_tclOsLinux::bMkDir(): Create dir '%s'.", strFile.c_str( ) ) );
      int status = mkdir( strFile.c_str( ), S_IRWXU | S_IRWXG | S_IRWXO );
      if ( ( status != - 1 ) || ( errno == EEXIST ) ){
         bRet = TRUE;
      }
   } else {
      ETG_TRACE_FATAL( ( "spm_tclOsLinux::bMkDir(): dir '%s' already created.", strFile.c_str( ) ) );
      bRet = TRUE;
   }
   return( bRet );
}
tVoid spm_tclOsLinux::vRunCommandLine( const std::string strCmdLine ){
    system( strCmdLine.c_str( ) );       // OK => This system() call runs WITHOUT root permission. Required group memberships : 'shell', 'adit_errmem_log'.
}
// EOF

