/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_IOsalProxy.h
  * @brief        This is header file contains interfaces to handle OSAL and notify LCM for system events.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include <string>
#include <vector>

class ISpmSubStateClient;
class ISpmCvmClient;

#include "spm_IWorkerClient.h"

#define SPM_S32_EMTRACE_READY_THREADS            ( (tS32)0 )
#define SPM_S32_EMTRACE_READY_THREADS_HIGH       ( (tS32)1 )
#define SPM_S32_EMTRACE_READY_THREADS_MID        ( (tS32)2 )
#define SPM_S32_EMTRACE_READY_THREADS_LOW        ( (tS32)3 )
#define SPM_S32_EMTRACE_TIMER_THREAD             ( (tS32)4 )
#define SPM_S32_EMTRACE_THREAD                   ( (tS32)5 )
#define SPM_S32_EMTRACE_SEM                      ( (tS32)6 )
#define SPM_S32_EMTRACE_TT_START                 ( (tS32)7 )
#define SPM_S32_EMTRACE_TT_DUMP                  ( (tS32)8 )
#define SPM_S32_EMTRACE_TT_STOP                  ( (tS32)9 )
#define SPM_S32_EMTRACE_TT_DUMP_WITH_CALLSTACK   ( (tS32)10 )

#define SPM_U8_OSALPROXY_STARTUPSTATE_UP         ( (tU8)0 )
#define SPM_U8_OSALPROXY_STARTUPSTATE_RUNNING    ( (tU8)1 )

#define SPM_BL_OSALPROXY_ALWAYS                  ( TRUE )
#define SPM_BL_OSALPROXY_IF_ENABLED              ( FALSE )
// -----------------------------------------------------------------------------
// class definition
// -----------------------------------------------------------------------------

/*!
  *  \class ISpmOsalProxy
  *  \brief This abstract class provides interfaces to handle OSAL and notify LCM for system events.
          This class also delivers pure virtual interfaces to set startup OK count, GPIO, reset reasons and shutdown override.
        This class also supplies pure virtual interface to get Wup versions, reset reason, timer difference, UTC time, reset information,
        GPIO and thread information by ID.
  */
class ISpmOsalProxy : public ISpmWorkerClient
{
public:
// constructor
ISpmOsalProxy( const ISpmFactory& factory ) : ISpmWorkerClient( factory ){ }
virtual ~ISpmOsalProxy( ){ }

virtual tVoid vStartCommunication( )                                                             = 0;

virtual tVoid vTerminate( )                                                                      = 0;

// common interface to OSAL
virtual tBool bGetWupVersion( tU32 *pu32Version )                                                = 0;

virtual tBool bSetWakeupConfig( tU32 u32WakeupConfig )                                           = 0;

virtual tBool bGetWakeupReason( tU32 *pu32WakeUpReason )                                         = 0;

virtual tU8 u8GetResetReason( tVoid )    = 0;
virtual tU8 u8GetResetReasonScc( tVoid ) = 0;
virtual tBool bGetStartType( tU8 *pU8Starttype )                                                 = 0;

virtual tBool bResetProcessor( tU8  u8Processor,
                               tU8  u8ResetMode,
                               tU16 u16ResetReason )                                             = 0;

virtual tBool bSetResetCounter( tU8 u8Counter )                                                  = 0;

virtual tU8   u8GetResetCounter( void )                                                          = 0;

virtual tU8   u8GetApplicationMode( void )                                                       = 0;

virtual tU8   u8GetOperationalState( void )                                                       = 0;

virtual tU8   u8GetDmVerityState( void )                                                          = 0;
virtual tBool bSetDmVerityResult( tU8 u8DmVerityResult )                                          = 0;
virtual tBool bSetFastShutdown( tU8 u8FastShutdownType )                                          = 0;

virtual tBool bStartupProcessed( tU8 u8StartupState = SPM_U8_OSALPROXY_STARTUPSTATE_UP )         = 0;

virtual tBool bPrepareForShutdown( )                                                             = 0;

virtual tBool bShutdownSystem( tU32 u32ShutdownOption )                                          = 0;

virtual tBool bIsShutdownActive( ) const                                                         = 0;

virtual tBool bGetUtcTime( OSAL_trTimeDate *tTimeDate )                                          = 0;

virtual tBool bTriggerWatchdog( )                                                                = 0;

virtual tBool bStartWatchdog( tU32 u32WatchdogInterval )                                         = 0;

virtual tBool bSetGpio( tU32  u32Gpio,
                        tBool bState )                                                           = 0;

virtual tBool bGetGpio( tU32   u32Gpio,
                        tBool *pbState )                                                         = 0;

virtual tBool bInstallGpioCallback( tU32                 u32Gpio,
                                    OSAL_tpfGPIOCallback pvCallback,
                                    tVoid               *pvArg = NULL,
                                    tU16                 u16TriggerLevel = OSAL_GPIO_EDGE_BOTH ) = 0;

virtual tBool bUbatSense( tU16 *u16UbatSense )                                                   = 0;

virtual tBool bCurrentSense( tU16 *u16UbatSense )                                                = 0;

virtual tBool bWdtTimeOutWarning( tS32         s32EmTraceType,
                                  const tChar *cThreadName = "" )                                = 0;

virtual tBool bWdtGetThreadInfoById( tU32  u32ThreadId,
                                     tBool fErrmemWrite = TRUE,
                                     tBool fTraceWrite = TRUE )                                  = 0;

virtual tBool bDumpThreadInfoReady( tS32  s32Prio = 0,
                                    tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED )                = 0;

virtual tBool bDumpThreadInfo( const std::string& cThreadName,
                               tBool              bAlways = SPM_BL_OSALPROXY_IF_ENABLED )        = 0;

virtual tBool bDumpProcessInfo( const std::string& cProcessName,
                                const std::string& strStartType,
                                tBool              bDumpCallstacks = TRUE,
                                tBool              bAlways = SPM_BL_OSALPROXY_IF_ENABLED )       = 0;

virtual tBool bDumpProcessInfo( tU32  u32ProcdId,
                                tBool bDumpCallstacks = TRUE,
                                tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED )                    = 0;

virtual tBool bDumpProcessList( tBool bErrMem,
                                tBool bTrace,
                                tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED )                    = 0;

virtual tBool bDumpSemInfo( const tChar *cSemName )                                              = 0;

virtual tBool bDumpTimeTrace( tBool bProcesses = FALSE,
                              tBool bAlways = SPM_BL_OSALPROXY_IF_ENABLED )                      = 0;

virtual tU32  u32GetSRamData( tU32 u32Type )                                                     = 0;

virtual tBool bSetSRamData( tU32 u32Type,
                            tU32 u32Data )                                                       = 0;

virtual tBool bExtendPowerOffTimeout( tU32  u32TimeOut,
                                      tU32 *pu32ResultTimeOut = NULL )                           = 0;

virtual tBool bConfigureSccResetHandling( tBool bDisable,
                                          tU8   u8ExceptionBitmask )                             = 0;

virtual const tChar*getInterfaceName( ) const { return( "ISpmOsalProxy" ); }
};

