/*!
  * \file spm_CriticalSection.cpp
  *  \brief
  *        Handles critical section of code .
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 24.01.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_trace.h"
#include "spm_CriticalSection.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_LAM
#include "trcGenProj/Header/spm_CriticalSection.cpp.trc.h"
#endif

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CRITICALSECTION


// ***************** F U N C T I O N  H E A D E R **************************
//
// DESCRIPTION:
//
// ! \brief
// !    Check if the critical section is locked.
// !
// ! \return
// !   true if the semphore is locked, else false in all other cases
// HISTORY:
// Date            |  Author                       | MODIFICATION
// -------------------------------------------------------------------------
// **************************************************************************
tBool spm_bIsCriticalSection( OSAL_tSemHandle hSemId ){
   tS32 s32SemValue;

   if ( hSemId != OSAL_C_INVALID_HANDLE ){
      if ( OSAL_OK == OSAL_s32SemaphoreGetValue( hSemId, &s32SemValue ) ){
         return( s32SemValue <= 0 );
      } else {
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         return( false );
      }
   } else {
         ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
      return( false );
   }
} // spm_bIsCriticalSection

tVoid spm_vEnterCritical( OSAL_tSemHandle hSemId ){
/*!
  * \fn
  *  \brief
  *    Get/wait for semaphore. If error detected --> ASSERT.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tS32 s32Ret;

   s32Ret = OSAL_s32SemaphoreWait( hSemId, OSAL_C_U32_INFINITE );

   if ( s32Ret != OSAL_OK ){
      // fatal situation -> Assert
                  ETG_TRACE_ERRMEM( ( "spm_vEnterCritical(): OSAL ErrorCode=%d (ErrText=%s)", (tUInt)OSAL_u32ErrorCode( ), OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
      TRACE_SPM_FAILURE;
   }
}

tBool spm_bEnterCriticalWithTimeout( OSAL_tSemHandle hSemId,
                                     tU32            u32TimeOut,
                                     tBool           fWaitTimeCheck ){
/*!
  * \fn
  *  \brief
  *    Get/wait for semaphore. If error detected --> ASSERT.
  *
  *  \version
  *    1.0   - Initial
  ******
  */
   tS32          s32Ret;


   OSAL_tMSecond msSemwaitStartTime = OSAL_ClockGetElapsedTime( );

   s32Ret = OSAL_s32SemaphoreWait( hSemId, u32TimeOut );
   OSAL_tMSecond msSemwaitStopTime  = OSAL_ClockGetElapsedTime( );
   if ( s32Ret != OSAL_OK ){

      /* Check if semamphore wait return with TIMEOUT error*/
      if ( OSAL_u32ErrorCode( ) == OSAL_E_TIMEOUT ){
         /* Just as a precaution: Check if the semwait call and the return of the setmwait call
            produced valid osaltimestamps:*/
         if ( msSemwaitStartTime <= msSemwaitStopTime ){
            OSAL_tMSecond msRealWaitTime = msSemwaitStopTime - msSemwaitStartTime;
            if ( msRealWaitTime < u32TimeOut ){
                  ETG_TRACE_ERRMEM( ( "spm_bEnterCriticalWithTimeout(): OSAL_s32SemaphoreWait() returned with OSAL_E_TIMEOUT, but to early." ) );
                  ETG_TRACE_ERRMEM( ( "SemWait call %d, return %d, real wait time=%d", (tUInt)msSemwaitStartTime, (tUInt)msSemwaitStopTime, (tUInt)msRealWaitTime ) );
               if ( fWaitTimeCheck ){
                  ETG_TRACE_ERRMEM( ( "Calling now (once) SemWait with remaing timeout %d", (tUInt)( u32TimeOut - msRealWaitTime ) ) );
                  return( spm_bEnterCriticalWithTimeout( hSemId, ( u32TimeOut - msRealWaitTime ), FALSE ) );
               } else {
                  ETG_TRACE_ERRMEM( ( "remaing timeout is still %d, but i dont wait for it a second time", (tUInt)( u32TimeOut - msRealWaitTime ) ) );
               }
            }
         } else {
                  ETG_TRACE_ERRMEM( ( "Starttime %d is bigger then stoptime %d ?", (tUInt)msSemwaitStartTime, (tUInt)msSemwaitStopTime ) );
         }
         return( FALSE );
      } else {
         // fatal situation -> Assert
                  ETG_TRACE_ERRMEM( ( "spm_bEnterCriticalWithTimeout(): OSAL ErrorCode=%d (ErrText=%s)", (tUInt)OSAL_u32ErrorCode( ), OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
         TRACE_SPM_FAILURE;
      }
      return( FALSE );
   } else {
      return( TRUE );
   }
} // spm_bEnterCriticalWithTimeout

tBool spm_bReleaseCritical( OSAL_tSemHandle hSemId ){
/*!
  * \fn
  *  \brief
  *    Method to release the lock of the given semaphore
  *
  *  \param hSemId - handle of the semaphore
  ******
  */
   tBool bSemPost = FALSE;

   tS32  s32Ret   = OSAL_s32SemaphorePost( hSemId );

   if ( s32Ret == OSAL_OK ){
      bSemPost = TRUE;
   } else {
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   return( bSemPost );
}

