/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_DownloadLockListManager.h
  * @brief        This is header file contains functions for CcaServiceHandler_DownloadLockStates.
  * @copyright    (C) 2017 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ISpmBase.h"

// FI-zone starts
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#include <list>
#include <map>
#include <algorithm>
#define SYSTEM_S_IMPORT_VECTOR

#include "spm_CcaServiceTypes.h"
#include "spm_SmcSystemStateManagerDownloadLockIf.h"

class ISpmOsLinux;
class TDownloadLock;
class TLockMsgContainer;

typedef std::list<TDownloadLock> TDownloadLockList;
typedef std::list<TDownloadLock>::iterator TDownloadLockIterator;
typedef std::vector<tU16>::iterator TDownloadLockUserIterator;
typedef std::map<tU32, spm_SmcSystemStateManagerDownloadLockIf::FsmEvent> TMapSmcState;
typedef std::pair<tU32, spm_SmcSystemStateManagerDownloadLockIf::FsmEvent> TPairSmcState;

///////////////////////////////////////////////////////////////////////////////
// 
// Struct: TLockMsgContainer
// 
// Contains request command information
// 
///////////////////////////////////////////////////////////////////////////////
struct TLockMsgContainer{
   public:
   TLockMsgContainer(tU32 u32Command = spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_USE
                    ,tU16 u16ReqAppId = AMT_C_U16_APPID_INVALID
                    ,std::string strLock = ""
                    ,tBool bRetCcaResponse = FALSE);
   tU32 _u32Command;
   tU16 _u16ReqAppId;
   std::string _strLock;
   tBool _bRetCcaResponse;
};

///////////////////////////////////////////////////////////////////////////////
// 
// Struct: TDownloadLock
// 
// Comprise with download lock and associated fsm
// 
///////////////////////////////////////////////////////////////////////////////
struct TDownloadLock {
   tBool _isReleased;
   spm_SmcSystemStateManagerDownloadLockIf::Fsm _oFsm;
   spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour* _poclFsmBehaviour;
   spm_fi_tcl_SPM_UpdateLockState _oDownloadLock;
   TDownloadLock(TLockMsgContainer* poLockMsgContainer
   	                  ,spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour* poFsmBehaviour);
   virtual ~TDownloadLock(){};
   tBool operator==(const std::string& roRef) const;
   tVoid operator=(const TDownloadLockStates& roRef);
   // Deep copy
   TDownloadLock(const TDownloadLock& coRef);
   // Assignment for fixing Coverity warning
   TDownloadLock& operator= (const TDownloadLock& coRef);
};

///////////////////////////////////////////////////////////////////////////////
// 
// Class: ISpmDownloadLockListManager
// 
// Interface of downloadlocklist manager
// 
///////////////////////////////////////////////////////////////////////////////
class ISpmDownloadLockListManager : public ISpmBase{
   public:
   ISpmDownloadLockListManager( const ISpmFactory& factory ) : ISpmBase( factory ){ };
   virtual ~ISpmDownloadLockListManager( ){};
   virtual tBool bOnLockCommandHandler(TLockMsgContainer* poLockMsgContainer) = 0;
   virtual tBool bRestoreLock(TLockMsgContainer* poLockMsgContainer) = 0;
   virtual tVoid vRetrieveAllLocks(spm_corefi_tclMsgUpdateLockStatesStatus& refLockListMsgUpdater) = 0;
   
   //ISpmBase
   virtual const tChar*getInterfaceName( ) const { return( "ISpmDownloadLockListManager" ); }   
};

///////////////////////////////////////////////////////////////////////////////
// 
// Class: spm_tclDownloadLockListManager
// 
// Manage locks list and states by handling requested lock commands
// 
///////////////////////////////////////////////////////////////////////////////
class spm_tclDownloadLockListManager : public ISpmDownloadLockListManager,
                                       public spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour {
   public:
   spm_tclDownloadLockListManager( const ISpmFactory& factory );
   virtual ~spm_tclDownloadLockListManager( ){ };
   tBool bOnLockCommandHandler(TLockMsgContainer* poLockMsgContainer);
   tBool bRestoreLock(TLockMsgContainer* poLockMsgContainer);
   tVoid vRetrieveAllLocks(spm_corefi_tclMsgUpdateLockStatesStatus& refLockListMsgUpdater);
   
   protected:
   tBool bIsLockUpdated( TLockMsgContainer* poLockMsgContainer );
   tBool bIsLockCreated( TLockMsgContainer* poLockMsgContainer );
   
   // FSM actions
   void EntryLocked ( void* pArg );
   void EntryLocking ( void* pArg );
   void EntryReserved ( void* pArg );
   void EntryUsed ( void* pArg );
   void ExitLocked ( void* pArg );
   void ExitLocking ( void* pArg );
   void ExitReserved ( void* pArg );
   void ExitUsed ( void* pArg );
   void vDeleteLockPersistently ( void* pArg );
   void vRemoveLock ( void* pArg );
   void vSaveLockPersistently ( void* pArg );
   void vUpdateOwner ( void* pArg );
   
   // FSM guards
   bool isExistingUser ( void* pArg );
   bool isNotExistingUser ( void* pArg );
   bool isNotOwner ( void* pArg );
   bool isNotUserListEmpty ( void* pArg );
   bool isOwner ( void* pArg );
   bool isUserListEmpty ( void* pArg );

   virtual const tChar*getName( ) const { return( "spm_tclDownloadLockListManager" ); }
   virtual tVoid vHandleTraceMessage( const tUChar *puchData );
   virtual tVoid vGetReferences( );
   virtual tVoid vStartCommunication( );

   tVoid initMapSmcState();
   tVoid vAddUser ( std::string strLockName, tU16 u16UserID );
   tVoid vClearOwner ( std::string strLockName );
   tVoid vRemoveUser ( std::string strLockName, tU16 u16UserID );
   tVoid vClearUser ( std::string strLockName );
   tVoid vLockStateUpdate ( std::string strLockName, spm_fi_tcl_e8_UpdateLockState::tenType state );

   TDownloadLockIterator getLockIterator( std::string strLockName );
   TDownloadLockUserIterator getLockUserIterator( std::string strLockName, tU16 u16UserID );

   TMapSmcState _mapSmcState;
   TDownloadLockList _tDownloadLockList;
};

