/*!
  * \file spm_DownloadLockListManager.cpp
  *  \brief
  *    Implements the SPM service for Method UploadLock.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 30.06.17  | Dinh Van Dung     | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_CcaServiceHandler.h"

// spm class definitions (wo if class)
#include "spm_ConnectedApps.h"

// interfaces class definitions
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ILocalAppManager.h"
#include "spm_ISyncHandler.h"

#include "spm_factory.h"

// spm helper
#include "spm_ClientHandlerBase.h"
#include "spm_IFactory.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_CSH
 #include "trcGenProj/Header/spm_DownloadLockListManager.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define DELETE_LOCK_FLAG "/dev/ffs2/persistent/deletelockflag.dnl"
#define GET_LOCK_MSG_CONTAINER_REFERENCE \
   TLockMsgContainer* poLockMsgContainer = static_cast< TLockMsgContainer* >(pArg); \
   SPM_NULL_POINTER_CHECK(poLockMsgContainer);
#define GET_LOCK_MSG_CONTAINER_REFERENCE_VAL \
   TLockMsgContainer* poLockMsgContainer = static_cast< TLockMsgContainer* >(pArg); \
   SPM_NULL_POINTER_CHECK_VAL(poLockMsgContainer);
///////////////////////////////////////////////////////////////////////////////
// 
// Struct: TLockMsgContainer
// 
// Contains request command information
// 
///////////////////////////////////////////////////////////////////////////////
TLockMsgContainer::TLockMsgContainer(tU32 u32Command
                                    ,tU16 u16ReqAppId
                                    ,std::string strLock
                                    ,tBool bRetCcaResponse):_u32Command(u32Command)
                                                            ,_u16ReqAppId(u16ReqAppId)
                                                            ,_strLock(strLock)
                                                            ,_bRetCcaResponse(bRetCcaResponse){
/*!
  * \fn
  *  \brief
  *   TLockMsgContainer constructor.
  *
  *  \param: u32Command -> initial lock command request.
  *          u16ReqAppId -> initial request application id.
  *          strLock -> initial lock name.
  *          bRetCcaResponse -> initial cca response request.
  *  \return
  ******
  */
}

///////////////////////////////////////////////////////////////////////////////
// 
// Struct: TDownloadLock
// 
// Comprise with download lock and associated fsm
// 
///////////////////////////////////////////////////////////////////////////////
TDownloadLock::TDownloadLock(TLockMsgContainer* poLockMsgContainer
                            ,spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour* poFsmBehaviour):_isReleased(FALSE)
                                                                                                   ,_oFsm(poFsmBehaviour)
                                                                                                   ,_poclFsmBehaviour(poFsmBehaviour){
/*!
  * \fn
  *  \brief
  *   TDownloadLock constructor.
  *   _isReleased is initialized to FALSE.
  *   _oFsm is initialized with poFsmBehaviour.
  *   _poclFsmBehaviour is initialized with poFsmBehaviour.
  *
  *  \param: poLockMsgContainer-> pointer to lock message container.
  *          poFsmBehaviour -> pointer to FsmBehaviour.
  *  \return
  ******
  */
   _oDownloadLock.strLockName = poLockMsgContainer->_strLock.c_str();
}
tBool TDownloadLock::operator==(const std::string& roRef) const{
/*!
  * \fn
  *  \brief
  *   Overloading opertor == to compare Download Lock object with a string lock name.
  *
  *  \param: roRef -> read only reference to string lock name.
  *  \return: TRUE: Download Lock matches with input string.
              TRUE: Download Lock doesn't match with input string.
  ******
  */
   return ( _oDownloadLock.strLockName == roRef.c_str() );
}
tVoid TDownloadLock::operator=(const TDownloadLockStates& roRef){
/*!
  * \fn
  *  \brief
  *   Overloading operator = to read from TDownloadLockStates.
  *
  *  \param: roRef -> read only reference to TDownloadLockStates.
  *  \return
  ******
  */
   _oDownloadLock.u16LockOwner = roRef.u16AppIdLockOwner;
   _oDownloadLock.strLockName = roRef.strDownloadLock;
}
TDownloadLock::TDownloadLock(const TDownloadLock& coRef):_isReleased(coRef._isReleased)
                                                        ,_oFsm(coRef._poclFsmBehaviour)
                                                        ,_poclFsmBehaviour(coRef._poclFsmBehaviour)
                                                        ,_oDownloadLock(coRef._oDownloadLock){
/*!
  * \fn
  *  \brief
  *   Copy constructor TDownloadLock.
  *   New _oFsm is initialized with coRef._poclFsmBehaviour
  *   New _poclFsmBehaviour is initialized with coRef._poclFsmBehaviour
  *   New _oDownloadLock is initialized with coRef._oDownloadLock
  *   New _isReleased is initialized with coRef._isReleased
  *
  *  \param: coRef -> reference to copied TDownloadLock object.
  *  \return
  ******
  */
}

///////////////////////////////////////////////////////////////////////////////
// 
// Class: spm_tclDownloadLockListManager
// 
// Manage locks list and states by handling requested lock commands
// 
///////////////////////////////////////////////////////////////////////////////
spm_tclDownloadLockListManager::spm_tclDownloadLockListManager( const ISpmFactory& factory ):ISpmDownloadLockListManager( factory ){
/*!
  * \fn
  *  \brief
  *   DownloadLockListManager constructor.
  *   DownloadLockListManager is constructed in spm factory.
  *   ISpmDownloadLockListManager is initialized with reference to factory
  *
  *  \param: factory -> reference to spm factory.
  *  \return
  ******
  */
   _tDownloadLockList.clear();
   _mapSmcState.clear();
   initMapSmcState();
}
tVoid spm_tclDownloadLockListManager::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *   Start communication with other objects.
  *
  *  \param.
  *  \return
  ******
  */
}
tBool spm_tclDownloadLockListManager::bOnLockCommandHandler(TLockMsgContainer* poLockMsgContainer){
/*!
  * \fn
  *  \brief
  *   Receive and handle update lock command.
  *   Decide to create new lock or update locks.
  *
  *  \param: poLockMsgContainer-> reference to requested lock message container.
  *  \return
  *
  ******
  */
   tBool bRet = FALSE;
   SPM_NULL_POINTER_CHECK_VAL(poLockMsgContainer);
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   if ( itLock == _tDownloadLockList.end( ) ){
      ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bOnLockCommandHandler(): Can not find lock --> Create new Lock name: %s",
                        ( poLockMsgContainer->_strLock ).c_str( ) ) );
      bRet = bIsLockCreated( poLockMsgContainer );
   } else {
      ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bOnLockCommandHandler(): Found lock --> Update available Lock: %s",
                        itLock->_oDownloadLock.strLockName.szValue ) );
      bRet = bIsLockUpdated( poLockMsgContainer );
   }
   return bRet;
}
tBool spm_tclDownloadLockListManager::bIsLockCreated( TLockMsgContainer* poLockMsgContainer ){
/*!
  * \fn
  *  \brief
  *   Create new lock with USED state initialized.
  *   New USED lock will be push back to Download Lock list.
  *
  *  \param.
  *  \return.
  ******
  */
   tBool bRet = FALSE;
   SPM_NULL_POINTER_CHECK_VAL( poLockMsgContainer );
   TDownloadLock newDownloadLock( poLockMsgContainer, dynamic_cast<spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour*>(this) );
   if ( _tDownloadLockList.size() >= SPM_DOWNLOAD_LOCK_MAX_NUM ){
      ETG_TRACE_FATAL( ( "spm_tclDownloadLockListManager::bIsLockCreated(): ERROR: DownloadLockList is handling more locks than capability of datapool. Maximum: %u",
                        SPM_DOWNLOAD_LOCK_MAX_NUM ) );
      ETG_TRACE_FATAL( ( "spm_tclDownloadLockListManager::bIsLockCreated(): Lock %s is discarded",
                       ( poLockMsgContainer->_strLock ).c_str( ) ) );
      bRet = FALSE;
   } else {
      if ( poLockMsgContainer->_u32Command == spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_USE
          || poLockMsgContainer->_u32Command == spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_MASTERLOCK ){
         _tDownloadLockList.push_back(newDownloadLock);
         TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
         itLock->_oFsm.init( static_cast<void*>( poLockMsgContainer ) );
         ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bIsLockCreated: New lock is created: State: %u, First User: %u, Name: %s",
                            ETG_ENUM( SPM_UPDATELOCKSTATE, itLock->_oDownloadLock.eLockState.enType ), ETG_ENUM( ail_u16AppId, itLock->_oDownloadLock.lu16LockUsers.back() ), itLock->_oDownloadLock.strLockName.szValue ) );
         bRet = TRUE;
         if ( poLockMsgContainer->_u32Command == spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_MASTERLOCK ){
            bRet = bIsLockUpdated( poLockMsgContainer );
         }
      } else {
         ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bIsLockCreated(): Invalid command from Application --> Lock: %s is not created",
                           ( poLockMsgContainer->_strLock ).c_str( ) ) );
         bRet = FALSE;
      }
   }
   return bRet;
}
tBool spm_tclDownloadLockListManager::bRestoreLock(TLockMsgContainer* poLockMsgContainer){
/*!
  * \fn
  *  \brief
  *   Restore all locks in persistent memory.
  *   If a delete download lock flag exists 
  *   then all locks are cleared and not restored.
  *
  *  \param: poLockMsgContainer-> reference to lock message container.
  *  \return.
  ******
  */
   const std::string strDeleteLockFlagFile = DELETE_LOCK_FLAG ;
   dp_tclSpmDpInternDataDownloadLockStates oDpDownloadLock;
   SPM_NULL_POINTER_CHECK_VAL(poLockMsgContainer);
   OSAL_tIODescriptor flagDLLock = OSAL_IOOpen( ( strDeleteLockFlagFile ).c_str( ), OSAL_EN_READONLY );
   if ( flagDLLock != OSAL_ERROR ){
      if ( OSAL_s32IOClose( flagDLLock ) != OSAL_OK ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_FATAL( ( "spm_tclDownloadLockListManager::bRestoreLock: Failed to close lock flag file. ERROR: %s.", OSAL_coszErrorText( u32ErrorReason ) ) );
      }
      oDpDownloadLock.vClearList();
      ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bRestoreLock(): Delete lock flag exists --> Persistent lock list is cleared"));
      if ( OSAL_s32IORemove( ( strDeleteLockFlagFile ).c_str( ) ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_FATAL( ( "spm_tclDownloadLockListManager::bRestoreLock: Failed to delete lock flag file. ERROR: %s.", OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   } else {
      tU32 u32Count = oDpDownloadLock.u32GetCount( );
      ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bRestoreLock(): Number of previous Locks stored persistently: %u", u32Count));
      for ( tU32 i = 0; i < u32Count; i++ ){
         TDownloadLockStates oNewDpEntry;
         TDownloadLock newDownloadLock(poLockMsgContainer, dynamic_cast<spm_SmcSystemStateManagerDownloadLockIf::FsmBehaviour*>(this));
         oDpDownloadLock.s32GetElem(i, oNewDpEntry);
         newDownloadLock = oNewDpEntry;
         poLockMsgContainer->_strLock = (std::string)newDownloadLock._oDownloadLock.strLockName.szValue;
         _tDownloadLockList.push_back(newDownloadLock);
         TDownloadLockIterator itLock = getLockIterator( (std::string)newDownloadLock._oDownloadLock.strLockName.szValue );
         itLock->_oFsm.init( static_cast<void*>( poLockMsgContainer ) );
         if ( itLock != _tDownloadLockList.end( ) ) {
            itLock->_oFsm.acceptEvent(spm_SmcSystemStateManagerDownloadLockIf::evRestoreLock, static_cast<void*>( poLockMsgContainer ) );
            itLock->_oDownloadLock.u16LockOwner = newDownloadLock._oDownloadLock.u16LockOwner;
            ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::bRestoreLock(): Lock is restored: Owner: %u, State: %u, Name: %s",
                              ETG_ENUM( ail_u16AppId,itLock->_oDownloadLock.u16LockOwner ),
                              ETG_ENUM( SPM_UPDATELOCKSTATE, itLock->_oDownloadLock.eLockState.enType ),
                              itLock->_oDownloadLock.strLockName.szValue ) );
         }
      }
      poLockMsgContainer->_bRetCcaResponse = TRUE;
   }
   return poLockMsgContainer->_bRetCcaResponse;
}
tVoid spm_tclDownloadLockListManager::vRetrieveAllLocks(spm_corefi_tclMsgUpdateLockStatesStatus& refLockListMsgUpdater){
/*!
  * \fn
  *  \brief
  *   Retrieve all lock from download lock list to update lock status cca message.
  *
  *  \param: refLockListMsgUpdater -> reference to MsgUpdateLockStatesStatus.
  *  \return
  ******
  */
   TDownloadLockIterator it;
   for (it = _tDownloadLockList.begin(); it != _tDownloadLockList.end(); it++){
      refLockListMsgUpdater.lLocks.push_back(it->_oDownloadLock);
   }
}
tBool spm_tclDownloadLockListManager::bIsLockUpdated( TLockMsgContainer* poLockMsgContainer ){
/*!
  * \fn
  *  \brief
  *   Update already existed lock in Download Lock list.
  *
  *  \param.
  *  \return.
  ******
  */
   tBool bRet = FALSE;
   SPM_NULL_POINTER_CHECK_VAL(poLockMsgContainer);
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   TMapSmcState::iterator itCommand = _mapSmcState.find(poLockMsgContainer->_u32Command);
   if ( itCommand != _mapSmcState.end() ){
      itLock->_oFsm.acceptEvent( itCommand->second, static_cast<void*>(poLockMsgContainer) );
      ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::bIsLockUpdated(): Lock %s is done updating", itLock->_oDownloadLock.strLockName.szValue ) );
      if ( itLock->_isReleased ){
         ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::bIsLockUpdated(): Lock %s is deleted persistenly", ( poLockMsgContainer->_strLock ).c_str( ) ) );
         _tDownloadLockList.erase( itLock );
         poLockMsgContainer->_bRetCcaResponse = TRUE;
      } else {
         ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::bIsLockUpdated(): Current state: %u --> Lock %s ",
                           ETG_ENUM( SPM_UPDATELOCKSTATE, itLock->_oDownloadLock.eLockState.enType ),
                           itLock->_oDownloadLock.strLockName.szValue ) );
      }
      bRet = poLockMsgContainer->_bRetCcaResponse;
   } else {
      ETG_TRACE_FATAL( ( "Invalid input lock command" ) );
      bRet = FALSE;
   }
   return bRet;
}
tVoid spm_tclDownloadLockListManager::initMapSmcState(){
/*!
  * \fn
  *  \brief
  *   Initial mapping event of SMC download lock state machine.
  *
  *  \param.
  *  \return
  ******
  */
   _mapSmcState.insert(TPairSmcState(spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_USE,
                                     spm_SmcSystemStateManagerDownloadLockIf::evUse));
   _mapSmcState.insert(TPairSmcState(spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_RESERVE,
                                     spm_SmcSystemStateManagerDownloadLockIf::evReserve));
   _mapSmcState.insert(TPairSmcState(spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_LOCK,
                                     spm_SmcSystemStateManagerDownloadLockIf::evLock));
   _mapSmcState.insert(TPairSmcState(spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_MASTERLOCK,
                                     spm_SmcSystemStateManagerDownloadLockIf::evMasterLock));
   _mapSmcState.insert(TPairSmcState(spm_fi_tcl_e8_UpdateLockCommand::FI_EN_SPM_E8_RELEASE,
                                     spm_SmcSystemStateManagerDownloadLockIf::evRelease));
}
void spm_tclDownloadLockListManager::EntryLocked(void* pArg){
/*!
  * \fn
  *  \brief
  *   Entry function of Locked state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   vLockStateUpdate( poLockMsgContainer->_strLock, spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_LOCKED );
   vClearUser( poLockMsgContainer->_strLock );
   poLockMsgContainer->_bRetCcaResponse = TRUE;
}
void spm_tclDownloadLockListManager::EntryLocking(void* pArg){
/*!
  * \fn
  *  \brief
  *   Entry function of Locking state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   vLockStateUpdate( poLockMsgContainer->_strLock, spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_LOCKING);
   vRemoveUser( poLockMsgContainer->_strLock, poLockMsgContainer->_u16ReqAppId );
   poLockMsgContainer->_bRetCcaResponse = TRUE;
}
void spm_tclDownloadLockListManager::EntryReserved(void* pArg){
/*!
  * \fn
  *  \brief
  *   Entry function of Reserved state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   vLockStateUpdate( poLockMsgContainer->_strLock, spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_RESERVED);
   poLockMsgContainer->_bRetCcaResponse = TRUE;
}
void spm_tclDownloadLockListManager::EntryUsed(void* pArg){
/*!
  * \fn
  *  \brief
  *   Entry function of Used state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   vLockStateUpdate( poLockMsgContainer->_strLock, spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_USED);
   vAddUser( poLockMsgContainer->_strLock, poLockMsgContainer->_u16ReqAppId );
   vClearOwner( poLockMsgContainer->_strLock );
   poLockMsgContainer->_bRetCcaResponse = TRUE;
}
void spm_tclDownloadLockListManager::ExitLocked(void* /*pArg*/){
/*!
  * \fn
  *  \brief
  *   Exit function of Locked state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
}
void spm_tclDownloadLockListManager::ExitLocking(void* /*pArg*/){
/*!
  * \fn
  *  \brief
  *   Exit function of Locking state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
}
void spm_tclDownloadLockListManager::ExitReserved(void* /*pArg*/){
/*!
  * \fn
  *  \brief
  *   Exit function of Reserved state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
}
void spm_tclDownloadLockListManager::ExitUsed(void* /*pArg*/){
/*!
  * \fn
  *  \brief
  *   Exit function of Used state.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
}
void spm_tclDownloadLockListManager::vDeleteLockPersistently(void* pArg){
/*!
  * \fn
  *  \brief
  *   Delete a lock from persistent memory.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE

   dp_tclSpmDpInternDataDownloadLockStates oDpDownloadLock;
   TDownloadLockStates oNewDpEntry;
   (tVoid)memset((tVoid*)oNewDpEntry.strDownloadLock, 0, SPM_CCA_AREA_MAX_LEN);
   oDpDownloadLock.vClearList();

   ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::vDeleteLockPersistently: Lock list datapool is cleared" ) );
   TDownloadLockIterator itLockSaved;
   for (itLockSaved = _tDownloadLockList.begin(); itLockSaved != _tDownloadLockList.end(); itLockSaved++){
      if(itLockSaved->_oDownloadLock.strLockName == (poLockMsgContainer->_strLock).c_str( ) ){
      } else if(spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_LOCKED == itLockSaved->_oDownloadLock.eLockState.enType){
         oNewDpEntry.u16AppIdLockOwner = itLockSaved->_oDownloadLock.u16LockOwner;
         OSAL_szStringNCopy(oNewDpEntry.strDownloadLock, itLockSaved->_oDownloadLock.strLockName.szValue, SPM_CCA_AREA_MAX_LEN-1);
         oDpDownloadLock.vPushBack(oNewDpEntry);
         ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::vDeleteLockPersistently: Lock %s is saved persistenly", itLockSaved->_oDownloadLock.strLockName.szValue ) );
      }
   }
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vDeleteLockPersistently: Lock %s is deleted persistenly", ( poLockMsgContainer->_strLock ).c_str( ) ) );
}
void spm_tclDownloadLockListManager::vRemoveLock(void* pArg){
/*!
  * \fn
  *  \brief
  *   Delete a lock object from lock list to release runtime resource.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   itLock->_isReleased= TRUE;
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vRemoveLock: Lock %s is removed from Lock list", ( poLockMsgContainer->_strLock ).c_str( ) ) );
}
void spm_tclDownloadLockListManager::vSaveLockPersistently(void* pArg){
/*!
  * \fn
  *  \brief
  *   Save a lock to memory persistently.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  *  \note: Persistent memory can hold 10 locks maximum.
            If more than 10 locks are saved, all exceeding locks will be discarded.
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   dp_tclSpmDpInternDataDownloadLockStates oDpDownloadLock;
   TDownloadLockStates oNewDpEntry;
   TDownloadLockIterator it;
   (tVoid)memset((tVoid*)oNewDpEntry.strDownloadLock, 0, SPM_CCA_AREA_MAX_LEN);
   tU32 u32Count = oDpDownloadLock.u32GetCount( );
   if ( u32Count >= SPM_DOWNLOAD_LOCK_MAX_NUM ){
      ETG_TRACE_FATAL( ( "Can not save more Lock to Datapool. Current numbers of Lock exceed maximum value" ) )
      return;
   }
   oDpDownloadLock.vClearList();
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vSaveLockPersistently: Lock list datapool is cleared" ) );
   for (it = _tDownloadLockList.begin(); it != _tDownloadLockList.end(); it++){
      if(spm_fi_tcl_e8_UpdateLockState::FI_EN_SPM_E8_LOCKED == it->_oDownloadLock.eLockState.enType
         && !(it->_oDownloadLock.strLockName == poLockMsgContainer->_strLock.c_str() ) ){
         oNewDpEntry.u16AppIdLockOwner = it->_oDownloadLock.u16LockOwner;
         OSAL_szStringNCopy(oNewDpEntry.strDownloadLock, it->_oDownloadLock.strLockName.szValue, SPM_CCA_AREA_MAX_LEN-1);
         oDpDownloadLock.vPushBack(oNewDpEntry);
         ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::vSaveLockPersistently: Lock %s is saved persistenly", it->_oDownloadLock.strLockName.szValue ) );
      }
   }
   oNewDpEntry.u16AppIdLockOwner = itLock->_oDownloadLock.u16LockOwner;
   OSAL_szStringNCopy(oNewDpEntry.strDownloadLock, itLock->_oDownloadLock.strLockName.szValue, SPM_CCA_AREA_MAX_LEN-1);
   oDpDownloadLock.vPushBack(oNewDpEntry);
   ETG_TRACE_USR4( ( "spm_tclDownloadLockListManager::vSaveLockPersistently: Lock %s is saved persistenly", itLock->_oDownloadLock.strLockName.szValue ) );
}
void spm_tclDownloadLockListManager::vUpdateOwner(void* pArg){
/*!
  * \fn
  *  \brief
  *   Update new owner of a Download Lock.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return
  ******
  */
   GET_LOCK_MSG_CONTAINER_REFERENCE
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   itLock->_oDownloadLock.u16LockOwner = poLockMsgContainer->_u16ReqAppId;
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vUpdateOwner: New owner %u is updated for Lock %s.", 
                     ETG_ENUM( ail_u16AppId,itLock->_oDownloadLock.u16LockOwner ), itLock->_oDownloadLock.strLockName.szValue ) );
}
bool spm_tclDownloadLockListManager::isExistingUser(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for existence of app id in user list of a lock.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE app id has already been existing.
  *           FALSE app id has not been existing.
  ******
  */
   bool bRet = FALSE;
   GET_LOCK_MSG_CONTAINER_REFERENCE_VAL
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   TDownloadLockUserIterator itLockUser = getLockUserIterator( poLockMsgContainer->_strLock, poLockMsgContainer->_u16ReqAppId );
   bRet = ( itLockUser != itLock->_oDownloadLock.lu16LockUsers.end() );
   return bRet;
}
bool spm_tclDownloadLockListManager::isNotExistingUser(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for non-existing app id in user list of a lock.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE app id has not been existing.
  *           FALSE app id has already been existing.
  ******
  */
   return !isExistingUser( pArg );
}
bool spm_tclDownloadLockListManager::isOwner(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for user list of lock empty.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE app id has already been owner of lock.
  *           FALSE app id has not been owner of lock.
  ******
  */
   bool bRet = FALSE;
   GET_LOCK_MSG_CONTAINER_REFERENCE_VAL
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   bRet = (poLockMsgContainer->_u16ReqAppId == itLock->_oDownloadLock.u16LockOwner);
   return bRet;
}
bool spm_tclDownloadLockListManager::isNotOwner(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for app id not current owner of lock.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE app id has not been owner of lock.
  *           FALSE app id has already been owner of lock.
  ******
  */
   return !isOwner( pArg );
}
bool spm_tclDownloadLockListManager::isUserListEmpty(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for user list of a lock empty.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE user list of lock will be empty after removing request app id.
  *           FALSE user list of lock will not be empty after removing request app id.
  *  \return
  ******
  */
   bool bRet = FALSE;
   GET_LOCK_MSG_CONTAINER_REFERENCE_VAL
   TDownloadLockIterator itLock = getLockIterator( poLockMsgContainer->_strLock );
   tU64 u32UserNumbers = itLock->_oDownloadLock.lu16LockUsers.size();
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::isUserListEmpty: Number of users is %u",u32UserNumbers ) );
   bRet = ( u32UserNumbers <= 1 );
   return bRet;
}
bool spm_tclDownloadLockListManager::isNotUserListEmpty(void* pArg){
/*!
  * \fn
  *  \brief
  *   Check for user list of lock not empty.
  *
  *  \param: pArg -> generic pointer to any type object.
  *  \return: TRUE user list of lock will not be empty after removing request app id.
  *           FALSE user list of lock will be empty after removing request app id.
  ******
  */
   return !isUserListEmpty( pArg );
}
tVoid spm_tclDownloadLockListManager::vAddUser( std::string strLockName, tU16 u16UserID ){
/*!
  * \fn
  *  \brief
  *   Add new user to user list of a Download Lock.
  *
  *  \param. strLockName: lock name.
  *          state: expected state to be updated.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   TDownloadLockUserIterator itLockUser = getLockUserIterator( strLockName, u16UserID );
   if( itLockUser == itLock->_oDownloadLock.lu16LockUsers.end( ) ){
      itLock->_oDownloadLock.lu16LockUsers.push_back( u16UserID );
   }
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vAddUser: New user %u is added to Lock %s ",
                     ETG_ENUM(ail_u16AppId,itLock->_oDownloadLock.lu16LockUsers.back() ), itLock->_oDownloadLock.strLockName.szValue ) );
}
tVoid spm_tclDownloadLockListManager::vClearOwner( std::string strLockName ){
/*!
  * \fn
  *  \brief
  *   Clear current owner of a Download Lock.
  *
  *  \param. strLockName: lock name.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   itLock->_oDownloadLock.u16LockOwner = AMT_C_U16_APPID_INVALID;
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vClearOwner: Lock %s: Owner is cleared", itLock->_oDownloadLock.strLockName.szValue ) );
}
tVoid spm_tclDownloadLockListManager::vClearUser( std::string strLockName ){
/*!
  * \fn
  *  \brief
  *   Clear all users from user list of a Download Lock.
  *
  *  \param. strLockName: lock name.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   itLock->_oDownloadLock.lu16LockUsers.clear();
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vClearUser: Lock %s: Users is clear", itLock->_oDownloadLock.strLockName.szValue ) );
}
tVoid spm_tclDownloadLockListManager::vRemoveUser( std::string strLockName, tU16 u16UserID ){
/*!
  * \fn
  *  \brief
  *   Remove existing user out of user list of a Download Lock.
  *
  *  \param. strLockName: lock name.
  *          u16UserID: user id.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   TDownloadLockUserIterator itLockUser = getLockUserIterator( strLockName, u16UserID );
   itLock->_oDownloadLock.lu16LockUsers.erase( itLockUser );
   ETG_TRACE_USR2( ( "spm_tclDownloadLockListManager::vRemoveUser: User %u is removed from Lock %s ",
                      ETG_ENUM( ail_u16AppId, u16UserID ), itLock->_oDownloadLock.strLockName.szValue ) );
}
tVoid spm_tclDownloadLockListManager::vLockStateUpdate(std::string strLockName, spm_fi_tcl_e8_UpdateLockState::tenType state){
/*!
  * \fn
  *  \brief
  *   Update state for lock in download lock list.
  *
  *  \param. strLockName: lock name.
  *          state: expected state to be updated.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   itLock->_oDownloadLock.eLockState.enType = state;
}
TDownloadLockIterator spm_tclDownloadLockListManager::getLockIterator( std::string strLockName ){
/*!
  * \fn
  *  \brief
  *   Get lock iterator from given lock name.
  *
  *  \param. strLockName: lock name.
  *  \return.
  ******
  */
   return std::find( _tDownloadLockList.begin(), _tDownloadLockList.end(), strLockName );
}
TDownloadLockUserIterator spm_tclDownloadLockListManager::getLockUserIterator( std::string strLockName
                                                                              ,tU16 u16UserID ){
/*!
  * \fn
  *  \brief
  *   Get lock user iterator from given lock name and user id.
  *
  *  \param. strLockName: lock name.
  *          u16UserID: user id.
  *  \return.
  ******
  */
   TDownloadLockIterator itLock = getLockIterator( strLockName );
   return std::find( itLock->_oDownloadLock.lu16LockUsers.begin()
                    ,itLock->_oDownloadLock.lu16LockUsers.end()
                    ,u16UserID);
}
tVoid spm_tclDownloadLockListManager::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *   Get reference to other objects through factory.
  *
  *  \param.
  *  \return
  ******
  */
}

