/*!
  * \file spm_CcaServiceHandler.cpp
  *  \brief
  *    Implements the SPM service.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 07.01.11  | TMS Fischer       | initial version
  ******
  */

#include <sstream>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_CcaServiceHandler.h"

// spm class definitions (wo if class)
#include "spm_ConnectedApps.h"

// interfaces class definitions
#include "spm_ISystemPowerManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ILocalAppManager.h"
#include "spm_ISyncHandler.h"
#include "spm_IOsLinux.h"

#include "spm_factory.h"

// spm helper
#include "spm_ClientHandlerBase.h"
#include "spm_IFactory.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_CSH
 #include "trcGenProj/Header/spm_CcaServiceHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CCASERVICEHANDLER


spm_tclCcaServiceHandler::spm_tclCcaServiceHandler( const ISpmFactory& factory )
   : ISpmCcaServiceServer( factory )
   , _u16NextRegisterId( 0 )
   , _bRestartTrigger( FALSE )
   , _bSysModeChangeActive( FALSE )
   , _poclCcaMsgHandler( NULL )
   , _poclLocalAppManager( NULL )
   , _poclSystemPowerManager( NULL )
   , _poclDb( NULL )
   , _poclWorkerServer( NULL )
   , _poclSyncHandler( NULL )
   , _poclLockListManager( NULL )
   , _oSpmProperties( SPM_FUNCARRAY_SIZE, CCA_C_U16_SRV_SPM, SPM_NOTTABLE_SEM_NAME )
   , _oSpmMethods( CCA_C_U16_SRV_SPM ){
/*!
  * \fn
  *  \brief
  *    Constructor.
  *
  *  \param
  *     spm_tclSystemPowerManager *poSpm: pointer to parent class
  ******
  */

   _tClientList.clear( );


   // lock list
   _tDlLockList.clear();
   //check for DP entries to restore Locks

   // add client handlers
   vInitProperties( ); // initialize own properties
   vInitMethods( );    // initialize own methods
}

spm_tclCcaServiceHandler::~spm_tclCcaServiceHandler( ){
/*!
  * \fn
  *  \brief
  *    Destructor.
  ******
  */
   _poclWorkerServer->vRemoveClient( this );

   _poclSystemPowerManager = NULL;
   _poclDb                 = NULL;
   _poclCcaMsgHandler      = NULL;
   _poclLocalAppManager    = NULL;
   _poclWorkerServer       = NULL;
   _poclSyncHandler        = NULL;
   _poclLockListManager    = NULL;
}

tVoid spm_tclCcaServiceHandler::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
      SPM_GET_IF_REFERENCE_USE_VAR( _poclLocalAppManager,    ISpmLocalAppManager );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaMsgHandler,      ISpmCcaServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclSystemPowerManager, ISpmSystemPowerManager );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclDb,                 ISpmApplicationDatabase );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclSyncHandler,        ISpmSyncHandler );
      SPM_GET_IF_REFERENCE_USE_VAR( _poclLockListManager,    ISpmDownloadLockListManager );

} // vGetReferences

tVoid spm_tclCcaServiceHandler::vStartCommunication( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );

   //restore DL restricted area list
   spm_corefi_tclMsgDownloadLockStatesStatus oDlLockList;
   oDlLockList.listRestrictedLock.clear();

   dp_tclSpmDpInternDataRestrictedDlLock oDpLock;
   tU32 u32Count = oDpLock.u32GetCount( );

   ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::vStartCommunication(): Number areas: %d", u32Count));

   for ( tUInt i = 0; i < u32Count; i++ ){
      TRestrictedDlLock oDpLockEntry;
      oDpLock.s32GetElem( i, oDpLockEntry );

      spm_fi_tcl_SPM_RestrictedDlLock oLock;
      oLock.strRestrictedLockArea   = oDpLockEntry.strRestrictedLockArea;
      oLock.u16AppIdLockOwner       = oDpLockEntry.u16AppIdLockOwner;
      oLock.u16AppIdLockRequester   = AMT_C_U16_APPID_INVALID;
      oDlLockList.listRestrictedLock.push_back(oLock);

      TDlLock newLock;
      newLock.u16AppIdOwner     = oDpLockEntry.u16AppIdLockOwner;
      newLock.u16AppIdRequester = AMT_C_U16_APPID_INVALID;

      _tDlLockList[oDpLockEntry.strRestrictedLockArea] = newLock;

      ETG_TRACE_FATAL( ( "spm_tclCcaServiceHandler::vStartCommunication(): Entry: area: %40s, Owner: %u", oLock.strRestrictedLockArea.szValue, ETG_ENUM( ail_u16AppId, oLock.u16AppIdLockOwner )));
   }
   vUpdateProperty(SPM_COREFI_C_U16_DOWNLOADLOCKSTATE, &oDlLockList);

   dp_tclSpmDpInternDataMapDlLockApp oDpMapDlLcokApp;
   spm_corefi_tclMsgMapDownloadLockStateStatus oMapDownloadLockState;

   oMapDownloadLockState.u16AppIdLockOwner = oDpMapDlLcokApp.tGetData( );

   vUpdateProperty( SPM_COREFI_C_U16_MAPDOWNLOADLOCKSTATE, &oMapDownloadLockState );

}

tVoid spm_tclCcaServiceHandler::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *  Start other services after LCM late is started.
  *  At this point, persistent partition is mounted
  *  and writable.
  *
  *  \param
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclLockListManager );
   TLockMsgContainer tLockMsgContainer;
   if(_poclLockListManager->bRestoreLock( &tLockMsgContainer ) ){
      vUpdateLockListProperty( );
   }
}

tVoid spm_tclCcaServiceHandler::vAddClientHandler( ISpmClientHandlerBase *pClient ){
   arClients.push_back( pClient );
}

tVoid spm_tclCcaServiceHandler::vRemoveClientHandler( ISpmClientHandlerBase *pClient ){
   TVectorClientHandlerBase::iterator it;

   it = std::find( arClients.begin( ), arClients.end( ), pClient );

   if ( it != arClients.end( ) ){
      arClients.erase( it );
   }
}

tBool spm_tclCcaServiceHandler::bRegisterServices( tU32 u32AppId ){
/*!
  * \fn
  *  \brief
  *  The function register services, which the spm uses (download, gateway).
  *  The bRegisterService-Function will be called if one component is initialized.
  *
  *  \param
  ******
  */
   (tVoid)u32AppId;

   tBool bSuccess = true;

   if ( _poclCcaMsgHandler == NULL ){
      // called before vSetReferences()
      TRACE_SPM_FAILURE;
      return( FALSE );
   }
   TVectorClientHandlerBase::iterator it;

   for ( it = arClients.begin( ); it != arClients.end( ); ++it ){
      bSuccess = ( * it )->bRegisterServices( u32AppId );
   }
   return( bSuccess );
} // bRegisterServices

tBool spm_tclCcaServiceHandler::bIsAppNotifiedToFid( tU32 u32AppId,
                                                     tU16 u16Fid ){
/*!
  * \fn
  *  \brief
  *    Check if application is already notified for spcific FID. Return True/False
  *
  *  \param
  ******
  */
   return( _oSpmProperties.bIsAppNotifiedToFid( u32AppId, u16Fid ) );
}

tVoid spm_tclCcaServiceHandler::vTraceNotificationTab( tU16 u16FID ){
   std::vector < spm_NotificationInfo >                 ar;
   std::vector < spm_NotificationInfo >::const_iterator it;

   _oSpmProperties.vTrace( u16FID, ar );

   for ( it = ar.begin( ); it != ar.end( ); ++it ){
      ETG_TRACE_FATAL( ( "Notification table for FID: %u, Application %u, RegisterId %d",
                         ETG_ENUM( SPM_CCA_FID_ID, u16FID ),
                         ETG_ENUM( ail_u16AppId,   (tU32)it->_u16AppID ),
                         it->_u16RegisterID ) );
   }
}

tVoid spm_tclCcaServiceHandler::vTraceAllNotificationTabs( ){
   std::vector < tU16 >                 tabs;
   std::vector < tU16 >::const_iterator it;

   _oSpmProperties.vGetAllProperties( tabs );

   for ( it = tabs.begin( ); it != tabs.end( ); ++it ){
      vTraceNotificationTab( * it );
   }
}

tVoid spm_tclCcaServiceHandler::vOnServiceDataMessage( amt_tclServiceData *poSrvData ){
/*!
  * \fn
  *  \brief
  *     This function receives ServiceData-Messages and dispatches them.
  *     The SPM uses some services of the following components:
  *     (Sensor, Download)
  *     This function dispatch the ServiceData-Messages to the
  *     special handler objects.
  *
  *     Also, the own SPM-Service with the SystemState-Propertie is
  *     realized in this functionblock.
  *
  *  \param
  ******
  */
   tU16 u16ServiceID = poSrvData->u16GetServiceID( );

   if ( poSrvData->u16GetSourceAppID( ) == CCA_C_U16_APP_SPM ){
      ETG_TRACE_USR4( ( "Handle service message from application ServiceId 0x%02x, %u, FunctionId %02x, Opcode %u",
                        poSrvData->u16GetServiceID( ),
                        ETG_ENUM( ail_u16AppId,   poSrvData->u16GetSourceAppID( ) ),
                        ETG_ENUM( SPM_CCA_FID_ID, poSrvData->u16GetFunctionID( ) ),
                        ETG_ENUM( CCA_OPCODE,     poSrvData->u8GetOpCode( ) )
                        ) );
   } else if ( poSrvData->u16GetSourceAppID( ) == CCA_C_U16_APP_SPM_SLV ){
                        ETG_TRACE_USR4( ( "Handle service message from application ServiceId 0x%02x, %u, FunctionId %02x, Opcode %u",
                        poSrvData->u16GetServiceID( ),
                        ETG_ENUM( ail_u16AppId,            poSrvData->u16GetSourceAppID( ) ),
                        ETG_ENUM( LCM_DBUSCTRL_CCA_FID_ID, poSrvData->u16GetFunctionID( ) ),
                        ETG_ENUM( CCA_OPCODE,              poSrvData->u8GetOpCode( ) )
                        ) );
   } else {
                        ETG_TRACE_USR4( ( "Handle service message from application ServiceId 0x%02x, %u, FunctionId %02x, Opcode %u",
                        poSrvData->u16GetServiceID( ),
                        ETG_ENUM( ail_u16AppId, poSrvData->u16GetSourceAppID( ) ),
                        poSrvData->u16GetFunctionID( ),
                        ETG_ENUM( CCA_OPCODE,   poSrvData->u8GetOpCode( ) )
                        ) );
   }
   bool                               bHandled = false;

   TVectorClientHandlerBase::iterator it;
   for ( it = arClients.begin( ); it != arClients.end( ); ++it ){
      if ( ( * it )->bOnServiceDataMessage( poSrvData ) ){
         bHandled = true;
         break;
      }
   }
   if ( !bHandled )
   {
      switch ( u16ServiceID ){
         case CCA_C_U16_SRV_SPM:
         {
            // handle own Service (CCA_C_U16_SRV_SPM)
            vHandleSPMService( poSrvData );
            break;
         }     // case CCA_C_U16_SRV_SPM

         default:
         {
            break;
         }
      }
   }
} // vOnServiceDataMessage

tVoid spm_tclCcaServiceHandler::vHandleSPMService( amt_tclServiceData *poSrvData ){
   spm_tclConnectedApp *poConApp = NULL;

   SPM_NULL_POINTER_CHECK( _poclDb );

   if ( _poclLocalAppManager == NULL ){
      // called before vSetReferences()
      TRACE_SPM_FAILURE;
      return;
   }

   poConApp = _poclDb->poGetByAppID( poSrvData->u16GetSourceAppID( ) );
   if ( poConApp != NULL ){
      if ( !poConApp->bIsRegistered( poSrvData->u16GetSourceSubID( ), poSrvData->u16GetRegisterID( ) ) ){
         // application is not registered for this service
         // \todo: Send response
         ETG_TRACE_FATAL( ( "vHandleSPMService not registered: AppId=0x%04x, RegisterID=0x%04x, SubID=0x%04x",
                            poSrvData->u16GetSourceAppID( ),
                            poSrvData->u16GetRegisterID( ),
                            poSrvData->u16GetSourceSubID( )
                            ) );
         return;
      }
   } else {
      if ( ( CCA_C_U16_APP_SPM == poSrvData->u16GetSourceAppID( ) )
           && ( CCA_C_U16_APP_SPM == poSrvData->u16GetTargetAppID( ) )
           && ( poSrvData->u8GetOpCode( ) == AMT_C_U8_CCAMSG_OPCODE_SET )
           ){
         //internal message --> dispatch
         ETG_TRACE_USR4( ( "vHandleSPMService, internal message received --> CmdId: %d.", ETG_ENUM( SPM_CCA_FID_ID, poSrvData->u16GetFunctionID( ) ) ) );
         SPM_NULL_POINTER_CHECK( _poclSyncHandler );
         _poclSyncHandler->vDispatchTask( poSrvData->u16GetFunctionID( ) );
         return;
      } else {
         _TCcaSrvClientList::iterator pIter;
         tBool                        bFound = FALSE;
         for ( pIter = _tClientList.begin( ); ( pIter != _tClientList.end( ) ) && !bFound; ++pIter ){
            ETG_TRACE_USR4( ( "vHandleSPMService, unknown app list: 0x%04x, 0x%04x, 0x%04x", pIter->u32AppId, pIter->u16SubId, pIter->u16RegId ) );
            if ( ( pIter->u16SubId == poSrvData->u16GetSourceSubID( ) ) && ( pIter->u32AppId == poSrvData->u16GetSourceAppID( ) ) ){
               ETG_TRACE_ERR( ( "vHandleSPMService, App found 0x%04x", poSrvData->u16GetSourceAppID( ) ) );
               bFound = TRUE;
            }
         }
         if ( bFound ){
            // bIsRegistered = TRUE;
         } else if ( CCA_C_U16_APP_SPM != poSrvData->u16GetSourceAppID( ) ){
               ETG_TRACE_ERR( ( "vHandleSPMService, unknown application: 0x%04x", poSrvData->u16GetSourceAppID( ) ) );
            return;
         }
      }
   }

   // handle properties and methods of own service
   if ( _poclCcaMsgHandler != NULL ){
      switch ( poSrvData->u8GetOpCode( ) ){
         case AMT_C_U8_CCAMSG_OPCODE_GET:
         case AMT_C_U8_CCAMSG_OPCODE_SET:
         case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         case AMT_C_U8_CCAMSG_OPCODE_STATUS:
            _oSpmProperties.vHandleProperty( _poclCcaMsgHandler, poSrvData );
            break;

         case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         case AMT_C_U8_CCAMSG_OPCODE_METHODRESULT:
         case AMT_C_U8_CCAMSG_OPCODE_METHODABORT:
            _oSpmMethods.vHandleMethod( _poclCcaMsgHandler, poSrvData );
            break;

         default:
            break;
      } // switch
   }
}       // vHandleSPMService

tBool spm_tclCcaServiceHandler::bHandleServiceRegister( const amt_tclServiceRegister *poSrvReg ){
/*!
  * \fn
  *  \brief
  *   Handles the SPM - Service - registrations.
  *   This Service is for applications, who are interested in systemstates
  *
  *  \param[in] poSrvReg: pointer to amt_tclServiceRegister.
  *  \return    TRUE if successfully register the service otherwise FALSE.
  ******
  */
   tU16        u16ServiceID;
   tU16        u16ClientSub;
   tU16        u16CurrentRegId = AMT_C_U16_REGID_INVALID;

   std::string strService;

   tBool       bRetVal;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );
   SPM_NULL_POINTER_CHECK_VAL( _poclCcaMsgHandler );
   SPM_NULL_POINTER_CHECK_VAL( _poclSystemPowerManager );
   SPM_NULL_POINTER_CHECK_VAL( _poclLocalAppManager );

   u16ServiceID = poSrvReg->u16GetServiceID( );

   // get sub sender
   u16ClientSub = poSrvReg->u16GetSourceSubID( );

   amt_tclServiceRegisterConf oSrvConf( CCA_C_U16_APP_SPM,
                                        poSrvReg->u16GetSourceAppID( ),
                                        u16ServiceID,
                                        0,                      // register ID
                                        0,                      // register state
                                        0,                      // major
                                        0,                      // minor
                                        0,                      // patch
                                        AMT_C_U16_DEFAULT_NULL, // source sub
                                        u16ClientSub            // target sub
                                        );

   ETG_TRACE_USR4( ( "bHandleServiceRegister: AppID=%d, ServiceID=%d, SubID=%d",
                     poSrvReg->u16GetSourceAppID( ),
                     poSrvReg->u16GetServiceID( ),
                     poSrvReg->u16GetSourceSubID( )
                     ) );


   switch ( u16ServiceID ){
      case CCA_C_U16_SRV_SPM:
      {
         spm_tclConnectedApp *poConApp;
         poConApp = _poclDb->poGetByAppID( poSrvReg->u16GetSourceAppID( ) );

         if ( poConApp != NULL ){
            if ( !poConApp->bSubIDAlwaysRegistered( u16ClientSub ) ){
               poConApp->vAddRegId( u16ClientSub, _u16NextRegisterId );
               // service registration acknowledge
            }
            oSrvConf.vSetRegisterID( _u16NextRegisterId );
            oSrvConf.vSetRegisterState( AMT_C_U8_REGCONF_SUCCESS );

            u16CurrentRegId = _u16NextRegisterId;

            ++_u16NextRegisterId;     // increment for next registration of client to spm service
         } else {

            _TCcaSrvClientList::iterator pIter;
            tBool                        bFound = FALSE;
            for ( pIter = _tClientList.begin( ); ( pIter != _tClientList.end( ) ) && !bFound; ++pIter ){
               if ( ( pIter->u16SubId == poSrvReg->u16GetSourceSubID( ) ) && ( pIter->u32AppId == poSrvReg->u16GetSourceAppID( ) ) ){
                  oSrvConf.vSetRegisterID( pIter->u16RegId );
                  oSrvConf.vSetRegisterState( AMT_C_U8_REGCONF_SUCCESS );
                  bFound = TRUE;
                  ETG_TRACE_USR1( ( "bHandleServiceRegister(): Application already registered to SPM service." ) );
               }
            }

            if ( !bFound ){
               _TCcaSrvClient tEntry;

               tEntry.u32AppId = poSrvReg->u16GetSourceAppID( );
               tEntry.u16RegId = _u16NextRegisterId++;
               tEntry.u16SubId = poSrvReg->u16GetSourceSubID( );

               u16CurrentRegId = tEntry.u16RegId;

               // service registration acknowledge
               oSrvConf.vSetRegisterID( tEntry.u16RegId );
               oSrvConf.vSetRegisterState( AMT_C_U8_REGCONF_SUCCESS );

               _tClientList.push_back( tEntry );

                  ETG_TRACE_USR1( ( "bHandleServiceRegister(): Unknown application added to service list." ) );
            }
         }
         break;

      }

      default:
      {
         // service not available
         std::stringstream stream;
         stream << "SPM: Service registration failed --> service 0x" << std::hex << u16ServiceID << " not available!!!! ";
         strService = stream.str();
         _poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strService.c_str( ), (tU16)strService.length( ) );

         oSrvConf.vSetRegisterState( AMT_C_U8_REGCONF_NO_SUCCESS );
      }
   } // switch

   if ( _poclCcaMsgHandler->bPostMessage( &oSrvConf )
        && ( oSrvConf.u8GetRegisterState( ) == AMT_C_U8_REGCONF_SUCCESS ) ){
      ETG_TRACE_USR4( ( "Registration from application %u succeeded, register id 0x%02x",
                        ETG_ENUM( ail_u16AppId, poSrvReg->u16GetSourceAppID( ) ),
                        u16CurrentRegId
                        ) );

      amt_tclServiceStatus oSrvStatus( CCA_C_U16_APP_SPM, poSrvReg->u16GetSourceAppID( ),
                                       u16ServiceID,
                                       u16CurrentRegId,
                                       AMT_C_U8_SVCSTATE_AVAILABLE,
                                       AMT_C_U16_DEFAULT_NULL,
                                       u16ClientSub
                                       );

      bRetVal = _poclCcaMsgHandler->bPostMessage( &oSrvStatus );
   } else {
      // \todo : error handling
      bRetVal = FALSE;
   }
   return( bRetVal );
} // bHandleServiceRegister

tBool spm_tclCcaServiceHandler::bHandleServiceUnregister( const amt_tclServiceUnregister *poSrvUnreg ){
/*!
  * \fn
  *  \brief
  *   Handles the SPM - Service - unregistrations.
  *
  *  \param
  *  \note
  *     All implementations up to now acknowledges the service unregister request. According to CCA
  *     this is not allowed and resulted in errors. The acknowledge is now removed.
  ******
  */
   tU16 u16ServiceID;
   tU16 u16ClientSub;

   SPM_NULL_POINTER_CHECK_VAL( _poclDb );

   u16ServiceID = poSrvUnreg->u16GetServiceID( );
   // get sub sender
   u16ClientSub = poSrvUnreg->u16GetSourceSubID( );

                  ETG_TRACE_USR4( ( "bHandleServiceUnRegister: AppID=%d, ServiceID=%d, SubID=%d",
                     poSrvUnreg->u16GetSourceAppID( ),
                     poSrvUnreg->u16GetServiceID( ),
                     poSrvUnreg->u16GetSourceSubID( )
                     ) );

   switch ( u16ServiceID ){
      case CCA_C_U16_SRV_SPM:
      {
         spm_tclConnectedApp *poConApp;

         poConApp = _poclDb->poGetByAppID( poSrvUnreg->u16GetSourceAppID( ) );

         if ( poConApp != NULL ){
            tU16 u16RegId;
            while ( ( u16RegId = poConApp->u16FindRegId( u16ClientSub ) ) != ( tU16 ) - 1 ){
               _oSpmProperties.bRemoveAllEntriesWithRegId( u16RegId );
               poConApp->vRemoveRegId( u16ClientSub );
            }
         } else {
            _TCcaSrvClientList::iterator pIter;
            for ( pIter = _tClientList.begin( ); ( pIter != _tClientList.end( ) ); ++pIter ){
               if ( ( pIter->u16SubId == poSrvUnreg->u16GetSourceSubID( ) ) && ( pIter->u32AppId == poSrvUnreg->u16GetSourceAppID( ) ) ){
                  ETG_TRACE_USR4( ( "spm_tclCcaServiceHandler::bHandleServiceUnregister --> Service unregister Remote." ) );
                  ETG_TRACE_USR4( ( "SourceAppID = %d RemoteClientRegisterID = 0x%04x, RemoteClientSubID = 0x%04x",
                                    pIter->u32AppId,
                                    pIter->u16RegId, pIter->u16SubId ) );

                  _oSpmProperties.bRemoveAllEntriesWithRegId( pIter->u16RegId );
                  _tClientList.erase( pIter );

                  break;
               }
            }
         }
         break;
      }

      default:
      {
                  ETG_TRACE_USR4( ( "bHandleServiceUnregister, unknown ServiceID: 0x%04x", u16ServiceID ) );
      }
      break;
   } // switch

   return( TRUE );
} // bHandleServiceUnregister

tBool spm_tclCcaServiceHandler::bHandleServiceConfirmation( const amt_tclServiceRegisterConf *poSrvConf ){
/*!
  * \fn
  *  \brief
  *   Handles the SPM - Service - Confirmation.
  *
  *  \param
  ******
  */
// tU16 u16ServiceID;

   if ( poSrvConf->u8GetRegisterState( ) == AMT_C_U8_REGCONF_NO_SUCCESS ){
      return( false );
   }
   TVectorClientHandlerBase::iterator it;

   for ( it = arClients.begin( ); it != arClients.end( ); ++it ){
      ( * it )->vHandleServiceConfirmation( poSrvConf );
   }

   return( true );
} // bHandleServiceConfirmation

tBool spm_tclCcaServiceHandler::bHandleServiceStatus( const amt_tclServiceStatus *poSrvStatus ){
/*!
  * \fn
  *  \brief
  *   Handles the SPM - Service - status.
  *
  *  \param
  ******
  */
   TVectorClientHandlerBase::iterator it;

   for ( it = arClients.begin( ); it != arClients.end( ); ++it ){
      ( * it )->vHandleServiceStatus( poSrvStatus );
   }

   return( TRUE );
}

tVoid spm_tclCcaServiceHandler::vTraceInfo( ){
/*!
  * \fn
  *  \brief
  *   Trace current state of CCA properties.
  ******
  */
   #ifndef LCM_UNIT_TESTS
      // trace system state
      dp_tclSpmDpConfigIgnitionOffTime           oIgnOff;
      dp_tclSpmDpConfigDoorOpenTime              oDoorOpen;
      dp_tclSpmDpConfigClockDisplayTime          oClockDis;
      dp_tclSpmDpInternDataConnectToBattery      oConnBat;
      dp_tclSpmDpInternDataConnectToBatteryCount oConnBatCount;

      tU32                                       u32StandbyTime = 0;
      dp_tclSpmDpConfigDevStandbyTime            oStdyTime;
      oStdyTime >> u32StandbyTime;

         ETG_TRACE_FATAL( ( "CCA property FID_SPM_SYSTEMSTATE:          %u", ETG_ENUM( SPM_SYSTEM_STATES, (tU32)_systemState.getProperty( ).SystemState.enType ) ) );
      //ETG_TRACE_FATAL( ( "CCA property FID_SPM_SUBSTATE:             0x%08x", (tU32)_onOffReason.getProperty().OnOffReason.enType ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_STANDBY_PERS_TIME:    %d sec", ( u32StandbyTime ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_IGNITION_OFF_TIME:    %d min", ( oIgnOff.tGetData( ) ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_DOOR_OPEN_TIME:       %d sec", ( oDoorOpen.tGetData( ) ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_CLOCK_DISPLAY_TIME:   %d sec", ( oClockDis.tGetData( ) ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_APP_HMISTATE:         %u", ETG_ENUM( SPM_HMI_STATE, (tU32)_hmiState.getProperty( ).HmiState.enType ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_APP_DIMSTATE:         %u", ETG_ENUM( SPM_DIM_STATE, (tU32)_dimState.getProperty( ).DimState.enType ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_CVM_EVENT:            %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_cvmEvent.getProperty( ).CvmEvent.enType ) ) );
         ETG_TRACE_FATAL( ( "CCA property FID_SPM_CONNECT_TO_BATTERY:   connect state: %u, count %d", ETG_ENUM( SPM_BOOL_STATE, oConnBat.tGetData( ) ), oConnBatCount.tGetData( ) ) );

         for (_oSpmProperties.vFirst(); !_oSpmProperties.bIsDone(); _oSpmProperties.vNext() ){
            tChar                    szStr[SPM_INTERNAL_BUFFER_LENGTH+1];
            spm_PropertyHandlingItem item;
            _oSpmProperties.vCurrentItem(item);

            if(OSALUTIL_s32SaveNPrintFormat(szStr, sizeof( szStr ), "CcaProperty: FID: 0x%04x (%26s): ", item._u16Fid, item._sName) == OSAL_ERROR ) {
               tU32 u32ErrorReason = OSAL_u32ErrorCode( );
               ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaServiceHandler::vTraceInfo !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                                   (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
            }
            if (!item._bUpReg){
               OSALUTIL_szSaveStringNConcat(szStr, "no UpregProperty", SPM_INTERNAL_BUFFER_LENGTH  - OSAL_u32StringLength(szStr) );
            } else {
               for (tU32 i = 0; i < item._ar.size() && i < 20; ++i){
                  tChar szTmpStr[10];
                  if (OSALUTIL_s32SaveNPrintFormat(szTmpStr, sizeof( szTmpStr ), "%02x", item._ar[i]) == OSAL_ERROR) {
                     tU32 u32ErrorReason = OSAL_u32ErrorCode( );
                     ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaServiceHandler::vTraceInfo !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                                         (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
                  }
                  if (OSALUTIL_szSaveStringNConcat(szStr, szTmpStr, SPM_INTERNAL_BUFFER_LENGTH  - OSAL_u32StringLength(szStr) ) == OSAL_NULL ) {
                     tU32 u32ErrorReason = OSAL_u32ErrorCode( );
                     ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaServiceHandler::vTraceInfo !!!!!! Error detected !!!!!! cannot create string: error 0x%08X (%s)",
                                         (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
                  }
               }
            }
            ETG_TRACE_FATAL( ( "%s", szStr ) );
         }
   #endif // ifndef LCM_UNIT_TESTS
} // vTraceInfo

tBool spm_tclCcaServiceHandler::bSetDlLockStateRequest(std::string strLockArea, tU16 u16ReqAppId, tU32 u32LockCommand) {

    tBool bRet = TRUE;
    tBool bChangeDetected = FALSE;

    ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): area: %40s, requester: %u, cmd: %d", strLockArea.c_str(), ETG_ENUM( ail_u16AppId, u16ReqAppId ), u32LockCommand));

    std::map <std::string, TDlLock>::iterator it = _tDlLockList.find(strLockArea);

    if (it == _tDlLockList.end()) {
        //no lock found for requested area
        if (u32LockCommand == (tU32)spm_fi_tcl_e8_DownloadLockStates::FI_EN_SPM_E8_DOWNLOADLOCK) {
            ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): No entry in list --> add lock."));
            TDlLock newLock;
            newLock.u16AppIdOwner     = u16ReqAppId;
            newLock.u16AppIdRequester = AMT_C_U16_APPID_INVALID;

            _tDlLockList[strLockArea] = newLock;
            bChangeDetected = TRUE;
        }
    } else {
        if (  ((u32LockCommand == (tU32)spm_fi_tcl_e8_DownloadLockStates::FI_EN_SPM_E8_DOWNLOADUNLOCK) && (u16ReqAppId == it->second.u16AppIdOwner))
           || ((u32LockCommand == (tU32)spm_fi_tcl_e8_DownloadLockStates::FI_EN_SPM_E8_DOWNLOADMASTERUNLOCK) && (u16ReqAppId == CCA_C_U16_APP_FC_SW_UPDATE))
        ) {
            ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): Entry in list --> delete now (request called by owner or master)."));
            _tDlLockList.erase(it);
            bChangeDetected = TRUE;
        } else if ((u32LockCommand == (tU32)spm_fi_tcl_e8_DownloadLockStates::FI_EN_SPM_E8_DOWNLOADUNLOCK) && (u16ReqAppId != it->second.u16AppIdOwner)) {
            ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): Entry in list --> currently locked by %d. Unlock only possible by owner or master", ETG_ENUM( ail_u16AppId, it->second.u16AppIdOwner )));
            it->second.u16AppIdRequester = u16ReqAppId;
            bRet = FALSE;
        } else if ((u32LockCommand == (tU32)spm_fi_tcl_e8_DownloadLockStates::FI_EN_SPM_E8_DOWNLOADLOCK) && (u16ReqAppId != it->second.u16AppIdOwner)) {
            ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): Entry in list --> but already locked by %d", ETG_ENUM( ail_u16AppId, it->second.u16AppIdOwner )));
            it->second.u16AppIdRequester = u16ReqAppId;
            bRet = FALSE;
        }
    }

    if (bChangeDetected) {
        dp_tclSpmDpInternDataRestrictedDlLock oDpLock;
        oDpLock.vClearList();

        ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): List changed: update CCA property"));
        spm_corefi_tclMsgDownloadLockStatesStatus oDlLockList;
        oDlLockList.listRestrictedLock.clear();
        for (it = _tDlLockList.begin();it != _tDlLockList.end();it++) {

            { //store in datapool
                TRestrictedDlLock oNewDpEntry;
                (tVoid)memset((tVoid*)oNewDpEntry.strRestrictedLockArea, 0, SPM_CCA_AREA_MAX_LEN);

                oNewDpEntry.u16AppIdLockOwner       = it->second.u16AppIdOwner;
                OSAL_szStringNCopy(oNewDpEntry.strRestrictedLockArea, it->first.c_str(), SPM_CCA_AREA_MAX_LEN-1);
                oDpLock.vPushBack(oNewDpEntry);
            }


            spm_fi_tcl_SPM_RestrictedDlLock oLock;
            oLock.strRestrictedLockArea = it->first.c_str();
            oLock.u16AppIdLockOwner     = it->second.u16AppIdOwner;
            oLock.u16AppIdLockRequester = it->second.u16AppIdRequester;
            oDlLockList.listRestrictedLock.push_back(oLock);
            ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bSetDlLockStateRequest(): Entry: area: %40s, Owner: %u, Requester: %u", it->first.c_str(), ETG_ENUM( ail_u16AppId, it->second.u16AppIdOwner ), ETG_ENUM( ail_u16AppId, it->second.u16AppIdRequester )));
        }

        vUpdateProperty(SPM_COREFI_C_U16_DOWNLOADLOCKSTATES, &oDlLockList);

    }
    return bRet;
}

tVoid spm_tclCcaServiceHandler::vUpdateProperty( tU16                  u16Fid,
                                                 const fi_tclTypeBase *pMsgBase ){
   if ( _poclCcaMsgHandler != NULL ){
      _oSpmProperties.vUpdateProperty( _poclCcaMsgHandler, u16Fid, pMsgBase );
   }
}

const fi_tclTypeBase*spm_tclCcaServiceHandler::poGetProperty( tU16 u16Fid ) const {
   return( _oSpmProperties.poGetContent( u16Fid ) );
}

tVoid spm_tclCcaServiceHandler::vUpdateMethodResult( tU16                  u16Fid,
                                                     const fi_tclTypeBase *pMsgBase,
                                                     tU8                   u8OpCode ){
   if ( _poclCcaMsgHandler != NULL ){
      _oSpmMethods.vUpdateMethodResult( _poclCcaMsgHandler, u16Fid, pMsgBase, u8OpCode );
   }
}

tVoid spm_tclCcaServiceHandler::vHandleMessage( tU32 u32Message,
                                                tU32 u32Parm ){
   (tVoid)u32Parm;

   ETG_TRACE_USR4( ( "spm_tclCcaServiceHandlerConfig::vHandleMessage(): Message received: %08x", u32Message ) );
   if ( u32Message == SPM_U32_WORKER_CSH_SYSTEM_MODE_DL ){
      if ( bGetSysModeState( ) ){
         #ifndef LCM_UNIT_TESTS
            // state change is active -> trigger result
            spm_corefi_tclMsgSystemModeMethodResult oSysModeResult;
            oSysModeResult.s32ReturnVal = 0;

            vUpdateMethodResult( SPM_COREFI_C_U16_SYSTEMMODE, &oSysModeResult );
         #endif
         vSetSysModeState( FALSE );
      }
   }
} // vHandleMessage

tVoid spm_tclCcaServiceHandler::vPostInternalMessage( tU16   u16Cmd,
                                                      tVoid *pInternalMsg,
                                                      tU32   u32Len ){
   ETG_TRACE_USR4( ( "spm_tclCcaServiceHandler::vPostInternalMessage(): post internal message %u with length = %u.", u16Cmd, u32Len ) );

   SPM_NULL_POINTER_CHECK( _poclCcaMsgHandler );
   #ifndef LCM_UNIT_TESTS
      gm_tclStreamMessage oStreamMsg( 0, 0, 0, 0, 256, u16Cmd, AMT_C_U8_CCAMSG_OPCODE_SET, u32Len );
      if ( pInternalMsg != NULL ){
         oStreamMsg.vSetData( (const tChar*)pInternalMsg );
      }
      if ( oStreamMsg.bIsValid( ) ){
         if ( _poclCcaMsgHandler->bPostMessage( &oStreamMsg ) != TRUE ){
            NORMAL_M_ASSERT_ALWAYS( );
         }
      } else {
            NORMAL_M_ASSERT_ALWAYS( );
      }
   #endif
} // vPostInternalMessage
tVoid spm_tclCcaServiceHandler::vUpdateLockListProperty(){
/*!
  * \fn
  *  \brief
  *   Update UpdateLockStates FI-Property from Download Locks list
  *
  *  \param.
  *  \return.
  ******
  */
   spm_corefi_tclMsgUpdateLockStatesStatus _tLockListMsgUpdater;
   SPM_NULL_POINTER_CHECK(_poclLockListManager);
   _poclLockListManager->vRetrieveAllLocks(_tLockListMsgUpdater);
   vUpdateProperty(SPM_COREFI_C_U16_UPDATELOCKSTATES, &_tLockListMsgUpdater);
   ETG_TRACE_USR2( ( "spm_tclCcaServiceHandler::vUpdateLockListProperty(): Lock list is updated to property" ) );
}
tBool spm_tclCcaServiceHandler::bUpdateLock(std::string strLock, tU32 u32LockCommand, tU16 u16ReqAppId){
/*!
  * \fn
  *  \brief
  *   Handle update lock requests from applications.
  *   Decide to create new lock or update available lock.
  *
  *  \param: strLock: requested Lock name.
  *          u32LockCommand: requested command.
  *          u16ReqAppId: ID of requesting App.
  *  \return: TRUE: requested command is valid and handled successfully.
  *           FALSE: requested command is invalid and failed to handle.
  ******
  */
   tBool bRet = FALSE;
   TLockMsgContainer tLockMsgContainer(u32LockCommand
                                      ,u16ReqAppId
                                      ,strLock);
   ETG_TRACE_USR2( ( "spm_tclCcaServiceHandler::bUpdateLock(): Received update lock request with Command: %u, Requester: %u, Lock String: %s",
                     ETG_ENUM( SPM_UPDATELOCKCOMMAND, u32LockCommand ), ETG_ENUM(ail_u16AppId,u16ReqAppId), strLock.c_str( ) ) );
   SPM_NULL_POINTER_CHECK_VAL(_poclLockListManager);
   bRet = _poclLockListManager->bOnLockCommandHandler(&tLockMsgContainer);
   if ( bRet ){
      vUpdateLockListProperty();
      ETG_TRACE_USR1( ( "spm_tclCcaServiceHandler::bUpdateLock(): Valid request from Application %u --> Updated Lock Property", ETG_ENUM(ail_u16AppId,u16ReqAppId) ) );
   }
   return bRet;
}

