/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_CcaProperties.h
  * @brief        This is header file contains framework to work with CCA properties.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include <string>
#include <map>

#include "spm_ICcaServer.h"
#include "spm_CcaServiceHandlerMsg.h"

// /////////////////////////////////////////////////////////////////////////////
// interface for abstract properties

/*!
  *  \class IProperty
  *  \brief This abstract class provides framework to work with CCA properties. This class provides interfaces
          for get and set operation on service data.

  */
class IProperty
{
public:
virtual ~IProperty( ){}

virtual tBool bUpdate( const fi_tclTypeBase *pMsg )        = 0;

virtual const fi_tclTypeBase*poGetContent( ) const         = 0;

virtual tVoid vGetContent( std::vector < tU8 >& ar ) const = 0;


// called if a get operation is triggered on a property
virtual tVoid vGetOperation( amt_tclServiceData */*poSrvData*/ ){
}  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

enum
{
   IPROPERTY_RETURN_SUCCESS,
   IPROPERTY_RETURN_FAILED,
   IPROPERTY_RETURN_DELAYED,
   IPROPERTY_RETURN_UNKNOWN
};

// called if a set operation is triggered on a property
virtual tU8 u8SetOperation( amt_tclServiceData */*poSrvData*/ ){
   return( (tU8)IPROPERTY_RETURN_SUCCESS );
}  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

// is called after a status update was made (for special purposes)
virtual tVoid vAfterUpdate( ){
}

};

/*!
  *  \class spm_Property
  *  \brief This template class provides framework to work with CCA properties. This class provides template interface for
          assignment operator and get property .
  */
template < typename T >
class spm_Property : public IProperty
{
protected:
T _property;

public:
spm_Property < T >( ){
}
spm_Property < T >( const spm_Property < T > &op ){
   _property = op._property;
}
spm_Property < T >( const T &op ){
   _property = op;
}
const spm_Property < T >& operator=( const spm_Property < T >& op ){
   _property = op._property;
   return( * this );
}

virtual const spm_Property < T >& operator=( const T& op ){
   _property = op;
   return( * this );
}

virtual ~spm_Property( ){
}

// Implementation of interface IProperty
virtual tBool bUpdate( const fi_tclTypeBase *pMsg ){
   const fi_tclTypeBase& val = * pMsg; // >corfoGetTypeBase();

   if ( !( _property == ( const T & )val ) ){
      _property = ( const T & )val;
      return( TRUE );
   }
   return( FALSE );
}

virtual const fi_tclTypeBase*poGetContent( ) const {
   return( (const fi_tclTypeBase*)&_property );
}

virtual tVoid vGetContent(std::vector < tU8 >& ar) const {
   tU8              buffer[256]; // temporary buffer for streamed object data
   fi_tclOutContext ctxt(&buffer[0], &buffer[256]);

   _property.oWrite(ctxt);

   // clear the vector, after that resize to required size
   // at last copy the contents from temporary buffer to vector
   ar.clear();
   ar.resize(ctxt.u32GetFilledSize() );
   std::copy(buffer, &buffer[ctxt.u32GetFilledSize()], ar.begin() );
}

const T& getProperty( ) const {
   return( _property );
}

};


// values to define the functionality of a property
#define SPM_PROP_ADMIN_GET      0x00000001
#define SPM_PROP_ADMIN_SET      0x00000002
#define SPM_PROP_ADMIN_UPREG    0x00000004

// convenience macros
#define SPM_PROP_ADMIN_GET_SET  ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_SET )
#define SPM_PROP_ADMIN_GET_UPREG ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_UPREG )
#define SPM_PROP_ADMIN_GET_SET_UPREG ( SPM_PROP_ADMIN_GET | SPM_PROP_ADMIN_SET | SPM_PROP_ADMIN_UPREG )

struct spm_AdminProperty
{
   IProperty *_pProperty;
   tU32 _u32Configuration;
   tCString _sName;
};

typedef std::map < tU16, spm_AdminProperty* > TMapProperties;
typedef std::map < tU16, ahl_tNotification > TMapPropertyNotification;

struct spm_PropertyHandlingItem
{
   tCString _sName;
   std::vector < tU8 > _ar;
   tU16 _u16Fid;
   tBool _bUpReg;
};

// needed to trace information
struct spm_NotificationInfo
{
   tU16 _u16AppID;
   tU16 _u16RegisterID;
};

/*!
  *  \class spm_PropertyHandling
  *  \brief This class provides framework to work with CCA properties. This class provides interfaces to
         add and remove property . It also provides interfaces to get all properties and update properties.
  */
class spm_PropertyHandling
{
private:
TMapProperties           _propertiesMap;
TMapPropertyNotification _notificationsMap;
ahl_tclNotificationTable _oNotificationTable;
OSAL_tSemHandle          _hNotificationTableSemaphore;
std::string              _strSemname;
TMapProperties::iterator _itMap;
tU16                     _u16ServiceId;

public:
spm_PropertyHandling( tU16               u16FuncArraySize,
                      tU16               u16ServiceId,
                      const std::string& strSemname );
virtual ~spm_PropertyHandling( );


tVoid vAddProperty( tU16               u16Fid,
                    spm_AdminProperty *p ){
   TMapProperties::iterator it;

   it = _propertiesMap.find( u16Fid );
   if ( it == _propertiesMap.end( ) ){
      _propertiesMap[u16Fid] = p;
   }
}

tVoid vRemoveProperty( tU16 u16Fid ){
   TMapProperties::iterator it = _propertiesMap.find( u16Fid );

   if ( it != _propertiesMap.end( ) ){
      _propertiesMap.erase( it );
   }
}

tVoid vGetAllProperties( std::vector < tU16 >& ar ){
   TMapProperties::iterator it;

   for ( it = _propertiesMap.begin( ); it != _propertiesMap.end( ); ++it ){
      ar.push_back( it->first );
   }
}

tVoid vUpdateProperty( ISpmCcaServer        *pServer,
                       tU16                  u16Fid,
                       const fi_tclTypeBase *pMsgBase );


const fi_tclTypeBase*poGetContent( tU16 u16Fid ) const {
   const spm_AdminProperty *p = poFindProperty( u16Fid );

   if ( p == NULL ){
      return( NULL );  // property not found
   }
   return( p->_pProperty->poGetContent( ) );
}

// iterator interface
tVoid vFirst( ){
   _itMap = _propertiesMap.begin( );
}

tBool bIsDone( ){
   return( _itMap == _propertiesMap.end( ) );
}

tVoid vNext( ){
   ++_itMap;
}

tVoid vCurrentItem( spm_PropertyHandlingItem& item ){
   if ( _itMap != _propertiesMap.end( ) ){
      item._sName  = _itMap->second->_sName;
      _itMap->second->_pProperty->vGetContent( item._ar );
      item._u16Fid = _itMap->first;
      item._bUpReg = ( _itMap->second->_u32Configuration & SPM_PROP_ADMIN_UPREG ) != 0;
   } else {
      item._sName  = NULL;
      item._u16Fid = 0xffff;
      item._bUpReg = FALSE;
      item._ar.clear( );
   }
}

private:
const spm_AdminProperty*poFindProperty( tU16 u16Fid ) const {
   TMapProperties::const_iterator it;

   it = _propertiesMap.find( u16Fid );

   if ( it != _propertiesMap.end( ) ){
      return( it->second );
   }
   return( NULL );
}

const spm_AdminProperty*poFindProperty( const amt_tclServiceData *poSrvData ) const {
   TMapProperties::const_iterator it;

   it = _propertiesMap.find( poSrvData->u16GetFunctionID( ) );

   if ( it != _propertiesMap.end( ) ){
      return( it->second );
   }
   return( NULL );
}

tVoid vStatusMessage( ISpmCcaServer           *pServer,
                      IProperty               *p,
                      tU16                     u16Fid,
                      const ahl_tNotification *pNot );

tVoid vErrorMessage( ISpmCcaServer           *pServer,
                     tU16                     u16Fid,
                     const ahl_tNotification *pNot,
                     tU16                     u16ErrorData );

public:
tVoid vHandleProperty( ISpmCcaServer      *pServer,
                       amt_tclServiceData *poSrvData );

tBool bIsAppNotifiedToFid( tU32 u32AppId,
                           tU16 u16Fid );

tVoid vTrace( tU16                                  u16Fid,
              std::vector < spm_NotificationInfo >& ar );


tBool bRemoveAllEntriesWithRegId( tU16 u16RegId ){
   return( _oNotificationTable.bRemoveAllEntriesWithRegID( u16RegId ) );
}

};

