/*!
  * \file spm_CcaProperties.cpp
  *  \brief
  *    Framework to help working with CCA properties.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 03.11.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#include <map>
#include <vector>
#include <string>
#include <algorithm>


#include "spm_Config.h"
#include "spm_CcaProperties.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_CcaProperties.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_CCAPROPERTIES


spm_PropertyHandling::spm_PropertyHandling( tU16               u16FuncArraySize,
                                            tU16               u16ServiceId,
                                            const std::string& strSemname )
   : _oNotificationTable( u16FuncArraySize )
   , _hNotificationTableSemaphore(OSAL_C_INVALID_HANDLE)
   , _u16ServiceId( u16ServiceId ){
   _strSemname = strSemname;
   if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( strSemname.c_str( ),
		                                       &_hNotificationTableSemaphore,
                                               1 ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_PropertyHandling !!!!!! Error detected !!!!!! cannot create Semaphore _hNotificationTableSemaphore: error 0x%08X (%s)"
               , (tUInt)u32ErrorReason
               , OSAL_coszErrorText( u32ErrorReason ) ) );
   };
   _itMap      = _propertiesMap.end( );

}

spm_PropertyHandling::~spm_PropertyHandling( ){
   OSAL_s32SemaphoreClose( _hNotificationTableSemaphore );
   OSAL_s32SemaphoreDelete( _strSemname.c_str( ) );
}

tVoid spm_PropertyHandling::vUpdateProperty( ISpmCcaServer        *pServer,
                                             tU16                  u16Fid,
                                             const fi_tclTypeBase *pMsgBase ){
   const spm_AdminProperty *p = poFindProperty( u16Fid );

   if ( p == NULL ){
      return;      // property not found
   }
   ahl_bEnterCritical( _hNotificationTableSemaphore );
   TMapPropertyNotification::iterator it = _notificationsMap.find( u16Fid );
   if ( p->_pProperty->bUpdate( pMsgBase ) || ( it != _notificationsMap.end( ) ) ){

      tBool bAlreadySent = FALSE;

      for ( ahl_tNotification *pNot = _oNotificationTable.poGetNotificationList( u16Fid );
            pNot; pNot = pNot->pNext ){
         // send the status update to all registered components
         vStatusMessage( pServer, p->_pProperty, u16Fid, pNot );

         if ( it != _notificationsMap.end( ) ){
            if ( it->second.u16AppID == pNot->u16AppID ){
               bAlreadySent = TRUE;
            }
         }
      }

      // normally you are not obliged to make this comparison because
      // erase on end is legal!
      if ( it != _notificationsMap.end( ) ){
         if ( !bAlreadySent ){
            vStatusMessage( pServer, p->_pProperty, u16Fid, &it->second );
         }
         _notificationsMap.erase( it );
      }
   }
   ahl_bReleaseCritical( _hNotificationTableSemaphore );
   // for special purposes after all status messages are sent out this function is called
   // to update internally the property
   p->_pProperty->vAfterUpdate( );
}     // vUpdateProperty

tVoid spm_PropertyHandling::vStatusMessage( ISpmCcaServer           *pServer,
                                            IProperty               *p,
                                            tU16                     u16Fid,
                                            const ahl_tNotification *pNot ){
   // fi_tclVisitorMessage oMyStatusMessage(p->poGetContent()->corfoGetTypeBase());
   fi_tclVisitorMessage oMyStatusMessage( * p->poGetContent( ) );

   // Now add targetting data
   oMyStatusMessage.vInitServiceData( CCA_C_U16_APP_SPM,
                                      pNot->u16AppID,
                                      AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                      0,
                                      pNot->u16RegisterID,
                                      pNot->u16CmdCounter,
                                      _u16ServiceId,
                                      u16Fid,
                                      AMT_C_U8_CCAMSG_OPCODE_STATUS,
                                      pNot->u8Act,
                                      0,
                                      pNot->u16SubID
                                      );

   // post message via message handler object
   pServer->bPostMessage( &oMyStatusMessage );

   ETG_TRACE_USR4( ( "spm_PropertyHandling::vStatusMessage FID=%04x, ServiceId=%04x, SendToAppId=%u(%d/0x%x)",
                     ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                     _u16ServiceId,
                     ETG_ENUM( ail_u16AppId,   pNot->u16AppID ), pNot->u16AppID, pNot->u16AppID
                     ) );
} // vStatusMessage

tVoid spm_PropertyHandling::vErrorMessage( ISpmCcaServer           *pServer,
                                           tU16                     u16Fid,
                                           const ahl_tNotification *pNot,
                                           tU16               u16ErrorData ){
   amt_tclServiceDataError oErrorMsg;

   oErrorMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                               pNot->u16AppID,
                               AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                               0,
                               pNot->u16RegisterID,
                               pNot->u16CmdCounter,
                               _u16ServiceId,
                               u16Fid,
                               AMT_C_U8_CCAMSG_OPCODE_ERROR,
                               pNot->u8Act,
                               0,
                               pNot->u16SubID
                               );

   oErrorMsg.vSetErrorData( u16ErrorData );

   // post message via message handler object
   pServer->bPostMessage( &oErrorMsg );

   ETG_TRACE_FATAL( ( "spm_PropertyHandling::vErrorMessage FID=%04x, ServiceId=%04x, SendToAppId=%u(%d/0x%x), ErrorCode=%u",
                      ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                      _u16ServiceId,
                      ETG_ENUM( ail_u16AppId,   pNot->u16AppID ), pNot->u16AppID, pNot->u16AppID,
                      ETG_ENUM( ail_u8OpCodeErrorNumber, u16ErrorData )
                      ) );
} // vErrorMessage

tVoid spm_PropertyHandling::vHandleProperty( ISpmCcaServer      *pServer,
                                             amt_tclServiceData *poSrvData ){
   const spm_AdminProperty *p      = poFindProperty( poSrvData );

//ETG_TRACE_FATAL(("spm_PropertyHandling::vHandleProperty: Property request with FID=%04x (requested by AppId=%04x)", ETG_ENUM(SPM_CCA_FID_ID, u16Fid), ETG_ENUM(ail_u16AppId,
// tNotification.u16AppID) ));

   tU16                     u16Fid = poSrvData->u16GetFunctionID( );

   ahl_tNotification        tNotification = {0,0,0,0,0,0,0,0};

   tNotification.bOccupied     = FALSE;
   tNotification.u16AppID      = poSrvData->u16GetSourceAppID( );
   tNotification.u16RegisterID = poSrvData->u16GetRegisterID( );
   tNotification.u16CmdCounter = poSrvData->u16GetCmdCounter( );
   tNotification.u16SubID      = poSrvData->u16GetSourceSubID( );
   tNotification.u8Act	       = poSrvData->u8GetACT( );

   if ( p == NULL ){
      ETG_TRACE_FATAL( ( "spm_PropertyHandling::vHandleProperty: Property with FID=%04x not supported by SPM (requesting by AppId=%u(%d/0x%x))",
                         ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                         ETG_ENUM( ail_u16AppId,   tNotification.u16AppID ), tNotification.u16AppID, tNotification.u16AppID
                         ) );
      ETG_TRACE_ERRMEM( ( "spm_PropertyHandling::vHandleProperty: Property with FID=%04x not supported by SPM (requesting by AppId=%d(0x%x))",
                          ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                          ETG_ENUM( ail_u16AppId,   tNotification.u16AppID ), tNotification.u16AppID
                          ) );

      vErrorMessage( pServer, u16Fid, &tNotification, AMT_C_U16_ERROR_UNKNOWN_FCT_ID );
      return;
   }


   switch ( poSrvData->u8GetOpCode( ) ){
      case AMT_C_U8_CCAMSG_OPCODE_GET:
         if ( p->_u32Configuration & SPM_PROP_ADMIN_GET ){
            p->_pProperty->vGetOperation( poSrvData );
            vStatusMessage( pServer, p->_pProperty, u16Fid, &tNotification );
            // for special purposes after all status messages are sent out this function is called
            // to update internally the property
            p->_pProperty->vAfterUpdate( );
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_SET:
         if ( p->_u32Configuration & SPM_PROP_ADMIN_SET ){
            tU32 result = p->_pProperty->u8SetOperation( poSrvData );
            if ( result == IProperty::IPROPERTY_RETURN_SUCCESS ){
               p->_pProperty->vGetOperation( poSrvData );
               vStatusMessage( pServer, p->_pProperty, u16Fid, &tNotification );
               // for special purposes after all status messages are sent out this function is called
               // to update internally the property
               p->_pProperty->vAfterUpdate( );
            } else if ( result == IProperty::IPROPERTY_RETURN_FAILED ){
               vErrorMessage( pServer, u16Fid, &tNotification, AMT_C_U16_ERROR_INTERNAL_FAILURE );
            } else {
               // the delay case means that the result is sent later because this is a real
               // long lasting aynchronous property change call.
               // Therefore the notification has to be stored in a map so that it is available
               // later for completing the result message.
               _notificationsMap[u16Fid] = tNotification;
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_PURESET:
         if ( p->_u32Configuration & SPM_PROP_ADMIN_SET ){
            if ( p->_pProperty->u8SetOperation( poSrvData ) != IProperty::IPROPERTY_RETURN_SUCCESS ){
               vErrorMessage( pServer, u16Fid, &tNotification, AMT_C_U16_ERROR_INTERNAL_FAILURE );
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_UPREG:
         if ( p->_u32Configuration & SPM_PROP_ADMIN_UPREG ){
            tBool bRet;

            // add interested application to notification table
            ahl_bEnterCritical( _hNotificationTableSemaphore );
            bRet = _oNotificationTable.bAddNotification( u16Fid,
                                                         poSrvData->u16GetSourceAppID( ),
                                                         poSrvData->u16GetRegisterID( ),
                                                         1,
                                                         poSrvData->u16GetCmdCounter( ),
                                                         poSrvData->u16GetSourceSubID( ) );
            ahl_bReleaseCritical( _hNotificationTableSemaphore );

            if ( bRet ){
               // send status data
               vStatusMessage( pServer, p->_pProperty, u16Fid, &tNotification );
               // for special purposes after all status messages are sent out this function is called
               // to update internally the property
               p->_pProperty->vAfterUpdate( );
            } else {
               // send error
               spm_tclServiceErrorStatus oUpregErr( CCA_C_U16_APP_SPM,
                                                    poSrvData->u16GetSourceAppID( ),
                                                    poSrvData->u16GetRegisterID( ),
                                                    poSrvData->u16GetCmdCounter( ),
                                                    _u16ServiceId,
                                                    u16Fid,
                                                    poSrvData->u8GetACT( ),
                                                    0,
                                                    poSrvData->u16GetSourceSubID( ) );
               pServer->bPostMessage( &oUpregErr );
            }
         }
         break;

      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG:
         if ( p->_u32Configuration & SPM_PROP_ADMIN_UPREG ){
            // remove application from notification table
            ahl_bEnterCritical( _hNotificationTableSemaphore );
            _oNotificationTable.bRemoveNotification( u16Fid,
                                                     poSrvData->u16GetSourceAppID( ),
                                                     poSrvData->u16GetRegisterID( ),
                                                     1,
                                                     poSrvData->u16GetCmdCounter( ),
                                                     poSrvData->u16GetSourceSubID( ) );
            ahl_bReleaseCritical( _hNotificationTableSemaphore );

            // send status data
            vStatusMessage( pServer, p->_pProperty, u16Fid, &tNotification );
            // for special purposes after all status messages are sent out this function is called
            // to update internally the property
            p->_pProperty->vAfterUpdate( );
         }
         break;

      default:
         break;
   }     // switch
} // vHandleProperty

tBool spm_PropertyHandling::bIsAppNotifiedToFid( tU32 u32AppId,
                                                 tU16 u16Fid ){
   tBool              bReturn = FALSE;

   ahl_bEnterCritical( _hNotificationTableSemaphore );
   ahl_tNotification *pNot    = _oNotificationTable.poGetNotificationList( u16Fid );
   for (; pNot != NULL; pNot = pNot->pNext ){
      if ( u32AppId == pNot->u16AppID ){
         bReturn = TRUE;
         break;
      }
   }
   ahl_bReleaseCritical( _hNotificationTableSemaphore );
   return( bReturn );
}

tVoid spm_PropertyHandling::vTrace( tU16                                  u16Fid,
                                    std::vector < spm_NotificationInfo >& ar ){
   ahl_bEnterCritical( _hNotificationTableSemaphore );
   ahl_tNotification *pNot = _oNotificationTable.poGetNotificationList( u16Fid );
   for (; pNot != NULL; pNot = pNot->pNext ){
      spm_NotificationInfo item;

      item._u16AppID      = pNot->u16AppID;
      item._u16RegisterID = pNot->u16RegisterID;
      ar.push_back( item );
   }
   ahl_bReleaseCritical( _hNotificationTableSemaphore );
}

