/*!
  * \file spm_CcaMethods.cpp
  *  \brief
  *    Framework to help working with CCA methods.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 03.11.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"

#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#include <map>
#include <vector>
#include <algorithm>


#include "spm_Config.h"
#include "spm_CcaMethods.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_CcaMethods.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_CCAMETHODS

spm_MethodHandling::spm_MethodHandling( tU16 u16ServiceId )
   : _u16ServiceId( u16ServiceId ){
}

spm_MethodHandling::~spm_MethodHandling( ){
}

tVoid spm_MethodHandling::vUpdateMethodResult( ISpmCcaServer        *pServer,
                                               tU16                  u16Fid,
                                               const fi_tclTypeBase *pMsgBase,
                                               tU8                   opCode ){
   const spm_AdminMethod *m = poFindMethod( u16Fid );

   if ( m == NULL ){
      return;                                 // method not found
   }
   m->_pMethod->vSetMethodResult( pMsgBase ); // update the method result
   TMapMethodNotification::iterator it = _notifications.find( u16Fid );
   if ( it != _notifications.end( ) ){
      vResultMessage( pServer, m->_pMethod, u16Fid, it->second );
      if ( ( opCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
           || ( opCode == AMT_C_U8_CCAMSG_OPCODE_METHODRESULTLAST )
           || ( opCode == AMT_C_U8_CCAMSG_OPCODE_METHODABORT ) ){
         // erase the notification entry in the map if the method call is finished.
         // if an opcode is missing above then please complete (maybe ABORT)
         _notifications.erase( it );
      }
   }
} // vUpdateMethodResult

tVoid spm_MethodHandling::vResultMessage( ISpmCcaServer           *pServer,
                                          IMethod                 *m,
                                          tU16                     u16Fid,
                                          const spm_tNotification& notif,
                                          tU8                      opCode ){
   // fi_tclVisitorMessage oMyResultMessage(m->poGetResult()->corfoGetTypeBase());
   fi_tclVisitorMessage oMyResultMessage( * m->poGetResult( ) );

   // Now add targetting data
   oMyResultMessage.vInitServiceData( CCA_C_U16_APP_SPM,
                                      notif.u16AppID,
                                      AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                                      0,
                                      notif.u16RegisterID,
                                      notif.u16CmdCounter,
                                      _u16ServiceId,
                                      u16Fid,
                                      opCode,
                                      notif.u8Act,
                                      0,
                                      notif.u16SubID
                                      );

   // post message via message handler object
   pServer->bPostMessage( &oMyResultMessage );

   ETG_TRACE_USR4( ( "spm_MethodHandling::vResultMessage FID=%04x, ServiceId=%04x, SendToAppId=%u(%d/0x%x), Opcode=%d",
                     ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                     _u16ServiceId,
                     ETG_ENUM( ail_u16AppId,   notif.u16AppID ), notif.u16AppID, notif.u16AppID,
                     opCode
                     ) );
} // vResultMessage

tVoid spm_MethodHandling::vErrorMessage( ISpmCcaServer     *pServer,
                                         tU16               u16Fid,
                                         spm_tNotification *pNot,
                                         tU16               u16ErrorData ){
   amt_tclServiceDataError oErrorMsg;

   oErrorMsg.vInitServiceData( CCA_C_U16_APP_SPM,
                               pNot->u16AppID,
                               AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,
                               0,
                               pNot->u16RegisterID,
                               pNot->u16CmdCounter,
                               _u16ServiceId,
                               u16Fid,
                               AMT_C_U8_CCAMSG_OPCODE_ERROR,
                               pNot->u8Act,
                               0,
                               pNot->u16SubID
                               );

   oErrorMsg.vSetErrorData( u16ErrorData );

   // post message via message handler object
   pServer->bPostMessage( &oErrorMsg );

   ETG_TRACE_FATAL( ( "spm_MethodHandling::vErrorMessage FID=%04x, ServiceId=%04x, SendToAppId=%u(%d/0x%x), ErrorCode=%u",
                      ETG_ENUM( SPM_CCA_FID_ID,          u16Fid ),
                      _u16ServiceId,
                      ETG_ENUM( ail_u16AppId,            pNot->u16AppID ), pNot->u16AppID, pNot->u16AppID,
                      ETG_ENUM( ail_u8OpCodeErrorNumber, u16ErrorData )
                      ) );
} // vErrorMessage

tVoid spm_MethodHandling::vHandleMethod( ISpmCcaServer      *pServer,
                                         amt_tclServiceData *poSrvData ){
   const spm_AdminMethod *m = poFindMethod( poSrvData );

   spm_tNotification      tNotification;

   tNotification.u16AppID      = poSrvData->u16GetSourceAppID( );
   tNotification.u16RegisterID = poSrvData->u16GetRegisterID( );
   tNotification.u16CmdCounter = poSrvData->u16GetCmdCounter( );
   tNotification.u8Act         = poSrvData->u8GetACT( );
   tNotification.u16SubID      = poSrvData->u16GetSourceSubID( );

   tU16 u16Fid = poSrvData->u16GetFunctionID( );

   if ( m == NULL ){
      ETG_TRACE_FATAL( ( "spm_MethodHandling::vHandleMethod: Method with FID=%04x not supported by SPM (requesting by AppId=%u(%d/0x%x))",
                         ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                         ETG_ENUM( ail_u16AppId,   tNotification.u16AppID ), tNotification.u16AppID, tNotification.u16AppID
                         ) );


      ETG_TRACE_ERRMEM( ( "spm_MethodHandling::vHandleMethod: Method with FID=%04x not supported by SPM (requesting by AppId=%u(%d/0x%x))",
                          ETG_ENUM( SPM_CCA_FID_ID, u16Fid ),
                          ETG_ENUM( ail_u16AppId,   tNotification.u16AppID ), tNotification.u16AppID, tNotification.u16AppID
                          ) );

      vErrorMessage( pServer, u16Fid, &tNotification, AMT_C_U16_ERROR_UNKNOWN_FCT_ID );
      return;
   }

   switch ( poSrvData->u8GetOpCode( ) ){
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
         if ( m->_u32Configuration & SPM_METHOD_ADMIN_METHODRESULT ){
            tU32 result = m->_pMethod->u32MethodStart( poSrvData );
            if ( result == IMethod::IMETHOD_RETURN_SUCCESS ){
               vResultMessage( pServer, m->_pMethod, u16Fid, tNotification );
            } else if ( result == IMethod::IMETHOD_RETURN_FAILED ){
               vErrorMessage( pServer, u16Fid, &tNotification, AMT_C_U16_ERROR_INTERNAL_FAILURE );
            } else {
               // the delay case means that the result is sent later because this is a real
               // long lasting aynchronous method call.
               // Therefore the notification has to be stored in a map so that it is available
               // later for completing the result message.
               _notifications[u16Fid] = tNotification;
            }
         }
         break;

      default:
         break;
   }     // switch
} // vHandleMethod

