/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_CcaMsgHandler.h
  * @brief        This is header file contains functions for CcaMsgHandler.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <map>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_ICcaServer.h"

class ISpmSupervisionManager;
class ISpmLocalAppManager;
class ISpmApplicationDatabase;
class ISpmCcaSupplierServer;
class ISpmCcaServiceServer;

typedef enum
{
   MBX_ERROR,
   MBX_TIMEOUT,
   MBX_MSG_RECEIVED
} eMbxResult;


class amt_tclBaseStreamMessage : public amt_tclBaseMessage
{
public:
amt_tclBaseStreamMessage ( tU8 *pStreamData ){
   tU32 u32Size = * ( (tU32*)( pStreamData + AMT_C_U32_BASEMSG_SIZE ) ); //lint !e826 Suspicious pointer-to-pointer conversion (area too small)

   vSetDynMsgSize( 0 );
   vAddDynMsgSize( u32Size );

   // Allocate in Shared Memory
   bAllocateMessage( );

   vSetStreamU8( 0, pStreamData, u32Size, u32Size );
}

};

/*!
  *  \class spm_tclCcaMsgHandler
  *  \brief MessageHandler is the first component to be created by FC_SPM factory among (CCA Server , Supplier and Messagehandler).
         It then runs in a separate thread and the thread exclusively interacts with FC_SPM clients through AMT
        message framework in conjunction with CCA .

         Its job is to wait for any incoming messages from AMT framework.
          Once the messages are received, depending on the type of messages, they are dispatched to the respective
        sub-component within FC_SPM for handling


          It uses these classes spm_tclApplicationDatabase, spm_tclWorkerServerConfig, spm_tclLocalApplicationManagerConfig,
        spm_tclCcaServiceHandlerConfig, spm_tclCcaSupplierHandlerConfig, spm_tclSupervisionManagerConfig for message communication.

        The spm_tclFactory object creates an instance of this class.

        This class is used by spm_tclCcaSupplierHandler, spm_tclLocalApplicationManager for message communication .


  */
class spm_tclCcaMsgHandler : public ISpmCcaServer
{
public:
spm_tclCcaMsgHandler( const ISpmFactory& factory,
                      tU32               u32AppId );
virtual ~spm_tclCcaMsgHandler( );

tVoid vTerminate( );

// implementation of interface ISpmBase
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

// implementation of ICcaServer
virtual tBool bPostMessage( amt_tclBaseMessage *poMessage,
                            tBool               bTrace = TRUE );

virtual tVoid vApplicationEndSuccessful( tU16 u16AppID );

virtual tBool bPostPowerMessage( tU16 u16TargetID,
                                 tU16 u16PowerType,
                                 tU32 u32PowerData1,
                                 tU32 u32PowerData2 );

// start the cca input thread
virtual tVoid vStart( const std::string& strName,
                      tU32               u32Prio,
                      tU32               u32Stack,
                      tU32               u32MaxMsgQueueLength );

public:
   // implementation of spm_tclActive
   virtual tVoid vOnStart( );

   virtual tVoid vOnTerminate( );

   virtual tVoid main( );

private:
   eMbxResult eMessageQueueWait( OSAL_tMQueueHandle  hMQ,
                                 amt_tclBaseMessage *poMsgObject,
                                 tU32               *pu32Prio,
                                 OSAL_tMSecond       rTimeout ) const;

   OSAL_tMQueueHandle _hSpmInQueue;

public:
   virtual tBool bCheckForCcaInContext( tVoid ){
      return ( hGetThreadId( ) == OSAL_ThreadWhoAmI( ) );
   }

private:
tVoid vHandlePowerMessage( const amt_tclPowerMessage *pPowerMsg );

tU16                                  _u16AppId;
tU32                                  _u32SuperVisionTimeOut;

OSAL_tSemHandle                       _hSemHandle;
std::map < tU16, OSAL_tMQueueHandle > _tMsgQueueContainer;

ISpmSupervisionManager               *_poclSupervisionManager;
ISpmLocalAppManager                  *_poclLocalAppManager;
ISpmWorkerServer                     *_poclWorkerServer;
ISpmCcaSupplierServer                *_poclCcaSupplierHandler;
ISpmCcaServiceServer                 *_poclCcaServiceHandler;

tVoid vOnNewCcaMessage( amt_tclBaseMessage *pBaseMsg );

protected:
// Implementation of interface ISpmWorkerClient
virtual tVoid vHandleMessage( tU32 /*u32Message*/,
                              tU32 /*u32Parm = 0*/ ){
}  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual tBool bHandleSynchrounousCall( tU32   /*u32Message*/,
                                       tVoid */*args*/ ){
   return( FALSE );
}  //lint !e715 Symbol 'xxx' not referenced --> CURRENTLY not used

virtual const tChar*getName( ) const { return( "spm_tclCcaMsgHandler" ); }
virtual tVoid vGetReferences( );

virtual tVoid vStartCommunication( );

// all system apps

};

