/*!
  * \file spm_CcaMsgHandler.cpp
  *  \brief
  *    Create the component SPM and the startup process
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 03.04.12  | CM-AI/VW32 kollai | Adaptation for NISSAN LCN2 KAI
  ******
  */

#include <sstream>
#include <iomanip>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// SPM  configuration
#include "spm_Config.h"

// my class header
#include "spm_CcaMsgHandler.h"

// spm class definitions

// interfaces class definitions
#include "spm_ISuperVisionManager.h"
#include "spm_IApplicationDatabase.h"
#include "spm_ICcaSupplierServer.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ILocalAppManager.h"
#include "spm_IApplicationErrorHandler.h"

#include "spm_IFactory.h"
#include "spm_CriticalSection.h"
#include "spm_GlobDefsExt.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_CcaMsgHandler.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define SPM_CCA_MSG_QUEUE_NAME_PREFIX          "mbx_"
#define SPM_CCA_MSG_QUEUE_NAME_POSTFIX_FORMAT  "%s%u"  // %u is filled with tU16 = 5 decimal places
#define SPM_CCA_MSG_QUEUE_NAME_LENGTH          ((tU8)(sizeof(SPM_CCA_MSG_QUEUE_NAME_PREFIX)+5))

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CCAMSGHANDLER

spm_tclCcaMsgHandler::spm_tclCcaMsgHandler( const ISpmFactory& factory,
                                            tU32               u32AppId )
   : ISpmCcaServer( factory )
   , _hSpmInQueue( OSAL_C_INVALID_HANDLE )
   , _u16AppId( (tU16)u32AppId )
   , _u32SuperVisionTimeOut( OSAL_C_U32_INFINITE )
   , _hSemHandle(OSAL_C_INVALID_HANDLE)
   , _poclSupervisionManager( NULL )
   , _poclLocalAppManager( NULL )
   , _poclWorkerServer( NULL )
   , _poclCcaSupplierHandler( NULL )
   , _poclCcaServiceHandler( NULL ){
   _tMsgQueueContainer.clear( );

   tChar strName[SPM_CCA_MSG_QUEUE_NAME_LENGTH];

   dp_tclSpmDpConfigCcaMsgWaitTimeout oCcaHBeatTimeout;
   oCcaHBeatTimeout >> _u32SuperVisionTimeOut;

   ETG_TRACE_USR4( ( "spm_tclCcaMsgHandler::vStart(): Wait on MSQ queue with timeout of %dms.", _u32SuperVisionTimeOut ) );

   if (OSALUTIL_s32SaveNPrintFormat(
      strName,
      sizeof(strName),
      SPM_CCA_MSG_QUEUE_NAME_POSTFIX_FORMAT,
      SPM_CCA_MSG_QUEUE_NAME_PREFIX,
      _u16AppId) == OSAL_ERROR) {
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler constructor !!!!!! Error detected !!!!!! cannot create MessageQueue name for _hSpmInQueue: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
     return;
   }

   // create msgbox
   if ( OSAL_ERROR == OSAL_s32MessageQueueCreate(
           strName,
           SPM_U32_CCA_MBX0_MSG_COUNT,
           SCD_MAILBOX_MAX_MESSAGE_LENGTH,
           OSAL_EN_READWRITE,
           &_hSpmInQueue
           )
        ){
      //try to open existing one
      if ( OSAL_ERROR == OSAL_s32MessageQueueOpen( strName, OSAL_EN_READWRITE, &_hSpmInQueue ) ){
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler !!!!!! Error detected !!!!!! cannot open MessageQueue _hSpmInQueue: error 0x%08X (%s)",
                                (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
      }
   }
   _tMsgQueueContainer[0] = _hSpmInQueue;       // add my own queue
   if ( OSAL_ERROR == OSAL_s32SemaphoreCreate( "spmCcaMhdl", &_hSemHandle, (tU32)1 ) ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler !!!!!! Error detected !!!!!! cannot create Semaphore spmCcaMhdl: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclCcaMsgHandler::~spm_tclCcaMsgHandler( ){
   tChar strName[SPM_CCA_MSG_QUEUE_NAME_LENGTH];

   if (OSALUTIL_s32SaveNPrintFormat(
      strName,
      sizeof(strName),
      SPM_CCA_MSG_QUEUE_NAME_POSTFIX_FORMAT,
      SPM_CCA_MSG_QUEUE_NAME_PREFIX,
      _u16AppId) == OSAL_ERROR) {
         tU32 u32ErrorReason = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "SPM: spm_tclCcaMsgHandler destructor !!!!!! Error detected !!!!!! cannot create MessageQueue name for _hSpmInQueue: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
     return;
   }

   OSAL_s32MessageQueueClose( _hSpmInQueue );
   OSAL_s32MessageQueueDelete( strName );

   _poclWorkerServer       = NULL;
   _poclLocalAppManager    = NULL;
   _poclCcaSupplierHandler = NULL;
   _poclCcaServiceHandler  = NULL;
   _poclSupervisionManager = NULL;

   OSAL_s32SemaphoreClose( _hSemHandle );
   OSAL_s32SemaphoreDelete( "spmCcaMhdl" );
}

tVoid spm_tclCcaMsgHandler::vGetReferences( ){
   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclWorkerServer,       ISpmWorkerServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclLocalAppManager,    ISpmLocalAppManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler,  ISpmCcaServiceServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaSupplierHandler, ISpmCcaSupplierServer );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSupervisionManager, ISpmSupervisionManager );
} // vGetReferences

tVoid spm_tclCcaMsgHandler::vStartCommunication( ){
}

tVoid spm_tclCcaMsgHandler::vStart( const std::string& strName,
                                    tU32               u32Prio,
                                    tU32               u32Stack,
                                    tU32               u32MaxMsgQueueLength ){

   (tVoid)strName;
   (tVoid)u32Prio;
   (tVoid)u32Stack;
   (tVoid)u32MaxMsgQueueLength;
   //spm_tclActive::vStartThread(strName, u32Prio, u32Stack);
} // vStart


tVoid spm_tclCcaMsgHandler::vOnStart( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vAddClient( this );
}

tVoid spm_tclCcaMsgHandler::vOnTerminate( ){
   SPM_NULL_POINTER_CHECK( _poclWorkerServer );
   _poclWorkerServer->vRemoveClient( this );
}

tVoid spm_tclCcaMsgHandler::main( ){
   amt_tclBaseMessage oMsgObject;
   tU32               u32Prio;
   eMbxResult         eResult;

   eResult = eMessageQueueWait( _hSpmInQueue,
                                &oMsgObject,
                                &u32Prio,
                                _u32SuperVisionTimeOut );

   // Thread shall be stopped?
   if ( bIsTerminated( ) ){
      return;
   }
   switch ( eResult ){
      case MBX_MSG_RECEIVED:
         // ETG_TRACE_USR4(("spm_tclCcaMsgHandler::main() -> eMessageQueueWait() returned with MSG_RECEIVED"));
         if ( _poclSupervisionManager ){
            _poclSupervisionManager->vTriggerCCAHeartBeat( );
         }

         vOnNewCcaMessage( &oMsgObject );
         // release memory of message object
         oMsgObject.bDelete( );
         break;

      case MBX_TIMEOUT:
         if ( _poclSupervisionManager ){
            _poclSupervisionManager->vTriggerCCAHeartBeat( );
            // ETG_TRACE_USR4(("spm_tclCcaMsgHandler::main() -> eMessageQueueWait() returned with TIMEOUT"));
         }
         break;

      case MBX_ERROR:
         ETG_TRACE_USR4( ( "spm_tclCcaMsgHandler::main() -> eMessageQueueWait() returned with ERROR" ) );
         break;

      default:
         ETG_TRACE_ERR_CLS( ( SPM_TRACE_CLASS_SPM, "vInputThread, message receive error" ) );
   } // switch

}    // main

eMbxResult spm_tclCcaMsgHandler::eMessageQueueWait( OSAL_tMQueueHandle  hMQ,
                                                    amt_tclBaseMessage *poMsgObject,
                                                    tU32               *pu32Prio,
                                                    OSAL_tMSecond       rTimeout )  const {
   eMbxResult eRetVal = MBX_ERROR;
   tS32       s32RetVal;

   // wait for a message
   tU32       u32MsgMax, u32MaxLen, u32MsgCount = 0;

   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSpmInQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   tU32       u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
   if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
      std::string strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue _hSpmInQueue (" + std::to_string(u32FillLevel) + "%) exceeded maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN)  + "%";
      ETG_TRACE_ERRMEM( ( "%s", strError.c_str() ) );
   }

   // writes the received OSALMsgHandle in the message object
   s32RetVal = OSAL_s32MessageQueueWait
               (
      hMQ,
      (tU8*)poMsgObject->prGetOSALMsgHandle( ), /*pointer of the MsgHandle field*/
      sizeof( OSAL_trMessage ),
      pu32Prio,
      rTimeout
               );

   if ( s32RetVal != OSAL_ERROR ){
      // gets the pointer to shared memory
      if ( sizeof( OSAL_trMessage ) == s32RetVal ){
         poMsgObject->pu8SharedMemBase = OSAL_pu8MessageContentGet
                                         (
            * ( poMsgObject->prGetOSALMsgHandle( ) ),
            OSAL_EN_READWRITE
                                         );

         eRetVal = ( ( ( poMsgObject->pu8SharedMemBase ) == OSAL_NULL ) ? MBX_ERROR : MBX_MSG_RECEIVED );
      } else {
         if ( OSAL_u32ErrorCode( ) != OSAL_E_TIMEOUT ){
            eRetVal = MBX_ERROR;
         } else {
            eRetVal = MBX_TIMEOUT;
         }
      }
   } else {
      ETG_TRACE_ERR_CLS( ( SPM_TRACE_CLASS_SPM, "bMessageQueueWait, receive error: %s", OSAL_coszErrorText( OSAL_u32ErrorCode( ) ) ) );
   }
   return( eRetVal );
} // eMessageQueueWait


void spm_tclCcaMsgHandler::vOnNewCcaMessage( amt_tclBaseMessage *pBaseMsg ){
   // get message type and dispatch
   switch ( pBaseMsg->u8GetType( ) ){
      case AMT_C_U8_CCAMSGTYPE_POWER:
      {
         amt_tclPowerMessage oPowerMsg( pBaseMsg );
         vHandlePowerMessage( &oPowerMsg );
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SVCREGISTER:
      {
         amt_tclServiceRegister oSrvReg( pBaseMsg );

         if ( _poclCcaServiceHandler ){
            _poclCcaServiceHandler->bHandleServiceRegister( &oSrvReg );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SVCUNREGISTER:
      {
         amt_tclServiceUnregister oSrvUnreg( pBaseMsg );

         if ( _poclCcaServiceHandler ){
            _poclCcaServiceHandler->bHandleServiceUnregister( &oSrvUnreg );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SVCREGCONF:
      {
         amt_tclServiceRegisterConf oSrvConf( pBaseMsg );

         if ( _poclCcaServiceHandler ){
            _poclCcaServiceHandler->bHandleServiceConfirmation( &oSrvConf );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SVSSTATUS:
      {
         amt_tclServiceStatus oSrvStatus( pBaseMsg );

         if ( _poclCcaServiceHandler ){
            _poclCcaServiceHandler->bHandleServiceStatus( &oSrvStatus );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_UNKNOWNREGID:
      {
         // \todo: UnkownRegisterID is no longer available in current AMT ++++++++++++++++++++++

         /*
            amt_tclUnknownRegisterID oUnkownRegID(pBaseMsg);
            bSuccess = bHandleUnkownRegID(&oUnkownRegID);
           ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SVCDATA:
      {
         amt_tclServiceData oSrvData( pBaseMsg );

         if ( _poclCcaServiceHandler ){
            _poclCcaServiceHandler->vOnServiceDataMessage( &oSrvData );
         }
         break;
      }

      // amt_tclServiceDataError
      case AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE_REGISTER:
      {
         amt_tclServiceSupplierRegister oSrvSupplierReg( pBaseMsg );

         if ( _poclCcaSupplierHandler ){
            _poclCcaSupplierHandler->bChangeServiceNotification( oSrvSupplierReg );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE:
      {
         amt_tclServiceSupplierStatus oSrvSupplierStatus( pBaseMsg );
         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM,
                               "Received supplier status message: AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE %04x, %u(%d/0x%x), %d",
                               oSrvSupplierStatus.u16GetServiceID( ),
                               ETG_ENUM( ail_u16AppId, oSrvSupplierStatus.u16GetAppID( ) ), oSrvSupplierStatus.u16GetAppID( ), oSrvSupplierStatus.u16GetAppID( ),
                               oSrvSupplierStatus.u8GetServiceSupplierState( )
                               ) );
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE_UNREGISTER:
      {
         amt_tclServiceSupplierUnregister oSrvSupplierReg( pBaseMsg );

         if ( _poclCcaSupplierHandler ){
            _poclCcaSupplierHandler->bChangeServiceNotification( oSrvSupplierReg, FALSE );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO_REGISTER:
      {
         amt_tclApplicationInfoRegister oAppInfoReg( pBaseMsg );
         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM,
                               "Received application info register message: AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO_REGISTER %u(%d/0x%x), %u(%d/0x%x)",
                               ETG_ENUM( ail_u16AppId, oAppInfoReg.u16GetAppID( ) ), oAppInfoReg.u16GetAppID( ), oAppInfoReg.u16GetAppID( ),
                               ETG_ENUM( ail_u16AppId, oAppInfoReg.u16GetSourceAppID( ) ), oAppInfoReg.u16GetSourceAppID( ), oAppInfoReg.u16GetSourceAppID( )
                               ) );

         if ( _poclCcaSupplierHandler ){
            _poclCcaSupplierHandler->bAddNotificationEntry( oAppInfoReg.u16GetAppID( ),
                                                            oAppInfoReg.u16GetSourceAppID( ),
                                                            AMT_C_U16_SERVICE_UNDEF,
                                                            oAppInfoReg.u16GetSourceSubID( ),
                                                            oAppInfoReg.u16GetTargetSubID( ) );
         }
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO:
      {
         amt_tclApplicationInfoStatus oAppInfoStatus( pBaseMsg );
         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM,
                               "Received application status message: AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO %u(%d/0x%x),%d ",
                               ETG_ENUM( ail_u16AppId, oAppInfoStatus.u16GetAppID( ) ), oAppInfoStatus.u16GetAppID( ), oAppInfoStatus.u16GetAppID( ),
                               oAppInfoStatus.u8GetApplicationInfoState( )
                               ) );
         break;
      }

      case AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO_UNREGISTER:
      {
         amt_tclApplicationInfoUnregister oAppInfoUnReg( pBaseMsg );
         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM,
                               "Received application info unregister message: AMT_C_U8_CCAMSGTYPE_APPLICATION_INFO_REGISTER %u(%d/0x%x), %u(%d/0x%x)",
                               ETG_ENUM( ail_u16AppId, oAppInfoUnReg.u16GetAppID( ) ), oAppInfoUnReg.u16GetAppID( ), oAppInfoUnReg.u16GetAppID( ),
                               ETG_ENUM( ail_u16AppId, oAppInfoUnReg.u16GetSourceAppID( ) ), oAppInfoUnReg.u16GetSourceAppID( ), oAppInfoUnReg.u16GetSourceAppID( )
                               ) );

         if ( _poclCcaSupplierHandler ){
            _poclCcaSupplierHandler->bRemoveNotificationEntry( oAppInfoUnReg.u16GetAppID( ), oAppInfoUnReg.u16GetSourceAppID( ) );
         }
         break;
      }

      default:
      {
         ETG_TRACE_ERR_CLS( ( SPM_TRACE_CLASS_SPM, "vOnNewCcaMessage, unknown message received" ) );
         break;
      }
   } // switch
}    // vOnNewCcaMessage

tVoid spm_tclCcaMsgHandler::vHandlePowerMessage( const amt_tclPowerMessage *pPowerMsg ){
   SPM_NULL_POINTER_CHECK( _poclLocalAppManager );

   switch ( pPowerMsg->u16GetPowerType( ) ){
      case AMT_C_U16_PWR_APP_START_REQ:
         // bSuccess = _poclLocalAppManager->bHandleAppStartReq( pPowerMsg);
         _poclLocalAppManager->bHandleAppStartReq( pPowerMsg->u16GetSourceAppID( ) );
         break;

      case AMT_C_U16_PWR_APP_INITIALIZED:
         // bSuccess = _poclLocalAppManager->bHandleAppInitialized( pPowerMsg);
         _poclLocalAppManager->bHandleAppInitialized( pPowerMsg->u16GetSourceAppID( ) );
         break;

      case AMT_C_U16_PWR_APP_RESTART:
         // bHandleAppRestart( pPowerMsg);
         break;

      case AMT_C_U16_PWR_APP_STATE_ACK:
         // bSuccess = _poclLocalAppManager->bHandleAppStateChangeAck( pPowerMsg);
         _poclLocalAppManager->bHandleAppStateChangeAck( pPowerMsg->u16GetSourceAppID( ), /*ChangedState*/ pPowerMsg->u32GetPowerData1( ), /*time extension*/ pPowerMsg->u32GetPowerData2( ) );
         break;

      case AMT_C_U16_PWR_APP_END_SUCCESSFUL:
         // bSuccess = _poclLocalAppManager->bHandleAppEndSuccess( pPowerMsg);
         _poclLocalAppManager->bHandleAppEndSuccess( pPowerMsg->u16GetSourceAppID( ) );
         break;

         #ifdef AMT_C_U16_PWR_APP_CCA_MSG_QUEUE_FULL
            case AMT_C_U16_PWR_APP_CCA_MSG_QUEUE_FULL:
               _poclLocalAppManager->bHandleAppQueueFull( pPowerMsg->u16GetSourceAppID( ), pPowerMsg->u32GetPowerData1( ) );
               break;
         #endif

      case AMT_C_U16_PWR_APP_ERROR_DETECT:
      case AMT_C_U16_PWR_APP_END_FAILURE:
         _poclLocalAppManager->bHandleAppEndFailure( pPowerMsg->u16GetSourceAppID( ), pPowerMsg->u32GetPowerData1( ) );
         break;

      // react on watchdog power messages
      case AMT_C_U16_PWR_SVM_APP_REGISTER:
         _poclLocalAppManager->bHandleSvmAppRegister( pPowerMsg->u16GetSourceAppID( ), /*u32RegRecFlags*/ pPowerMsg->u32GetPowerData1( ), /*u32NotifyInterval*/ pPowerMsg->u32GetPowerData2( ) );
         break;

      case AMT_C_U16_PWR_SVM_APP_UNREGISTER:
         _poclLocalAppManager->bHandleSvmAppUnRegister( pPowerMsg->u16GetSourceAppID( ) );
         break;

      case AMT_C_U16_PWR_SVM_APP_NOTIFY:
         _poclLocalAppManager->bHandleSvmAppNotify( pPowerMsg->u16GetSourceAppID( ) );
         break;

      case AMT_C_U16_PWR_SVM_APP_NOTIFY_PROBLEM_RESPONSE:
         _poclLocalAppManager->bHandleSvmAppNotifyProblemResponse( pPowerMsg->u16GetSourceAppID( ), /*AppState*/ pPowerMsg->u32GetPowerData1( ) );
         break;

      default:
         ETG_TRACE_ERR_CLS( ( SPM_TRACE_CLASS_SPM, "bHandlePowerMessage, unknown power message received" ) );
         break;
   } // switch
}    // vHandlePowerMessage

tBool spm_tclCcaMsgHandler::bPostMessage( amt_tclBaseMessage *poMessage,
                                          tBool               bTrace ){
   tBool              bSuccess     = TRUE;
   tS32               s32Retval    = OSAL_ERROR;
   OSAL_tMQueueHandle hAppMsgQueue = OSAL_C_INVALID_HANDLE;
   tU16               u16TargetID;

   u16TargetID = poMessage->u16GetTargetAppID( );

   if ( bTrace ){
      if ( AMT_C_U8_CCAMSGTYPE_SVCDATA == poMessage->u8GetType( ) ){
         amt_tclServiceData *poSrvMsg = dynamic_cast < amt_tclServiceData* >( poMessage );
         SPM_NULL_POINTER_CHECK_VAL( poSrvMsg );

         // trace message
         if ( poSrvMsg->u16GetTargetAppID( ) == CCA_C_U16_APP_SPM ){
            ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM_CSH,
                                  "bPostMessage, send service data message with FID: %u(%x) to app %u(%x)",
                                  ETG_ENUM( SPM_CCA_FID_ID, poSrvMsg->u16GetFunctionID( ) ),
                                  ETG_ENUM( SPM_CCA_FID_ID, poSrvMsg->u16GetFunctionID( ) ),
                                  ETG_ENUM( ail_u16AppId,   poSrvMsg->u16GetTargetAppID( ) ),
                                  ETG_ENUM( ail_u16AppId,   poSrvMsg->u16GetTargetAppID( ) )
                                  ) );
         } else if ( poSrvMsg->u16GetTargetAppID( ) == CCA_C_U16_APP_SPM_SLV ){
                                  ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM_CSH,
                                  "bPostMessage, send service data message with FID: %u(%x) to app %u(%x)",
                                  ETG_ENUM( LCM_DBUSCTRL_CCA_FID_ID, poSrvMsg->u16GetFunctionID( ) ),
                                  ETG_ENUM( LCM_DBUSCTRL_CCA_FID_ID, poSrvMsg->u16GetFunctionID( ) ),
                                  ETG_ENUM( ail_u16AppId,            poSrvMsg->u16GetTargetAppID( ) ),
                                  ETG_ENUM( ail_u16AppId,            poSrvMsg->u16GetTargetAppID( ) )
                                  ) );
         } else {
                                  ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM_CSH,
                                  "bPostMessage, send service data message with FID: %u(%x) to app %u(%x)",
                                  poSrvMsg->u16GetFunctionID( ),
                                  poSrvMsg->u16GetFunctionID( ),
                                  ETG_ENUM( ail_u16AppId, poSrvMsg->u16GetTargetAppID( ) ),
                                  ETG_ENUM( ail_u16AppId, poSrvMsg->u16GetTargetAppID( ) )
                                  ) );
         }

      } else if ( AMT_C_U8_CCAMSGTYPE_SUPPLIER_STATE == poMessage->u8GetType( ) ){
         amt_tclServiceSupplierStatus *poSupplierStateMsg = dynamic_cast < amt_tclServiceSupplierStatus* >( poMessage );
         SPM_NULL_POINTER_CHECK_VAL( poSupplierStateMsg );

         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM_CSH, "bPostMessage, send supplier state message: AppID: %u(%d/0x%x) ServiceID: 0x%04x ServiceState: %d",
                               ETG_ENUM( ail_u16AppId, poSupplierStateMsg->u16GetTargetAppID( ) ), poSupplierStateMsg->u16GetTargetAppID( ), poSupplierStateMsg->u16GetTargetAppID( ),
                               poSupplierStateMsg->u16GetServiceID( ),
                               poSupplierStateMsg->u8GetServiceSupplierState( )
                               ) );
      } else if ( AMT_C_U8_CCAMSGTYPE_SVCREGISTER == poMessage->u8GetType( ) ){
         amt_tclServiceRegister *poRegMsg = dynamic_cast < amt_tclServiceRegister* >( poMessage );
         SPM_NULL_POINTER_CHECK_VAL( poRegMsg );

         ETG_TRACE_USR4_CLS( ( SPM_TRACE_CLASS_SPM_CSH, "bPostMessage, send service register message: AppID: %u(%d/0x%x) ServiceID: 0x%04x",
                               ETG_ENUM( ail_u16AppId, poRegMsg->u16GetTargetAppID( ) ), poRegMsg->u16GetTargetAppID( ), poRegMsg->u16GetTargetAppID( ),
                               poRegMsg->u16GetServiceID( )
                               ) );
      }
   }

   /* get queue */
   spm_tclHandleSemaphore                          hdl( _hSemHandle );
   std::map < tU16, OSAL_tMQueueHandle >::iterator pPosQueue = _tMsgQueueContainer.find( u16TargetID );
   if ( pPosQueue != _tMsgQueueContainer.end( ) ){
      hAppMsgQueue = pPosQueue->second;
   } else {
      hAppMsgQueue = scd_OpenQueue( u16TargetID );
      if ( hAppMsgQueue != OSAL_C_INVALID_HANDLE ){
         _tMsgQueueContainer[u16TargetID] = hAppMsgQueue;
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclCcaMsgHandler::bPostMessage() => Open message queue for app=%d(0x%x) failed!", u16TargetID, u16TargetID ) );
      }
   }

   /* send message */
   if ( hAppMsgQueue != OSAL_C_INVALID_HANDLE ){
      s32Retval = OSAL_s32MessageQueuePost( hAppMsgQueue,
                                            (tCU8*)poMessage->prGetOSALMsgHandle( ),
                                            sizeof( OSAL_trMessage ),
                                            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST );
      if ( s32Retval == OSAL_OK ){
         tU32 u32MsgMax, u32MaxLen, u32MsgCount = 0;
         if ( OSAL_OK != OSAL_s32MessageQueueStatus( hAppMsgQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
            ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
         }
         tU32 u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
         if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
            std::stringstream stream ;
            stream << "0x" << std::setw(4) << std::setfill('0') << std::hex << u16TargetID ;
            std::string strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue for app " + stream.str() + " = " + std::to_string(u32FillLevel) + "% which exceeds maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN) + "%";
            ETG_TRACE_ERRMEM( ( "%s", strError.c_str() ) );
         }

         bSuccess = TRUE;
      } else {
         tU32 u32OsalErrorCode = OSAL_u32ErrorCode( );
         ETG_TRACE_ERRMEM( ( "spm_tclCcaMsgHandler::bPostMessage() to app=%d(0x%x) failed! MsgQueueHandle=0x%X, OSAL ErrorCode=%d (ErrText=%s)", u16TargetID, u16TargetID, (tUInt)hAppMsgQueue, (tUInt)u32OsalErrorCode, OSAL_coszErrorText( u32OsalErrorCode ) ) );

         if ( u32OsalErrorCode == OSAL_E_QUEUEFULL ){
            // get status info and reset
            SPM_GET_IF_REFERENCE_NEW_VAR_VAL( _poclAppErrHandler, ISpmApplicationErrorHandler );
            _poclAppErrHandler->vMsgQueueFull( (tU32)u16TargetID, false, true );
         }

         bSuccess = FALSE;
         poMessage->bDelete( );
      }
   } else {
      ETG_TRACE_ERR_CLS( ( SPM_TRACE_CLASS_SPM_LAM, "bPostMessage, invalid msg-queue" ) );
      poMessage->bDelete( );
      bSuccess = FALSE;
   }

   return( bSuccess );
} // bPostMessage

tVoid spm_tclCcaMsgHandler::vApplicationEndSuccessful( tU16 u16AppID ){
   std::map < tU16, OSAL_tMQueueHandle >::iterator pPosQueue = _tMsgQueueContainer.find( u16AppID );

   if ( pPosQueue != _tMsgQueueContainer.end( ) ){
      scd_s32CloseQueue( pPosQueue->second );
      _tMsgQueueContainer.erase( pPosQueue );
   }
} // vApplicationEndSuccessful

tBool spm_tclCcaMsgHandler::bPostPowerMessage( tU16 u16TargetID,
                                               tU16 u16PowerType,
                                               tU32 u32PowerData1,
                                               tU32 u32PowerData2 ){
   tBool               bSuccess = FALSE;

   /* create and send message */
   amt_tclPowerMessage oAnswerPowerMsg( _u16AppId, u16TargetID,
                                        u16PowerType, u32PowerData1,
                                        u32PowerData2 );

   bSuccess = bPostMessage( &oAnswerPowerMsg );

   return( bSuccess );
}

tVoid spm_tclCcaMsgHandler::vTerminate( ){
   // set termination of thread
   vSetTerminate( );

   tU8 terminateMsg = 0;
   // wakeup thread
   if ( OSAL_ERROR == OSAL_s32MessageQueuePost( _hSpmInQueue,
                                                (tCU8*)&terminateMsg,
                                                sizeof( tU8 ),
                                                OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST ) ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   tU32 u32MsgMax, u32MaxLen, u32MsgCount = 0;
   if ( OSAL_OK != OSAL_s32MessageQueueStatus( _hSpmInQueue, &u32MsgMax, &u32MaxLen, &u32MsgCount ) ){
      ETG_TRACE_ERR( ( "SPM: !!!!!! Error detected !!!!!!" ) );
   }
   tU32 u32FillLevel = ( u32MsgCount * 100 ) / u32MsgMax;
   if ( u32FillLevel > SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN ){
      std::string strError = "SPM: !!!!!! Error detected !!!!!! Fill level of queue _hSpmInQueue (" + std::to_string(u32FillLevel) + "%) exceeded maximum of " + std::to_string(SPM_WARNING_QUEUE_FILL_LEVEL_PERCENTAGE_WARN)+ "%";
      ETG_TRACE_ERRMEM( ( "%s", strError.c_str()) );
   }
} // vTerminate

