/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_factory.h
  * @brief        This is header file contains functions for spm_factory.
  * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include <string>
#include <map>
#include <list>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

#include "spm_IFactory.h"

#define SPM_EV_SHUTDOWN      0x00000001
#define SPM_EV_SHUTDOWN_NAME "SpmEvEnd"

/******************************************************************************
 | local #define (scope: module-local)
 |*****************************************************************************/
#define SPM_THREAD_STARTUP_TENGINE      "SpmSt1"
#define SPM_THREAD_STARTUP_LINUX        "SpmSt2"
#define SPM_THREAD_STARTUP_LINUX2       "SpmSt3"

#define SPM_THREAD_WORKER_SERVER        "SpmWork"
#define SPM_THREAD_OSAL_PROXY           "SpmPrx"

#define SPM_SEMAPHORE_STARTUP_INVEST    "SpmStInv"

// forward declaration
class ISpmApplicationConfiguration;
class ISpmApplicationDatabase;
class ISpmApplicationRequestSupervisor;
class ISpmCcaServer;
class ISpmDownloadLockListManager;
class ISpmCcaServiceServer;
class ISpmCcaSupplierServer;
class ISpmCvmClient;
class ISpmGlobalApplicationManager;
class ISpmGlobalApplicationManager;
class ISpmLamAppManager;
class ISpmLocalAppManager;
class ISpmApplicationErrorHandler;
class ISpmOsalCvm;
class ISpmOsalProxy;
class ISpmOsLinux;
class ISpmProcessDatabase;
class ISpmProcessSupervision;
class ISpmStartupCommon;
class ISpmStartupInvestigationServer;
class ISpmStartupSystem;
class ISpmWorkerClient;
class ISpmStartupSystemVariant;
class ISpmStartupSystemGroupServer;
class ISpmStatistics;
class ISpmPowerSupplyStatistics;
class ISpmSystemLoadStatistics;
class ISpmSoftwareVersionStatistics;
class ISpmSubStateClient;
class ISpmSupervisionClient;
class ISpmSupervisionManager;
class ISpmSupervisionEnableSupervisor;
class ISpmSystemPowerManager;
class ISpmSystemStateManager;
class ISpmThermalMgntIncClientHandler;
class ISpmSupplyMgntIncClientHandler;
class ISpmSystemStateMachineIncClientHandler;
class ISpmVersionInfo;
class ISpmWakeupHandler;
class ISpmTimerHandler;
class ISpmWatchdog;
class ISpmWorkerServer;
class ISpmSyncHandler;
class ISpmWupOnOffEvents;
class ISpmWupSupervisionErrorWarning;
class ISpmStartupSupervisor;
class spm_tclActive;
class ISpmTrace;
class ISpmRegistry;
class ISpmClientHandlerBase;

// //////////////////////////////////////////////////////////////////////////////
// class definition
//
// creates the FC SPM components, client handlers and the startup processes neeeded in project
//

/*!
  *  \class spm_tclFactory
  *  \brief This class creates FC SPM components and client handlers and performs the startup of processes needed in the project.
  *         It acts interface point for ProcBase.
  *
  *         While creating FC SPM components and client handlers, it calls constructor first. Then it calls vGetReferences of
  *         FC SPM components and client handlers to initialize all dependencies. Then it calls vStartCommunication to initiate all actions after
  *         initializing the dependencies.
  *         To integrate with this class, any other class has to be derived from ISpmBase.
  *         This class has to be extended to meet the project specific configuration.
  *
  *         All spm components and client handlers are using this class for creation and to initialize all dependencies.
  *
  *         FC SPM components initialize the dependencies using getSpmObjHandler by passing the dependent class name.
  *         FC SPM client handlers initialize the dependencies using getClientHandler by passing the dependent class name.
  */
class spm_tclFactory : public ISpmFactory
{
public:

spm_tclFactory( );

virtual ~spm_tclFactory( );

tBool bCreate( ); // create SystemPowerManager + Startup Processes

tVoid vStart( );  // system startup, all in parallel threads

tVoid vRunEnv( ); // system startup, all in parallel threads

tBool bGetFastShutdown( ) const;

tVoid vAddFactoryClient( ISpmBase *client );

tVoid vRemoveFactoryClient( ISpmBase *client );

tVoid vAddFactoryClientHandler( ISpmClientHandlerBase *client );

tVoid vRemoveFactoryClientHandler( ISpmClientHandlerBase *client );

tVoid vStartAllLateCommunication( ) const;

protected:

// not allow copy-constructor
spm_tclFactory( const spm_tclFactory &factory );

tVoid vCreateComponents( );

// implementation is defined at derived class of project part
virtual tVoid vCreatePrjComponents( ) = 0;

// implementation is defined at derived class of project part
virtual tVoid vCreateClientHandling( ) = 0;

tVoid vStartComponents( );

// implementation is defined at derived class of project part
virtual tVoid vStartPrjComponents( ) = 0;

tVoid vPreStart( );

// implementation is defined at derived class of project part
virtual tVoid vPrePrjStart( ) = 0;

// implementation is defined at derived class of project part
virtual tVoid vPreStartEvaluateProcessStartup( ) = 0;

tVoid vPostStart( );

// implementation is defined at derived class of project part
virtual tVoid vPostPrjStart( ) = 0;

tVoid vSystemProcessStart( );

virtual tVoid vGetReferences( ){ }
virtual tVoid vStartCommunication( ){ }

tVoid vAddAllClientHandler( ) const;

tVoid vGetAllReferences( ) const;

tVoid vStartAllCommunication( ) const;



// trace handling
virtual tVoid vHandleTraceMessage( const tUChar *puchData );

// implementation is defined at derived class of project part
virtual tVoid vHandleTraceMessagePrj( const tUChar *puchData ) = 0;


tVoid vShowSystemInformation( );

// implementation is defined at derived class of project part
virtual tVoid vShowSystemInformationPrj( ) = 0;

std::list < ISpmBase* >              SpmComponentPointerList;

std::list < ISpmClientHandlerBase* > SpmClientHdlList;
static tVoid vDpCallBackFunc( tU32 u32Event );

protected:
// used to add project specific components of derived class
std::string                          _boardtype;
std::string                          _boardtype2;
tU32                                 _sdram;
tBool                                _bSpecialProjectStartRegistry;
tBool                                _bUnconfigProjectStartRegistry;
tBool                                _bIsSWVersionUpdate;

ISpmStartupInvestigationServer            *_poclStartupInvest;
ISpmTrace                                 *_poclTrace;
ISpmGlobalApplicationManager              *_poclGlobalApplicationManager;
ISpmCcaServer                             *_poclCcaMsgHandler;
ISpmDownloadLockListManager               *_poclDownloadLockListManager;
ISpmOsLinux                               *_poOsLinuxProxy;
ISpmWakeupHandler                         *_poWakeupHandler;
ISpmTimerHandler                          *_poTimerHandler;
ISpmSystemPowerManager                    *_poSPM;
ISpmWorkerServer                          *_poWorker;
ISpmSubStateClient                        *_poSubStateHdl;
ISpmCcaServiceServer                      *_poServiceHdl;
ISpmLamAppManager                         *_poLamAppCfg;
ISpmLocalAppManager                       *_poclLocalAppManager;
ISpmApplicationErrorHandler               *_poclAppErrorHdl;
ISpmSupervisionManager                    *_poclSupervisionManager;
ISpmSupervisionClient                     *_poclIgnitionOffSupervisor;
ISpmSupervisionClient                     *_poclSysStateSupervisor;
ISpmSupervisionClient                     *_poclCpuLoadSupervisor;
ISpmSupervisionClient                     *_poclRamLoadSupervisor;
ISpmStartupSupervisor                     *_poclStartupSupervisor;
ISpmOsalProxy                             *_poclOsalProxy;
ISpmWupOnOffEvents                        *_poclOsalWupOnOffEvents;
ISpmWupSupervisionErrorWarning            *_poclOsalWupSupervisionErrorWarning;
ISpmOsalCvm                               *_poclOsalCvm;
ISpmApplicationDatabase                   *_poclDb;
ISpmSyncHandler                           *_poclSyncHdl;
ISpmStatistics                            *_poclWakeupStatistic;
ISpmStatistics                            *_poclSwitchOnStatistic;
ISpmStatistics                            *_poResetStatistic;
ISpmStatistics                            *_poSubStateStatistic;
ISpmStatistics                            *_poPowerOnStatistic;
ISpmStatistics                            *_poTransportModeStatistic;
ISpmStatistics                            *_poSystemStateStatistic;
ISpmPowerSupplyStatistics                 *_poPowerSupplyStatistic;
ISpmSystemLoadStatistics                  *_poCpuUsageStatistic;
ISpmSystemLoadStatistics                  *_poRamUsageStatistic;
ISpmSystemLoadStatistics                  *_poStartupTimeStatistic;
ISpmStatistics                            *_poSWVersionWakeupStatistic;
ISpmStatistics                            *_poSWVersionSwitchOnStatistic;
ISpmStatistics                            *_poSWVersionResetStatistic;
ISpmStatistics                            *_poSWVersionSubStateStatistic;
ISpmStatistics                            *_poSWVersionPowerOnStatistic;
ISpmStatistics                            *_poSWVersionSystemStateStatistic;
ISpmSystemLoadStatistics                  *_poSWVersionCpuUsageStatistic;
ISpmSystemLoadStatistics                  *_poSWVersionRamUsageStatistic;
ISpmSystemLoadStatistics                  *_poSWVersionStartupTimeStatistic;
ISpmSoftwareVersionStatistics             *_poSWVersionStatistic;
ISpmStartupSystem                         *_poStartupLocal;
ISpmApplicationConfiguration              *_poclAppCfg;
ISpmCcaSupplierServer                     *_poclCcaSupplierHandler;
ISpmApplicationRequestSupervisor          *_poclApplicationRequestSupervisor;
ISpmStartupCommon                         *_poclStartupCommon;
ISpmVersionInfo                           *_poclVersionInfo;
ISpmCvmClient                             *_poclCriticalVoltageManagerIntern;
ISpmRegistry                              *_poclRegistry;
#ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
   ISpmStartupSystemVariant               *_variant;
#endif

ISpmProcessDatabase                       *_poProcessDatabase;
ISpmProcessSupervision                    *_poProcessSupervision;
ISpmSupervisionEnableSupervisor           *_poSupervisionEnableSupervisor;

#ifdef VARIANT_S_FTR_ENABLE_INC_THERMALMANAGEMENT
   ISpmThermalMgntIncClientHandler        *_poclThermalMgmtServiceHandler;
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SUPPLYMANAGEMENT
   ISpmSupplyMgntIncClientHandler         *_poclSupplyMgmtServiceHandler;
#endif
#ifdef VARIANT_S_FTR_ENABLE_INC_SYSTEMSTATEMACHINE
   ISpmSystemStateMachineIncClientHandler *_poclSystemStateMachineServiceHandler;
#endif

tVoid vCreateCustomerVersionReg( const std::string& strCustomerVersionRegFile,
                                 const std::string& strCustomerVersion ) const;

tVoid vHandleCustomerVersionReg( ) const;

ISpmSystemStateManager *_poSystemStateManager;

public:
virtual const tChar*getName( ) const { return( "spm_tclFactory" ); }
virtual const tChar*getInterfaceName( ) const { return( "ISpmFactory" ); }

ISpmClientHandlerBase*getClientHandler( const std::string& strName ) const;

ISpmBase*getSpmObjHandler( const std::string& strName ) const;

tBool   bIsSpecialStartupRegActive( ) const { return( _bSpecialProjectStartRegistry ); }

tBool   bIsTargetUnconfigured( ) const { return( _bUnconfigProjectStartRegistry ); }

tBool   bIsSWVersionUpdate( ) const { return( _bIsSWVersionUpdate ); }

// include the creator functions to be overwritten by UnitTests
#include "spm_factory_creator.h"
};

