/*!
  * \file  spm_Factory_trace.cpp
  *  \brief
  *    Method definitions for testing via trace purposes.
  *               This list is generated.
  *
  *  \note
  *  PROJECT: NextGen
  *  SW-COMPONENT: FC SPM
  *  COPYRIGHT:    (C) 2013 - 2020 Robert Bosch GmbH, Hildesheim

  *  The reproduction, distribution and utilization of this file as
  *  well as the communication of its contents to others without express
  *  authorization is prohibited. Offenders will be held liable for the
  *  payment of damages. All rights reserved in the event of the grant
  *  of a patent, utility model or design.
  *
  ******
  */

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#include "spm_Config.h"

// Trace specific includes can be put here
// spm_tclFactory_include_ext: user specific code start
// my class header
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"
#include "dp_generic_if.h"

#include "spm_factory.h"
#include "spm_IOsalProxy.h"
#include "spm_LocalApplicationManagerConfig.h"
#include "spm_LamAppManagerConfig.h"
#include "spm_ApplicationConfigurationConfig.h"
#include "spm_StartupCommon.h"
#include "spm_ApplicationDatabase.h"
#include "spm_SubStateHandlerConfig.h"
#include "spm_SystemStateManagerConfig.h"
#include "spm_CriticalVoltageManagerIntern.h"
#include "spm_GlobalApplicationManager.h"
#include "spm_IStartupInvestigationServer.h"
#include "spm_StartupSystem.h"
#include "spm_CcaMsgHandler.h"
#include "spm_ICcaServiceServer.h"
#include "spm_ApplicationRequestSupervisor.h"
#include "spm_SuperVisionManager.h"
#include "spm_ISuperVisionClient.h"
#include "spm_appsim.h"
#include "spm_IStartupSystemVariant.h"

#include "spm_service_csm.h"
#include "spm_security.h"

#include "spm_IRegistry.h"
#include "spm_Registry_pathes.h"
#include "spm_VersionInfo.h"
#include "spm_ResetStatistics.h"
#include "spm_SubStatesStatistics.h"
#include "spm_TransportmodeStatistics.h"
#include "spm_PowerOnCount.h"
#include "spm_SystemStateStatistics.h"
#include "spm_WakeupStatistics.h"
#include "spm_SoftwareVersionStatistics.h"
#ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
   #include "spm_IStartupSystemVariant.h"
#endif
// my class header
// spm_tclFactory_include_ext: user specific code end


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
#include "trcGenProj/Header/spm_Factory_trace.cpp.trc.h"
#endif

// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

// Spm trace specific code can be put here
// spm_tclFactory_include_spm: user specific code start
// spm_tclFactory_include_spm: user specific code end


tVoid spm_tclFactory::vHandleTraceMessage(const tUChar *puchData){
/*!
  * \fn
  *  \brief
  *    Interface function for the tracechannel.
  *
  *  \param[in]
  *    puchData - Trace command data.
  ******
  */
   tU32 u32MsgCode = ( puchData[1] << 8 ) | puchData[2];

   switch (u32MsgCode){

      case SPM_GETREMAININGSTANDBYTIME:
      {
         // Returns some current active timer values (e.g. SystemTimeout, IgnitionOff time, Time to shutdown in low voltage).

         // GetRemainingStandbyTime: user specific code start
            ETG_TRACE_FATAL( ( "Remaining standby time:          %d sec", _poSystemStateManager->u32GetRemainingStateTimeoutTime() / 1000 ) );
         // ETG_TRACE_FATAL(("Ignition Off count:              %d min", (dynamic_cast<spm_tclSubStateHandlerConfig*>(_poSubStateHdl))->u32GetCurrentIgnitionOffTime()));
            ETG_TRACE_FATAL( ( "Remaining Critical voltage time: %d sec", _poclCriticalVoltageManagerIntern->u32GetRemainingCriticalVoltageTime() / 1000 ) );
         SPM_GET_CLASS_REFERENCE_NEW_VAR(poclSupervisionClientIgnOff, spm_tclIgnitionOffSupervisor, ISpmSupervisionClient);
         poclSupervisionClientIgnOff->vTraceInfo();
         _poclGlobalApplicationManager->vTraceRemainingTime();
         // GetRemainingStandbyTime: user specific code end
      }
      break;

      case SPM_GETSYSTEMSTATE:
      {

         // GetSystemState: user specific code start
         tU32 u32SystemState    = _poSystemStateManager->u32GetSystemState();
         tU32 u32NewSystemState = _poSystemStateManager->u32GetNewSystemState();

         if (u32SystemState == u32NewSystemState){
            ETG_TRACE_FATAL( ( "SystemStateInfo: current system state %u.", ETG_ENUM(SPM_SYSTEM_STATES, u32SystemState) ) );
         } else {
            ETG_TRACE_FATAL( ( "SystemStateInfo: system state change active %u --> %u", ETG_ENUM(SPM_SYSTEM_STATES, u32SystemState), ETG_ENUM(SPM_SYSTEM_STATES, u32NewSystemState) ) );
         }
         if (_poSubStateHdl){
            TTriggerMsg tTrigger = _poSubStateHdl->tGetTrigger();
            ( dynamic_cast < spm_tclSubStateHandlerConfig* >( _poSubStateHdl ) )->vTraceTrigger(SSH_vCurrentTrigger, &tTrigger, TR_LEVEL_FATAL);
         }
            ETG_TRACE_FATAL( ( "Remaining timeout time for current system state: %d sec", _poSystemStateManager->u32GetRemainingStateTimeoutTime() / 1000 ) );
         // GetSystemState: user specific code end
      }
      break;

      case SPM_STARTUPINVESTIGATION:
      {

         // StartupInvestigation: user specific code start
         vShowSystemInformation();
         _poclStartupInvest->vTraceItems();
         // StartupInvestigation: user specific code end
      }
      break;

      case SPM_GETSHOWRESETMEM:
      {

         // GetShowResetMem: user specific code start
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****        RESET MEMORY INFO        *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poResetStatistic){
            _poResetStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****        WAKEUP MEMORY INFO       *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poclWakeupStatistic){
            _poclWakeupStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****        SWITCH ON MEMORY INFO    *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poclSwitchOnStatistic){
            _poclSwitchOnStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****    TRANSPORT_MODE MEMORY INFO   *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poTransportModeStatistic){
            _poTransportModeStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****    SYSTEMSTATE MEMORY INFO      *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poSystemStateStatistic){
            _poSystemStateStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****    SUBSTATE STATISTIC INFO      *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poSubStateStatistic){
            _poSubStateStatistic->vTrace();
         }
            ETG_TRACE_FATAL( ( " " ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );
            ETG_TRACE_FATAL( ( "****         POWER ON TIME           *****" ) );
            ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poPowerOnStatistic){
            _poPowerOnStatistic->vTrace();
         }

         if (_poSWVersionStatistic){
            _poSWVersionStatistic->vTrace();
         }
         // GetShowResetMem: user specific code end
      }
      break;

      case SPM_GETERASERESETMEM:
      {

         // GetEraseResetMem: user specific code start
         if (_poResetStatistic){
            _poResetStatistic->vErase();
         }
         if (_poclWakeupStatistic){
            _poclWakeupStatistic->vErase();
         }
         if (_poclSwitchOnStatistic){
            _poclSwitchOnStatistic->vErase();
         }
         if (_poTransportModeStatistic){
            _poTransportModeStatistic->vErase();
         }
         if (_poSystemStateStatistic){
            _poSystemStateStatistic->vErase();
            spm_tclSystemStateStatistics *poSystemStateStatistics = dynamic_cast < spm_tclSystemStateStatistics* >( _poSystemStateStatistic );
            if (poSystemStateStatistics) {
               poSystemStateStatistics->vEraseErrorHistory();
            }
         }
         if (_poSubStateStatistic){
            _poSubStateStatistic->vErase();
         }
         if (_poSWVersionStatistic){
            _poSWVersionStatistic->vErase();
         }
         dp_tclSrvIf dp;
         dp.s32StoreNow();
         // GetEraseResetMem: user specific code end
      }
      break;

      case SPM_GETPOWERONCOUNTER:
      {

         // GetPowerOnCounter: user specific code start
         if (_poPowerOnStatistic){
            _poPowerOnStatistic->vTrace();
         }
         // GetPowerOnCounter: user specific code end
      }
      break;

      case SPM_GETBUILDVERSION:
      {

         // GetBuildVersion: user specific code start
         _poclVersionInfo->TraceBuildversion();
         // GetBuildVersion: user specific code end
      }
      break;

      case SPM_GETTHREADS:
      {

         // GetThreads: user specific code start
            ETG_TRACE_FATAL( ( "List of spm threads" ) );
         // startup threads
         if (_poStartupLocal){
            _poStartupLocal->vTraceActive();
         }

         // supervision threads
         if (_poclSupervisionManager){
            _poclSupervisionManager->vTraceActive();
            // state request supervison
         }
         if (_poclApplicationRequestSupervisor){
            _poclApplicationRequestSupervisor->vTraceActive();
            // substate handling
         }
         if (_poSubStateHdl){
            _poSubStateHdl->vTraceActive();
            // cca message handling
         }
         if (_poclCcaMsgHandler){
            _poclCcaMsgHandler->vTraceActive();
            // worker server
         }
         if (_poWorker){
            _poWorker->vTraceActive();
         }
         // GetThreads: user specific code end
      }
      break;

      case SPM_GETVARIANTVARIABLEVALUES:
      {

         // GetVariantVariableValues: user specific code start
         #ifdef VARIANT_S_FTR_ENABLE_VARIANT_HANDLING
            _variant->vTrace();
         #endif
         // GetVariantVariableValues: user specific code end
      }
      break;

      case SPM_GETSTARTUPTIME:
      {

         // GetStartupTime: user specific code start
         tUInt nStartupTime = ( dynamic_cast < ISpmWorkerServer* >( _poWorker ) )->nGetStartupTime();
         if (nStartupTime == 0){
            ETG_TRACE_FATAL( ( "Startup processes are still running" ) );
         } else {
            ETG_TRACE_FATAL( ( "Startup has finished after %dms", nStartupTime ) );
         }
         // GetStartupTime: user specific code end
      }
      break;

      case SPM_GETSYSTEMCONFIGURATION:
      {

         // GetSystemConfiguration: user specific code start
         vShowSystemInformation();
         // GetSystemConfiguration: user specific code end
      }
      break;

      case SPM_GETVOLTAGEEVENTCOUNT:
      {

         // GetVoltageEventCount: user specific code start
         TSpmSubStatesStatisticsData    s;
         dp_tclSpmDpEngDataStatisticEng oStat;
         s = oStat.tGetData( );
         ETG_TRACE_FATAL(("LowVoltageCount   : %u", s._u32LowVoltageCount));
         ETG_TRACE_FATAL(("CrLowVoltageCount : %u", s._u32CrLowVoltageCount ));
         ETG_TRACE_FATAL(("HighVoltageCount  : %u", s._u32HighVoltageCount));
         ETG_TRACE_FATAL(("CrHighVoltageCount: %u", s._u32CrHighVoltageCount));
         // GetVoltageEventCount: user specific code end
      }
      break;

      case SPM_SETPOWERONCOUNTER:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SetPowerOnCounter' was called: param1 = 0x%x", param1 ) );

         // SetPowerOnCounter: user specific code start
         if (_poPowerOnStatistic){
            ( dynamic_cast < spm_tclPowerOnCount* >( _poPowerOnStatistic ) )->vSetPwrOnTime(param1);
         }
         // SetPowerOnCounter: user specific code end
      }
      break;

      case SPM_SPMGETCCAAPPINFO:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);

         // SpmGetCcaAppInfo: user specific code start
         amt_tclDebugAppInfo tclInfoMessage( (tU16)param1, (tU16)param1, 0);
         OSAL_tMQueueHandle  hMQ    = scd_OpenQueue( (tU16)param1);
         ail_bIpcMessagePost( (tU16)param1, hMQ, &tclInfoMessage, OSAL_C_U32_MQUEUE_PRIORITY_LOWEST);
         scd_s32CloseQueue(hMQ);
         // SpmGetCcaAppInfo: user specific code end
      }
      break;

      case SPM_SPMSENDOVERTEMPSHUTDOWNEVENT:
      {
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSendOvertempShutdownEvent' was called" ) );

         // SpmSendOvertempShutdownEvent: user specific code start
         // _poclOsalProxy->bSendOvertempShutdownEvent();
         // SpmSendOvertempShutdownEvent: user specific code end
      }
      break;

      case SPM_SPMHELP:
      {

         // SpmHelp: user specific code start
         #ifndef LCM_UNIT_TESTS
         for (tU16 u16TraceId = 0x1144; u16TraceId <=0x114d; u16TraceId++) {
            // check generated file spmhelp.trc for limits
            etg_vTraceBinary(SPM_CALC_ETG_TRACE_CLASS(SPM_TRACE_CLASS_SPM, TR_LEVEL_FATAL) , 0, 0, SPM_ETG_FORMAT_VALUE_EN_T16, u16TraceId);
         }
         #endif
         // SpmHelp: user specific code end
      }
      break;

      case SPM_SPMSETPERSTODEFAULT:
      {
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetPersToDefault' was called" ) );

         // SpmSetPersToDefault: user specific code start
         // _dp->vSetToDefault();
         // SpmSetPersToDefault: user specific code end
      }
      break;

      case SPM_SPMSETPRAMDATA:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetPRamData' was called: param1 = 0x%x, param2 = 0x%x", param1, param2 ) );

         // SpmSetPRamData: user specific code start
         _poclOsalProxy->bSetSRamData(param1, param2);
         // SpmSetPRamData: user specific code end
      }
      break;

      case SPM_SPMGETPRAMDATA:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);

         // SpmGetPRamData: user specific code start
         ETG_TRACE_FATAL( ( "Sram data: %d", _poclOsalProxy->u32GetSRamData(param1) ) );
         // SpmGetPRamData: user specific code end
      }
      break;

      case SPM_SPMSETSHUTDOWNALLOWED:
      {
         tU8 param1 = puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetShutdownAllowed' was called: param1 = 0x%x", param1 ) );

         // SpmSetShutdownAllowed: user specific code start
         param1++;
         (tVoid)param1;
         // SpmSetShutdownAllowed: user specific code end
      }
      break;

      case SPM_SPMREMOVEREGISTRYAPPLICATION:
      {
         // Removes the entry for a specific application from registry. Pass the SWBlock and ApplicationName e.g. BASE/SPM
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmRemoveRegistryApplication' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmRemoveRegistryApplication: user specific code start
         SPM_NULL_POINTER_CHECK(_poclRegistry);
         _poclRegistry->vRemoveEntry( (const tChar*)pString1, "APPID");
         // SpmRemoveRegistryApplication: user specific code end
      }
      break;

      case SPM_SPMREMOVEREGISTRYSERVICE:
      {
         // Removes the entry for a specific service from registry. Pass the SWBlock and ApplicationName and service e.g. BASE/SPM/SERVICES/SPM 
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmRemoveRegistryService' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmRemoveRegistryService: user specific code start
         SPM_NULL_POINTER_CHECK(_poclRegistry);
         _poclRegistry->vRemoveEntry( (const tChar*)pString1, "SERVICEID");
         // SpmRemoveRegistryService: user specific code end
      }
      break;

      case SPM_SPMSHOWPROCESSINREGISTRY:
      {
         // Prints the process configuration of "SwBlock". Pass a SwBlockName as parameter e.g. BASE
         const tChar* pString1 = (const tChar*)&puchData[3];

         // SpmShowProcessInRegistry: user specific code start
         std::string  strName;

         strName = (std::string)SPM_REG_PROCESS_BASE_PATH + pString1;

         SPM_NULL_POINTER_CHECK(_poclRegistry);
         _poclRegistry->vTraceOutRegistry(strName);
         // SpmShowProcessInRegistry: user specific code end
      }
      break;

      case SPM_SPMSHOWBASEINREGISTRY:
      {
         // Prints the registry starting with "String". Pass the directory path e.g. PROCESS/CONFIG
         const tChar* pString1 = (const tChar*)&puchData[3];

         // SpmShowBaseInRegistry: user specific code start
         std::string  strName;

         strName = (std::string)SPM_REG_BASE_PATH + pString1;

         SPM_NULL_POINTER_CHECK(_poclRegistry);
         _poclRegistry->vTraceOutRegistry(strName);
         // SpmShowBaseInRegistry: user specific code end
      }
      break;

      case SPM_SPMTRIGGERAPPSTATE:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);
         tU32 param3 = (tU32)((puchData[11]<<24) | (puchData[12]<<16) | (puchData[13]<<8) | puchData[14]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmTriggerAppState' was called: param1 = 0x%x, param2 = 0x%x, param3 = 0x%x", param1, param2, param3 ) );

         // SpmTriggerAppState: user specific code start
         if (_poclAppCfg){
            _poclAppCfg->vSetValue(param1, param2, param3);
         }
         _poclLocalAppManager->bSetAllSwBlocksToNonBlockMode();
         _poclLocalAppManager->bSetAllApplicationsInProfile();
         // SpmTriggerAppState: user specific code end
      }
      break;

      case SPM_SPMSETSYSTEMTIME:
      {
         // Set system state time in ms.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetSystemTime' was called: param1 = 0x%x, param2 = 0x%x", param1, param2 ) );

         // SpmSetSystemTime: user specific code start
         tU32 u32State      = param1;
         tU32 u32TimeOutSec = param2;
               ETG_TRACE_FATAL( ( "spm_tclFactory(): Set Timer value for state %u: %d s.", ETG_ENUM(CFC_FI_SPM_SYSTEM_STATES, u32State), u32TimeOutSec ) );
         _poSystemStateManager->vSetStateTime(u32State, u32TimeOutSec * 1000);

         if (u32State == _poSystemStateManager->u32GetSystemState() ){
            _poSystemStateManager->vStartStateTimeoutTimer(u32TimeOutSec * 1000);
         }
         // SpmSetSystemTime: user specific code end
      }
      break;

      case SPM_SPMGETUBAT:
      {
         // Get current voltage level in mV.

         // SpmGetUBat: user specific code start
         tU16 u16Ubat = 0;
         _poclOsalProxy->bUbatSense(&u16Ubat);
               ETG_TRACE_FATAL( ( "spm_tclFactory(): Current voltage level: %d mV.", u16Ubat ) );
         // _poclOsalProxy->bCurrentSense(&u16Ubat);
         // ETG_TRACE_FATAL(("spm_tclFactory():              Current : %d mA.",  u16Ubat));

         // SpmGetUBat: user specific code end
      }
      break;

      case SPM_SPMGETSYSTEMTIME:
      {
         // Get system state time in ms.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);

         // SpmGetSystemTime: user specific code start
               ETG_TRACE_FATAL( ( "spm_tclFactory(): Timer value for state %u: %d s.", ETG_ENUM(CFC_FI_SPM_SYSTEM_STATES, param1), _poSystemStateManager->u32GetStateTime(param1) / 1000 ) );
         // SpmGetSystemTime: user specific code end
      }
      break;

      case SPM_SPMSETWATCHDOGCOUNT:
      {
         // Sets the supervision/watchdog count of a supervision client (class name is used to address client, e.g. spm_tclSysStateSupervisor).
         const tChar* pString1 = (const tChar*)&puchData[3];
         tU32 param2 = (tU32)((puchData[53]<<24) | (puchData[54]<<16) | (puchData[55]<<8) | puchData[56]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetWatchdogCount' was called: param1 = '%20s', param2 = 0x%x", (const tChar*)pString1, param2 ) );

         // SpmSetWatchdogCount: user specific code start
         ISpmSupervisionClient *poclSupervisionClient = dynamic_cast < ISpmSupervisionClient* >( _crfFactory.getSpmObjHandler(pString1) );
         if (poclSupervisionClient){
               ETG_TRACE_FATAL( ( "spm_tclFactory(): Set watchdog count to %d (for supervisor client '%s'.", param2, pString1 ) );
            poclSupervisionClient->vSetCount(param2);
         } else {
               ETG_TRACE_FATAL( ( "spm_tclFactory(): No supervisor client found with classname '%s'.", pString1 ) );
         }
         // SpmSetWatchdogCount: user specific code end
      }
      break;

      case SPM_SPMSETSTARTUPCOUNTER:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetStartUpCounter' was called: param1 = 0x%x", param1 ) );

         // SpmSetStartUpCounter: user specific code start
         (void)param1;
               ETG_TRACE_FATAL( ( "SETSTARTUPCOUNTER is not supported in Gen3" ) );
         // SpmSetStartUpCounter: user specific code end
      }
      break;

      case SPM_SPMSYSTEMSTATEHISTORY:
      {
         // Last 200 state transition will be displayed.

         // SpmSystemStateHistory: user specific code start
               ETG_TRACE_FATAL( ( " " ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );
               ETG_TRACE_FATAL( ( "****    SYSTEMSTATE MEMORY INFO      *****" ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poSystemStateStatistic){
            _poSystemStateStatistic->vTrace();
         }
               ETG_TRACE_FATAL( ( " " ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );
               ETG_TRACE_FATAL( ( "****    SUBSTATE STATISTIC INFO      *****" ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poSubStateStatistic){
            _poSubStateStatistic->vTrace();
         }
         // SpmSystemStateHistory: user specific code end
      }
      break;

      case SPM_SPMERRORSTATEHISTORY:
      {
         // List of State transition history for requested error number will be displayed.
         tU8 param1 = puchData[3];

         // SpmErrorStateHistory: user specific code start
         if (_poSystemStateStatistic){
            ( dynamic_cast < spm_tclSystemStateStatistics* >( _poSystemStateStatistic ) )->vTraceErrorHistory( (tU8)param1);
         }
         // SpmErrorStateHistory: user specific code end
      }
      break;

      case SPM_SPMADDERRORSTATEHISTORY:
      {
         // Add current StateTransition list to error state list.

         // SpmAddErrorStateHistory: user specific code start
         if (_poSystemStateStatistic){
            (tVoid)(( dynamic_cast < spm_tclSystemStateStatistics* >( _poSystemStateStatistic ) )->u32AddErrorHistory());
         }
         // SpmAddErrorStateHistory: user specific code end
      }
      break;

      case SPM_SPMWAKEUPLIST:
      {
         // Last wakeup reasons will be dislayed.

         // SpmWakeupList: user specific code start
               ETG_TRACE_FATAL( ( " " ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );
               ETG_TRACE_FATAL( ( "****        WAKEUP MEMORY INFO       *****" ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poclWakeupStatistic){
            _poclWakeupStatistic->vTrace();
         }

         // SpmWakeupList: user specific code end
      }
      break;

      case SPM_SPMRESETLIST:
      {
         // Last reset reasons will be dislayed.

         // SpmResetList: user specific code start
               ETG_TRACE_FATAL( ( "******************************************" ) );
               ETG_TRACE_FATAL( ( "****        RESET MEMORY INFO        *****" ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poResetStatistic){
            _poResetStatistic->vTrace();
         }
         // SpmResetList: user specific code end
      }
      break;

      case SPM_SPMSUBSTATESTATISTIC:
      {
         // Substate statistics will be dislayed.

         // SpmSubstateStatistic: user specific code start
               ETG_TRACE_FATAL( ( " " ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );
               ETG_TRACE_FATAL( ( "****    SUBSTATE STATISTIC INFO      *****" ) );
               ETG_TRACE_FATAL( ( "******************************************" ) );

         if (_poSubStateStatistic){
            _poSubStateStatistic->vTrace();
         }
         // SpmSubstateStatistic: user specific code end
      }
      break;

      case SPM_SPMERASEVOLTAGEEVENTCOUNT:
      {
         // Erase Voltage Event Count in Substate statistics.

         // SpmEraseVoltageEventCount: user specific code start
         TSpmSubStatesStatisticsData    s;
         dp_tclSpmDpEngDataStatisticEng oStat;
         s = oStat.tGetData( );
         s._u32LowVoltageCount = 0;
         s._u32CrLowVoltageCount = 0;
         s._u32HighVoltageCount = 0;
         s._u32CrHighVoltageCount = 0;
         oStat << s;
         // SpmEraseVoltageEventCount: user specific code end
      }
      break;

      case SPM_SPMDISABLEHWWATCHDOG:
      {
         tU8 param1 = puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmDisableHwWatchdog' was called: param1 = 0x%x", param1 ) );

         // SpmDisableHwWatchdog: user specific code start
         param1 = param1;
               ETG_TRACE_FATAL( ( "SpmDisableHwWatchdog --> not possible vi SPM anymore. Must be changed in SYSTEMD service-file!" ) );
               ETG_TRACE_FATAL( ( "To disable resets triggered by SPM use cmd 'SpmWdtResetDisable'!" ) );
         // SpmDisableHwWatchdog: user specific code end
      }
      break;

      case SPM_SPMSETSTARTUPCONFIG:
      {
         // Set Startup configuration.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetStartupConfig' was called: param1 = 0x%x", param1 ) );

         // SpmSetStartupConfig: user specific code start
         dp_tclSpmDpInternDataStartupConfig oCfgConfig;
         oCfgConfig << param1;
         // SpmSetStartupConfig: user specific code end
      }
      break;

      case SPM_SPMGETSTARTUPCONFIG:
      {
         // Display current Startup configuration info.

         // SpmGetStartupConfig: user specific code start
         dp_tclSpmDpInternDataStartupConfig oStartupConfig;
         ETG_TRACE_FATAL( ( "Current startup config: %d", oStartupConfig.tGetData( ) ) );
         // SpmGetStartupConfig: user specific code end
      }
      break;

      case SPM_SPMGETPROCESSINFO:
      {
         // Display current process configuration info.

         // SpmGetProcessInfo: user specific code start
         _poclStartupCommon->vTraceProcessConfiguration();
         // SpmGetProcessInfo: user specific code end
      }
      break;

      case SPM_SPMKILLSWBLOCK:
      {
         // Specific SwBlock will be killed.
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmKillSwBlock' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmKillSwBlock: user specific code start
         _poclLocalAppManager->vKillSoftwareBlock(pString1);
         // SpmKillSwBlock: user specific code end
      }
      break;

      case SPM_SPMSTARTPROC:
      {
         // Start a process from current configuration -> for process information call TTFIS command 'SpmGetProcessInfo'.
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmStartProc' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmStartProc: user specific code start
         _poclStartupCommon->bStartProcess(pString1);
         // SpmStartProc: user specific code end
      }
      break;

      case SPM_SPMKDSWRITE:
      {
         // Write KDS entry -> 'SpmKdsWrite 0d5d 19 29 91 ab 12 11 22 11'.
         const tChar* pString1 = (const tChar*)&puchData[3];
         const tUChar* pStream2 = (const tUChar*)&puchData[7];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmKdsWrite' was called: param1 = '%20s', param2 = 0x%x", (const tChar*)pString1, (tU8)*pStream2 ) );

         // SpmKdsWrite: user specific code start
         tU32          u32EntryLen = (tU32)(puchData[0] - ( pStream2 - &puchData[0] ) + 1U);
         {
            tU32 u32KeyEntry = (tU32)OSAL_u32StringToU32(pString1, NULL, 16);
            if (_poclVersionInfo->bWriteKdsEntry((tU16)u32KeyEntry, (tU8*)pStream2, (tU16)u32EntryLen) ){  //lint !e1773 Attempt to cast away const (or volatile)
               ETG_TRACE_FATAL( ( "spm_tclFactory(), Write KDS Key %04x: (len: %u) succeded (data: %*p).", (tUInt)u32KeyEntry, (tUInt)u32EntryLen, ETG_LIST_LEN(u32EntryLen), ETG_LIST_PTR_T8(pStream2) ) );
            } else {
               ETG_TRACE_FATAL( ( "spm_tclFactory(), Write KDS Key %04x: (len: %u) FAILED.", (tUInt)u32KeyEntry, (tUInt)u32EntryLen ) );
            }
         }

         // SpmKdsWrite: user specific code end
      }
      break;

      case SPM_SPMKDSREAD:
      {
         // Read KDS entry -> 'SpmKdsRead 0d5d 10'.
         const tChar* pString1 = (const tChar*)&puchData[3];
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);

         // SpmKdsRead: user specific code start
         tU32         u32KeyEntry = (tU32)OSAL_u32StringToU32(pString1, NULL, 16);
         tLcmUString  ustrData(255, 0);
         if (_poclVersionInfo->bReadKdsEntry(u32KeyEntry, param2, &ustrData) ){
               ETG_TRACE_FATAL( ( "spm_tclFactory(), Read KDS Key %04x: (len: %d): %*p.", u32KeyEntry, param2, ETG_LIST_LEN(param2), ETG_LIST_PTR_T8(ustrData.c_str() ) ) );
         }
         // SpmKdsRead: user specific code end
      }
      break;

      case SPM_SPMREADDIR:
      {
         // Get content of giver directory.
         const tChar* pString1 = (const tChar*)&puchData[3];

         // SpmReadDir: user specific code start

         OSAL_trIOCtrlDir    *pDir     = OSALUTIL_prOpenDir(pString1);
         OSAL_trIOCtrlDirent *pEntry   = OSAL_NULL;

         if(pDir != OSAL_NULL){
            pEntry = OSALUTIL_prReadDir(pDir);
            while ( pEntry != OSAL_NULL ){
               ETG_TRACE_FATAL( ( "%s", pEntry->s8Name ) );
               pEntry = OSALUTIL_prReadDir( pDir );
            } /*End of While*/
               OSALUTIL_s32CloseDir( pDir );
         } else {
            ETG_TRACE_FATAL( ( "Cannot oden directory: %s", pString1 ) );
         }
         // SpmReadDir: user specific code end
      }
      break;

      case SPM_SPMKDSITEMGET:
      {
         // Read KDS entry via datapool-> 'SpmKdsItemGet [NbBytes] [KdsElement] [KdsItem]' (SpmKdsItemGet 1 VehicleInformation PowerPlatform).
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         const tChar* pString2 = (const tChar*)&puchData[5];
         const tChar* pString3 = (const tChar*)&puchData[45];

         // SpmKdsItemGet: user specific code start
         tU8* pu8Data = new tU8[param1];
         memset(pu8Data, 0, param1);

         ETG_TRACE_FATAL(("SpmKdsItemGet: '%40s' , '%40s'", pString2, pString3));

         if ( DP_S32_NO_ERR == DP_s32GetConfigItem(pString2, pString3, pu8Data, param1) ){
             ETG_TRACE_FATAL(("SpmKdsItemGetData: %02x ", ETG_LIST_LEN(param1), ETG_LIST_PTR_T8(pu8Data) ));
         } else {
             ETG_TRACE_FATAL(("SpmKdsItemGetData: FAILED"));
         }
         // SpmKdsItemGet: user specific code end
      }
      break;

      case SPM_SPMSETTESTMODE:
      {
         // Set test mode for module test. Evaluation of HW signals will be disabled.
         tU8 param1 = puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmSetTestMode' was called: param1 = 0x%x", param1 ) );

         // SpmSetTestMode: user specific code start
         // SPM_GET_CLASS_REFERENCE_NEW_VAR(poHwSigs, spm_tclHwSig, ISpmHwSigs);
         // poHwSigs->vSetTestMode( (tBool)param1);
         // SpmSetTestMode: user specific code end
      }
      break;

      case SPM_SPMSTARTCCAAPP:
      {
         // Creates a CCA message queue for given CcaAppId and waits for incomming messages in a thread --> Parameter [AppName] [AddId] [SrvId].
         const tChar* pString1 = (const tChar*)&puchData[3];
         tU16 param2 = (tU16)((puchData[53]<<8) | puchData[54]);
         tU16 param3 = (tU16)((puchData[55]<<8) | puchData[56]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmStartCcaApp' was called: param1 = '%20s', param2 = 0x%x, param3 = 0x%x", (const tChar*)pString1, param2, param3 ) );

         // SpmStartCcaApp: user specific code start
         {
            spm_tclAppSim *poSimApp = new spm_tclAppSim(* this, pString1, (tU16)param2, (tU16)param3);
            SPM_NULL_POINTER_CHECK(poSimApp);
            vAddFactoryClient(poSimApp);
         }
         // SpmStartCcaApp: user specific code end
      }
      break;

      case SPM_SPMSTOPCCAAPP:
      {
         // Stops aready started app --> Parameter [AppName].
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmStopCcaApp' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmStopCcaApp: user specific code start
         spm_tclAppSim *poSimApp = dynamic_cast < spm_tclAppSim* >( _crfFactory.getSpmObjHandler(pString1) );
         if (poSimApp){
            vRemoveFactoryClient(poSimApp);
            ETG_TRACE_FATAL( ( "spm_tclFactory(): SpmStopCcaApp '%s' will be deleted now.", pString1 ) );
            delete poSimApp;
         }
         // SpmStopCcaApp: user specific code end
      }
      break;

      case SPM_SPMPOSTCCAMSG:
      {
         // post the given stream as cca message.
         const tUChar* pStream1 = (const tUChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmPostCcaMsg' was called: param1 = 0x%x", (tU8)*pStream1 ) );

         // SpmPostCcaMsg: user specific code start
         amt_tclBaseStreamMessage oMyBaseMsg( (tU8*)pStream1); //lint !e1773 PQM_authorized_373
         SPM_GET_IF_REFERENCE_NEW_VAR(poCcaMsg, ISpmCcaServer);
         poCcaMsg->bPostMessage(&oMyBaseMsg, FALSE);
         // SpmPostCcaMsg: user specific code end
      }
      break;

      case SPM_DLLOCKAREA:
      {
         // Lock area (DL feature).
         const tChar* pString1 = (const tChar*)&puchData[3];
         tU16 param2 = (tU16)((puchData[53]<<8) | puchData[54]);
         tU32 param3 = (tU32)((puchData[55]<<24) | (puchData[56]<<16) | (puchData[57]<<8) | puchData[58]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DlLockArea' was called: param1 = '%20s', param2 = 0x%x, param3 = 0x%x", (const tChar*)pString1, param2, param3 ) );

         // DlLockArea: user specific code start
         SPM_GET_IF_REFERENCE_NEW_VAR(poCcaMsg, ISpmCcaServiceServer);
         if (poCcaMsg->bSetDlLockStateRequest(pString1, (tU16)param2, param3)) {
             ETG_TRACE_FATAL( ( "SPM_DLLOCKAREA: passed to add/remove lock area %50s",  pString1) );
         } else {
             ETG_TRACE_FATAL( ( "SPM_DLLOCKAREA: Failed to add/remove lock area %50s",  pString1) );
         }

         // DlLockArea: user specific code end
      }
      break;

      case SPM_SECURITYINFO:
      {
         // State of Security object will be traces out.

         // SecurityInfo: user specific code start
         ISpmSecurity *poSecurity = dynamic_cast < ISpmSecurity* >( _crfFactory.getSpmObjHandler("spm_tclSecurity") );
         if (poSecurity){
            poSecurity->vSecurityInfo();
         }
         // SecurityInfo: user specific code end
      }
      break;

      case SPM_SECURITYDEFSET:
      {
         // Security will be set to default.
         ETG_TRACE_ERRMEM( ( "SPM command 'SecurityDefSet' was called" ) );

         // SecurityDefSet: user specific code start
         ISpmSecurity *poSecurity = dynamic_cast < ISpmSecurity* >( _crfFactory.getSpmObjHandler("spm_tclSecurity") );
         if (poSecurity){
            poSecurity->vSetFactoryDef();
         }
         // SecurityDefSet: user specific code end
      }
      break;

      case SPM_SECURITYENABLE:
      {
         // Enable or disable security feature. Possible only if disabled by KDS.
         tU8 param1 = puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SecurityEnable' was called: param1 = 0x%x", param1 ) );

         // SecurityEnable: user specific code start
         ISpmSecurity *poSecurity = dynamic_cast < ISpmSecurity* >( _crfFactory.getSpmObjHandler("spm_tclSecurity") );
         if (poSecurity){
            poSecurity->vSetSecurityMode( (tBool)param1);
         }
         // SecurityEnable: user specific code end
      }
      break;

      case SPM_DATAPOOLTRACE:
      {
         // Compete datapool will be traced..

         // DatapoolTrace: user specific code start
         dp_tclSrvIf dp;
         dp.vTracePoolElement();
         // DatapoolTrace: user specific code end
      }
      break;

      case SPM_DATAPOOLTRACEPOOL:
      {
         // Specific datapool will be traced. 'DatapoolTracePool CCA_APP_ID SubpoolId'.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU16 param2 = (tU16)((puchData[5]<<8) | puchData[6]);

         // DatapoolTracePool: user specific code start
         dp_tclSrvIf dp;
         tU32        u32PoolId = ( param1 << 16 ) | ( param2 );
         dp.vTracePoolElement(u32PoolId);
         // DatapoolTracePool: user specific code end
      }
      break;

      case SPM_DATAPOOLTRACEELEMENT:
      {
         // Datapool element with given name will be traced.
         const tChar* pString1 = (const tChar*)&puchData[3];

         // DatapoolTraceElement: user specific code start
         dp_tclSrvIf  dp;
         dp.vTracePoolElementByName(pString1);
         // DatapoolTraceElement: user specific code end
      }
      break;

      case SPM_DATAPOOLTRACEHISTORY:
      {
         // Datapool history will be traced..

         // DatapoolTraceHistory: user specific code start
         dp_tclSrvIf dp;
         dp.vTracePoolHistory();
         // DatapoolTraceHistory: user specific code end
      }
      break;

      case SPM_DATAPOOLSTORE:
      {
         // Force storing of the complete datapool.
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolStore' was called" ) );

         // DatapoolStore: user specific code start
         dp_tclSrvIf dp;
            ETG_TRACE_FATAL( ( "Datapool: Start 'StorePool' (time: %dms).", OSAL_ClockGetElapsedTime() ) );
         dp.s32StoreNow();
            ETG_TRACE_FATAL( ( "Datapool: pool stored (%dms).", OSAL_ClockGetElapsedTime() ) );
         // DatapoolStore: user specific code end
      }
      break;

      case SPM_DATAPOOLSETDEFAULT:
      {
         // Datapool: set to default.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolSetDefault' was called: param1 = 0x%x, param2 = 0x%x", param1, param2 ) );

         // DatapoolSetDefault: user specific code start
         dp_tclSrvIf dp;
            ETG_TRACE_FATAL( ( "Datapool: Start SetDefault with %d (time: %dms).", param1, OSAL_ClockGetElapsedTime() ) );
         dp.s32SetDefault(param1, param2);
            ETG_TRACE_FATAL( ( "Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime() ) );
         // DatapoolSetDefault: user specific code end
      }
      break;

      case SPM_DATAPOOLLOCK:
      {
         // Datapool: set to lock state.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         tU32 param2 = (tU32)((puchData[7]<<24) | (puchData[8]<<16) | (puchData[9]<<8) | puchData[10]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolLock' was called: param1 = 0x%x, param2 = 0x%x", param1, param2 ) );

         // DatapoolLock: user specific code start
         dp_tclSrvIf dp;
            ETG_TRACE_FATAL( ( "Datapool: Start SetDefault with %d (time: %dms).", param1, OSAL_ClockGetElapsedTime() ) );
         dp.s32Lock(param1, param2);
            ETG_TRACE_FATAL( ( "Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime() ) );
         // DatapoolLock: user specific code end
      }
      break;

      case SPM_DATAPOOLUNLOCK:
      {
         // Datapool: set to lock state.
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolUnlock' was called: param1 = 0x%x", param1 ) );

         // DatapoolUnlock: user specific code start
         dp_tclSrvIf dp;
            ETG_TRACE_FATAL( ( "Datapool: Start SetDefault with %d (time: %dms).", param1, OSAL_ClockGetElapsedTime() ) );
         dp.s32Unlock(param1);
            ETG_TRACE_FATAL( ( "Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime() ) );
         // DatapoolUnlock: user specific code end
      }
      break;

      case SPM_DATAPOOLCALCHASH:
      {
         // Datapool: trigger datapool to calculate a hash over all pool files (SHA1).
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolCalcHash' was called" ) );

         // DatapoolCalcHash: user specific code start
         dp_tclSrvIf dp;
         tU32        u32Data[5];
            ETG_TRACE_FATAL( ( "Datapool: Start CalcHash (time: %dms).", OSAL_ClockGetElapsedTime() ) );
         if (0 < dp.s32CalcHash(u32Data, 5) ){
            ETG_TRACE_FATAL( ( "Datapool:: DP HASH: %08x%08x%08x%08x%08x.",
                               u32Data[0],
                               u32Data[1],
                               u32Data[2],
                               u32Data[3],
                               u32Data[4] ) );
         }
            ETG_TRACE_FATAL( ( "Datapool: CalcHash finished (%dms).", OSAL_ClockGetElapsedTime() ) );
         // DatapoolCalcHash: user specific code end
      }
      break;

      case SPM_DATAPOOLSETDEFAULTRAW:
      {
         // Datapool: set to default.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU32 param2 = (tU32)((puchData[5]<<24) | (puchData[6]<<16) | (puchData[7]<<8) | puchData[8]);
         tU32 param3 = (tU32)((puchData[9]<<24) | (puchData[10]<<16) | (puchData[11]<<8) | puchData[12]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolSetDefaultRaw' was called: param1 = 0x%x, param2 = 0x%x, param3 = 0x%x", param1, param2, param3 ) );

         // DatapoolSetDefaultRaw: user specific code start
         tU32 u32Starttime = OSAL_ClockGetElapsedTime();
         tS32 s32Ret       = DP_s32SetDefault(param1, param2, param3);
            ETG_TRACE_FATAL( ( "Datapool: DefSet returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime() - u32Starttime ) );
         // DatapoolSetDefaultRaw: user specific code end
      }
      break;

      case SPM_DATAPOOLLOCKRAW:
      {
         // Datapool: set to lock state.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU32 param2 = (tU32)((puchData[5]<<24) | (puchData[6]<<16) | (puchData[7]<<8) | puchData[8]);
         tU32 param3 = (tU32)((puchData[9]<<24) | (puchData[10]<<16) | (puchData[11]<<8) | puchData[12]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolLockRaw' was called: param1 = 0x%x, param2 = 0x%x, param3 = 0x%x", param1, param2, param3 ) );

         // DatapoolLockRaw: user specific code start
         tU32 u32Starttime = OSAL_ClockGetElapsedTime();
         tS32 s32Ret       = DP_s32Lock(param1, param2, param3);
            ETG_TRACE_FATAL( ( "Datapool: Lock returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime() - u32Starttime ) );
         // DatapoolLockRaw: user specific code end
      }
      break;

      case SPM_DATAPOOLUNLOCKRAW:
      {
         // Datapool: set to lock state.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU32 param2 = (tU32)((puchData[5]<<24) | (puchData[6]<<16) | (puchData[7]<<8) | puchData[8]);
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolUnlockRaw' was called: param1 = 0x%x, param2 = 0x%x", param1, param2 ) );

         // DatapoolUnlockRaw: user specific code start
         tU32 u32Starttime = OSAL_ClockGetElapsedTime();
         tS32 s32Ret       = DP_s32Unlock(param1, param2);
            ETG_TRACE_FATAL( ( "Datapool: Unlock returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime() - u32Starttime ) );
         // DatapoolUnlockRaw: user specific code end
      }
      break;

      case SPM_DATAPOOLWRITEELEMENT:
      {
         // Datapool: Test interface to write one element.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU16 param2 = (tU16)((puchData[5]<<8) | puchData[6]);
         tU16 param3 = (tU16)((puchData[7]<<8) | puchData[8]);
         const tChar* pString4 = (const tChar*)&puchData[9];
         const tUChar* pStream5 = (const tUChar*)&puchData[39];
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolWriteElement' was called: param1 = 0x%x, param2 = 0x%x, param3 = 0x%x, param4 = '%20s', param5 = 0x%x", param1, param2, param3, (const tChar*)pString4, (tU8)*pStream5 ) );

         // DatapoolWriteElement: user specific code start
         tU32              u32EntryLen = (tU32)(puchData[0] - ( pStream5 - &puchData[0] ) + 1);
         dp_tclBaseElement myDpElem(pString4);
         myDpElem.bFillData( (tVoid*)pStream5, u32EntryLen); //lint !e1773 Attempt to cast away const (or volatile)
         tU32              s32Status   = DP_s32SetElement( ( param1 << 16 | param2 ), &myDpElem, (tU16)param3);
            ETG_TRACE_FATAL( ( "Datapool: write element status: '%u' for '%s'", ETG_ENUM(EDpErrorCodes, s32Status), pString4 ) );
         // DatapoolWriteElement: user specific code end
      }
      break;

      case SPM_DATAPOOLREADELEMENT:
      {
         // Datapool: Test interface to read one element.
         tU16 param1 = (tU16)((puchData[3]<<8) | puchData[4]);
         tU16 param2 = (tU16)((puchData[5]<<8) | puchData[6]);
         tU16 param3 = (tU16)((puchData[7]<<8) | puchData[8]);
         const tChar* pString4 = (const tChar*)&puchData[9];

         // DatapoolReadElement: user specific code start
         tChar            *strData   = (tChar*)pString4; //lint !e1773

         dp_tclBaseElement myDpElem(pString4);
         tU32              s32Status = DP_s32GetElement( ( param1 << 16 | param2 ), &myDpElem, (tU16)param3);
            ETG_TRACE_FATAL( ( "Datapool: read element status: '%u' for '%s'", ETG_ENUM(EDpErrorCodes, s32Status), strData ) );
            ETG_TRACE_FATAL( ( "Datapool: data: %*p", ETG_LIST_LEN(myDpElem.u32GetDataLength() ), ETG_LIST_PTR_T8(myDpElem.pvGetData() ) ) );
         // DatapoolReadElement: user specific code end
      }
      break;

      case SPM_DATAPOOLENABLEELEMENTFORTRACE:
      {
         // Datapool element will be traced if Datapool trace class is enabled.
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolEnableElementForTrace' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // DatapoolEnableElementForTrace: user specific code start
         DP_vEnableElementForTrace(pString1);
         ETG_TRACE_FATAL( ( "Datapool: Element: '%s' is enabled for Trace now.", pString1 ) );
         // DatapoolEnableElementForTrace: user specific code end
      }
      break;

      case SPM_DATAPOOLENABLEALLELEMENTSFORTRACE:
      {
         // All datapool elements will be traced if Datapool trace class is enabled.
         ETG_TRACE_ERRMEM( ( "SPM command 'DatapoolEnableAllElementsForTrace' was called" ) );

         // DatapoolEnableAllElementsForTrace: user specific code start
         DP_vEnableElementForTrace(NULL);
         // DatapoolEnableAllElementsForTrace: user specific code end
      }
      break;

      case SPM_SPMDPWRITETESTSTRING:
      {
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmDpWriteTestString' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmDpWriteTestString: user specific code start
         pString1 = pString1; // useless LINT stuff
         // SpmDpWriteTestString: user specific code end
      }
      break;

      case SPM_SPMDPADDNOTTESTSTRING:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmDpAddNotTestString' was called: param1 = 0x%x", param1 ) );

         // SpmDpAddNotTestString: user specific code start
         param1 = param1; // useless LINT stuff
         // SpmDpAddNotTestString: user specific code end
      }
      break;

      case SPM_SPMDPREMOVENOTTESTSTRING:
      {
         tU32 param1 = (tU32)((puchData[3]<<24) | (puchData[4]<<16) | (puchData[5]<<8) | puchData[6]);
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmDpRemoveNotTestString' was called: param1 = 0x%x", param1 ) );

         // SpmDpRemoveNotTestString: user specific code start
         param1 = param1; // useless LINT stuff
         // SpmDpRemoveNotTestString: user specific code end
      }
      break;

      case SPM_SPMDPSETFAKEDEVICECONFIG:
      {
         tU8 param1 = puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmDpSetFakeDeviceConfig' was called: param1 = 0x%x", param1 ) );

         // SpmDpSetFakeDeviceConfig: user specific code start
         dp_tclSpmDpConfigFakeDeviceConfig oFakeDeviceConnect;
         ETG_TRACE_FATAL( ( "spm_tclFactory::SpmDpSetFakeDeviceConfig: before bConnect2Fake= %u", ETG_ENUM(SPM_BOOL_STATE, oFakeDeviceConnect.tGetData() ) ) );
         oFakeDeviceConnect << (tBool)param1;
         ETG_TRACE_FATAL( ( "spm_tclFactory::SpmDpSetFakeDeviceConfig: after bConnect2Fake= %u", ETG_ENUM(SPM_BOOL_STATE, oFakeDeviceConnect.tGetData() ) ) );
         // SpmDpSetFakeDeviceConfig: user specific code end
      }
      break;

      case SPM_SPMTRIGGERSYNCPOINT:
      {
         const tChar* pString1 = (const tChar*)&puchData[3];
         ETG_TRACE_ERRMEM( ( "SPM command 'SpmTriggerSyncPoint' was called: param1 = '%20s'", (const tChar*)pString1 ) );

         // SpmTriggerSyncPoint: user specific code start
         _poclStartupCommon->vOnSyncPointReached(pString1);
         // SpmTriggerSyncPoint: user specific code end
      }
      break;

      default:
         //nothing to do
         break;
   } //lint !e764: switch statement possibly does not have a case --> generated trace command function
     // spm_tclFactoryFunction: user specific code start
   vHandleTraceMessagePrj(puchData);
     // spm_tclFactoryFunction: user specific code end

   return;
} // vHandleTraceMessage

// Trace specific code can be put here
// spm_tclFactoryFile: user specific code start
// spm_tclFactoryFile: user specific code end

