/*!
  * \file spm_WakeupStatistics.cpp
  *  \brief
  *    Wakeup statistics of the system
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 09.11.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_WakeupStatistics.h"

#include "spm_IOsalProxy.h"
#include "timeConvert.h"

#include "spm_IFactory.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_WakeupStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"


/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

// //#define SPM_TRACE_FILE_ID   SPM_FILE_WAKEUPSTATISTICS
template < typename T1, typename T2>
spm_tclWakeupStatisticsTemplate<T1, T2>::spm_tclWakeupStatisticsTemplate( const ISpmFactory& factory, enStatisticsType eStatType )
   : ISpmStatistics( factory ){
   _eStatType = eStatType;
}
template < typename T1, typename T2>
spm_tclWakeupStatisticsTemplate<T1, T2>::~spm_tclWakeupStatisticsTemplate( ){
}

template < typename T1, typename T2>
tVoid spm_tclWakeupStatisticsTemplate<T1, T2>::vGetData( void *pStatistics, tU32 u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    Get the wakeup history from data pool.
  *  \param[out] pStatistics:  pointer to TSpmWakeupStatisticsData buffer of wakeup history.
  *  \param[in]  u32ObjectSize: buffer size.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of wakeUp history list.
  ******
  */
   if ( pStatistics != NULL ){
      T1    oHistory;
      tU32  u32Count = oHistory.u32GetCount( );
      std::list< TSpmWakeupStatisticsData > tWakeupList;
      if ( u32ObjectSize >= u32Count * sizeof( TSpmWakeupStatisticsData ) ){
         for ( tUInt i = 0; i < u32Count; ++i ){
            TSpmWakeupStatisticsData tWupElem ={0,0};

            oHistory.s32GetElem( i, tWupElem );
            tWakeupList.push_back( tWupElem );
         }
         memset( pStatistics, 0, u32ObjectSize );
         std::copy( tWakeupList.begin(), tWakeupList.end(), ( TSpmWakeupStatisticsData * )pStatistics );
      } else {
         ETG_TRACE_FATAL( ("WakeupStatistics::vGetData(): buffer size is not valid") );
      }
   } else {
	   ETG_TRACE_ERRMEM( ("WakeupStatistics::vGetData() pStatistics buffer is NULL") );
   }
} // vGetData

template < typename T1, typename T2>
tVoid spm_tclWakeupStatisticsTemplate<T1, T2>::vSetNewData( tU32 u32StatisticType ) const {
/*!
  * \fn
  *  \brief
  *    update the wakeup history & wakeup counter to data pool.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   OSAL_trTimeDate rCurrentTime = { 0,0,0,0,0,0,0,0,0 };

   SPM_GET_IF_REFERENCE_NEW_VAR( poclOsalProxy, ISpmOsalProxy );
   (tVoid)poclOsalProxy->bGetUtcTime( &rCurrentTime );
   rCurrentTime.s32Year += 1900;

   tclTimeConvert               oTimeConv;
   T1 oHistory;

   tU32                         u32Seconds = oTimeConv.u32GetSeconds( &rCurrentTime );
   TSpmWakeupStatisticsData     tWupElem;
   tWupElem.u32Time       = u32Seconds;
   tWupElem.u32WakeUpType = u32StatisticType;
   oHistory.vPushBack( tWupElem );

                         ETG_TRACE_FATAL( ( "Detected wakeup reason: %d: %04d-%02d-%02d  %02d:%02d:%02d (seconds since 1970: %d)", ETG_ENUM( SPM_WAKEUP_REASON, u32StatisticType ),
                      rCurrentTime.s32Year, rCurrentTime.s32Month, rCurrentTime.s32Day,
                      rCurrentTime.s32Hour, rCurrentTime.s32Minute, rCurrentTime.s32Second,
                      u32Seconds ) );

   T2 oDevWakeUpCount;
   tU32                                u32Count = oDevWakeUpCount.tGetData( ) + 1;
   oDevWakeUpCount << u32Count;

   switch ( _eStatType )
   {
      case SPM_U32_STATISTIC_GLOBAL:
      {
         SPM_GET_CLASS_REFERENCE_NEW_VAR( poclSWVersionWakeupStatistics, spm_tclSWVersionWakeupStatistics, ISpmStatistics );
         poclSWVersionWakeupStatistics->vSetNewData( u32StatisticType );
         break;
      }
      default:
         break;
   }
} // vSetNewData
template < typename T1, typename T2>
tVoid spm_tclWakeupStatisticsTemplate<T1, T2>::vErase( ) const {
/*!
  * \fn
  *  \brief
  *    Clear wakeup history in data pool.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   T1 oHistory;

   oHistory.vClearList( );
}
template < typename T1, typename T2>
tVoid spm_tclWakeupStatisticsTemplate<T1, T2>::vTrace( ) const {
/*!
  * \fn
  *  \brief
  *     Trace info of wakeup history & wakeup counter.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */

   T2 oDevWakeUpCount;

                         ETG_TRACE_FATAL( ( "Device WAKEUP overall count :            %d", oDevWakeUpCount.tGetData( ) ) );

   T1        oHistory;

   tU32                                u32Count = oHistory.u32GetCount( );

   for ( tUInt i = 0; i < u32Count; ++i ){
      TSpmWakeupStatisticsData tWupElem;
      tWupElem.u32Time = 0;
     // tWupElem.u32Time = 0;
     tWupElem.u32WakeUpType = 0;
      tclTimeConvert           oTimeConv;
      OSAL_trTimeDate          tTimeDate = { 0,0,0,0,0,0,0,0,0 };

      oHistory.s32GetElem( i, tWupElem );
      oTimeConv.vGetDateFromElapsedSeconds( tWupElem.u32Time, &tTimeDate );

                         ETG_TRACE_FATAL( ( "WakeUp reason %2d:                %3d (%04d.%02d.%02d %02d:%02d:%02d (UTC))",
                         i + 1,
                         ETG_ENUM( SPM_WAKEUP_REASON, tWupElem.u32WakeUpType ),
                         tTimeDate.s32Year, tTimeDate.s32Month, tTimeDate.s32Day,
                         tTimeDate.s32Hour, tTimeDate.s32Minute, tTimeDate.s32Second
                         ) );
   }

} // vTrace

spm_tclWakeupStatistics::spm_tclWakeupStatistics( const ISpmFactory& factory )
   : spm_tclWakeupStatisticsTemplate < dp_tclSpmDpEngDataWupHistory, dp_tclSpmDpEngDataDeviceWakeupCount >( factory, SPM_U32_STATISTIC_GLOBAL )
{
}

spm_tclSWVersionWakeupStatistics::spm_tclSWVersionWakeupStatistics( const ISpmFactory& factory )
   : spm_tclWakeupStatisticsTemplate < dp_tclSpmDpEngDataCurrentSWVersionWupHistory, dp_tclSpmDpEngDataCurrentSWVersionDeviceWakeupCount >( factory, SPM_U32_STATISTIC_SOFTWARE_VERSION )
{
}

