/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_StatisticTypes.h
  * @brief        This is header file contains Statistic Types.
  * @copyright    (C) 2015 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#include "../statehandling/spm_SubStateDefines.h"
#include <cstring>
// -----------------------------------------------------------------------------
// data structure definition
// -----------------------------------------------------------------------------
#define SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION 40
struct TSpmResetElem
{
   tU32 u32Time; // seconds since 1.1.2012
   tU16 u16ResetType;
   tU8 strBuildVersion[SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION];
   tU8 strCustomerVersion[SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION];
};

struct TAppProcReset
{
   tU32 u32ResetCount;
   tU32 u32LastResetType;
   tU32 u32LastResetTime;

   tU32 u32ApplicationError;
   tU32 u32Software;
   tU32 u32EcuReset;
   tU32 u32Download;
   tU32 u32DownloadError;
   tU32 u32Overtemperature;
   tU32 u32LcmIntern;
   tU32 u32User;
   tU32 u32PowerOn;
   tU32 u32PowerLoss;
   tU32 u32PowerOff;
   tU32 u32Scc;
   tU32 u32Unknown;
};

struct TSccReset
{
   tU32 u32ResetCount;
   tU32 u32LastResetType;
   tU32 u32LastResetTime;

   tU32 u32PowerOn;
   tU32 u32HwWdg;
   tU32 u32Coldstart;
   tU32 u32AppmodeChange;
   tU32 u32DuringLpw;
   tU32 u32DuringPllOsz;
   tU32 u32DuringSw;
   tU32 u32User;
   tU32 u32Warmstart;
};

struct TAdrReset
{
   tU32 _u32Reset;
};

struct TSpmResetStatisticsData
{
   TAppProcReset tHmiReset;
   TSccReset tSccReset;
   TAdrReset tAdrReset;

   tBool _bMyOwnResetTrigger;
};


struct TSpmSubStatesStatisticsData
{
   tU32 _u32OnTipperCount;
   tU32 _u32SContCount;
   tU32 _u32IgnitionCount;
   tU32 _u32DoorOpenCount;
   tU32 _u32DiagCount;
   tU32 _u32DownloadCount;
   tU32 _u32PhoneCount;

   tU32 _u32LowVoltageCount;
   tU32 _u32CrLowVoltageCount;
   tU32 _u32HighVoltageCount;
   tU32 _u32CrHighVoltageCount;

   tU32 _u32LowVoltageShutdownCount;
   tU32 _u32LowVoltageCurrentShutdownCount;

   tU32 _u32OnReachedCount;
   tU32 _u32OnReachedSecurityCount;
};

struct TSpmSwitchOnStatisticsData
{
   tU32 u32Time;                 // seconds since 1.1.2012
   tU32 u32WakeUpType;
};

struct TSpmSystemStateStatisticsData
{ 
   TSpmSystemStateStatisticsData ()
    {
       u32Time = 0;
       u32TimeMs = 0;
       u8SystemState = 0;
       u8EntryType =0;
    }

   tU32 u32Time;                    // seconds since 1.1.2012
   tU32 u32TimeMs;
   tU8 u8SystemState;
   tU8 u8EntryType;
   union
   {
      TTriggerMsg tTrigger;
      tU8 u8Wakeup;
      tU8 u8Reset;
   }
   u;

};

#define SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS          60

struct TSpmWakeupStatisticsData
{
   tU32 u32Time;               // seconds since 1.1.2012
   tU32 u32WakeUpType;
};

/*!
   \struct TSpmVoltageHistogramRangeStatisticsData
   structure prepresent the voltage element in voltage histogram.
   u16VoltageValue: Voltage value in mV
   u32OperationTime: in the operating time in sec corresponding to voltage.
   u32OperationTimeMs: remainder of operating time in milisec after divided by 1 sec .
  */

struct TSpmVoltageHistogramRangeStatisticsData
{
   tU16 u16VoltageValue;
   tU32 u32OperationTime;
   tU32 u32OperationTimeMs;
};

/*!
   \struct TSpmSystemLoadStatisticsData
   structure for keeping the system load (CPU, RAM, startup time,...) of system.
   u32MaxValue: Maximum value of loading
   u32AvgValue: Average value of loading
   u32MinValue: Minimum value of loading
   u32CounterValue: Counter (Number of times) getting loading from system.
   u64SumOfLoading: Sum of all collection loading.
  */

struct TSpmSystemLoadStatisticsData
{
   tU32 u32MaxValue;
   tU32 u32AvgValue;
   tU32 u32MinValue;
   tBool bStatisticDataDisabled;
   tU32 u32CounterValue;
   tU64 u64SumOfLoading;
};

/*!
  *  \enum enStatisticsType
  *  \brief type of statistics data (global, specific software version)
  *
  */
enum enStatisticsType
{
   SPM_U32_STATISTIC_GLOBAL,
   SPM_U32_STATISTIC_SOFTWARE_VERSION,
};

/*!
   Maximum keeping number of wakeup history element
  */
#define SPM_U32_STATISTIC_MAX_WAKEUP_HISTORY_LIST           10
/*!
   Maximum keeping number of switch on history element
  */
#define SPM_U32_STATISTIC_MAX_SWITCH_ON_HISTORY_LIST        10
/*!
   Maximum keeping number of reset history element
  */
#define SPM_U32_STATISTIC_MAX_RESET_HISTORY_LIST            20
/*!
   Maximum keeping number of system state history element
  */
#define SPM_U32_STATISTIC_MAX_SYS_STATE_HISTORY_LIST        60

/*!
   \struct TSpmSoftwareVersion
   structure for keeping software version information.
   strBuildVersion: build version label  	[HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\VERSIONS\BUILDVERSION_LABEL]
   strBuildVersionCustomer: build version  	[HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\VERSIONS\BUILDVERSION_CUSTVERSTRING]
   strCustomerVersion: build customer version [HKEY_LOCAL_MACHINE\SOFTWARE\BLAUPUNKT\VERSIONS\CUSTOMERVERSION]
  */

struct TSpmSoftwareVersion
{
public:
   tChar strBuildVersion[SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION];
   tChar strBuildVersionCustomer[SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION];
   tChar strCustomerVersion[SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION];
   TSpmSoftwareVersion& operator= (const TSpmSoftwareVersion& others) {
      std::memcpy(strBuildVersion, others.strBuildVersion, SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION);
      std::memcpy(strBuildVersionCustomer, others.strBuildVersionCustomer, SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION);
      std::memcpy(strCustomerVersion, others.strCustomerVersion, SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION);
      return *this;
   }
};

/*!
   \struct TSpmSoftwareVersionStatisticsData
   structure for keeping statistics data of software version.
   tSWVersion: keep software version information (build label, build version, build customer version)
   tResetStatisticsData: keep system reset information
   tResetHistory: keep system reset history
   tSubStatesStatistics: keep system state information
   tSystemStateHistory: keep system state history
   tSystemStateErrorHistory[0-4]: keep 5 system state history stacks
   tWakeupStatisticsData: keep Wakeup information
   tSwitchOnStatisticsData: keep switch on information
   u32SystemOperationTime: keep power on time
   tCpuUsage: keep cpu load statistic
   tRamUsage: keep ram load statistic
   tStartupTime: keep startup time load statistic
  */
struct TSpmSoftwareVersionStatisticsData
{
   TSpmSoftwareVersion tSWVersion;
   TSpmResetStatisticsData tResetStatisticsData;
   TSpmResetElem tResetHistory[SPM_U32_STATISTIC_MAX_RESET_HISTORY_LIST];
   tU32 u32ResetHistoryCount;
   TSpmSubStatesStatisticsData tSubStatesStatistics;
   TSpmSystemStateStatisticsData tSystemStateHistory[SPM_U32_STATISTIC_MAX_SYS_STATE_HISTORY_LIST];
   TSpmSystemStateStatisticsData tSystemStateErrorHistory0[SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS];
   TSpmSystemStateStatisticsData tSystemStateErrorHistory1[SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS];
   TSpmSystemStateStatisticsData tSystemStateErrorHistory2[SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS];
   TSpmSystemStateStatisticsData tSystemStateErrorHistory3[SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS];
   TSpmSystemStateStatisticsData tSystemStateErrorHistory4[SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS];
   TSpmWakeupStatisticsData tWakeupStatisticsData[SPM_U32_STATISTIC_MAX_WAKEUP_HISTORY_LIST];
   TSpmSwitchOnStatisticsData tSwitchOnStatisticsData[SPM_U32_STATISTIC_MAX_SWITCH_ON_HISTORY_LIST];
   tU32 u32SystemOperationTime;
   TSpmSystemLoadStatisticsData tCpuUsage;
   TSpmSystemLoadStatisticsData tRamUsage;
   TSpmSystemLoadStatisticsData tStartupTime;
};
