/**
  * @swcomponent   Life Cycle Management
  * @{
  * @file          spm_SoftwareVersionStatistics.cpp
  * @PROJECT:      CMD project
  * @brief         implementation of spm_tclSoftwareVersionStatistics class
  *
  * ----------------------------------------------------------------------------
  * @copyright     (C) 2017 Robert Bosch GmbH.
  *                The reproduction, distribution and utilization of this file as well as the
  *                communication of its contents to others without express authorization is prohibited.
  *                Offenders will be held liable for the payment of damages.
  *                All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_STDVISITORS
#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_FUNCTIONIDS
#include "spm_fi_if.h"

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_spm_if.h"

// -----------------------------------------------------------------------------
// includes FC SPM
// -----------------------------------------------------------------------------
#include "spm_Config.h"
#include "spm_SoftwareVersionStatistics.h"
#include "spm_SystemStateStatistics.h"
#include "spm_SubStateHandler.h"
#include "spm_IOsalProxy.h"

#include "spm_IFactory.h"
#include "spm_IRegistry.h"
#include "spm_SubStatesStatistics.h"
#include "spm_ISystemLoadStatistics.h"

#include "timeConvert.h"
#include <algorithm>
#include <cstring>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_SoftwareVersionStatistics.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/

spm_tclSoftwareVersionStatistics::spm_tclSoftwareVersionStatistics( const ISpmFactory& factory )
   : ISpmSoftwareVersionStatistics( factory )
   , _poclSWVersionPowerOnCount( NULL )
   , _poclSWVersionResetStatistics( NULL )
   , _poclSWVersionSubStateStatistics( NULL )
   , _poclSWVersionCpuUsageStatistics( NULL )
   , _poclSWVersionRamUsageStatistics( NULL )
   , _poclSWVersionStartupTimeStatistics( NULL )
   , _poclSWVersionSystemStateStatistics( NULL )
   , _poclSWVersionWakeupStatistics( NULL )
   , _poclSWVersionSwitchOnStatistics( NULL ){
} // spm_tclSoftwareVersionStatistics

spm_tclSoftwareVersionStatistics::~spm_tclSoftwareVersionStatistics( ){
   _poclSWVersionPowerOnCount          = NULL;
   _poclSWVersionResetStatistics       = NULL;
   _poclSWVersionSubStateStatistics    = NULL;
   _poclSWVersionCpuUsageStatistics    = NULL;
   _poclSWVersionRamUsageStatistics    = NULL;
   _poclSWVersionStartupTimeStatistics = NULL;
   _poclSWVersionSystemStateStatistics = NULL;
   _poclSWVersionWakeupStatistics      = NULL;
   _poclSWVersionSwitchOnStatistics    = NULL;
} // ~spm_tclSoftwareVersionStatistics

tVoid spm_tclSoftwareVersionStatistics::vGetReferences( ){
/*!
  * \fn
  *  \brief
  *    Call when system starts up.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details load all the statistics information from datapool of current running software version.
  ******
  */
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionPowerOnCount,            spm_tclSWVersionPowerOnCount,          ISpmPowerOnStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionResetStatistics,         spm_tclSWVersionResetStatistics,       ISpmResetStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionSubStateStatistics,      spm_tclSWVersionSubStateStatistics,    ISpmStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionCpuUsageStatistics,      spm_tclSWVersionCpuUsageStatistics,    ISpmSystemLoadStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionRamUsageStatistics,      spm_tclSWVersionRamUsageStatistics,    ISpmSystemLoadStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionStartupTimeStatistics,   spm_tclSWVersionStartupTimeStatistics, ISpmSystemLoadStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionSystemStateStatistics,   spm_tclSWVersionSystemStateStatistics, ISpmSystemStateStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionWakeupStatistics,        spm_tclSWVersionWakeupStatistics,      ISpmStatistics );
   SPM_GET_CLASS_REFERENCE_USE_VAR( _poclSWVersionSwitchOnStatistics,      spm_tclSWVersionSwitchOnStatistics,    ISpmStatistics );
   vOnSWVersionUpdate( );
} // vGetReferences

tVoid spm_tclSoftwareVersionStatistics::vStartLateCommunication( ){
/*!
  * \fn
  *  \brief
  *    Call when system starts up.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details load all the software version information from registry and store in datapool
  ******
  */
   if ( _crfFactory.bIsSWVersionUpdate( ) ){
      dp_tclSpmDpEngDataCurrentSWVersion oSWVersion;
      TSpmSoftwareVersion tSWVersion;
      //Update new sw version
      vGetSoftwareVersionReg( &tSWVersion );
      if ( DP_S32_NO_ERR != oSWVersion.s32SetData( tSWVersion ) ){
         ETG_TRACE_FATAL(( "spm_tclSoftwareVersionStatistics::vStartLateCommunication: write software version to Datapool fail" ));
      }
      //Update SWVersion in data
      _tSoftwareVersionStatisticsData.tSWVersion = tSWVersion;
   }
} // vStartLateCommunication

tVoid spm_tclSoftwareVersionStatistics::vOnSWVersionUpdate( ){
/*!
  * \fn
  *  \brief
  *    Handle the software version datapools while system starts up and new software updated.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   // New SW Update
   if ( _crfFactory.bIsSWVersionUpdate( ) ){
      dp_tclSpmDpEngDataSWVersionUpdateCount oSWVersionUpdateCounter;
      tU32  u32SWVersionUpdateCounter = 0;
      oSWVersionUpdateCounter >> u32SWVersionUpdateCounter;
      // Get all elements data of last software version and push to the stack
      vGetElemDataFromDataPool( );

      // push to the SWVersion datapool statck.
      if ( ( u32SWVersionUpdateCounter >= 1 ) && ( u32SWVersionUpdateCounter <= SPM_U32_STATISTIC_SW_VERSION_STACK_01_MAX_INDEX ) ){
         dp_tclSpmDpEngDataSoftwareVersionStatistics0 oSWVersionStatistics0;
         oSWVersionStatistics0.vPushBack( _tSoftwareVersionStatisticsData );
      } else if ( u32SWVersionUpdateCounter > SPM_U32_STATISTIC_SW_VERSION_STACK_01_MAX_INDEX ){
         dp_tclSpmDpEngDataSoftwareVersionStatistics1 oSWVersionStatistics1;
         oSWVersionStatistics1.vPushBack( _tSoftwareVersionStatisticsData );
      }

      // update counter
      ++u32SWVersionUpdateCounter;
      oSWVersionUpdateCounter << u32SWVersionUpdateCounter;

      // clean up all the datapool and private swversion statistics data
      _poclSWVersionPowerOnCount->vErase( );
      _poclSWVersionResetStatistics->vErase( );
      _poclSWVersionSubStateStatistics->vErase( );
      _poclSWVersionCpuUsageStatistics->vEraseLoading( );
      _poclSWVersionRamUsageStatistics->vEraseLoading( );
      _poclSWVersionStartupTimeStatistics->vEraseLoading( );
      _poclSWVersionSystemStateStatistics->vErase( );
      _poclSWVersionSystemStateStatistics->vEraseErrorHistory( );
      _poclSWVersionWakeupStatistics->vErase( );
      _poclSWVersionSwitchOnStatistics->vErase( );
      memset( &_tSoftwareVersionStatisticsData, 0, sizeof( TSpmSoftwareVersionStatisticsData ) );
   } else {
      vGetElemDataFromDataPool( );
   }
} // vOnSWVersionUpdate

tVoid spm_tclSoftwareVersionStatistics::vGetElemDataFromDataPool( ){
/*!
  * \fn
  *  \brief
  *    Wrapper of getting information from current running software version datapool.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   // Get the data from all the elements and push to data pool then
   vGetSoftwareVersionDP( &_tSoftwareVersionStatisticsData.tSWVersion );
   _tSoftwareVersionStatisticsData.u32SystemOperationTime  = vGetPowerOnCount( _poclSWVersionPowerOnCount );
   vGetResetStatisticsData( &_tSoftwareVersionStatisticsData.tResetStatisticsData,   _poclSWVersionResetStatistics );
   vGetResetHistory( _tSoftwareVersionStatisticsData.tResetHistory, _poclSWVersionResetStatistics );
   vGetSubStateStatisticsData( &_tSoftwareVersionStatisticsData.tSubStatesStatistics, _poclSWVersionSubStateStatistics );
   vGetSystemLoadUsageData( &_tSoftwareVersionStatisticsData.tCpuUsage,    _poclSWVersionCpuUsageStatistics );
   vGetSystemLoadUsageData( &_tSoftwareVersionStatisticsData.tRamUsage,    _poclSWVersionRamUsageStatistics );
   vGetSystemLoadUsageData( &_tSoftwareVersionStatisticsData.tStartupTime, _poclSWVersionStartupTimeStatistics );
   vGetSystemStateHistory( _tSoftwareVersionStatisticsData.tSystemStateHistory, _poclSWVersionSystemStateStatistics );
   vGetErrSystemDataStateHistoryStack01( _tSoftwareVersionStatisticsData.tSystemStateErrorHistory0, _poclSWVersionSystemStateStatistics );
   vGetErrSystemDataStateHistoryStack02( _tSoftwareVersionStatisticsData.tSystemStateErrorHistory1, _poclSWVersionSystemStateStatistics );
   vGetErrSystemDataStateHistoryStack03( _tSoftwareVersionStatisticsData.tSystemStateErrorHistory2, _poclSWVersionSystemStateStatistics );
   vGetErrSystemDataStateHistoryStack04( _tSoftwareVersionStatisticsData.tSystemStateErrorHistory3, _poclSWVersionSystemStateStatistics );
   vGetErrSystemDataStateHistoryStack05( _tSoftwareVersionStatisticsData.tSystemStateErrorHistory4, _poclSWVersionSystemStateStatistics );
   vGetWakeupHistory( _tSoftwareVersionStatisticsData.tWakeupStatisticsData, _poclSWVersionWakeupStatistics );
   vGetSwitchOnHistory( _tSoftwareVersionStatisticsData.tSwitchOnStatisticsData, _poclSWVersionSwitchOnStatistics );
} // vGetElemDataFromDataPool

tVoid spm_tclSoftwareVersionStatistics::vGetSoftwareVersionReg( TSpmSoftwareVersion *pSWVersion ){
   SPM_GET_IF_REFERENCE_NEW_VAR( poclRegistry, ISpmRegistry );
   std::string    strBuildVersion;
   std::string    strBuildVersionCustomer;
   std::string    strCustomerVersion;
   if ( false == poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_LABEL", strBuildVersion ) ){
      ETG_TRACE_ERR( ( "BUILDVERSION_LABEL not available in registry" ) );
   } else {
      ETG_TRACE_USR1( ( "BUILDVERSION_LABEL=%s", strBuildVersion.c_str( ) ) );
   }
   if ( false == poclRegistry->bGetStringFromRegistry( "VERSIONS", "BUILDVERSION_CUSTVERSTRING", strBuildVersionCustomer ) ){
      ETG_TRACE_ERR( ( "BUILDVERSION_CUSTVERSTRING not available in registry" ) );
   } else {
      ETG_TRACE_USR1( ( "BUILDVERSION_CUSTVERSTRING=%s", strBuildVersionCustomer.c_str( ) ) );
   }
   if ( false == poclRegistry->bGetStringFromRegistry( "VERSIONS", "CUSTOMERVERSION", strCustomerVersion ) ){
      ETG_TRACE_ERR( ( "CUSTOMERVERSION not available in registry" ) );
   } else {
      ETG_TRACE_USR1( ( "CUSTOMERVERSION=%s", strCustomerVersion.c_str( ) ) );
   }
   OSAL_szStringNCopy( pSWVersion->strBuildVersion,    strBuildVersion.c_str( ),    SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION - 1 );
   OSAL_szStringNCopy( pSWVersion->strBuildVersionCustomer, strBuildVersionCustomer.c_str( ), SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION - 1 );
   OSAL_szStringNCopy( pSWVersion->strCustomerVersion, strCustomerVersion.c_str( ), SPM_RESET_STATISTIC_MAX_LENGTH_SW_VERSION - 1 );
} // vGetSoftwareVersionReg

tVoid spm_tclSoftwareVersionStatistics::vGetSoftwareVersionDP( TSpmSoftwareVersion *pSWVersion ){
   dp_tclSpmDpEngDataCurrentSWVersion oSWVersion;
   TSpmSoftwareVersion tSWVersion;
   oSWVersion >> tSWVersion;
   memcpy( pSWVersion, &tSWVersion, sizeof( TSpmSoftwareVersion ) );
   _tSoftwareVersionStatisticsData.tSWVersion = tSWVersion;
}

tVoid spm_tclSoftwareVersionStatistics::vGetSystemLoadUsageData( TSpmSystemLoadStatisticsData *pSystemLoadData, ISpmSystemLoadStatistics *pSystemLoadStatistic ){
   pSystemLoadStatistic->vGetData( pSystemLoadData, (tU32) sizeof( TSpmSystemLoadStatisticsData ) );
} // vGetSystemLoadUsageData

tVoid spm_tclSoftwareVersionStatistics::vGetResetStatisticsData( TSpmResetStatisticsData *pResetStatisticsData, ISpmResetStatistics *pResetStatistics ){
   pResetStatistics->vGetData( pResetStatisticsData, (tU32) sizeof( TSpmResetStatisticsData ) );
} // vGetResetStatisticsData

tVoid spm_tclSoftwareVersionStatistics::vGetSubStateStatisticsData( TSpmSubStatesStatisticsData *pSubStatesStatisticsData, ISpmStatistics *pSubStateStatistics ){
   pSubStateStatistics->vGetData( pSubStatesStatisticsData, (tU32) sizeof( TSpmSubStatesStatisticsData ) );
} // vGetSubStateStatisticsData

tU32 spm_tclSoftwareVersionStatistics::vGetPowerOnCount( ISpmPowerOnStatistics *pPowerOnCount ){
   return pPowerOnCount->u32GetCurrentPwrOnTime( );
} // vGetPowerOnCount

tVoid spm_tclSoftwareVersionStatistics::vGetResetHistory( TSpmResetElem *pResetHistoryStatisticsData, ISpmResetStatistics *pResetStatistics ){
   pResetStatistics->vGetData( pResetHistoryStatisticsData, (tU32)( sizeof( TSpmResetElem ) * SPM_U32_STATISTIC_MAX_RESET_HISTORY_LIST ) );
} // vGetResetHistory

tVoid spm_tclSoftwareVersionStatistics::vGetWakeupHistory( TSpmWakeupStatisticsData *pWakeupStatisticsData, ISpmStatistics *pWakeupStatistics ){
   pWakeupStatistics->vGetData( pWakeupStatisticsData, (tU32)( sizeof( TSpmWakeupStatisticsData ) * SPM_U32_STATISTIC_MAX_WAKEUP_HISTORY_LIST ) );
} // vGetWakeupHistory

tVoid spm_tclSoftwareVersionStatistics::vGetSwitchOnHistory( TSpmSwitchOnStatisticsData *pSwitchOnStatisticsData, ISpmStatistics *pSwitchOnStatistics ){
   pSwitchOnStatistics->vGetData( pSwitchOnStatisticsData, (tU32)( sizeof( TSpmSwitchOnStatisticsData ) * SPM_U32_STATISTIC_MAX_SWITCH_ON_HISTORY_LIST ) );
} // vGetSwitchOnHistory

tVoid spm_tclSoftwareVersionStatistics::vGetSystemStateHistory( TSpmSystemStateStatisticsData *pSystemStateStatisticsData, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataSysStateHistory( pSystemStateStatisticsData, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_SYS_STATE_HISTORY_LIST ) );
} // vGetSystemStateHistory

tVoid spm_tclSoftwareVersionStatistics::vGetErrSystemDataStateHistoryStack01( TSpmSystemStateStatisticsData *pSystemStateStatisticsDataErr01, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataErrSysStateStateHistoryStack01( pSystemStateStatisticsDataErr01, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ) );
} // vGetErrSystemDataStateHistoryStack01

tVoid spm_tclSoftwareVersionStatistics::vGetErrSystemDataStateHistoryStack02( TSpmSystemStateStatisticsData *pSystemStateStatisticsDataErr02, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataErrSysStateStateHistoryStack02( pSystemStateStatisticsDataErr02, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ) );
} // vGetErrSystemDataStateHistoryStack02

tVoid spm_tclSoftwareVersionStatistics::vGetErrSystemDataStateHistoryStack03( TSpmSystemStateStatisticsData *pSystemStateStatisticsDataErr03, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataErrSysStateStateHistoryStack03( pSystemStateStatisticsDataErr03, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ) );
} // vGetErrSystemDataStateHistoryStack03

tVoid spm_tclSoftwareVersionStatistics::vGetErrSystemDataStateHistoryStack04( TSpmSystemStateStatisticsData *pSystemStateStatisticsDataErr04, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataErrSysStateStateHistoryStack04( pSystemStateStatisticsDataErr04, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ));
} // vGetErrSystemDataStateHistoryStack04

tVoid spm_tclSoftwareVersionStatistics::vGetErrSystemDataStateHistoryStack05( TSpmSystemStateStatisticsData *pSystemStateStatisticsDataErr05, ISpmSystemStateStatistics *pSystemStateStatistics ){
   pSystemStateStatistics->vGetDataErrSysStateStateHistoryStack05( pSystemStateStatisticsDataErr05, (tU32)( sizeof( TSpmSystemStateStatisticsData ) * SPM_U32_STATISTIC_MAX_ERR_ENTRIES_PER_STACKS ) );
} // vGetErrSystemDataStateHistoryStack05

tVoid spm_tclSoftwareVersionStatistics::vGetData( void *pStatistics,
                                                  tU32  u32ObjectSize ) const {
/*!
  * \fn
  *  \brief
  *    Get the all current store software version statistics from datapool.
  *
  *  \param [out] void *pStatistics: buffer for getting the system load statistics.
  *  \param [in] tU32  u32ObjectSize: size of buffer.
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  *  \details  All the exceed buffer data will be set to 0 if the u32ObjectSize > size of software version datapools count.
  ******
  */
   if ( pStatistics != NULL ){
      dp_tclSpmDpEngDataSoftwareVersionStatistics0 oSWVersionStatistics0;
      dp_tclSpmDpEngDataSoftwareVersionStatistics1 oSWVersionStatistics1;
      tU32 u32SWVersionStatistics0Count = oSWVersionStatistics0.u32GetCount( );
      tU32 u32SWVersionStatistics1Count = oSWVersionStatistics1.u32GetCount( );
      tU32 u32BufferSize = tU32( ( u32SWVersionStatistics0Count + u32SWVersionStatistics1Count ) * sizeof ( TSpmSoftwareVersionStatisticsData ) );
      if (  u32ObjectSize >= u32BufferSize ) {
         std::list< TSpmSoftwareVersionStatisticsData > tSoftwareVersionStatisticsDataList;
         for (tUInt i = 0; i < u32SWVersionStatistics0Count; ++i){
            TSpmSoftwareVersionStatisticsData tSWVersionStatisticsData0;
            oSWVersionStatistics0.s32GetElem( i, tSWVersionStatisticsData0 );
            tSoftwareVersionStatisticsDataList.push_back( tSWVersionStatisticsData0 );
         }
         for (tUInt i = 0; i < u32SWVersionStatistics1Count; ++i){
            TSpmSoftwareVersionStatisticsData tSWVersionStatisticsData1;
            oSWVersionStatistics1.s32GetElem( i, tSWVersionStatisticsData1 );
            tSoftwareVersionStatisticsDataList.push_back( tSWVersionStatisticsData1 );
         }
         memset( pStatistics, 0, u32ObjectSize );
         std::copy( tSoftwareVersionStatisticsDataList.begin(), tSoftwareVersionStatisticsDataList.end(), ( TSpmSoftwareVersionStatisticsData * )pStatistics );
      } else {
         ETG_TRACE_ERRMEM( ( "spm_tclSoftwareVersionStatistics::vGetData() buffer size is not valid size should be %u ", u32BufferSize ) );
      }
   } else {
      ETG_TRACE_ERRMEM( ( "spm_tclSoftwareVersionStatistics::vGetData() pStatistics buffer is NULL " ) );
   }
} // vGetData

tBool spm_tclSoftwareVersionStatistics::vGetDataWithIndex( TSpmSoftwareVersionStatisticsData *pStatisticsData,
                                                           tU32 u32Index ) {
/*!
  * \fn
  *  \brief
  *    Get the software version statistics with specific index
  *
  *  \param [out] TSpmSoftwareVersionStatisticsData *pStatisticsData: buffer for getting the software version statistics.
  *  \param [in] tU32  u32Index: Index in datapool (0x00 - 0x13, for current running will 0xFF).
  *  \return TRUE if succuess else false
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   if ( pStatisticsData != NULL ){
      if ( u32Index == SPM_U32_STATISTIC_CURRENT_SW_VERSION ){
         vGetElemDataFromDataPool( );
         memcpy( pStatisticsData, &_tSoftwareVersionStatisticsData, sizeof( TSpmSoftwareVersionStatisticsData ) );
         return TRUE;
      }
      if ( u32Index >= SPM_U32_STATISTIC_SW_VERSION_MAX_INDEX ){
         return FALSE;
      }
      if ( u32Index < SPM_U32_STATISTIC_SW_VERSION_STACK_01_MAX_INDEX ){
         dp_tclSpmDpEngDataSoftwareVersionStatistics0 oSWVersionStatistics0;
         TSpmSoftwareVersionStatisticsData tSoftwareVersionStatisticsData0;
         if ( oSWVersionStatistics0.s32GetElem( u32Index, tSoftwareVersionStatisticsData0 ) < 0 ){
            ETG_TRACE_ERRMEM( ( "spm_tclSoftwareVersionStatistics::vGetDataWithIndex() oSWVersionStatistics0 get elem fail with index %u", u32Index ) );
            return FALSE;
         }
         memcpy( pStatisticsData, &tSoftwareVersionStatisticsData0, sizeof( TSpmSoftwareVersionStatisticsData ) );
      } else {
         dp_tclSpmDpEngDataSoftwareVersionStatistics1 oSWVersionStatistics1;
         TSpmSoftwareVersionStatisticsData tSoftwareVersionStatisticsData1;
         if ( oSWVersionStatistics1.s32GetElem( ( u32Index - SPM_U32_STATISTIC_SW_VERSION_STACK_01_MAX_INDEX ), tSoftwareVersionStatisticsData1 ) < 0 ){
            ETG_TRACE_ERRMEM( ( "spm_tclSoftwareVersionStatistics::vGetDataWithIndex() oSWVersionStatistics1 get elem fail with index %u", u32Index ) );
            return FALSE;
         }
         memcpy( pStatisticsData, &tSoftwareVersionStatisticsData1, sizeof( TSpmSoftwareVersionStatisticsData ) );
      }
      return TRUE;
   } else {
      ETG_TRACE_ERRMEM( ( "spm_tclSoftwareVersionStatistics::vGetDataWithIndex() pStatistics buffer is NULL with index %u", u32Index ) );
      return FALSE;
   }
} // vGetDataWithIndex

tVoid spm_tclSoftwareVersionStatistics::vTrace( ) const {
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   TSpmSoftwareVersion tSWVersion;
   dp_tclSpmDpEngDataCurrentSWVersion oSWVersion;
   oSWVersion >> tSWVersion;
   ETG_TRACE_FATAL( ( " " ) );
   ETG_TRACE_FATAL( ( "******************************************" ) );
   ETG_TRACE_FATAL( ( "****   Current Software Version      *****" ) );
   ETG_TRACE_FATAL( ( "******************************************" ) );
   ETG_TRACE_FATAL( ( "Current SW Version BUILD_LABEL                = %s", tSWVersion.strBuildVersion ) );
   ETG_TRACE_FATAL( ( "Current SW Version BUILDVERSION_CUSTVERSTRING = %s", tSWVersion.strBuildVersionCustomer ) );
   ETG_TRACE_FATAL( ( "Current SW Version CUSTOMERVERSION            = %s", tSWVersion.strCustomerVersion ) );
   ETG_TRACE_FATAL( ( "---------------------------------------------------------------------------" ) );
   _poclSWVersionPowerOnCount->vTrace();
   _poclSWVersionResetStatistics->vTrace();
   _poclSWVersionSubStateStatistics->vTrace();
   _poclSWVersionCpuUsageStatistics->vTrace();
   _poclSWVersionRamUsageStatistics->vTrace();
   _poclSWVersionStartupTimeStatistics->vTrace();
   _poclSWVersionSystemStateStatistics->vTrace();
   for ( tU8 i = 0; i < 5; ++i ) {
      _poclSWVersionSystemStateStatistics->vTraceErrorHistory( i );
   }
   _poclSWVersionWakeupStatistics->vTrace();
   _poclSWVersionSwitchOnStatistics->vTrace();
}


tVoid spm_tclSoftwareVersionStatistics::vErase( ) const {
/*!
  * \fn
  *  \brief
  *    Erase all data inside software version statistics datapool
  *
  *  \note
  *  \version
  *  \callgraph
  *  \callergraph
  ******
  */
   dp_tclSpmDpEngDataSWVersionUpdateCount oSWVersionUpdateCounter;
   oSWVersionUpdateCounter << 0;
   dp_tclSpmDpEngDataSoftwareVersionStatistics0 oSWVersionStatistics0;
   oSWVersionStatistics0.vClearList( );
   dp_tclSpmDpEngDataSoftwareVersionStatistics1 oSWVersionStatistics1;
   oSWVersionStatistics1.vClearList( );
   dp_tclSpmDpEngDataCurrentSWVersion oSWVersion;
   TSpmSoftwareVersion tSWVersion;
   memset( &tSWVersion, 0, sizeof( TSpmSoftwareVersion ) );
   oSWVersion << tSWVersion;
   _poclSWVersionPowerOnCount->vErase( );
   _poclSWVersionResetStatistics->vErase( );
   _poclSWVersionSubStateStatistics->vErase( );
   _poclSWVersionCpuUsageStatistics->vEraseLoading( );
   _poclSWVersionRamUsageStatistics->vEraseLoading( );
   _poclSWVersionStartupTimeStatistics->vEraseLoading( );
   _poclSWVersionSystemStateStatistics->vErase( );
   _poclSWVersionSystemStateStatistics->vEraseErrorHistory( );
   _poclSWVersionWakeupStatistics->vErase( );
   _poclSWVersionSwitchOnStatistics->vErase( );
} // vErase
