/*!
  * \file spm_ProcessDatabase.cpp
  *  \brief
  *    Contains the access point to all process supervision relevant information.
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  *  \callgraph
  *  \callergraph
  * 1.0 | 14.03.2013 | Petroglou Johannis (BSOT/ENG), Bipin Krishnan(RBEI/ECG4) | Initial Version \n
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#include "OsalConf.h"
#include "osal_public.h"

// SPM  configuration
#include "spm_Config.h"

#define SPM_FI_S_IMPORT_INTERFACE_SPM_COREFI_TYPES
#include "spm_fi_if.h"

#include <string>

#include "spm_ProcessDatabase.h"
#include "spm_CriticalSection.h"
#include "spm_IFactory.h"
#include "spm_ISystemPowerManager.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM
 #include "trcGenProj/Header/spm_ProcessDatabase.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

#define SPM_SEM_PROC_DB       "SpmPrcDb"

spm_tclProcessDatabase::spm_tclProcessDatabase( const ISpmFactory& factory )
   : ISpmProcessDatabase( factory )
   , _hProcessConfigSem( OSAL_C_INVALID_HANDLE )
   , _idt( - 1 )
   , _szThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 ){
/*!
  * \fn
  *  \brief
  *    Constructor
  *
  *  \param[in]
     factory: spm factory object.
  ******
  */
   _idp = OSAL_ProcessWhoAmI( );

   if ( OSAL_s32SemaphoreCreate( SPM_SEM_PROC_DB, &_hProcessConfigSem, (tU32)1 ) != OSAL_OK ){
      tU32 u32ErrorReason = OSAL_u32ErrorCode( );
      ETG_TRACE_ERRMEM( ( "SPM: spm_tclProcessDatabase !!!!!! Error detected !!!!!! cannot create Semaphore _hProcessConfigSem: error 0x%08X (%s)",
                             (tUInt)u32ErrorReason, OSAL_coszErrorText( u32ErrorReason ) ) );
   }
}

spm_tclProcessDatabase::~spm_tclProcessDatabase( ){
/*!
  * \fn
  *  \brief
  *    Destructor
  *
  *  \param none
  ******
  */
   if ( _hProcessConfigSem != OSAL_C_INVALID_HANDLE ){
      OSAL_s32SemaphoreClose( _hProcessConfigSem );
      OSAL_s32SemaphoreDelete( SPM_SEM_PROC_DB );
   }
}

tVoid spm_tclProcessDatabase::vAccessDatabaseIntern( ){
/*!
  * \fn
  *  \brief
  *    Method to get the lock of Process Supervision relevant information.
  *
  *  \param none
  ******
  */
   OSAL_tThreadID myThreadId = OSAL_ThreadWhoAmI( );

   if ( ( myThreadId == _idt ) && ( _idt != - 1 ) ){
      ETG_TRACE_FATAL( ( "spm_tclProcessDatabase::vAccessDatabaseIntern() tried to enter a 2nd time. Thread: (tid=%d) '%s'",
                         _idt,
                         _szThreadName.c_str( ) ) );
   } else {
      /* Try for max. 5s to obtain the semaphore*/
      ETG_TRACE_USR4( ( "spm_tclProcessDatabase::vAccessDatabaseIntern() waiting for 5000s. Thread-Id: %d", myThreadId ) );
      if ( spm_bEnterCriticalWithTimeout( _hProcessConfigSem, 5000 ) == FALSE ){
         std::string   strSpmError = "SPM: spm_tclProcessDatabase::vAccessDatabaseIntern() blocked for more than 5s! Dumping Spm callstacks now.";
         ETG_TRACE_FATAL( ( "spm_tclProcessDatabase::vAccessDatabaseIntern() warning: cant get ProcessDataBaseLock after 5s, waiting now forever!" ) );
         SPM_GET_IF_REFERENCE_NEW_VAR( poclSystemPowerManager, ISpmSystemPowerManager );
         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strSpmError.c_str( ), (tU16)strSpmError.length( ), FALSE );

         std::string   myThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 );
         (tVoid)bGetTaskName( _idp, myThreadId, &myThreadName[0] );

         strSpmError = "spm_tclApplicationDatabase::vAccessDatabaseIntern blocked for Thread ";
         strSpmError += myThreadName;
         strSpmError += " (tid=";
         strSpmError += std::to_string(myThreadId);
         strSpmError += ") (blocked by Thread ";
         strSpmError += _szThreadName;
         strSpmError += " (tid=";
         strSpmError += std::to_string(_idt);
         strSpmError += "))";

         poclSystemPowerManager->vWriteErrmem( U16_M_ERRMEM_SPM_ERROR( SPM_U8_ERRMEM_TYPE_STRING ), (const tU8*)strSpmError.c_str( ), (tU16)strSpmError.length( ), FALSE );
         TRACE_SPM_ERROR_MY_THREAD( poclSystemPowerManager );

         spm_vEnterCritical( _hProcessConfigSem );
      }

      _idt = myThreadId;
      (tVoid)bGetTaskName( _idp, _idt, &_szThreadName[0] );
   }

} // vAccessDatabaseIntern

tVoid spm_tclProcessDatabase::vReleaseDatabaseIntern( tVoid ){
/*!
  * \fn
  *  \brief
  *    Method to release the lock of Process Supervision relevant information.
  *
  *  \param none
  ******
  */
   OSAL_tThreadID myThreadId = OSAL_ThreadWhoAmI( );

   std::string    myThreadName( OSAL_C_U32_MAX_NAMELENGTH, 0 );

   if ( _idt != - 1 ){
      if ( myThreadId == _idt ){
         _idt = - 1;
         spm_bReleaseCritical( _hProcessConfigSem );
      } else {
         ETG_TRACE_FATAL( ( "spm_tclProcessDatabase::vReleaseDatabaseIntern(): Cannot release the database by thread (tid=%d) '%s'",
                            myThreadId,
                            myThreadName.c_str( ) ) );
         ETG_TRACE_FATAL( ( "because currently Semaphore is with other thread (tid=%d) '%s'",
                            _idt,
                            _szThreadName.c_str( ) ) );
      }
   } else {
         ETG_TRACE_FATAL( ( "spm_tclApplicationDatabase::vReleaseDatabaseIntern(): Thread (tid=%d) '%s' is tried to release database without vReleaseDatabaseInternIntern",
                         myThreadId,
                         myThreadName.c_str( ) ) );
   }
} // vReleaseDatabaseIntern

tBool spm_tclProcessDatabase::bUpdateMapStartedProc( const TProcessConfiguration& processConfig ){
/*!
  * \fn
  *  \brief
  *   Updates Process Configuration information of a SPM Started Process in respective map.
  *
  ******
  */
   vAccessDatabaseIntern( );
   _oStartedProcesses[processConfig.handle] = processConfig;
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tBool spm_tclProcessDatabase::bUpdateMapAdditionalProc( const TProcessConfiguration& processConfig ){
/*!
  * \fn
  *  \brief
  *   Updates Process Configuration information of an additional process in respective map.
  *
  ******
  */
   vAccessDatabaseIntern( );
   _oAdditionalProcesses[processConfig.handle] = processConfig;
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tBool spm_tclProcessDatabase::bUpdateExternalProcess( const TProcessSupervisionConfiguration& processConfig ){
/*!
  * \fn
  *  \brief
  *   Updates Process Configuration information of an external process in respective vector.
  *
  *  \param[in] processConfig: Process Configuration.
  ******
  */
   vAccessDatabaseIntern( );
   _vecExternalProcessesList.push_back( processConfig );
   vReleaseDatabaseIntern( );
   return( TRUE );
}

tBool spm_tclProcessDatabase::bRemoveMapProc( const std::string strProcessName ){
/*!
  * \fn
  *  \brief
  *     Removes a Process Configuration information from the map(started processes or additional processes) based on the Process name.
  *
  *  \param[in] strProcessName: Process Name.
  ******
  */
   tBool                          fFoundProcess = FALSE;

   vAccessDatabaseIntern( );
   TMapStartedProcesses::iterator it;

   for ( it = _oStartedProcesses.begin( ); it != _oStartedProcesses.end( ); ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( processConfig.strProcessName == strProcessName ){
         fFoundProcess = TRUE;
         _oStartedProcesses.erase( it++ );
      } else {
         ++it;
      }
   }

   for ( it = _oAdditionalProcesses.begin( ); it != _oAdditionalProcesses.end( ); ){
      TProcessConfiguration& processConfig = ( * it ).second;
      if ( processConfig.strProcessName == strProcessName ){
         fFoundProcess = TRUE;
         _oAdditionalProcesses.erase( it++ );

      } else {
         ++it;
      }
   }

   /* its possible that a process which has to be removed, is also added to the "ExternalProcess" list
      registry key. So then the process is listed in the _vecExternalProcessesList, which is taken over after
      end of startup to the above _oAdditionalProcesses. If kill of process is received before this take over
      is done, its possible that we get a process supervision reset caused by the missing (because killed) process.
      To avoid this, we remove the process also from this list*/
   std::vector < TProcessSupervisionConfiguration >::iterator itExternal;
   for ( itExternal = _vecExternalProcessesList.begin( ); itExternal != _vecExternalProcessesList.end( ); ){
      if ( ( * itExternal ).procName == strProcessName ){
         fFoundProcess = TRUE;
         _vecExternalProcessesList.erase( itExternal++ );
      } else {
         ++itExternal;
      }
   }


   vReleaseDatabaseIntern( );
   return( fFoundProcess );
} // bRemoveMapProc

