/**
  * @swcomponent  Life Cycle Management
  * @{
  * @file         spm_ConnectedApps.h
  * @brief        This is header file contains functions for storing the information about connected applications..
  * @copyright    (C) 2013 - 2016 Robert Bosch GmbH.
  *               The reproduction, distribution and utilization of this file as well as the
  *               communication of its contents to others without express authorization is prohibited.
  *               Offenders will be held liable for the payment of damages.
  *               All rights reserved in the event of the grant of a patent, utility model or design.
  * @}
  */

#pragma once

#define AMT_S_IMPORT_INTERFACE_GENERIC
#include "amt_if.h"    // only for the state defines

#include <map>
#include <string>

#ifndef _SPM_CONFIG_H
   #error spm_Config.h must be included!
#endif

// -----------------------------------------------------------------------------
// includes
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// stores information about connected applications
// -----------------------------------------------------------------------------

/*!
  *  \class spm_tclConnectedApp
  *  \brief This class stores the information about connected applications. If a SoftwareBlock has multiple applications , then,
          these applications are said to be 'connected', hence, giving rise to the term 'ConnectedApps'
          It stores application type specific ID for different states , information related to monitoring applications ,
        application state change  and application state start/end time .

        This class is used by spm_tclLocalApplicationManager, spm_tclApplicationDatabase, spm_tclSoftwareBlock .
  */
class spm_tclConnectedApp
{
private:
tU32                    _u32AppStateInvalid;       // application type specific ID for the Invalid state
tU32                    _u32AppStateUninitialized; // application type specific ID for the Uninitialized state
tU32                    _u32AppStateInitialized;   // application type specific ID for the Initialized state

tBool                   _bIsConnected;             // Is application connected?
tBool                   _bIsLocal;                 // Is application local to FC SPM or remote?

tU32                    _u32AppID;                 // Application ID

tBool                   _bWatchdog;                // monitoring application ?
tU32                    _u32NotifyInterval;        // permissible max time between two Watchdog-Signals

OSAL_tMSecond           _u32LastLifeSignalTime;    // stores time of the last Watchdog Signal
tBool                   _bWatchdogError;           // monitoring error
tBool                   _bNotifyProblemRequest;    // is notify_problem request active ?

std::map < tU16, tU16 > _mapServiceReg;            // map from subid to register id


tU32                    _u32AppStateCurrent;                        // state of the application
tU32                    _u32AppStateRequest;                        // current state request
tU32                    _u32AppStateNewRequest;                     // already new state in the queue
tU32                    _u32AppStateForceReq;                       // already new state in the queue and we want to force a specific app state

tU32                    _u32AppStateReqStartTime;                   // appstate request-start-time
tU32                    _u32AppStateReqEndTime;                     // appstate request-end-time
tU32                    _u32FatalErrorState;

public:
spm_tclConnectedApp( tU32  u32AppID = AMT_C_U16_APPID_INVALID,
                     tBool bLocal = TRUE );
virtual ~spm_tclConnectedApp( );

tVoid vSetPowerStateOk( ){
   _bWatchdogError        = FALSE;
   _bNotifyProblemRequest = FALSE;
   _u32LastLifeSignalTime = OSAL_ClockGetElapsedTime( );
}

tVoid vSetNotifyProblemRequest( tBool bNotifyProblemRequest = TRUE ){ _bNotifyProblemRequest = bNotifyProblemRequest; }
tBool bHasNotifiedProblemRequest( ) const { return( _bNotifyProblemRequest ); }
tBool bIsWatchdogActive( ) const { return( _bWatchdog ); }
tVoid vSetWatchdogActive( tU32 u32NotifyInterval ){
   _bWatchdog             = TRUE;
   _u32NotifyInterval     = u32NotifyInterval;
   _u32LastLifeSignalTime = OSAL_ClockGetElapsedTime( );
}

tVoid vSetWatchdogInactive( ){
   _bWatchdog             = FALSE;
   _u32NotifyInterval     = 0;
   _u32LastLifeSignalTime = 0;
}

tVoid vSetWatchdogError( tBool bWatchdogError = TRUE ){ _bWatchdogError = bWatchdogError; }
tBool bIsWatchdogError( ) const { return( _bWatchdogError ); }
tVoid vUpdateLastLifeSignalTime( ){
   _u32LastLifeSignalTime = OSAL_ClockGetElapsedTime( );
}

tU32 u32GetLastLifeSignalTime( ) const { return( _u32LastLifeSignalTime ); }
tU32 u32GetNotifyInterval( ) const { return( _u32NotifyInterval ); }

tVoid vSetIsConnected( tBool bConnected = TRUE ){ _bIsConnected = bConnected; }
tBool bIsConnected( ) const { return( _bIsConnected ); }

tVoid vSetIsLocal( tBool bLocal = TRUE ){ _bIsLocal = bLocal; }
tBool bIsLocal( ) const { return( _bIsLocal ); }

tVoid vSetAppStateCurrent( tU32 u32Current ){ _u32AppStateCurrent = u32Current; }
tU32 u32GetAppStateCurrent( ) const { return( _u32AppStateCurrent ); }

tVoid vSetAppStateRequest( tU32 u32AppStateRequest,
                           tU32 u32AppStateReqStartTime = OSAL_ClockGetElapsedTime( ) ){
   _u32AppStateRequest      = u32AppStateRequest;
   _u32AppStateReqStartTime = u32AppStateReqStartTime;
}

tVoid vUpdateAppStateReqStartTime( ){
   _u32AppStateReqStartTime = OSAL_ClockGetElapsedTime( );
}

tU32 u32GetAppStateReqStartTime( ) const { return( _u32AppStateReqStartTime ); }
tVoid vExtendAppStateReqEndTime( tU32 u32TimeExtension ){ _u32AppStateReqEndTime += u32TimeExtension; }
tVoid vSetAppStateReqEndTime( tU32 u32AppStateReqEndTime ){ _u32AppStateReqEndTime = u32AppStateReqEndTime; }
tU32  u32GetAppStateReqEndTime( ) const { return( _u32AppStateReqEndTime ); }

tVoid vSetAppStateNewRequest( tU32 u32AppStateRequest ){
   _u32AppStateNewRequest = u32AppStateRequest;
}

tU32 u32GetAppStateRequest( ) const { return( _u32AppStateRequest ); }
tU32 u32GetAppStateNewRequest( ) const { return( _u32AppStateNewRequest ); }


tU32 u32GetAppStateForceReq( ) const { return( _u32AppStateForceReq ); }
tVoid vSetAppStateForceReq( tU32 u32AppStateRequest ){ _u32AppStateForceReq = u32AppStateRequest; }

tBool bHandleAppInitialized( );

tU32  u32GetInvalidState( ) const {
   return( _u32AppStateInvalid );
}

tU32  u32GetUninitializedState( ) const {
   return( _u32AppStateUninitialized );
}

tU32  u32GetInitializedState( ) const {
   return( _u32AppStateInitialized );
}

tU32 u32GetAppId( ) const { return( _u32AppID ); }

tBool bIsRegistered( tU16 u16SubID,
                     tU16 u16RegisterID ) const {
   std::map < tU16, tU16 >::const_iterator it;

   it = _mapServiceReg.find( u16SubID );
   if ( it == _mapServiceReg.end( ) ){
      return( FALSE );
   }
   return( it->second == u16RegisterID );
}

tBool bSubIDAlwaysRegistered( tU16 u16SubID ) const {
   std::map < tU16, tU16 >::const_iterator it;

   it = _mapServiceReg.find( u16SubID );

   return( it != _mapServiceReg.end( ) );
}

tVoid vAddRegId( tU16 u16SubID,
                 tU16 u16NewRegId ){
   _mapServiceReg[u16SubID] = u16NewRegId;
}

tU16 u16FindRegId( tU16 u16SubID ){
   std::map < tU16, tU16 >::const_iterator it;

   it = _mapServiceReg.find( u16SubID );

   if ( it == _mapServiceReg.end( ) ){
      return( ( tU16 ) - 1 );
   }
   return( it->second );
}

tVoid vRemoveRegId( tU16 u16SubID ){
   std::map < tU16, tU16 >::iterator it;

   it = _mapServiceReg.find( u16SubID );
   if ( it != _mapServiceReg.end( ) ){
      _mapServiceReg.erase( it );
   }
}

tVoid vSetFatalErrorState( tU32 u32FatalErrorState ){ _u32FatalErrorState = u32FatalErrorState; }
tU32 u32GetFatalErrorState( tVoid ) const { return( _u32FatalErrorState ); }
};

