/*!
  * \file spm_CriticalVoltageManagerIntern.cpp
  *  \brief
  *    handle critical voltage related issues .
  *
  *  \note
  *  \b PROJECT: NextGen \n
   \b SW-COMPONENT: FC SPM \n
   \b COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim \n
  *  \see
  *  \version
  * Date      | Author            | Modification
  * 05.01.11  | TMS Fischer       | initial version
  ******
  */

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

// SPM  configuration
#include "spm_Config.h"
#include "spm_GlobDefs.h"

// my class header
#include "spm_CriticalVoltageManagerIntern.h"

// interfaces class definitions
#include "spm_ICcaServiceServer.h"
#include "spm_IOsalProxy.h"
#include "spm_ILocalAppManager.h"
#include "spm_ISubStateClient.h"

#include "spm_IFactory.h"

// spm helper
#include "spm_SubStatesStatistics.h"

#ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
   #define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#include "ICcaDiaglogProxy.h"
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
   #define ETG_DEFAULT_TRACE_CLASS SPM_TRACE_CLASS_SPM_SSH
#include "trcGenProj/Header/spm_CriticalVoltageManagerIntern.cpp.trc.h"
#endif
// has to come after etg include because redefinition of macros takes place
// to meet special spm requirements of blocking early spm traces
#include "spm_trace.h"

/******************************************************************************
  | local #define (scope: module-local)
  |-----------------------------------------------------------------------*/
// #define SPM_TRACE_FILE_ID   SPM_FILE_CRITICALVOLTAGEMANAGERINTERN

spm_tclCriticalVoltageManagerIntern::spm_tclCriticalVoltageManagerIntern( const ISpmFactory& factory )
   : spm_tclCriticalVoltageManager( factory )
   , _poclLocalAppManager( NULL )
   , _poclSubStateHandler( NULL )
   , _poclCcaServiceHandler( NULL )
   , _poclOsalProxy( NULL )
   , _poDiagLog( NULL ){
}

spm_tclCriticalVoltageManagerIntern::~spm_tclCriticalVoltageManagerIntern( ){
   _poclLocalAppManager   = NULL;
   _poclCcaServiceHandler = NULL;
   _poclSubStateHandler   = NULL;
   _poclOsalProxy         = NULL;
   _poDiagLog             = NULL;
}

tVoid spm_tclCriticalVoltageManagerIntern::vGetReferences( ){
   spm_tclCriticalVoltageManager::vGetReferences( );

   // get all needed references now -> all SPM objects are now available
   SPM_GET_IF_REFERENCE_USE_VAR( _poclLocalAppManager,   ISpmLocalAppManager );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclOsalProxy,         ISpmOsalProxy );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclSubStateHandler,   ISpmSubStateClient );
   SPM_GET_IF_REFERENCE_USE_VAR( _poclCcaServiceHandler, ISpmCcaServiceServer );
#ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
   SPM_GET_CLIENT_HANDLER_IF_REFERENCE_USE_VAR ( _poDiagLog, ICcaDiaglogProxy);
#endif
}

tVoid spm_tclCriticalVoltageManagerIntern::vStartCommunication( ){
/*!
  * \fn
  *  \brief
  *    Synchronize the startup. Until now the FC_SPM is fully running.
  *
  *  \param
  *  \note
  *  \version
  *    1.0   - Initial
  ******
  */
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vAddItc( VDL_ITC_SPM_UNDER_VOLTAGE );
      _poDiagLog->vAddItc( VDL_ITC_SPM_CRITICAL_UNDER_VOLTAGE );
      _poDiagLog->vAddItc( VDL_ITC_SPM_HIGH_VOLTAGE );
      _poDiagLog->vAddItc( VDL_ITC_SPM_CRITICAL_HIGH_VOLTAGE );
   #endif

   spm_corefi_tclMsgCvmEventStatus cvmstatus;

   cvmstatus.CvmEvent.enType = spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_NO_EVENT;

   ETG_TRACE_USR1( ( "spm_tclCriticalVoltageManagerIntern::vStartCommunication(): %u", ETG_ENUM( CFC_FI_SPM_CVM_EVENT, cvmstatus.CvmEvent.enType ) ) );
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_CVMEVENT, &cvmstatus );

   _u32CcaCvmEvent           = (tU32)cvmstatus.CvmEvent.enType;

   spm_tclCriticalVoltageManager::vStartCommunication( );
} // vStartCommunication

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageStart( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageStart(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_LOW_VOLTAGE, TRUE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_UNDER_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_FAILED );
   #endif

   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( pSubStateStat, spm_tclSubStateStatistics, ISpmStatistics );
      pSubStateStat->vSetNewData( SPM_U32_STATISTIC_CVM_LOW );
   }
   #ifdef SPM_FEATURE_ENABLE_OSAL_CVM_BROADCAST
      SPM_NULL_POINTER_CHECK( _poclLocalAppManager );
      _poclLocalAppManager->vSendCriticalVoltageEvent( AMT_C_U32_LOW_VOLTAGE );
   #endif
} // vHandleOnLowVoltageStart

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageEnd( ){
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
      ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageEnd(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_LOW_VOLTAGE, FALSE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_UNDER_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_PASSED );
   #endif

   #ifdef SPM_FEATURE_ENABLE_OSAL_CVM_BROADCAST
      SPM_NULL_POINTER_CHECK( _poclLocalAppManager );
      _poclLocalAppManager->vSendCriticalVoltageEvent( AMT_C_U32_NORMAL_VOLTAGE );
   #endif
} // vHandleOnLowVoltageEnd

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalLowVoltageStart( ){
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
      ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalLowVoltageStart(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_CRITICAL_LOW_VOLTAGE, TRUE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_CRITICAL_UNDER_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_FAILED );
   #endif

   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( pSubStateStat, spm_tclSubStateStatistics, ISpmStatistics );
      pSubStateStat->vSetNewData( SPM_U32_STATISTIC_CVM_CR_LOW );
   }
} // vHandleOnCriticalLowVoltageStart

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalLowVoltageEnd( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalLowVoltageEnd(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_CRITICAL_LOW_VOLTAGE, FALSE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_CRITICAL_UNDER_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_PASSED );
   #endif
} // vHandleOnCriticalLowVoltageEnd

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnHighVoltageStart( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnHighVoltageStart(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_HIGH_VOLTAGE, TRUE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_HIGH_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_FAILED );
   #endif

   #ifdef SPM_FEATURE_ENABLE_OSAL_CVM_BROADCAST
      SPM_NULL_POINTER_CHECK( _poclLocalAppManager );
      _poclLocalAppManager->vSendCriticalVoltageEvent( AMT_C_U32_HIGH_VOLTAGE );
   #endif
   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( pSubStateStat, spm_tclSubStateStatistics, ISpmStatistics );
      pSubStateStat->vSetNewData( SPM_U32_STATISTIC_CVM_HIGH );
   }
} // vHandleOnHighVoltageStart

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnHighVoltageEnd( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnHighVoltageEnd(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_HIGH_VOLTAGE, FALSE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_HIGH_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_PASSED );
   #endif

   #ifdef SPM_FEATURE_ENABLE_OSAL_CVM_BROADCAST
      SPM_NULL_POINTER_CHECK( _poclLocalAppManager );
      _poclLocalAppManager->vSendCriticalVoltageEvent( AMT_C_U32_NORMAL_VOLTAGE );
   #endif
} // vHandleOnHighVoltageEnd

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalHighVoltageStart( ){
      SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
      ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalHighVoltageStart(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_CRITICAL_HIGH_VOLTAGE, TRUE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_CRITICAL_HIGH_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_FAILED );
   #endif

   {
      SPM_GET_CLASS_REFERENCE_NEW_VAR( pSubStateStat, spm_tclSubStateStatistics, ISpmStatistics );
      pSubStateStat->vSetNewData( SPM_U32_STATISTIC_CVM_CR_HIGH );
   }
} // vHandleOnCriticalHighVoltageStart

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalHighVoltageEnd( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalHighVoltageEnd(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_CRITICAL_HIGH_VOLTAGE, FALSE );

   #ifdef VARIANT_S_FTR_ENABLE_USE_SPM_DIAGLOG
      _poDiagLog->vSetItcError( VDL_ITC_SPM_CRITICAL_HIGH_VOLTAGE, SPM_U8_CCAPROXY_DIAGLOG_PASSED );
   #endif
}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageAtStartup( ){
   ETG_TRACE_USR4( ( "spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageAtStartup(): VoltageState %u", ETG_ENUM( SPM_CVM_EVENT, (tU32)_u32VoltageState ) ) );
}

tVoid spm_tclCriticalVoltageManagerIntern::vUpdateCvmEvent( tU32 cvmEvent ){
   spm_corefi_tclMsgCvmEventStatus cvmstatus;

   ETG_TRACE_USR1( ( "spm_tclCriticalVoltageManagerIntern::vUpdateCvmEvent(): %u", ETG_ENUM( CFC_FI_SPM_CVM_EVENT, cvmEvent ) ) );

   cvmstatus.CvmEvent.enType = (spm_fi_tcl_SPM_e32_CVM_EVENT::tenType)cvmEvent;
   SPM_NULL_POINTER_CHECK( _poclCcaServiceHandler );
   _poclCcaServiceHandler->vUpdateProperty( SPM_COREFI_C_U16_CVMEVENT, &cvmstatus );
}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageEndTimeout( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   // stop emergency shutdown
   _poclSubStateHandler->vSetSubStateType( SPM_U32_ENDURING_LOW_VOLTAGE, FALSE );
}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalLowVoltageEndTimeout( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   // stop emergency shutdown
   _poclSubStateHandler->vSetSubStateType( SPM_U32_ENDURING_CR_VOLTAGE, FALSE );

}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnLowVoltageTimeout( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_ENDURING_LOW_VOLTAGE, TRUE );
}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnHighVoltageTimeout( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_ENDURING_HIGH_VOLTAGE, TRUE );
}

tVoid spm_tclCriticalVoltageManagerIntern::vHandleOnCriticalVoltageTimeout( ){
   SPM_NULL_POINTER_CHECK( _poclSubStateHandler );

   _poclSubStateHandler->vSetSubStateType( SPM_U32_ENDURING_CR_VOLTAGE, TRUE );
}

